/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Screening/ExecutionBase.cs#3 $
 * $DateTime: 2008/05/14 13:05:12 $
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.Diagnostics;
using System.Drawing;

using Poderosa;

using Bellagio.Environment;
using Bellagio.Forms;
using Bellagio.Data;
using Bellagio.Values;

//sɍ\z鋤ʕ
namespace Bellagio.Screening {
    //ꎟtB^
    public enum ScreeningFirstFilter {
        NoFilter,
        SingleStock
    }

    //XL[}parameterDescriptionɏp[X́@XN[jOA؋p
    public class ScreeningParameterUI : StringWithParameterParser.ISite {
        private string _rawDescription;
        private List<ParamTag> _data;

        public class ParamTag {
            public Label label;
            public NumericUpDown valueControl;
            public string fullID; //p[^̃tID
        }

        public ScreeningParameterUI(ExtensionKitItemWithParameter owner, string rawDescription, EventHandler value_change_handler) {
            _rawDescription = rawDescription;
            _data = new List<ParamTag>();
            if(rawDescription.Length > 0) {
                int i = 0;
                foreach(string description_elem in rawDescription.Split(',')) {
                    StringWithParameterParser parser = new StringWithParameterParser(this);
                    ParamTag pt = new ParamTag();
                    ExtensionParameter ep = owner.parameter[i];
                    pt.fullID = owner.FullID + "." + ep.PreferenceID;
                    pt.label = FormUtil.CreateLabel(parser.Parse(description_elem));
                    pt.label.Size = new Size(4 + pt.label.Text.Length*9, 19); //{̓tHgvZׂ
                    pt.valueControl = new NumericUpDown();
                    pt.valueControl.Size = new Size(48, 19);
                    pt.valueControl.Minimum = 1;
                    pt.valueControl.Maximum = 999;
                    pt.valueControl.Value = ParseUtil.ParseInt(ep.Value, 1);
                    pt.valueControl.ValueChanged += value_change_handler;

                    i++;
                    _data.Add(pt);
                }
            }
        }
        public IEnumerable<ParamTag> Tags {
            get {
                return _data;
            }
        }
        public void DisposeControls() {
            foreach(ParamTag tag in _data) {
                tag.label.Dispose();
                tag.valueControl.Dispose();
            }
        }
        public void EnableControls(bool value) {
            foreach(ParamTag tag in _data) {
                tag.valueControl.Enabled = value;
            }
        }

        string StringWithParameterParser.ISite.FindParameter(string name) {
            if(name=="unit") return ""; //̂
            else return "";
        }

    }

 
    //fĉNX
    public abstract class ScreeningExecutorBase {
        public interface IProgressNotifier {
            void InitProgress(string comment, int limit_count);
            void OnProgress(int count);
            void OnComplete();
            void OnFail(string message);
            void OnAbort();
        }

        protected ScreeningFirstFilter _firstFilter;
        protected AbstractStockProfile _singleTargetStock; //_firstFilter==SingleStockł̂ݎg

        protected IProgressNotifier _progressNotifier;
        protected Thread _thread;

        protected List<AbstractStockProfile> _filteredStocks;
        protected List<AbstractStockProfile> _insufficientDataStocks;
        protected List<AbstractStockProfile> _errorStocks;

        protected ScreeningExecutorBase() {
        }

        public IProgressNotifier ProgressNotifier {
            get {
                return _progressNotifier;
            }
            set {
                _progressNotifier = value;
            }
        }

        //񓯊s
        public void AsyncExecute() {
            _thread = new Thread(new ThreadStart(Run));
            _thread.Start();
        }

        protected IEnumerable<AbstractStockProfile> GetStockEnumerable() {
            if(_firstFilter==ScreeningFirstFilter.NoFilter) {
                return BellagioRoot.GlobalStockCollection;
            }
            else {
                Debug.Assert(_singleTargetStock!=null);
                List<AbstractStockProfile> p = new List<AbstractStockProfile>();
                p.Add(_singleTargetStock);
                return p;
            }
        }
        protected int GetTargetStockCount() {
            if(_firstFilter==ScreeningFirstFilter.NoFilter)
                return BellagioRoot.GlobalStockCollection.Count;
            else
                return 1;
        }

        private void Execute() {

            _filteredStocks = new List<AbstractStockProfile>(); //tB^ŏOꂽ
            _insufficientDataStocks = new List<AbstractStockProfile>(); //f[^s\ŏO
            _errorStocks = new List<AbstractStockProfile>(); //G[̔

            ExecuteMain();
        }
        
        protected abstract void ExecuteMain(); //PɂĎs

        public void Run() {
            try {
                Execute();
                if(_progressNotifier!=null)
                    _progressNotifier.OnComplete();
            }
            catch(ThreadAbortException ) {
                if(_progressNotifier!=null)
                    _progressNotifier.OnAbort();
            }
            catch(Exception ex) {
                RuntimeUtil.SilentReportException(ex);
                if(_progressNotifier!=null)
                    _progressNotifier.OnFail(ex.Message);
            }
        }

        public void Abort() {
            if(_thread!=null)
                _thread.Abort();
        }

        protected bool ExcludesByStaticFilter(BasicStockProfile prof, ScreeningFilterType filter) {
            switch(filter) {
                case ScreeningFilterType.Nikkei225:
                    return (prof.Flags & StockProfileFlags.Nikkei225)==StockProfileFlags.None;
                case ScreeningFilterType.Div1:
                case ScreeningFilterType.MO_HE_JQ: {
                        StockExchange e = prof.Primary.Market;
                        StockExchangeSubType s = prof.Primary.MarketSubType;
                        if(filter==ScreeningFilterType.Div1)
                            return !((e==StockExchange.T || e==StockExchange.O) && s==StockExchangeSubType.Ichibu);
                        else
                            return !(s==StockExchangeSubType.Mothers || e==StockExchange.Hercules ||  e==StockExchange.J);
                    }
                default:
                    return false;
            }

        }

        //XN[jOE؂ł̂ݗLȊ֐o^
        private static bool _functionRegistered;
        public static void RegisterFunctions() {
            if(!_functionRegistered) {
                FunctionLibrary lib = BellagioRoot.Functions.BuiltIn;
                lib.DefineInternalFunction("refbrand", QuoteType.instance, QuoteType.instance, new BT[] { BT.String }, new BInternalExecution(RefBrand));
                lib.DefineInternalFunction("refbrand", null, QuoteType.instance, new BT[] { BT.String }, new BInternalExecution(GlobalRefBrand));
                _functionRegistered = true;
            }
        }
        private static ExecResult RefBrand(BV target, EvalContext context, BV[] args, BV result) {
            Quote q = target as Quote;
            if(!q.IsNil && q.Count>0) {
                //tQƂQuoteLbV̒擾ĕԂ
                string code = ((BString)args[0]).Value;
                AbstractStockProfile prof = BellagioRoot.GlobalStockCollection.FindExact(code);
                if(prof!=null) {
                    ConcreteQuote cq = ScreeningPlugin.Instance.DailyDataCache.Request(prof);
                    if(cq!=null) {
                        int index = cq.FindExactly(q.LastCandle.Time);
                        if(index!=-1) {
                            ((SubseqQuote)result).Set(cq, 0, index+1);
                            return ExecResult.OK;
                        }
                    }
                }
            }
            return ExecResult.Nil;
        }
        private static ExecResult GlobalRefBrand(BV target, EvalContext context, BV[] args, BV result) {
            //ŐVt܂ł̓Ԃ
            string code = ((BString)args[0]).Value;
            AbstractStockProfile prof = BellagioRoot.GlobalStockCollection.FindExact(code);
            if(prof!=null) {
                ConcreteQuote cq = ScreeningPlugin.Instance.DailyDataCache.Request(prof);
                if(cq!=null) {
                    ((SubseqQuote)result).Set(cq, 0, cq.Count);
                    return ExecResult.OK;
                }
            }
            return ExecResult.Nil;
        }
    }

    public enum ExecuteBrandResult {
        Succeeded,
        Ignored,
        InsufficientData,
        Filtered,
        Stop
    }

    public abstract class ScreeningResultBase<T> : IEnumerable<T> where T:ScreeningResultItemBase  {
        protected List<T> _data;
        protected bool _sorted;
        protected List<AbstractStockProfile> _dataErrorStocks;
        protected List<AbstractStockProfile> _filteredStocks;
        protected List<AbstractStockProfile> _insufficientsStocks;

        public ScreeningResultBase() {
            _data = new List<T>();
            _sorted = false;
        }
        public void SetExcludedStocks(List<AbstractStockProfile> insufficients, List<AbstractStockProfile> filtered, List<AbstractStockProfile> error) {
            _dataErrorStocks = error;
            _filteredStocks = filtered;
            _insufficientsStocks = insufficients;
        }
        public virtual void AddResult(T item) {
            _data.Add(item);
            _sorted = false;
        }
        public int Count {
            get {
                return _data.Count;
            }
        }
        public T this[int index] {
            get {
                return _data[index];
            }
        }

        public void Sort() {
            if(_sorted) return;

            Comparison<T> c = GetSortComparison();
            if(c!=null)
                _data.Sort(c);
            _sorted = true;
        }
        public void SortExternal(Comparison<T> c) {
            _data.Sort(c);
        }
        protected abstract Comparison<T> GetSortComparison();

        public IEnumerator<T> GetEnumerator() {
            Debug.Assert(_sorted);
            return _data.GetEnumerator();
        }
        IEnumerator IEnumerable.GetEnumerator() {
            return GetEnumerator();
        }

        //G[܂
        public List<AbstractStockProfile> ErrorStocks {
            get {
                return _dataErrorStocks;
            }
        }
        public List<AbstractStockProfile> FilteredStocks {
            get {
                return _filteredStocks;
            }
        }
        public List<AbstractStockProfile> InsufficientDataStocks {
            get {
                return _insufficientsStocks;
            }
        }
        public int TotalCount {
            get {
                return _data.Count + _dataErrorStocks.Count + _insufficientsStocks.Count + _filteredStocks.Count;
            }
        }

    }

    public abstract class ScreeningResultItemBase {
        protected AbstractStockProfile _stock;

        public ScreeningResultItemBase(AbstractStockProfile stock) {
            _stock = stock;
        }

        public AbstractStockProfile StockProfile {
            get {
                return _stock;
            }
        }

    }
}
