/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id$
 * $DateTime$
 * 
 * SE̓IɎ@\BŃXN[jOB
 * - \z͒ʏ̓sB
 * - ̍\͖->ConcreteQuotẽnbVe[u
 * - Xg[ [Œ}[N   ^ [R[h [R[h]*<>] * <>]
 * - ꂩԂȂȂǁAȂĂf[^͌Œ蒷Ŏcɂ̓[Ŗ߂
 * - R[h́A㉺QoCgɕĎwɂΉ
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.IO;

using Poderosa;
using Bellagio.Values;
using Bellagio.Data;
using Travis.IO;


#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Screening {
    //OŒ񋟂AԌ̓̍擾VXe{LbV
    public interface IScreeningDailyDataProvider {
        void LoadLatestDailyData(AbstractStockProfile prof, int max_length, IDailyDataWriteStream strm);
        SplitReflection DefaultSplitReflection { get; }
        DateTime GuessLatestTradeDate { get; }
    }

    public class RecentDailyDataCache {
        private const uint HEAD_MARK = 0xCAC0E001; //擪̌Œ 'CACHE 001'̈

        private class DataTag {
            public ConcreteQuote quote;
            public bool fully_read; //f[^͂邾ǂ܂ꂽƂ

            public DataTag(ConcreteQuote q) {
                quote = q;
            }
        }

        private int _dayCountPerStock; //P̓̒
        private Dictionary<AbstractStockProfile, DataTag> _data; //Cf[^
        private IScreeningDailyDataProvider _provider;
        private bool _prohibitDataProvider;

        public RecentDailyDataCache() {
            _data = new Dictionary<AbstractStockProfile, DataTag>();
            Clear();
        }
        public void ProhibitDataProvider() {
            _prohibitDataProvider = true;
        }

        public int Count {
            get {
                return _data.Count;
            }
        }
        public int DailyDataLength {
            get {
                return _dayCountPerStock;
            }
        }

        public ConcreteQuote Request(AbstractStockProfile prof) {
            DataTag dt;
            if(_data.TryGetValue(prof, out dt))
                return dt.quote; //LbVɂ
            else if(!_prohibitDataProvider) {
                AssureProvider();
                SimpleDailyDataProvider d = new SimpleDailyDataProvider();
                _provider.LoadLatestDailyData(prof, _dayCountPerStock, d);
                ConcreteQuote result = d.Result;
                if(_provider.DefaultSplitReflection==SplitReflection.NotRefrected && prof.SplitInfo!=null && prof.SplitInfo.Length>0)
                    result = QuoteConverter.ReflectSplit(result, prof.SplitInfo);
                dt = new DataTag(result);
                dt.fully_read = result.Count < _dayCountPerStock;
                _data.Add(prof, dt);
                return result;
            }
            else
                return null;
        }

        //ƒ̂ł̂ł΂trueԂAłȂfalseԂB̂Ƃquote͔j󂳂邱Ƃ
        public bool ExtendDataIfPossible(AbstractStockProfile prof, ref ConcreteQuote quote) {
            if(_prohibitDataProvider) return false;

            int max = ScreeningPlugin.Instance.Preferences.MaximumDailyDataLength;
            if(_dayCountPerStock>=max)
                return false; //K̏ɒBĂ
            else {
                DataTag dt = _data[prof];
                if(dt.fully_read) return false;

                _dayCountPerStock += max/10; //擾𑝂₷
                _data.Remove(prof); //f[^폜čĎ擾
                quote = this.Request(prof);
                return true;
            }
        }

        public void AddDirect(AbstractStockProfile prof, ConcreteQuote q) {
            DataTag dt = new DataTag(q);
            dt.fully_read = true;
            _data.Add(prof, dt);
        }

        public void Clear() {
            _dayCountPerStock = 300; //l
            _data.Clear();
        }

        private void AssureProvider() {
            _provider = ScreeningPlugin.Instance.ScreeningDailyDataProvider;
        }

#if false
        //SS\z
        public void RecreateCacheFile() {
            _data.Clear();
            BellagioRoot.DateBasedQuoteProvider.CheckLoader();
            int count = 0;
            foreach(AbstractStockProfile prof in BellagioRoot.GlobalStockCollection) {
                BasicStockProfile bp = prof as BasicStockProfile;
                if(bp!=null) {
                    if((++count % 100)==0) Debug.WriteLine("process " + count);

                    SimpleDailyDataProvider dp = new SimpleDailyDataProvider();
                    BellagioRoot.DateBasedQuoteProvider.GetDataLoader().Load(bp.Primary, dp, DailyDataLoadOption.None);
                    ConcreteQuote q = dp.Result;
                    if(q!=null) {
                        q = q.Count<=_dayCountPerStock? q : CreateSubConcreteQuote(q, q.Count-_dayCountPerStock, _dayCountPerStock);
                        _data.Add(prof, q);
                    }
                }
            }

            _dirty = true;
            FileStream fs = new FileStream(GetCacheFileName(), FileMode.Create);
            this.WriteTo(fs);
            fs.Close();
            Debug.WriteLine("FINISH!");
        }

        //[hȂ烍[h
        public void AssureActive() {
            if(_status==InternalStatus.Initial) {
                try {
                    Debug.WriteLine("CacheLoad Start");
                    FileStream fs = new FileStream(GetCacheFileName(), FileMode.Open);
                    ReadFrom(fs, 0);
                    fs.Close();
                    Debug.WriteLine("CacheLoad End");
                }
                catch(Exception ex) {
                    Debug.WriteLine(ex.Message);
                    _status = InternalStatus.Unavailable;
                }
            }
        }
        
        //ŏȊǉ@f[^̏ꍇAŐV̑łKv
        public void UpdateLatestCandle(AbstractStockProfile prof, Candle c) {
            ConcreteQuote q;
            if(!_data.TryGetValue(prof, out q)) {
                q = new ConcreteQuote(Quote.QuoteUnit.Daily);
                _data.Add(prof, q);
            }

            Candle last = q.LastCandle;
            Debug.Assert(last==null || last.Time<c.Time);
            q.Add(c);
        }

        public void ReadFrom(Stream strm, int capacity) {
            _data.Clear();
            BinaryReader re = new BinaryReader(strm);
            if(re.ReadUInt32()!=HEAD_MARK) throw new FormatException("Stream format error(head mark)");
            int quote_len = re.ReadInt32(); //_dailyDataLengthƂ͈قȂ邩ȂƂɒ
            int stock_count = re.ReadInt32();
            RecordTypeE recordtype = (RecordTypeE)re.ReadInt32();
            int CANDLE_LEN = GetDailyDataLength(recordtype);

            for(int i=0; i<stock_count; i++) {
                AbstractStockProfile prof = FindStockProfile(re.ReadInt32());
                //ꂪnull̃P[X邱ƂɒӁBǂݔ΂͖ʓ|

                ConcreteQuote q = new ConcreteQuote(Quote.QuoteUnit.Daily);
                for(int j=0; j<quote_len; j++) {
                    int time = re.ReadInt32();
                    if(prof!=null && time!=0) {
                        int open = re.ReadInt32();
                        int high = re.ReadInt32();
                        int low = re.ReadInt32();
                        int close = re.ReadInt32();
                        int vol = re.ReadInt32();
                        re.ReadInt32(); //Mpc
                        re.ReadInt32();
                        q.Add(new Candle(time, open, high, low, close, vol, CandleFlags.None));
                    }
                    else {
                        for(int k=0; k<CANDLE_LEN-4; k+=4)
                            re.ReadInt32(); //ǂݔ΂
                    }
                }

                if(prof!=null)
                    _data.Add(prof, q);
            }

            re.Close();
            _dirty = false;
            _status = InternalStatus.Consumed;
        }

        public void WriteTo(Stream strm) {

            BinaryWriter wr = new BinaryWriter(strm);
            wr.Write(HEAD_MARK);
            wr.Write(_dayCountPerStock);
            wr.Write(_data.Count);
            wr.Write((int)RecordTypeE.OHLCV_Credit);
            int CANDLE_LEN = GetDailyDataLength(RecordTypeE.OHLCV_Credit);
            //oCiT[`ɂ͂ȂDictionary񋓂Ă܂
            foreach(KeyValuePair<AbstractStockProfile, ConcreteQuote> en in _data) {
                int code;
                if(!FormatCodeAsInt(en.Key, out code)) continue; //LbV쐬Ȃ̂̓XLbv
                wr.Write(code);
                ConcreteQuote q = en.Value;
                //Xg_dailyDataLengthR[hL^
                int index = Math.Max(0, q.Count-_dayCountPerStock);
                while(index < q.Count)
                    WriteCandle(wr, q.CandleAt(index++));
                //sXLbv
                if(q.Count < _dayCountPerStock)
                    wr.Seek((_dayCountPerStock - q.Count) * CANDLE_LEN, SeekOrigin.Current);
            }

            //ŌɃV[NĂƃN[YƂɐ؂̂Ă̂ŃPcɂ}[N
            wr.Write(HEAD_MARK);

            wr.Close();

            _dirty = false;
        }


        private void WriteCandle(BinaryWriter wr, Candle c) {
            wr.Write(c.Time);
            wr.Write(c.Open);
            wr.Write(c.High);
            wr.Write(c.Low);
            wr.Write(c.Close);
            wr.Write(c.Volume);
            wr.Write((int)0);
            wr.Write((int)0);
        }

        private AbstractStockProfile FindStockProfile(int rawcode) {
            string name;
            if(rawcode >= 10000) {
                MarketIndex.Builtins t = (MarketIndex.Builtins)(rawcode - 10000);
                name = t.ToString();
            }
            else
                name = rawcode.ToString();

            //UnitTestłGlobalStockCollectionɂȂȂ
#if UNITTEST
            if(RecentDailyDataCacheTests.running)
                return RecentDailyDataCacheTests.FindStockProfile(name);
#endif
            //ʏ펞
            return BellagioRoot.GlobalStockCollection.FindExact(name);
        }
        //LbVɏR[hԂBȂꍇfalse
        private bool FormatCodeAsInt(AbstractStockProfile prof, out int code) {
            if(prof is BasicStockProfile) {
                return Int32.TryParse(prof.Code, out code);
            }
            else if(prof is MarketIndex) {
                code = 10000 + (int)(((MarketIndex)prof).EnumCode);
                return true;
            }
            else {
                code = 0;
                return false;
            }
        }

        private static int GetDailyDataLength(RecordTypeE t) {
            Debug.Assert(t==RecordTypeE.OHLCV_Credit);
            return 4 * 8;  //tEOHLC4EoEMpc*2
        }

        private static ConcreteQuote CreateSubConcreteQuote(ConcreteQuote source, int start, int count) {
            ConcreteQuote q = new ConcreteQuote(source.Unit);
            for(int i=0; i<count; i++)
                q.Add(source.CandleAt(start + i));
            return q;
        }

        private string GetCacheFileName() {
            return BellagioRoot.PathInfo.DataHomeDir + "dailydatacache";
        }

#endif
        //f[^̃[h{̂̑OɕKvȏs
        private class SimpleDailyDataProvider : IDailyDataWriteStream {
            private ConcreteQuote _daily;
            private bool _completed;

            public SimpleDailyDataProvider() {
                _completed = false;
            }
            public ConcreteQuote Result {
                get {
                    return _daily;
                }
            }
            public bool IsCompleted {
                get {
                    return _completed;
                }
            }

            public void InitializeStream(int record_count_capacity, bool reverse, bool volumeAvailable) { //ʏ폇(reverse==false)ł͓t͌Â̂ZbgBt̂Ƃ͐V̂B
                _daily = new ConcreteQuote(Quote.QuoteUnit.Daily, 1);
                _daily.SetCapacity(record_count_capacity);
                Debug.Assert(!reverse); //ЂԂ͖̂T|[g
                Debug.Assert(volumeAvailable);
            }
    

            public void AddRecord(int date, int open, int high, int low, int close, int volume, int creditlong, int creditshort) {
                _daily.Add(date, open, high, low, close, volume, creditlong, creditshort);
            }

            //PԂ̓f[^̍쐬I
            public void Complete() {
                _completed = true;
            }
            

            public void FailLoading(string message) {
                throw new IOException(message);
            }


            public IAdaptable GetAdapter(Type adapter) {
                return BUtil.DefaultGetAdapter(this, adapter);
            }
        }

    }
#if false //UNITTEST
    [TestFixture]
    public class RecentDailyDataCacheTests {
        public static bool running; //̃eXgs^CɒmKv
        private static Dictionary<string, AbstractStockProfile> _testStocks;

        [TestFixtureSetUp]
        public void SetUp() {
            running = true;
            _testStocks = new Dictionary<string, AbstractStockProfile>();
        }
        [TestFixtureTearDown]
        public void TearDown() {
            running = false;
        }

        public static AbstractStockProfile FindStockProfile(string code) {
            AbstractStockProfile p;
            if(_testStocks.TryGetValue(code, out p))
                return p;
            else
                return null;
        }
        private static BasicStockProfile CreateBasicStockProfile(string code) {
            BasicStockProfile p = new BasicStockProfile(code, code, StockProfileFlags.None, 1);
            _testStocks.Add(p.Code, p);
            return p;
        }
        private ConcreteQuote CreateConcreteQuote(int length, int date, int price) {
            ConcreteQuote q = new ConcreteQuote(Quote.QuoteUnit.Daily);
            for(int i=0; i<length; i++) {
                Candle c = new Candle(date+i, price+i-1, price+i+1, price+i-2, price+i, 1, CandleFlags.None);
                q.Add(c);
            }
            return q;
        }

        private void AssertQuotesAreEqual(ConcreteQuote q1, ConcreteQuote q2) {
            Assert.AreEqual(q1.Count, q2.Count);
            for(int i=0; i<q1.Count; i++) {
                Candle c1 = q1.CandleAt(i);
                Candle c2 = q2.CandleAt(i);
                Assert.AreEqual(c1.Time, c2.Time);
                Assert.AreEqual(c1.Open, c2.Open);
                Assert.AreEqual(c1.High, c2.High);
                Assert.AreEqual(c1.Low, c2.Low);
                Assert.AreEqual(c1.Close, c2.Close);
                Assert.AreEqual(c1.Volume, c2.Volume);
            }
        }

        [Test]
        public void NormalReadWrite() {
            //Kɍ\z
            _testStocks.Clear();
            BasicStockProfile p1 = CreateBasicStockProfile("1");
            BasicStockProfile p2 = CreateBasicStockProfile("2");
            ConcreteQuote q01 = CreateConcreteQuote(10, 20080101, 100);
            ConcreteQuote q02 = CreateConcreteQuote(20, 20080101, 500);

            RecentDailyDataCache dc = new RecentDailyDataCache();
            dc.AddDirect(p1, q01);
            dc.AddDirect(p2, q02);
            MemoryStream buf = new MemoryStream();
            dc.WriteTo(buf);
            buf.Close();

            byte[] rawdata = buf.ToArray();
            buf = new MemoryStream(rawdata);
            dc = new RecentDailyDataCache();
            dc.ReadFrom(buf, 0x10000); //ǂ񂾃f[^č\z

            //gƂmF
            ConcreteQuote q11 = dc.Request(p1);
            ConcreteQuote q12 = dc.Request(p2);
            Assert.IsNotNull(q11);
            Assert.IsNotNull(q12);
            Assert.AreEqual(2, dc.Count);

            AssertQuotesAreEqual(q01, q11);
            AssertQuotesAreEqual(q02, q12);
        }

        [Test]
        public void TailShouldBeCut() {
            //Kɍ\z
            _testStocks.Clear();
            BasicStockProfile p1 = CreateBasicStockProfile("1");
            ConcreteQuote qorig = CreateConcreteQuote(120, 20080101, 100);
            
            RecentDailyDataCache dc = new RecentDailyDataCache();
            dc.AddDirect(p1, qorig);
            MemoryStream buf = new MemoryStream();
            dc.WriteTo(buf);
            buf.Close();

            byte[] rawdata = buf.ToArray();
            buf = new MemoryStream(rawdata);
            dc = new RecentDailyDataCache();
            dc.ReadFrom(buf, 0x10000); //ǂ񂾃f[^č\z

            ConcreteQuote qread = dc.Request(p1);
            Assert.IsNotNull(qread);
            Assert.AreEqual(1, dc.Count);

            //q01(IWi)̐KLbVɓƂmF
            ConcreteQuote qresult = new ConcreteQuote(Quote.QuoteUnit.Daily);
            for(int i = qorig.Count - dc.DailyDataLength; i < qorig.Count; i++)
                qresult.Add(qorig.CandleAt(i));
            AssertQuotesAreEqual(qresult, qread);
        }

        [Test]
        public void Ignore() {
            //Kɍ\z
            _testStocks.Clear();
            BasicStockProfile p1 = CreateBasicStockProfile("1");
            BasicStockProfile p2 = CreateBasicStockProfile("2");
            ConcreteQuote q01 = CreateConcreteQuote(10, 20080101, 100);
            ConcreteQuote q02 = CreateConcreteQuote(20, 20080101, 500);

            RecentDailyDataCache dc = new RecentDailyDataCache();
            dc.AddDirect(p1, q01);
            dc.AddDirect(p2, q02);
            MemoryStream buf = new MemoryStream();
            dc.WriteTo(buf);
            buf.Close();

            byte[] rawdata = buf.ToArray();
            buf = new MemoryStream(rawdata);
            //݂Ȃf[^ɂĂmF
            _testStocks.Remove("2");
            dc = new RecentDailyDataCache();
            dc.ReadFrom(buf, 0x10000); //ǂ񂾃f[^č\z

            //gƂmF@p2ɑΉ̂͂Ȃ͂Ȃ̂Ŗ̂
            ConcreteQuote q11 = dc.Request(p1);
            ConcreteQuote q12 = dc.Request(p2);
            Assert.IsNotNull(q11);
            Assert.IsNull(q12);
            Assert.AreEqual(1, dc.Count);

            AssertQuotesAreEqual(q01, q11);
        }

        //̃eXǵAQuoteĂK݂̂LbV錏ALbVɂۂɂ͌ȂƂ̏ADailyDataXVif[^TCY邩ǂłQƂjA
    }
#endif
}
