/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Travis/SimpleMemoryStream.cs#4 $
 * $DateTime: 2007/12/28 19:23:38 $
 * 
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.IO;

namespace Travis.IO {
    public class DataFragment {
        private byte[] _data;
        private int _offset;
        private int _length;

        public DataFragment(byte[] data, int offset, int length) {
            Init(data, offset, length);
        }
        public DataFragment(byte[] data) {
            Init(data, 0, data.Length);
        }
        public DataFragment(int capacity) {
            _data = new byte[capacity];
            _offset = 0;
            _length = 0;
        }

        public int Length {
            get {
                return _length;
            }
        }
        public int Capacity {
            get {
                return _data.Length;
            }
        }
        public int Offset {
            get {
                return _offset;
            }
        }
        public byte[] Data {
            get {
                return _data;
            }
        }

        public byte ByteAt(int offset) {
            return _data[_offset + offset];
        }

        public void Append(byte[] data, int offset, int length) {
            int newcapacity = _offset + _length + length;

            AssureCapacity(RoundUp(newcapacity));
            Array.Copy(data, offset, _data, _offset+_length, length);
            _length += length;
        }
        public void Append(DataFragment data) {
            if(_length==0) {
                AssureCapacity(RoundUp(data.Length));
                Array.Copy(data.Data, data.Offset, _data, 0, data.Length);
                _offset = 0;
                _length = data.Length;
            }
            else {
                Append(data.Data, data.Offset, data.Length);
            }
        }

        //reuse this instance
        public void Init(byte[] data, int offset, int length) {
            _data = data;
            _offset = offset;
            _length = length;
        }
        public void Init(DataFragment data) {
            Init(data.Data, data.Offset, data.Length);
        }

        //clear
        public void Clear() {
            _offset = 0;
            _length = 0;
        }

        public DataFragment Isolate() {
            int newcapacity = RoundUp(_length);
            byte[] t = new byte[newcapacity];
            Array.Copy(_data, _offset, t, 0, _length);
            DataFragment f = new DataFragment(t, 0, _length);
            return f;
        }

        //be careful!
        public void Consume(int length) {
            _offset += length;
            _length -= length;
            Debug.Assert(_length >= 0);
        }
        //be careful!
        public void SetLength(int offset, int length) {
            _offset = offset;
            _length = length;
            Debug.Assert(_offset+_length <= this.Capacity);
        }

        public void AssureCapacity(int size) {
            size = RoundUp(size);
            if(_data.Length < size) {
                byte[] t = new byte[size];
                Array.Copy(_data, 0, t, 0, _data.Length);
                _data = t;
            }
        }

        public byte[] ToNewArray() {
            byte[] t = new byte[_length];
            Array.Copy(_data, _offset, t, 0, _length);
            return t;
        }

        private static int RoundUp(int size) {
            uint t = 0xFFFFFFF0;
            while((size & t)!=0)
                t <<= 1;

            return (int)((~t) + 1);
        }

        //Stream\zB͎Oɂ͂킩Ȃ̂ŃqgĊg
        public static DataFragment CreateFromStream(Stream src, int length_hint) {
            SimpleMemoryStream ms = new SimpleMemoryStream(length_hint);
            int len = 0;
            int max, read;
            do {
                max = ms.UnderlyingBuffer.Length-len;
                read = src.Read(ms.UnderlyingBuffer, len, max);
                len += read;
                if(max==read) ms.AssureCapacity(ms.UnderlyingBuffer.Length*2);
            } while(max==read); //Sǂ߂Ă邤͌p

            Debug.WriteLine(String.Format("Deflate hint={0} actual={1}", length_hint, len));
            return new DataFragment(ms.UnderlyingBuffer, 0, len);
        }

    }


    public class SimpleMemoryStream {
        private byte[] _buffer;
        private int _offset;

        public SimpleMemoryStream(int capacity) {
            Init(capacity);
        }
        public SimpleMemoryStream() {
            Init(512);
        }
        private void Init(int capacity) {
            _buffer = new byte[capacity];
            Reset();
        }

        public int Length {
            get {
                return _offset;
            }
        }
        public byte[] UnderlyingBuffer {
            get {
                return _buffer;
            }
        }
        public void Reset() {
            _offset = 0;
        }
        public void SetOffset(int value) {
            _offset = value;
        }
        public byte[] ToNewArray() {
            byte[] r = new byte[_offset];
            Array.Copy(_buffer, 0, r, 0, _offset);
            return r;
        }

        public void AssureCapacity(int size) {
            if(_buffer.Length < size) {
                byte[] t = new byte[Math.Max(size, _buffer.Length*2)];
                Array.Copy(_buffer, 0, t, 0, _buffer.Length);
                _buffer = t;
            }
        }

        public void Write(byte[] data, int offset, int length) {
            AssureCapacity(_offset + length);
            Array.Copy(data, offset, _buffer, _offset, length);
            _offset += length;
        }
        public void Write(byte[] data) {
            Write(data, 0, data.Length);
        }
        public void WriteByte(byte b) {
            AssureCapacity(_offset+1);
            _buffer[_offset++] = b;
        }
        
    }
}
