/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/BType.cs#8 $
 * $DateTime: 2007/12/28 19:35:14 $
 * 
 * ^IuWFNgQ
 */

using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

#if UNITTEST
using NUnit.Framework;
#endif

using Bellagio.Evaluators;
using Travis;

namespace Bellagio.Values {
    //BV̌^
    //TESTINFO ̃t@C
    public abstract class BT {
        private string _name;
        protected BArrayT _arrayType; //x쐬z^
        protected BV _nilInstance;

        public delegate BV CreateInstanceDelegate();
        public delegate int CompareDelegate(BV left, BV right);

        public BT(string name) {
            _name = name;
            //̂nil instanceǂŃZbgKv
        }

        public string Name {
            get {
                return _name;
            }
        }
        //Name`
        public static string ListNames(BT[] ts) {
            StringBuilder bld = new StringBuilder();
            for(int i=0; i<ts.Length; i++) {
                StringUtil.AppendSeparatedList(bld, ", ", ts[i].Name);
            }
            return bld.ToString();
        }

        public abstract BV CreateInstance(); //^̂ݍ킹

        //LXg
        public virtual BArrayT AsArray() {
            return null;
        }
        public virtual BArrayT ArrayType() {
            if(_arrayType==null)
                _arrayType = new BArrayT(this);
            return _arrayType;
        }

        public virtual BV Nil {
            get {
                Debug.Assert(_nilInstance!=null);
                Debug.Assert(_nilInstance.IsNil);
                return _nilInstance;
            }
        }

        //zȂǂł͓Koverride
        public virtual bool IsConvertibleTo(BT type) {
            return this==type;
        }

        public bool IsPrimitive {
            get {
                return this is BPrimitiveType;
            }
        }

        //primitives
        private static BPrimitiveType _int;
        private static BPrimitiveType _double;
        private static BPrimitiveType _bool;
        private static BPrimitiveType _time;
        private static BPrimitiveType _date;
        private static BPrimitiveType _string;
        private static BPrimitiveType[] _primitives;

        static BT() {
            _int = new BPrimitiveType("int", delegate { return new BInt(0); }, delegate(BV left, BV right) { return ((BInt)left).Value - ((BInt)right).Value; });
            _double = new BPrimitiveType("double", delegate { return new BDouble(0); }, delegate(BV left, BV right) { return ((BDouble)left).Value.CompareTo(((BDouble)right).Value); });
            _bool = new BPrimitiveType("bool", delegate { return new BBoolean(false); }, delegate(BV left, BV right) { return ((BBoolean)left).Value.CompareTo(((BBoolean)right).Value); });
            _time = new BPrimitiveType("time", delegate { return new BTime(0); }, delegate(BV left, BV right) { return ((BTime)left).AsInt() - ((BTime)right).AsInt(); });
            _date = new BPrimitiveType("date", delegate { return new BDate(0); }, delegate(BV left, BV right) { return ((BDate)left).AsInt() - ((BDate)right).AsInt(); });
            _string = new BPrimitiveType("string", delegate { return new BString(""); }, delegate(BV left, BV right) { return ((BString)left).Value.CompareTo(((BString)right).Value); });

            _primitives = new BPrimitiveType[] { _int, _double, _bool, _time, _date, _string };
        }

        public static BPrimitiveType Int {
            get {
                return _int;
            }
        }
        public static BPrimitiveType Double {
            get {
                return _double;
            }
        }
        public static BPrimitiveType Bool {
            get {
                return _bool;
            }
        }
        public static BPrimitiveType Time {
            get {
                return _time;
            }
        }
        public static BPrimitiveType Date {
            get {
                return _date;
            }
        }
        public static BPrimitiveType String {
            get {
                return _string;
            }
        }
        public static BPrimitiveType[] PrimitiveTypes {
            get {
                return _primitives;
            }
        }


    }

    public class BPrimitiveType : BT {
        private CreateInstanceDelegate _createInstance;
        private CompareDelegate _comparator;

        private class NilV : BV {
            private BT _type;
            public NilV(BT type) {
                _type = type;
            }
            public override bool IsNil {
                get {
                    return true;
                }
            }
            public override BT BT {
                get {
                    return _type;
                }
            }
            public override void Format(BVFormatter formatter) {
                formatter.Nil(this);
            }
            public override void Let(BV value) {
                Debug.Assert(value.IsNil); //Ȃ
            }
        }

        internal BPrimitiveType(string name, CreateInstanceDelegate ci, CompareDelegate comp)
            : base(name) {
            _createInstance = ci;
            _comparator = comp;
            _nilInstance = new NilV(this);
        }
        public int Compare(BV left, BV right) {
            Debug.Assert(left.BT==this);
            Debug.Assert(right.BT==this);

            //Nil͑SPrimitiveType
            if(left.IsNil)
                return right.IsNil? 0 : -1;
            else if(right.IsNil) //!left.IsNil
                return 1;
            else
                return _comparator(left, right);
        }
        public override BV  CreateInstance() {
            return _createInstance();
        }
    }

    public class BArrayT : BT {
        private BT _elementType;
        private int _length; //RpCɌłꍇLBłȂ-1

        public BArrayT(BT et)
            : base(et.Name + "[]") {
            _elementType = et;
            _length = -1;
            _nilInstance = new BRegularArray(this, 0).MakeNil();
        }
        /*
        //RpCɒł̂publicRXgN^
        public BArrayT(BT et, int length) : base(et.Name + "[]") {
            _elementType = et;
            _length = length;
            _nilInstance = new BRegularArray(this).MakeNil();
        }
        */
        //BArrayObjectTpRXgN^
        protected BArrayT(string name, BT et) : base(name) {
            _elementType = et;
            _length = -1;
            _nilInstance = new BRegularArray(this, 0).MakeNil();
        }

        public override BArrayT AsArray() {
            return this;
        }
        public override BV CreateInstance() {
            return new BRegularArray(this, _length);   
        }
        public BT ElementType {
            get {
                return _elementType;
            }
        }
        public override bool IsConvertibleTo(BT type) {
            if(type==null) return false;
            BArrayT array = type.AsArray();
            return array!=null && (this==array || _elementType.IsConvertibleTo(array.ElementType));
        }
        /*
        public int Length {
            get {
                return _length;
            }
        }
        */
    }

    public interface IBObjectT {
        void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys);
    }

    //IuWFNg
    public abstract class BObjectT : BT, IBObjectT {
        public BObjectT(string name, BV nil)
            : base(name) {
            _nilInstance = nil;
        }

        public abstract void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys);
    }
    //złAŗL̃\bh^Cv
    public abstract class BArrayObjectT : BArrayT, IBObjectT {
        public BArrayObjectT(string name, BT element)
            : base(name, element) {
        }
        public abstract void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys);
    }

    public class BFunctionT : BT {
        private BT _targetType;
        private BT _retType;
        private BT[] _argTypes;

        public BFunctionT(BT target, BT[] argTypes, BT retType) : base(Format(target, argTypes, retType)) {
            Debug.Assert(argTypes!=null);
            Debug.Assert(retType!=null);
            _targetType = target;
            _argTypes = argTypes;
            _retType = retType;
            _nilInstance = new BVNilFunction(this);
        }
        public BT TargetType {
            get {
                return _targetType;
            }
        }
        public BT[] ArgTypes {
            get {
                return _argTypes;
            }
        }
        public BT ReturnType {
            get {
                return _retType;
            }
        }

        public override BV CreateInstance() {
            return new BMethodValue(this);
        }

        public override bool IsConvertibleTo(BT type) {
            BFunctionT ft = type as BFunctionT;
            if(ft==null) return false;
            if(!_retType.IsConvertibleTo(ft._retType)) return false;
            return IsMatchArgs(ft._targetType, ft._argTypes);
        }
        public bool IsMatchArgs(BT target, BT[] args) {
            if(_targetType!=null && !_targetType.IsConvertibleTo(target)) return false;
            if(_argTypes.Length!=args.Length)
                return false;
            for(int i=0; i<_argTypes.Length; i++)
                if(!_argTypes[i].IsConvertibleTo(args[i])) return false;
            return true;
        }

        private static string Format(BT target, BT[] args, BT ret) {
            StringBuilder bld = new StringBuilder();
            if(target!=null) {
                bld.Append(target.Name);
                bld.Append(".");
            }

            bld.Append("(");
            for(int i=0; i<args.Length; i++) {
                if(i>0) bld.Append(", ");
                bld.Append(args[i].Name);
            }
            bld.Append(")");
            bld.Append(" -> ");
            bld.Append(ret.Name);
            return bld.ToString();
        }
    }

#if UNITTEST
    [TestFixture]
    public class BTypeTests {
        [Test]
        public void Primitive1() {
            BT[] types = new BT[] { BT.Int, BT.Double, BT.Bool, BT.Time, BT.Date, BT.String };
            for(int i=0; i<types.Length; i++) {
                BT t = types[i];
                BV v = t.CreateInstance();
                Assert.AreSame(v.BT, t);
                Assert.IsFalse(v.IsNil);

                BV n = t.Nil;
                Assert.IsTrue(n.IsNil);

                for(int j=0; j<types.Length; j++)
                    Assert.AreEqual(i==j, t.IsConvertibleTo(types[j])); //^̂
            }
        }
        [Test]
        public void Primitive2() {
            //QlƂĕmF
            int i = 1;
            Assert.IsTrue(i.CompareTo(2) < 0);
            Assert.IsTrue("a".CompareTo("b") < 0);

            AssertPrimitiveComparation(BT.Int, new BInt(1), new BInt(2));
            AssertPrimitiveComparation(BT.Double, new BDouble(1.0), new BDouble(2.0));
            AssertPrimitiveComparation(BT.Time, new BTime(1), new BTime(2));
            AssertPrimitiveComparation(BT.Date, new BDate(1), new BDate(2));
            AssertPrimitiveComparation(BT.String, new BString("a"), new BString("b"));
        }
        private void AssertPrimitiveComparation(BPrimitiveType t, BV s, BV l) {
            Assert.IsTrue(t.Compare(s, l) < 0);
            Assert.IsTrue(t.Compare(l, s) > 0);
            Assert.IsTrue(t.Compare(s, s) == 0);

            Assert.IsTrue(t.Compare(t.Nil, t.Nil) == 0); //Nilm͓
            Assert.IsTrue(t.Compare(t.Nil, l) < 0);
            Assert.IsTrue(t.Compare(s, t.Nil) > 0);
        }
        [Test]
        public void ArrayType1() {
            BArrayT ia1 = BT.Int.ArrayType();
            BArrayT ia2 = new BArrayT(BT.Int); 

            BArray v1 = (BArray)ia1.CreateInstance();
            BArray v2 = (BArray)ia2.CreateInstance();
            Assert.AreSame(v1.BT, ia1);
            Assert.AreSame(v2.BT, ia2);
            Assert.AreSame(v1.BArrayT.ElementType, BT.Int);
            Assert.AreSame(v2.BArrayT.ElementType, BT.Int);
            Assert.IsTrue(ia1.IsConvertibleTo(ia2)); //ϊ͉\
            Assert.IsTrue(ia2.IsConvertibleTo(ia1));

        }
        [Test]
        public void ArrayType2() {
            //ϊsmF
            BArrayT ia1 = BT.Int.ArrayType();
            BArrayT ta2 = BT.Time.ArrayType();
            Assert.IsFalse(ia1.IsConvertibleTo(BT.Int));
            Assert.IsFalse(ia1.IsConvertibleTo(BT.Bool));
            Assert.IsFalse(ia1.IsConvertibleTo(ta2));
        }
        [Test]
        public void ArrayType3() {
            //vf̕ύXAύX
            BArrayT ia = new BArrayT(BT.Int);
            BRegularArray a = (BRegularArray)ia.CreateInstance();
            a.Extend(3);
            a.SetAt(2, new BInt(123));
            Assert.AreEqual(123, ((BInt)a[2]).Value);

            a.Extend(20);
            Assert.AreEqual(20, a.Count);
            Assert.IsNotNull(a[15]);
            Assert.AreSame(BT.Int, a[15].BT);
            Assert.AreEqual(123, ((BInt)a[2]).Value); //̒l͕ςĂȂ

            a.Extend(10);
            Assert.AreEqual(10, a.Count);
            Assert.AreEqual(123, ((BInt)a[2]).Value);
        }
        [Test]
        public void CompareInt() {
            List<BInt> ints = new List<BInt>();
            ints.Add(new BInt(2));
            ints.Add(new BInt(3));
            ints.Add(new BInt(1));
            ints.Sort(new Comparison<BInt>(BT.Int.Compare));
            Assert.AreEqual(1, ints[0].Value);
            Assert.AreEqual(2, ints[1].Value);
            Assert.AreEqual(3, ints[2].Value);
        }
        [Test]
        public void CompareDouble() {
            List<BDouble> doubles = new List<BDouble>();
            doubles.Add(new BDouble(2));
            doubles.Add(new BDouble(Math.PI));
            doubles.Add(new BDouble(1));
            doubles.Sort(new Comparison<BDouble>(BT.Double.Compare));
            Assert.AreEqual(1, doubles[0].Value);
            Assert.AreEqual(2, doubles[1].Value);
            Assert.AreEqual(Math.PI, doubles[2].Value);
        }
        [Test]
        public void Let1() {
            BInt i = new BInt(123);
            BV j = new BInt(0);
            int c = BV.instanceCount;
            BV.Let(ref j, i);
            Assert.AreEqual(i.Value, ((BInt)j).Value);
            Assert.AreEqual(c, BV.instanceCount); //^ȂVK쐬͂Ȃ

            //nullnilł
            BV r = null;
            BV.Let(ref r, i);
            Assert.IsNotNull(r);
            Assert.AreSame(BT.Int, r.BT);
            Assert.AreEqual(i.Value, ((BInt)r).Value);

            r = BT.Int.Nil;
            BV.Let(ref r, i);
            Assert.IsNotNull(r);
            Assert.AreSame(BT.Int, r.BT);
            Assert.AreEqual(i.Value, ((BInt)r).Value);
        }
        [Test]
        public void Let2() {
            //zւ̑S
            BArrayT ia = new BArrayT(BT.Int);
            BRegularArray a = new BRegularArray(ia, 10);
            a.SetAt(2, new BInt(99));

            BV r = null;
            BV.Let(ref r, a);
            a.SetAt(2, new BInt(1));

            Assert.IsTrue(r is BRegularArray);
            BRegularArray rr = (BRegularArray)r;
            Assert.AreEqual(10, rr.Count);
            Assert.AreEqual(99, ((BInt)rr[2]).Value); //Ƃ̒l̂܂܂̂͂
        }
    }
#endif
}
