/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Alert/AlertDocument.cs#12 $
 * $DateTime: 2008/01/30 09:23:51 $
 */
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

using Poderosa;
using Poderosa.UI;
using Poderosa.Forms;
using Poderosa.Preferences;
using Poderosa.Commands;
using Poderosa.Plugins;
using Poderosa.Sessions;

using Travis.ORT;
using Sansa.Runtime;

using Bellagio.Environment;
using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Drawing;
using Bellagio.Forms;
using Bellagio.Evaluators;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Alert {
    public class AlertDocument : IPeriodicalDataSubscriber {
        
        private EvalContext _evalContext;

        private List<AlertData> _newSignals; //VKVOi͂܂ɂ܂if[^XbhjACXbhŒʒm|bvAbvƃr[XV
        private List<AlertData> _signals;
        private List<RuntimeAlertCondition> _conditions;
        private Dictionary<Stock, StockTag> _stockData;
        private int _duplicativeAlertSuppressTimeSec; //dA[gh~鎞
        private DataThreadToMainThread _notifier;

        //̂̓K؂ȍ\͂܂smBRuntimeAlertCondition̍\͋NɌł邪AĎQuote͎sɕωB
        //ɁAQuote̐GMOł͐yVRSSł͐S肤B
        //Condition̎boolԂA̒lfalsetrueɕωuԂVOiłBȂ̂őO̒loĂȂƂȂB
        private class StockTag {
            public Stock stock;
            public BBoolean[] values; //RuntimeAlertConditioñCfNXƓ

            public StockTag(Stock s, int c) {
                values = new BBoolean[c];
                stock = s;
            }
            public bool IsFalseValue(int index) { //falseł邱Ƃ̊mFBnilfalseƈႤ̂Œ
                BBoolean x = values[index];
                return x!=null && !x.IsNil && !x.Value;
            }
            public void Clear() {
                for(int i=0; i<values.Length; i++)
                    if(values[i]!=null) values[i].Value = false;
            }
        }

        public AlertDocument() {
            _signals = new List<AlertData>();
            _conditions = new List<RuntimeAlertCondition>();
            _stockData = new Dictionary<Stock, StockTag>();
            _evalContext = new EvalContext();
            _evalContext.CurrentStock = new StockRef();
            _newSignals = new List<AlertData>();
            _notifier = new DataThreadToMainThread(DataThreadNotifier);
            _duplicativeAlertSuppressTimeSec = 180;
        }

        public void AddConditon(RuntimeAlertCondition cond) {
            cond.Index = _conditions.Count;
            _conditions.Add(cond);
        }

        public void CompleteCreation() {
        }

        public IEnumerable<AlertData> Signals {
            get {
                return _signals;
            }
        }

        public AlertData GetAlertDataAt(int index) {
            return _signals[index];
        }
        public RuntimeAlertCondition GetConditionAt(int index) {
            return _conditions[index];
        }

#if UNITTEST
        internal List<AlertData> NewSignals {
            get {
                return _newSignals;
            }
        }
        internal int _unittest_singal_suppressed_count;
        internal void ClearStockTag(Stock s) {
            StockTag t = _stockData[s];
            t.Clear();
        }
#endif


        #region IDataSubscriber

        DataThreadToMainThread IDataSubscriber.NotifyDelegate {
            get {
                return _notifier;
            }
        }

        int IPeriodicalDataSubscriber.IntervalSec {
            get {
                return 5;
            }
        }
        void IPeriodicalDataSubscriber.PeriodicalProcess(DataProcessArg arg) {

            BTime time = arg.LocalTime;

            foreach(IntraDayTrade q in BellagioRoot.IntraDayTradeProvider.GetAllActives())
                ProcessIntraDayTrade(q, time);
        }

        //P̃eXgŎĝinternal\bh
        private IntraDayTradeRef _tempIntraDayTradeRef = new IntraDayTradeRef();
        internal void ProcessIntraDayTrade(IntraDayTrade q, BTime time) {

            StockTag tag;
            if(!_stockData.TryGetValue(q.Stock, out tag)) {
                tag = new StockTag(q.Stock, _conditions.Count);
                _stockData.Add(q.Stock, tag);
            }

            for(int i=0; i<_conditions.Count; i++) {
                RuntimeAlertCondition rc = _conditions[i];

                //false->true̕ω
                _tempIntraDayTradeRef.Value = q;
                _evalContext.ContextValue = _tempIntraDayTradeRef;
                _evalContext.CurrentStock.Stock = q.Stock;
                _evalContext.CurrentTime = time;
                BV value = rc.Evaluator.Eval(_evalContext);

                if(tag.IsFalseValue(i) && !value.IsNil && ((BBoolean)value).Value) { //VOi
                    bool c = ContainsDuplicatives(_newSignals, q.Stock, time, rc);
#if UNITTEST //@\mF̂߃JEg
                    if(c) _unittest_singal_suppressed_count++;
#endif
                    if(!c)
                        _newSignals.Add(RaiseAlert(q, time, rc));
                }

                //Os̒lZbg
                BV x = tag.values[i];
                BV.Let(ref x, value);
                tag.values[i] = x.IsNil? null : (BBoolean)x; //Ȃ񂩉
                Debug.Assert(x.IsNil || !Object.ReferenceEquals(x, value));
            }
        }

        #endregion

        public AlertData ForceRaiseAlert(IntraDayTrade q, BTime time, RuntimeAlertCondition rc) {
            if(ContainsDuplicatives(_newSignals, q.Stock, time, rc)) return null;

            AlertData ad = RaiseAlert(q, time, rc);
            _newSignals.Add(ad);
            DataThreadNotifier();
            return ad;

        }

        //VAlert̍쐬
        private AlertData RaiseAlert(IntraDayTrade q, BTime time, RuntimeAlertCondition rc) {
            BTime start = new BTime(time);
            int open_time = MarketUtil.GetZenbaOpenTime(q.Stock.Market).AsInt();
            
            ConcreteQuote src_pr = q.Minutely;
            ConcreteQuote pr = new ConcreteQuote(Quote.QuoteUnit.Minutely);
            int index = 0;
            while(index<src_pr.Count && src_pr.CandleAt(index).Time<start.AsInt()) {
                pr.Add(new Candle(src_pr.CandleAt(index)));
                index++;
            }

            TimeAndSales src_ts = q.TimeAndSales;
            TimeAndSales ts = new TimeAndSales();
            index = src_ts.FindByTime(start.AsInt());
            if(index!=-1) {
                while(index < src_ts.Count && src_ts.TickAt(index).Time<=time.AsInt())
                    ts.Add(new TickData(src_ts.TickAt(index++)));
            }

            return new AlertData(rc, q.Stock, new BTime(time), pr, ts, null); //ItaHistory͂܂
        }

        private void DataThreadNotifier() {
            try {
                foreach(AlertData ad in _newSignals) {
                    AlertPlugin.Instance.OpenAlertCommand(BellagioPlugin.Instance.ActivePoderosaWindow, ad);
                    _signals.Add(ad);
                }

                if(AlertPlugin.Instance.AlertContent!=null)
                    AlertPlugin.Instance.AlertContent.UpdateContent();
                _newSignals.Clear();
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

        //Eł̓VOi莞Ԗh~
        private bool ContainsDuplicatives(List<AlertData> collection, Stock stock, BTime time, RuntimeAlertCondition cond) {
            foreach(AlertData d in collection) {
                if(d.Stock==stock && d.SourceCondition==cond) {
                    int t = time.AsInt() - d.AlertTime.AsInt();
                    if(t>=0 && t<_duplicativeAlertSuppressTimeSec) return true;
                }
            }
            return false;
        }



    }

    public class RuntimeAlertCondition {
        private int _index;
        private string _title;
        private string _soundFile; //Ȃnull
        private IEvaluator _evaluator;

        public RuntimeAlertCondition(AlertConditionSchema schema) {
            _title = schema.title.ParseMandatoryString();
            _soundFile = schema.soundFile.ParseOptionalString(null);
            if(_soundFile!=null) {
                _soundFile = Path.GetFullPath(Path.Combine(BellagioRoot.FixedPreferences.SoundDir, _soundFile));
                if(!File.Exists(_soundFile)) throw new BellagioException(String.Format("{0} ݂͑܂", _soundFile));
            }
            EvaluatorBuildContext bc = new EvaluatorBuildContext(schema);
            bc.Initialize(IntraDayTradeType.instance, LocalVariable.EmptyArray);
            _evaluator = EvaluatorBuilderFromText.Build(schema.condition, bc);
        }
        public RuntimeAlertCondition(string title, IEvaluator ev) {
            _title = title;
            _evaluator = ev;
        }

        public string Title {
            get {
                return _title;
            }
        }
        public string SoundFile {
            get {
                return _soundFile;
            }
        }
        public IEvaluator Evaluator {
            get {
                return _evaluator;
            }
        }

        //AlertDocumentݒ肷
        public int Index {
            get {
                return _index;
            }
            set {
                _index = value;
            }
        }
    }

#if UNITTEST
    [TestFixture]
    public class AlertTests {
        [TestFixtureSetUp]
        public void Setup() {
            BellagioEnvironmentParam ep = new BellagioEnvironmentParam();
            ep.SetupForUnitTest();
            ep.RUN_DATA_SUBSCRIBER_ENGINE = true;
            ep.USE_DEMONSTRATION_STOCKS = false;
            ep.UNITTEST_STOCK_COUNT = 0;
            BellagioRoot.Init(ep);
        }
        [TestFixtureTearDown]
        public void TearDown() {
            BellagioRoot.Terminate();
        }

        private IntraDayTrade PrepareQuote() {
            Stock st = new BasicStockProfile("T", "T", StockProfileFlags.None, 1000).CreatePrimary(StockExchange.T, StockExchangeSubType.Ichibu, StockFlags.None);
            IntraDayTrade q = new IntraDayTrade(st);

            BTime t = new BTime(9, 0, 0);
            //30bƂɂP~オeXgf[^
            for(int i=0; i<20; i++)
                q.UpdateBySingleTick(new TickData(t.AsInt() + i*30, 990 + i, 1000, TickItaRelation.Unknown));

            return q;
        }
        private RuntimeAlertCondition PrepareCondition() {
            EvaluatorBuildContext bc = new EvaluatorBuildContext(null);
            bc.Initialize(IntraDayTradeType.instance, new LocalVariable[0]);
            IEvaluator ev = EvaluatorBuilderFromText.Build("price() > 1000", bc);
            return new RuntimeAlertCondition("test", ev);
        }
        private AlertDocument PrepareDocument() {
            AlertDocument doc = new AlertDocument();
            doc.AddConditon(PrepareCondition());
            doc.CompleteCreation();
            return doc;
        }

        [Test]
        public void Basic() {
            AlertDocument doc = PrepareDocument();
            IntraDayTrade q = PrepareQuote();

            BTime t = new BTime(9, 0, 0);
            //30bƂɂP~オ̂ŁA9:05:001000A9:05:301001ƂȂ͂
            for(int i=0; i<6; i++) {
                doc.ProcessIntraDayTrade(q, t);
                Assert.AreEqual(0, doc.NewSignals.Count);

                t.AddSec(60);
            }

            doc.ProcessIntraDayTrade(q, t);
            Assert.AreEqual(0, doc._unittest_singal_suppressed_count);
            Assert.AreEqual(1, doc.NewSignals.Count);
            AlertData ad = doc.NewSignals[0];
            Assert.AreEqual(t.AsInt(), ad.AlertTime.AsInt());

            //ł̎͂XUAŌ̂Q^ȂS{̑ƂT{Tick
            Assert.AreEqual(6, ad.Quote.Count);
            Assert.AreEqual(1, ad.TickData.Count);

            //ɐi߂B̂ɂ͍vĂ邪ԂߐڂĂ̂ŃXLbv͂
            t.AddSec(60);
            doc.ClearStockTag(q.Stock);
            doc.ProcessIntraDayTrade(q, t);
            Assert.AreEqual(1, doc._unittest_singal_suppressed_count);
            Assert.AreEqual(1, doc.NewSignals.Count);
            t.AddSec(60);
            doc.ClearStockTag(q.Stock);
            doc.ProcessIntraDayTrade(q, t);
            Assert.AreEqual(2, doc._unittest_singal_suppressed_count);
            Assert.AreEqual(1, doc.NewSignals.Count);
        }

    }
#endif
}
