/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Chart/CurrentPriceInfoDrawing.cs#6 $
 * $DateTime: 2008/03/13 13:20:43 $
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Windows.Forms;
using System.Drawing;

using Bellagio.Drawing;
using Bellagio.Values;
using Bellagio.Forms;
using Bellagio.Data;

using Travis;
using Travis.LocationOperation;
using Sansa.Runtime;

namespace Bellagio.Chart {
    //`NX
    public class CurrentPriceInfoGraphics : ViewContentGraphics {
        public enum ModeT {
            Stock, Future
        }
        //\邩̃IvV
        public enum LayoutT {
            Default, VolumeAndCredit, PriceOnly
        }

        private LayoutT _layout;
        private CurrentPriceInfoGraphicsSettings _settings;
        private CurrentPriceInfo _data;
        private BRect _bound; //Ŝ̋E
        private BRect _priceRect;
        private BRect _volumeRect; //AVWAP
        private BRect _ohlRect; //OAOpen/High/Low
        private BRect _bidAskRect;
        private BRect _creditRect;
        private ModeT _mode;

        //f[^vZł
        private RealtimeInfoString<int> _lastCloseDiff; //O

        public CurrentPriceInfoGraphics(Control control, IPlatformBridge platform, CurrentPriceInfoGraphicsSettings settings, LayoutT layout)
            : base(platform, control) {
            _settings = settings;
            _layout = layout;
            _bound = new BRect();
            _priceRect = new BRect();
            _volumeRect = new BRect();
            _ohlRect = new BRect();
            _bidAskRect = new BRect();
            _creditRect = new BRect();

            int m2 = _settings.margin.V*2;
            RectangleOperatorBuilder bld = new RectangleOperatorBuilder();
            bld.ProbeBRect(_bound).Shrink(3);
            if(layout==LayoutT.Default || layout==LayoutT.PriceOnly)
                bld.SplitTop(_settings.rowPitch.V*2+m2).ProbeBRect(_priceRect).CloseBranch();
            if(layout==LayoutT.Default || layout==LayoutT.VolumeAndCredit) {
                bld.SplitFixed(SplitDir.Top, _settings.rowPitch.V*4+m2)
                    .SplitRatio(SplitDir.Left, 0.5F, 80).ProbeBRect(_volumeRect).CloseBranch().ProbeBRect(_ohlRect)
                .CloseBranch()
                .SplitFixed(SplitDir.Top, _settings.rowPitch.V*2+m2)
                    .SplitRatio(SplitDir.Left, 0.5F, 80).ProbeBRect(_bidAskRect).CloseBranch().ProbeBRect(_creditRect)
                .CloseBranch();
            }

            _rectangleOperator = bld.Finish();
            _lastCloseDiff = new RealtimeInfoString<int>();
        }

        public ModeT Mode {
            get {
                return _mode;
            }
            set {
                _mode = value;
            }
        }
        public override void Dispose() {
        }
        public BRect Bound {
            get {
                return _bound;
            }
        }

        protected override void BuildScaleInfo() {
        }
        protected override void OnEndAnimation() {
            base.OnEndAnimation();
            _data.SetAllMoveToStable();
        }

        protected override void InvalidateForDataAnimationTick() {
            InvalidateRect(_bound); //ꉞAǂ̃f[^ڂXVꂽ݂ĕKvȕInvalidateΕ׌ył
        }
        public void StartMotion(CurrentPriceInfo data) {
            _data = data;
            base.DataUpdateDirect();
            if(BellagioRoot.FixedPreferences.Chart.EnablePriceInfoAnimation)
                _dataAnimationTimer.Initialize();
        }
        public override void Paint(Graphics g, Rectangle clip) {
            if(_data==null) return;

            if(_priceRect.IntersectsWith(ref clip)) PaintPriceAndDiff(g);
            if(_volumeRect.IntersectsWith(ref clip)) PaintVolumes(g);
            if(_ohlRect.IntersectsWith(ref clip)) PaintPrevCloseAndOHL(g);
            if(_bidAskRect.IntersectsWith(ref clip)) PaintBestBidAsk(g);
            if(_creditRect.IntersectsWith(ref clip)) PaintCredits(g);
        }

        private void PaintPriceAndDiff(Graphics g) {
            //delimiter
            if(_layout==LayoutT.Default)
                DrawHDelimiter(g, _priceRect.Left, _priceRect.Bottom, _priceRect.Width);

            if(_data.CurrentPrice.IsEmpty) return;

            Rectangle rect = CutMargin(_priceRect);
            if(!_data.CurrentPrice.IsEmpty)
                DrawItem(g, _data.CurrentPrice, new Rectangle(rect.X, rect.Y, rect.Width/2, rect.Height), _settings.priceColor.col, _settings.largeValueFont.F, SplitDir.Left);
            if(!_data.CurrentPriceTime.IsEmpty)
                DrawItem(g, _data.CurrentPriceTime, new Rectangle(rect.X+rect.Width/2, rect.Y, rect.Width/2, rect.Height), _settings.timeColor.col, _settings.valueFont.F, SplitDir.Left);
            //O
            if(!_data.LastClose.IsEmpty) {
                bool up = _data.CurrentPrice.Value >= _data.LastClose.Value;
                DrawItem2(g, _data.CurrentPrice.Move,
                    _data.CurrentPrice.PrevString==null? "" : FormatDiff(_data.CurrentPrice.PrevValue, _data.LastClose.Value),
                    FormatDiff(_data.CurrentPrice.Value, _data.LastClose.Value),
                    new Rectangle(rect.X+rect.Width/2, rect.Y+15, rect.Width/2, rect.Height-5), //tHgȂ̂ŏ
                    up? _settings.plusColor.col : _settings.minusColor.col,
                    _settings.valueFont.F, SplitDir.Left);
            }
        }
        private string FormatDiff(int value, int lastclose) {
            string sign = value>=lastclose? "+" : "";
            return String.Format("{0}{1} ({2}{3:F2}%)", sign, value-lastclose, sign, ((double)(value-lastclose)) / lastclose * 100);
        }
        private void PaintVolumes(Graphics g) {
            DrawHDelimiter(g, _volumeRect.Left, _volumeRect.Bottom, _volumeRect.Width);
            DrawVDelimiter(g, _volumeRect.Right, _volumeRect.Top, _volumeRect.Height);

            Brush lb = _settings.labelColor.B;
            Color vc = _settings.valueColor.col;
            Rectangle rect = CutMargin(_volumeRect);
            rect.Height = _settings.rowPitch.V;

            g.DrawString("o", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.Volume.IsEmpty) DrawItemNoAnimation(g, _data.Volume, rect, vc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;

            g.DrawString("VWAP", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.VWAP.IsEmpty) DrawItemNoAnimation(g, _data.VWAP, rect, vc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;

            g.DrawString("", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.TotalAmount.IsEmpty) DrawItemNoAnimation(g, _data.TotalAmount, rect, vc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;
        }
        private void PaintCredits(Graphics g) {
            Brush lb = _settings.labelColor.B;
            Color vc = _settings.valueColor.col;
            Rectangle rect = CutMargin(_creditRect);
            rect.Height = _settings.rowPitch.V;

            if(_mode==ModeT.Stock) {
                g.DrawString("Mpc", _settings.labelFont.F, lb, rect.X, rect.Y);
                if(!_data.CreditLongVolume.IsEmpty) DrawItemNoAnimation(g, _data.CreditLongVolume, rect, vc, _settings.valueFont.F, SplitDir.Right);
                rect.Y += _settings.rowPitch.V;
                g.DrawString("Mpc", _settings.labelFont.F, lb, rect.X, rect.Y);
                if(!_data.CreditShortVolume.IsEmpty) DrawItemNoAnimation(g, _data.CreditShortVolume, rect, vc, _settings.valueFont.F, SplitDir.Right);
                rect.Y += _settings.rowPitch.V;
            }
            else { //敨ł͔cʕ\ɂȂ
                g.DrawString("ʎc", _settings.labelFont.F, lb, rect.X, rect.Y);
                if(!_data.CreditLongVolume.IsEmpty) DrawItemNoAnimation(g, _data.CreditLongVolume, rect, vc, _settings.valueFont.F, SplitDir.Right);
                rect.Y += _settings.rowPitch.V;
            }
        }

        private void PaintPrevCloseAndOHL(Graphics g) {
            DrawHDelimiter(g, _ohlRect.Left, _ohlRect.Bottom, _ohlRect.Width);

            Brush lb = _settings.labelColor.B;
            Color vc = _settings.valueColor.col;
            Color tc = _settings.timeColor.col;
            Rectangle rect = CutMargin(_ohlRect);
            rect.Height = _settings.rowPitch.V;

            g.DrawString("OIl", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.LastClose.IsEmpty) DrawItem(g, _data.LastClose, rect, vc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;

            //\pGAvZ
            Rectangle time_rect = new Rectangle(rect.Right - _settings.timeAreaWidth.V, rect.Y, _settings.timeAreaWidth.V, rect.Height);
            rect.Width -= _settings.timeAreaWidth.V;

            g.DrawString("nl", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.OpenPrice.IsEmpty) DrawItem(g, _data.OpenPrice, rect, vc, _settings.valueFont.F, SplitDir.Right);
            if(!_data.OpenTime.IsEmpty) DrawItemNoAnimation(g, _data.OpenTime, time_rect, tc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;
            time_rect.Y += _settings.rowPitch.V;

            g.DrawString("l", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.HighPrice.IsEmpty) DrawItem(g, _data.HighPrice, rect, vc, _settings.valueFont.F, SplitDir.Right);
            if(!_data.HighTime.IsEmpty) DrawItemNoAnimation(g, _data.HighTime, time_rect, tc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;
            time_rect.Y += _settings.rowPitch.V;

            g.DrawString("l", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.LowPrice.IsEmpty) DrawItem(g, _data.LowPrice, rect, vc, _settings.valueFont.F, SplitDir.Right);
            if(!_data.LowTime.IsEmpty) DrawItemNoAnimation(g, _data.LowTime, time_rect, tc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;
            time_rect.Y += _settings.rowPitch.V;

        }

        private void PaintBestBidAsk(Graphics g) {
            DrawVDelimiter(g, _bidAskRect.Right, _bidAskRect.Top, _bidAskRect.Height);

            Brush lb = _settings.labelColor.B;
            Color vc = _settings.valueColor.col;
            Color pc = _settings.priceColor.col;
            Rectangle rect = CutMargin(_bidAskRect);
            rect.Height = _settings.rowPitch.V;

            Rectangle volume_rect = new Rectangle(rect.Right - _settings.bestBidAskVolumeAreaWidth.V, rect.Y, _settings.bestBidAskVolumeAreaWidth.V, rect.Height);
            rect.Width -= _settings.bestBidAskVolumeAreaWidth.V;

            g.DrawString("", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.AskPrice.IsEmpty) DrawItem(g, _data.AskPrice, rect, pc, _settings.valueFont.F, SplitDir.Right);
            if(!_data.AskVolume.IsEmpty) DrawItemNoAnimation(g, _data.AskVolume, volume_rect, vc, _settings.valueFont.F, SplitDir.Right);
            rect.Y += _settings.rowPitch.V;
            volume_rect.Y += _settings.rowPitch.V;

            g.DrawString("", _settings.labelFont.F, lb, rect.X, rect.Y);
            if(!_data.BidPrice.IsEmpty) DrawItem(g, _data.BidPrice, rect, pc, _settings.valueFont.F, SplitDir.Right);
            if(!_data.BidVolume.IsEmpty) DrawItemNoAnimation(g, _data.BidVolume, volume_rect, vc, _settings.valueFont.F, SplitDir.Right);
        }


        //`
        private void DrawItem(Graphics g, RealtimeInfoStringBase item, Rectangle rect, Color col, Font font, SplitDir align) {
            DrawItem2(g, item.Move, item.PrevString, item.CurrentString, rect, col, font, align);
        }
        private void DrawItem2(Graphics g, RealtimeInfoStringBase.MoveT move, string prev, string current, Rectangle rect, Color col, Font font, SplitDir align) {
            double animation = BellagioRoot.FixedPreferences.Chart.EnablePriceInfoAnimation? _dataAnimationTimer.value() : 1.0;

            Brush br = null;
            if(move==RealtimeInfoStringBase.MoveT.Stable) { //Î~
                br = new SolidBrush(col);
                DrawClippedString(g, current, font, br, rect, rect.Y, align);
            }
            else if(move==RealtimeInfoStringBase.MoveT.FadeIn) { //tF[hC
                br = new SolidBrush(DrawingUtil.ProportionalColor(_settings.backColor.col, col, animation));
                DrawClippedString(g, current, font, br, rect, rect.Y, align);
            }
            else if(move==RealtimeInfoStringBase.MoveT.FadeOut) {//tF[hAEg
                br = new SolidBrush(DrawingUtil.ProportionalColor(_settings.backColor.col, col, 1.0-animation));
                DrawClippedString(g, prev, font, br, rect, rect.Y, align);
            }
            else {
                //increasêƂ prev 0...1, current -1...0
                //decreasêƂ prev 0...-1, current 1...0
                int h = (int)(rect.Height * animation);
                int d = move==RealtimeInfoStringBase.MoveT.Increasing? -1 : 1;

                if(animation < 1.0) {
                    br = new SolidBrush(DrawingUtil.ProportionalColor(_settings.backColor.col, col, 1.0-animation));
                    DrawClippedString(g, prev, font, br, rect, rect.Y - d*h, align);
                }
                if(animation > 0) {
                    if(br!=null) br.Dispose();
                    br = new SolidBrush(DrawingUtil.ProportionalColor(_settings.backColor.col, col, animation));
                    DrawClippedString(g, current, font, br, rect, rect.Y + d*(rect.Height-h), align);
                }

            }
            if(br!=null) br.Dispose();
        }
        private void DrawItemNoAnimation(Graphics g, RealtimeInfoStringBase item, Rectangle rect, Color col, Font font, SplitDir align) {
            Brush br = new SolidBrush(col);
            DrawClippedString(g, item.CurrentString, font, br, rect, rect.Y, align);
            br.Dispose();
        }

        //ǂɕ\邩͍Eǂ炩BSplitDirőp
        private void DrawClippedString(Graphics g, string text, Font font, Brush br, Rectangle clip, int y, SplitDir align) {
            if(align==SplitDir.Left)
                DrawStringUtil.Clipped(g, text, font, br, clip, clip.X, y);
            else {
                SizeF sz = g.MeasureString(text, font);
                DrawStringUtil.Clipped(g, text, font, br, clip, clip.Right - (int)sz.Width, y);
            }
        }

        //Kȃ}[WĐ
        private void DrawHDelimiter(Graphics g, int x, int y, int width) {
            const int margin = 4;
            Pen pen = _settings.delimiterPen.P;
            g.DrawLine(pen, x+margin, y, x+width-margin, y);
        }
        private void DrawVDelimiter(Graphics g, int x, int y, int height) {
            const int margin = 4;
            Pen pen = _settings.delimiterPen.P;
            g.DrawLine(pen, x, y+margin, x, y+height-margin);
        }

        private Rectangle CutMargin(BRect rect) {
            int m = _settings.margin.V;
            return new Rectangle(rect.Left + m, rect.Top + m, rect.Width - m*2, rect.Height - m*2);
        }

    }

    public class CurrentPriceInfoGraphicsSettings : DrawingSettingsBase {
        public enum LayoutStyle {
            None,
            Normal
        }
        public DSInt layoutStyleI;
        public LayoutStyle layoutStyle {
            get {
                return (LayoutStyle)layoutStyleI.V;
            }
            set {
                layoutStyleI.UnderlyingItem.AsInt().Value = (int)value;
            }
        }
        public DSBrush backColor; //wiF
        public DSFont labelFont; //x̃tHg
        public DSFont valueFont; //l̃tHg
        public DSFont largeValueFont; //傫\̒l̃tHg
        public DSBrush borderColor; //g̐F
        public DSPenStyle delimiterPen; //؂
        public DSBrush plusColor; //OvX̐F
        public DSBrush minusColor; //O}CiX̐F
        public DSBrush labelColor; //e퍀ږ̐F
        public DSBrush valueColor;      //el̐F
        public DSBrush timeColor;  //\̐F
        public DSBrush priceColor; //li\̐F
        public DSInt rowPitch; //ŴPs̍
        public DSInt margin; //gƒl\̃Xy[X
        public DSInt timeAreaWidth; //\̕
        public DSInt bestBidAskVolumeAreaWidth; //BestBidAskʕ\̕

        public CurrentPriceInfoGraphicsSettings(string id)
            : base(id) {
        }

        protected override void CreateElements() {
            layoutStyleI = DefineInt("layoutStyle", 1);
            labelFont = DefineFont("labelFont", DSFont.GenericFontName, 9);
            valueFont = DefineFont("valueFont", DSFont.GenericFontName, 9);
            largeValueFont = DefineFont("largeValueFont", "OCR A Std", 18, FontStyle.Bold);

            backColor = DefineBrush("backColor", Color.Black);
            borderColor = DefineBrush("borderColor", Color.Ivory);
            plusColor = DefineBrush("plusColor", Color.Red);
            minusColor = DefineBrush("minusColor", Color.Cyan);
            labelColor = DefineBrush("labelColor", Color.White);
            valueColor = DefineBrush("valueColor", Color.LightCyan);
            timeColor = DefineBrush("timeColor", Color.Yellow);
            priceColor = DefineBrush("priceColor", Color.Gold);

            delimiterPen = DefinePenStyle("delimiterPen", Color.Ivory, BPenStyle.DashStyle.Dot, 1);

            rowPitch = DefinePositiveInt("rowPitch", 16);
            margin = DefineInt("margin", 3);
            timeAreaWidth = DefinePositiveInt("timeAreaWidth", 40);
            bestBidAskVolumeAreaWidth = DefinePositiveInt("bestBidAskVolumeAreaWidth", 50);
        }

    }
}
