/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/DaytimeTrade.cs#14 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * i󂯎AB
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Threading;

using Poderosa;
using Poderosa.Util.Collections;
using Bellagio.Values; //NSO

#if UNITTEST
using NUnit.Framework;
#endif


namespace Bellagio.Data {


    //A^Cf[^BTime&SalesĕƂ܂łz
    //DataSubscriber̈킾ASubscriberQuotef[^擾邱Ƃ͂߂Ă悭邱ƂȂ̂ŁADataSubscriberManagerQuoteɏ悤ɏɋCĂ
    //vC}f[^񋟂ĂȂꍇ̂IPeriodicalDataSubscriberƂĂ̋@\KvɂȂĂ
    //UpdateBy***n\bhŒgXVB̓Xbht[BĂԂDataSubscriber.ReserveExecł̖̍XVsB
    //ʂ̃f[^ŏƂResetBy***n\bhB
    //̂Am肵Ă̂"fixed"łƂBxm肵́ATickDatał͏㏑łȂ
    //TESTINFO ̃t@C

    [Flags]
    public enum IntraDayTradeWarnings {
        None = 0,
        TickSpilled = 1
    }

    public class IntraDayTrade : StockPrimaryData, IPeriodicalDataSubscriber {

        private CurrentPriceInfo _currentPriceInfo;
        private TimeAndSales _timeAndSales;
        private ConcreteQuote _minutely; // m蕪݂̂^
        private int _fixedLength; //̂擪̉{m肵Ă邩
        private Candle _lastMinuteCandle; //ݍXV̂P
        private int _intervalSec;
        private IntraDayTradeWarnings _warningStatus; //G[ł͂ȂʒmׂԂƂ̃tO

        public IntraDayTrade(Stock stock) : base(stock) {
            _intervalSec = 60; //ܕ݂̂Ȃ̂60b
            Reset();
        }
        internal void ReplaceQuotes(ConcreteQuote q) {
            _minutely = q;
        }
#if UNITTEST
        internal void ReplaceTimeAndSales(TimeAndSales ts) {
            _timeAndSales = ts;
        }
#endif
        internal void Reset() {
            if(_currentPriceInfo==null)
                _currentPriceInfo = new CurrentPriceInfo();
            else
                _currentPriceInfo.Reset();
            _timeAndSales = new TimeAndSales();
            _minutely = new ConcreteQuote(Quote.QuoteUnit.Minutely);
            _lastMinuteCandle = new Candle();
            _lastMinuteCandle.Let(CandleType.instance.Nil);
            _fixedLength = 0;
        }
        public TimeAndSales TimeAndSales {
            get {
                return _timeAndSales;
            }
        }
        public ConcreteQuote Minutely {
            get {
                return _minutely;
            }
        }
        public int FixedLength {
            get {
                return _fixedLength;
            }
        }
        public Candle LastMinuteCandle {
            get {
                return _lastMinuteCandle;
            }
        }
        public bool IsLastMinuteCandleAvailable {
            get {
                return !_lastMinuteCandle.IsNil;
            }
        }
        public CurrentPriceInfo CurrentPriceInfo {
            get {
                return _currentPriceInfo;
            }
        }
        public IntraDayTradeWarnings WarningStatus {
            get {
                return _warningStatus;
            }
            set {
                _warningStatus = value;
            }
        }
#if DEBUG
        //`p
        public void SetCurrentCandle(Candle y) {
            _lastMinuteCandle = y;
        }
#endif

        #region IDataSubscriber
        DataThreadToMainThread IDataSubscriber.NotifyDelegate {
            get {
                return null;
            }
        }

        int IPeriodicalDataSubscriber.IntervalSec {
            get {
                return _intervalSec;
            }
        }


        void IPeriodicalDataSubscriber.PeriodicalProcess(DataProcessArg args) {
            UpdateByTime(args.LocalTime);
        }

        #endregion

        public override bool FireEventToSubscriberManager() {
            BellagioRoot.DataSubscriberManager.ReserveExecByStock(_stock, SubscribeDataKind.Ticks);
            return false;
        }
        public override void FireErrorEventToSubscriberManager(string msg) {
            BellagioRoot.DataSubscriberManager.ReserveErrorByStock(_stock, SubscribeDataKind.Ticks, msg);
        }


        //łĂmF
        public void AssertTickDataInitial() {

        }

        //̍XVBi͂낢날B̓Xbht[Ȃ̂QuotebN

        //Quoteŏ㏑
        public void ResetByQuote(Quote record) {
            lock(this) {
                if(record.Count==0) return;

                int last_time = record.LastCandle.Time;
                AssureQuoteSize(last_time);
                for(int i=0; i<record.Count; i++)
                    ImportCandle(record.CandleAt(i));

                //LastMinuteCandleݍ܂ꂽƂɂ...
                if(!_lastMinuteCandle.IsNil && _lastMinuteCandle.Time<=last_time) {
                    ResetLastMinuteCandle(last_time+_intervalSec);
                }
            }
        }
        //TimeAndSalesŏ㏑
        //end_time͕PʂŊm肵ŐVw肷B
        //̎Ȃ炿傤Ǒ邵AłȂend_time𒴉߂TickData_lastMinuteɕғ
        public void ResetByTimeAndSales(TimeAndSales ts, BTime end_time) {
            Debug.Assert(end_time.Second==0);
            lock(this) {
                ConcreteQuote r = new ConcreteQuote(Quote.QuoteUnit.Minutely);
                _timeAndSales = new TimeAndSales();

                MarketOpenTimeRange time_range = MarketOpenTimeRange.GetFor(_stock);
                int time = time_range.StartTimeAt(0);
                int tick_index = 0;
                int end;
                bool valid_candle_created = false;
                Candle yonhonne = null;
                while(time < end_time.AsInt()) {
                    if(FindTickRange(ts, time, tick_index, out end)) {
                        yonhonne = CreateMinutelyCandle(ts, time, tick_index, end);
                        tick_index = end; //͂`FbNJn
                        valid_candle_created = true;
                    }
                    else {
                        yonhonne = new Candle(r, r.Count, time, tick_index==0? 0 : ts.TickAt(tick_index-1).Price, 0);
                        if(!valid_candle_created) yonhonne.Flags = CandleFlags.Nil; //܂̂łĂȂƂNilƂ
                    }

                    r.Add(yonhonne);
                    time += _intervalSec;
                    int section_index;
                    if(time_range.IsHikeTime(time, out section_index) && section_index<time_range.SectionCount-1)
                        time = time_range.StartTimeAt(section_index+1);
                }

                _minutely = r;
                _fixedLength = _minutely.Count;

                if(FindTickRange(ts, time, tick_index, out end)) 
                    _lastMinuteCandle = CreateMinutelyCandle(ts, time, tick_index, end);
                else
                    _lastMinuteCandle.Let(CandleType.instance.Nil);
            }
        }
        //timen܂Ptick_indexTickDatatrueԂȀI[endɊi[
        private bool FindTickRange(TimeAndSales ts, int time, int start_index, out int end) {
            int index = start_index;
            end = start_index;
            TickData td = ts.TickAtOrNull(index);
            if(td==null || td.Time < time) return false;

            int next_time = time + _intervalSec;
            bool inclusive = IsHikeTime(next_time);
            while(td!=null && (inclusive? (td.Time <= next_time) : (td.Time < next_time))) {
                index++;
                td = ts.TickAtOrNull(index);
            }
            end = index;
            return index!=start_index;
        }
        //TimeAndSalesCfNX͈͂w肵ĂPԂ̃f[^B[JTimeAndSalesɂC|[ĝŒӁBResetByTimeAndSalesĂ΂
        private Candle CreateMinutelyCandle(TimeAndSales ts, int time, int begin, int end) {
            int index = begin;
            TickData td = ts.TickAt(index++);
            _timeAndSales.Add(new TickData(td));
            Candle y = new Candle(_minutely, -1, time, td.Price, td.Volume); //TODO indexvZ
            while(index < end) {
                td = ts.TickAt(index++);
                _timeAndSales.Add(new TickData(td));
                y.Update(td.Price, td.Volume);
            }
            return y;
        }

        //PɂXV
        public void UpdateBySingleTick(TickData tick) {
            lock(this) {
                ImportSingleTick(tick);
            }
            EndOfUpdate();
        }
        public void UpdateByTicks(TimeAndSales ts) { 
            lock(this) {
                for(int i=0; i<ts.Count; i++)
                    ImportSingleTick(ts.TickAt(i));
            }
            EndOfUpdate();
        }
        private void ImportSingleTick(TickData tick) {
            int time = tick.Time;
            CheckCorrectTime(time, true);
            if(time >= GetNextMinute()) {
                if(IsHikeTime(time)) {
                    if(_lastMinuteCandle.IsNil || _lastMinuteCandle.Volume==0)
                        _lastMinuteCandle.Set(_minutely, _minutely.Count-1, time-60, tick.Price, tick.Volume); //_lastMinuteCandleNil͂Tick݂̂Ȃ郍E\N͂POBz̃P[X
                    else
                        _lastMinuteCandle.Update(tick.Price, tick.Volume); //Tick͂Ŏ荞ށBƎInternalUpdateByTimeœo^
                }

                InternalUpdateByTime(time); //XV
            }

            _timeAndSales.Add(tick);

            int index = MarketOpenTimeRange.GetFor(_stock).TimeToIndex(time);
            if(index < _fixedLength) { //fixedLengthȉ̂Ƃ͑kXV
                Candle c = _minutely.CandleAt(index);
                if(c.IsNil || c.Volume==0) //̑ɑ΂XV
                    c.Set(_minutely, index, c.Time, tick.Price, tick.Volume);
                else
                    c.Update(tick.Price, tick.Volume);
            }
            else if(!IsHikeTime(time)) {
                //_lastMinuteCandlȅȂ炱tick̒ĝōXV
                if(_lastMinuteCandle.IsNil || _lastMinuteCandle.Volume==0) //Ȍ̕Il+oOƂ`_lastMinuteCandleP[X݂
                    _lastMinuteCandle.Set(_minutely, index, BTime.AsMinutelyInt(tick.Time), tick.Price, tick.Volume);
                else
                    _lastMinuteCandle.Update(tick.Price, tick.Volume);
            }
        }

        //O񋟂̎ɂXV
        public void UpdateByTime(BTime time) {
            if(!IsCorrectTime(time.AsInt(), true)) return;

            lock(this) {
                InternalUpdateByTime(time.AsInt());
            }
            EndOfUpdate();
        }
        private void InternalUpdateByTime(int time) {
            //LastMinutě`FbN
            bool valid_candle_exists = _minutely.Count>0 && !_minutely.LastCandle.IsNil;
            if(!_lastMinuteCandle.IsNil && _lastMinuteCandle.Time+_intervalSec <= time) {
                Candle c = new Candle(_lastMinuteCandle);
                c.Flags = (valid_candle_exists || c.Volume>0)? CandleFlags.None : CandleFlags.Nil;
                _minutely.Add(c);
                _lastMinuteCandle.Let(CandleType.instance.Nil);
                valid_candle_exists = !c.IsNil;
            }

            //ԂWv邱Ƃ̂œK؂ɖ߂
            int next_minute = GetNextMinute();
            while(next_minute + _intervalSec <= time) {
                int p = _minutely.Count>0? _minutely.LastCandle.Close : 0; //f[^ȂƂ0
                Candle c = _minutely.Add(next_minute, p, 0); //oO
                if(!valid_candle_exists) c.Flags = CandleFlags.Nil;

                Debug.WriteLineIf(BDebugOpts.TRACE_MINUTELY_QUOTE, String.Format("Minutely Quote code={0} time={1} price={2}", this.Stock.Profile.Code, BTime.DefaultFormat(new BTime(_minutely.LastCandle.Time)), _minutely.LastCandle.Close));
                next_minute = GetNextMinute();
            }

            if(_lastMinuteCandle.IsNil && next_minute<=time && !IsHikeTime(time)) {
                ResetLastMinuteCandle(next_minute);
            }

            _fixedLength = Math.Max(_fixedLength, _minutely.Count);
        }

        //S{l㏑BFixedB
        private void ImportCandle(Candle y) {
            CheckCorrectTime(y.Time, false);
            int index = MarketOpenTimeRange.GetFor(_stock).TimeToIndex(y.Time);
            Candle t = _minutely.CandleAt(index);
            if(t==null)
                _minutely.SetDirect(index, new Candle(y));
            else
                t.Let(y);
            _fixedLength = Math.Max(_fixedLength, index+1);
        }
        private void CheckCorrectTime(int time, bool allow_close) { //allow_closéAOE̎ǂ̎w
            if(!IsCorrectTime(time, allow_close)) {
                throw new BellagioException(String.Format("Invalid time {0}", time));
            }
        }
        private bool IsCorrectTime(int time, bool allow_close) {
            bool ok;
            StockExchange m = this.Stock.Market;
            int zo = MarketUtil.GetZenbaOpenTime(m).AsInt();
            int zc = MarketUtil.GetZenbaCloseTime(m).AsInt();
            int go = MarketUtil.GetGobaOpenTime(m).AsInt();
            int gc = MarketUtil.GetGobaCloseTime(m).AsInt();
            int eo = MarketUtil.GetEveningOpenTime(m).AsInt();
            int ec = MarketUtil.GetEveningCloseTime(m).AsInt();
            bool derivative = _stock.Profile.IsDerivative;

            if(time < eo)
                ok = false;
            else if(derivative && time < ec)
                return true;
            else if(derivative && time==ec)
                return allow_close;
            else if(time < zo)
                ok = false;
            else if(time < zc)
                ok = true;
            else if(time==zc)
                ok = allow_close;
            else if(time < go)
                ok = false;
            else if(time < gc)
                ok = true;
            else if(time==gc)
                ok = allow_close;
            else
                ok = false;

            return ok;
        }
        private void AssureQuoteSize(int time) {
            CheckCorrectTime(time, true);
            _minutely.SetSize(MarketOpenTimeRange.GetFor(_stock).TimeToIndex(time)+1);
        }
        private void ResetLastMinuteCandle(int time) {
            //ɏ]LastMinuteCandleXV
            if(IsHikeTime(time)) //̂ƂLastMinuteCandle݂Ȃ悤ɂ
                _lastMinuteCandle.Let(CandleType.instance.Nil);
            else if(_minutely.Count > 0) //Ƃ͒ȌIlp
                _lastMinuteCandle.Set(_minutely, _minutely.Count, time, _minutely.LastCandle.Close, 0);
            else if(!_currentPriceInfo.LastClose.IsEmpty) //ꂪȂƂ͑OIlőpAȂƂ݂͑Ȃ
                _lastMinuteCandle.Set(_minutely, MarketOpenTimeRange.GetFor(_stock).TimeToIndex(time), time, _currentPriceInfo.LastClose.Value, 0);
            else
                _lastMinuteCandle.Let(CandleType.instance.Nil); //not available
        }

        private void EndOfUpdate() {
            BellagioRoot.DataSubscriberManager.ReserveExecByStock(this.Stock, SubscribeDataKind.Ticks);
        }


        //̏ԂÂPn܂邩vZ
        private int GetNextMinute() {
            if(!_lastMinuteCandle.IsNil)
                return _lastMinuteCandle.Time + _intervalSec;
            else {
                MarketOpenTimeRange time_range = MarketOpenTimeRange.GetFor(_stock);

                if(_minutely.Count==0)
                    return time_range.StartTimeAt(0);
                else {
                    int it = _minutely.LastCandle.Time + _intervalSec;
                    for(int i=0; i<time_range.SectionCount; i++) {
                        if(it < time_range.StartTimeAt(i))
                            return time_range.StartTimeAt(i);
                        else if(it < time_range.EndTimeAt(i))
                            return it;
                    }
                    return BTime.MaxAsInt;
                }
            }
        }
        private bool IsHikeTime(int time) {
            StockExchange m = this.Stock.Market;
            return time==MarketUtil.GetZenbaCloseTime(m).AsInt() || time==MarketUtil.GetGobaCloseTime(m).AsInt() || time==MarketUtil.GetEveningCloseTime(m).AsInt();
        }
        //Time&Sales擾͈͎̔̏I[CBʏ́AuObv͎̑AꂪƂ͍Ō̑Ɋ܂ށBႦ΁A15:00TickData14:59̑Ɋ܂܂B
        private int ModifyMinuteEndTimeForTS(int time) {
            StockExchange m = this.Stock.Market;
            if(IsHikeTime(time))
                return time+1; //Pb点΂낵
            else
                return time;
        }
    }



#if UNITTEST
    [TestFixture]
    public class IntraDayTradeTests {
        private FixedDataStock _stock;
        private IntraDayTrade _quotes;

        [TestFixtureSetUp]
        public void Setup() {
            BellagioEnvironmentParam p = new BellagioEnvironmentParam();
            p.SetupForUnitTest();
            p.RUN_DATA_SUBSCRIBER_ENGINE = true;
            p.USE_DEMONSTRATION_STOCKS = false;
            p.UNITTEST_STOCK_COUNT = 0;
            BellagioRoot.Init(p);
            BellagioRoot.BootForTest();
            _stock = new FixedDataStock("1", "1", StockExchange.T, 1000, new TimeAndSales());
            BellagioRoot.GlobalStockCollection.Add(_stock);

            BellagioRoot.IntraDayTradeProvider.SetTickDataSource(new UnitTestSimpleDataSource());

            //GetOrOpenSubscribero^OKȂƂɒ
            string message = null;
            BellagioRoot.IntraDayTradeProvider.Open(_stock.Primary, ref _quotes, ref message);
            Debug.Assert(_quotes!=null);
        }
        [TestFixtureTearDown]
        public void TearDown() {
            BellagioRoot.DataSubscriberManager.Terminate();
        }

        private void ResetTimeAndSales(BTime current, string text_ts) {
            _quotes.Reset();
            TextReader strm = new StringReader(text_ts);
            _stock.Reset(new TextTimeAndSalesReader().Read(strm));
            _stock.Run();
            strm.Close();

            SetTime(current);
            Assert.AreEqual(1, BellagioRoot.DataSubscriberManager.SubscriberCount); //Quotê݁AȑO͔
        }
        private void SetTime(BTime time) {
            BellagioRoot.TimeManager.ManualSetTime(time);
            _quotes.UpdateByTime(time);
        }

        [Test]
        public void Basic() {
            ResetTimeAndSales(new BTime(8, 59, 0), "9:00:00 100 1000\n9:00:30 101 1000\n9:01:00 102 1000");

            BTime time = new BTime(9, 0, 0);
            SetTime(time);
            Assert.AreEqual(1, _quotes.TimeAndSales.Count); //9:00:00̂P͂
            Assert.AreEqual(0, _quotes.Minutely.Count); //͂܂ĂȂ
            Assert.AreEqual(time.AsInt(), _quotes.LastMinuteCandle.Time); //݂S{l͂Ă͂
            Assert.AreEqual(100, _quotes.LastMinuteCandle.Close);
            Assert.AreEqual(1000, _quotes.LastMinuteCandle.Volume);
            Assert.AreEqual(0, _quotes.FixedLength);
            
            time.AddSec(30);
            SetTime(time);
            Assert.AreEqual(2, _quotes.TimeAndSales.Count); //9:00:30̂P͂
            Assert.AreEqual(0, _quotes.Minutely.Count); //͂܂ĂȂ
            Assert.AreEqual(time.AsInt(), _quotes.LastMinuteCandle.Time+30); //ɂ͕ωȂ
            Assert.AreEqual(101, _quotes.LastMinuteCandle.Close);
            Assert.AreEqual(2000, _quotes.LastMinuteCandle.Volume);  //XV͂Ă
            Assert.AreEqual(0, _quotes.FixedLength);

            time.AddSec(30);
            SetTime(time);
            Assert.AreEqual(3, _quotes.TimeAndSales.Count); //9:01:00
            Assert.AreEqual(1, _quotes.Minutely.Count); //
            Candle y = _quotes.Minutely.CandleAt(0);
            Assert.AreEqual(time.AsInt()-60, y.Time);
            Assert.AreEqual(100, y.Open);
            Assert.AreEqual(100, y.Low);
            Assert.AreEqual(101, y.High);
            Assert.AreEqual(101, y.Close);
            Assert.AreEqual(2000, y.Volume);
            //݂S{l9:01:00̂P݂̂̂̂Ă͂
            Assert.AreEqual(102, _quotes.LastMinuteCandle.Close);
            Assert.AreEqual(1000, _quotes.LastMinuteCandle.Volume);
            Assert.AreEqual(time.AsInt(), _quotes.LastMinuteCandle.Time);
            Assert.AreEqual(1, _quotes.FixedLength);

            Assert.AreEqual(1, BellagioRoot.TimeManager.TimerListenerCount); //ImF(DataSubscriberManager̂)

            //łQʒmƂĂ͋NȂƂmF
            SetTime(time);
            Assert.AreEqual(3, _quotes.TimeAndSales.Count); //9:01:00
            Assert.AreEqual(1, _quotes.Minutely.Count); //
        }

        
        [Test]
        public void Sukima() {
            //oONil̑}邱ƂmF
            ResetTimeAndSales(new BTime(9, 0, 0), "9:10:00 100 1000\n9:10:30 101 1000\n9:14:00 102 3000");
            Assert.AreEqual(0, _quotes.TimeAndSales.Count);

            BTime time = new BTime(9, 10, 0);
            SetTime(time);
            Assert.AreEqual(1, _quotes.TimeAndSales.Count);
            Assert.AreEqual(10, _quotes.Minutely.Count); //܂܂ł0910{
            for(int i=0; i<_quotes.Minutely.Count; i++ ) {
                Candle c = _quotes.Minutely.CandleAt(i);
                Assert.IsTrue(c.IsNil);
                Assert.AreEqual(0, c.Volume);
            }

            time.AddSec(120); //9:12
            SetTime(time);
            Assert.AreEqual(2, _quotes.TimeAndSales.Count);
            Assert.AreEqual(12, _quotes.Minutely.Count);
            Candle y = _quotes.Minutely.LastCandle;
            Assert.AreEqual(time.AsInt()-60, y.Time);
            Assert.AreEqual(0, y.Volume);
            Assert.AreEqual(101, y.Open); //S{l͒O̒l
            Assert.AreEqual(101, y.High);
            Assert.AreEqual(101, y.Low);
            Assert.AreEqual(101, y.Close);
            
            time.AddSec(180); //9:15
            SetTime(time);
            Assert.AreEqual(3, _quotes.TimeAndSales.Count);
            Assert.AreEqual(15, _quotes.Minutely.Count);
            y = _quotes.Minutely.LastCandle;
            Assert.AreEqual(3000, y.Volume);
            Assert.AreEqual(102, y.Open);
            Assert.AreEqual(102, y.High);
            Assert.AreEqual(102, y.Low);
            Assert.AreEqual(102, y.Close);
            Assert.AreEqual(15, _quotes.FixedLength);

            Assert.AreEqual(1, BellagioRoot.TimeManager.TimerListenerCount); //ImF
        }

        [Test]
        public void LocalDataTest() {
            //Time&Sales𒼐ڗ^ĈCQuotes쐬邱ƂmFBꂪłȂƃfobNeXgŎ
            BTime time = new BTime(9, 0, 0);
            ResetTimeAndSales(time, "9:00:00 100 1000\n9:00:10 101 1000\n9:01:30 102 3000\n9:01:50 103 2000");
            
            time.AddSec(120); //9:02
            SetTime(time);
            Assert.AreEqual(4, _quotes.TimeAndSales.Count);
            Assert.AreEqual(2, _quotes.Minutely.Count);
            Candle y = _quotes.Minutely.CandleAt(0);
            Assert.AreEqual(time.AsInt()-120, y.Time);
            Assert.AreEqual(2000, y.Volume);
            Assert.AreEqual(100, y.Open);
            Assert.AreEqual(101, y.High);
            Assert.AreEqual(100, y.Low);
            Assert.AreEqual(101, y.Close);

            y = _quotes.Minutely.CandleAt(1);
            Assert.AreEqual(time.AsInt()-60, y.Time);
            Assert.AreEqual(5000, y.Volume);
            Assert.AreEqual(102, y.Open); //S{l͒O̒l
            Assert.AreEqual(103, y.High);
            Assert.AreEqual(102, y.Low);
            Assert.AreEqual(103, y.Close);
            Assert.AreEqual(2, _quotes.FixedLength);

            Assert.AreEqual(1, BellagioRoot.TimeManager.TimerListenerCount); //ImF

        }

        [Test]
        public void CloseTest() {
            //ɊւT&SĂ邱ƂmF
            BTime time = new BTime(8, 59, 0);
            ResetTimeAndSales(time, "9:00:00 100 1000\n10:58:00 100 1000\n10:59:00 100 1000\n11:00:00 101 1000");
            Assert.IsFalse(_quotes.IsLastMinuteCandleAvailable); //psɂȂĂ

            time = new BTime(10, 59, 0);
            SetTime(time);
            Assert.AreEqual(3, _quotes.TimeAndSales.Count);
            Assert.AreEqual(119, _quotes.Minutely.Count); //10:59̑ȊO͂łĂ͂
            Assert.IsTrue(_quotes.IsLastMinuteCandleAvailable);
            Assert.AreEqual(119, _quotes.FixedLength);

            time.AddSec(60); //11:00ɂȂ
            SetTime(time);
            Assert.AreEqual(4, _quotes.TimeAndSales.Count);
            Assert.AreEqual(120, _quotes.Minutely.Count);
            Candle y = _quotes.Minutely.LastCandle;
            Assert.AreEqual(101, y.Close);
            Assert.AreEqual(2000, y.Volume); //11:00̃f[^܂ł
            Assert.IsFalse(_quotes.IsLastMinuteCandleAvailable); //psɂȂĂ
            Assert.AreEqual(120, _quotes.FixedLength);

            time.AddSec(60); //11:01
            SetTime(time);
            Assert.AreEqual(120, _quotes.Minutely.Count);
            Assert.IsFalse(_quotes.IsLastMinuteCandleAvailable);

            time = new BTime(12, 30, 0);
            SetTime(time);
            Assert.IsTrue(_quotes.IsLastMinuteCandleAvailable); //ōŐV͂
            Assert.AreEqual(0, _quotes.LastMinuteCandle.Volume);
            Assert.AreEqual(time.AsInt(), _quotes.LastMinuteCandle.Time);
            Assert.AreEqual(101, _quotes.LastMinuteCandle.Open); //OIl
            Assert.AreEqual(120, _quotes.Minutely.Count);
            Assert.AreEqual(120, _quotes.FixedLength);

            time.AddSec(60); //12:31
            SetTime(time);
            Assert.AreEqual(121, _quotes.Minutely.Count);
            Assert.IsTrue(_quotes.IsLastMinuteCandleAvailable);
            Assert.AreEqual(121, _quotes.FixedLength);

            time = new BTime(14, 59, 0);
            SetTime(time);
            Assert.AreEqual(101, _quotes.LastMinuteCandle.Open); //܂OIl
            Assert.AreEqual(269, _quotes.FixedLength);
            Assert.IsTrue(_quotes.IsLastMinuteCandleAvailable);

            time = new BTime(15, 0, 0);
            SetTime(time);
            Assert.AreEqual(270, _quotes.Minutely.Count);
            Assert.AreEqual(101, _quotes.Minutely.LastCandle.Close);
            Assert.AreEqual(270, _quotes.FixedLength);
            Assert.IsFalse(_quotes.IsLastMinuteCandleAvailable);

            time.AddSec(60*10); //15:10
            SetTime(time);
            Assert.IsFalse(_quotes.IsLastMinuteCandleAvailable);
        }

        private void ResetDirectTimeAndSales(BTime time, string text) {
            TextReader strm = new StringReader(text);
            TimeAndSales ts = new TextTimeAndSalesReader().Read(strm);
            strm.Close();
            _quotes.Reset();

            _quotes.ResetByTimeAndSales(ts, time);
        }

        [Test]
        public void DirectTimeAndSales1() {
            Candle y;
            //ӂɊm肵P
            ResetDirectTimeAndSales(new BTime(9, 1, 0), "9:00:00 100 1000\n9:00:10 101 1000\n9:00:20 102 3000");
            Assert.IsTrue(_quotes.LastMinuteCandle.IsNil);
            Assert.AreEqual(1, _quotes.Minutely.Count);
            y = _quotes.Minutely.CandleAt(0);
            Assert.AreEqual(3600*9, y.Time);
            Assert.AreEqual(100, y.Open);
            Assert.AreEqual(102, y.High);
            Assert.AreEqual(100, y.Low);
            Assert.AreEqual(102, y.Close);
            Assert.AreEqual(5000, y.Volume);


            //ӂɊm肵T@f[^͒Ȃ̂Œ
            ResetDirectTimeAndSales(new BTime(9, 5, 0), "9:01:00 100 1000\n9:02:00 101 1000\n9:03:00 102 3000");

            Assert.IsTrue(_quotes.LastMinuteCandle.IsNil);
            Assert.AreEqual(5, _quotes.Minutely.Count);
            y = _quotes.Minutely.CandleAt(0);
            Assert.AreEqual(3600*9, y.Time);
            Assert.AreEqual(0, y.Volume);

            y = _quotes.Minutely.CandleAt(1);
            Assert.AreEqual(3600*9 + 60, y.Time);
            Assert.AreEqual(100, y.Open);
            Assert.AreEqual(1000, y.Volume);

            y = _quotes.Minutely.CandleAt(2);
            Assert.AreEqual(3600*9 + 120, y.Time);
            Assert.AreEqual(101, y.Open);
            Assert.AreEqual(1000, y.Volume);

            y = _quotes.Minutely.CandleAt(3);
            Assert.AreEqual(3600*9 + 180, y.Time);
            Assert.AreEqual(102, y.Open);
            Assert.AreEqual(3000, y.Volume);

            y = _quotes.Minutely.CandleAt(4);
            Assert.AreEqual(3600*9 + 240, y.Time);
            Assert.AreEqual(0, y.Volume);
        }
        [Test]
        public void DirectTimeAndSales2() {
            Candle y;
            //O
            ResetDirectTimeAndSales(new BTime(11, 0, 0), "9:00:00 100 1000\n10:59:00 101 1000\n11:00:00 102 3000");
            Assert.IsTrue(_quotes.LastMinuteCandle.IsNil);
            Assert.AreEqual(120, _quotes.Minutely.Count);
            y = _quotes.Minutely.LastCandle;
            Assert.AreEqual(new BTime(10, 59, 0).AsInt(), y.Time);
            Assert.AreEqual(102, y.Close);
            Assert.AreEqual(4000, y.Volume);

            //
            ResetDirectTimeAndSales(new BTime(12, 31, 0), "10:59:00 101 1000\n11:00:00 102 3000\n12:30:00 110 10000");
            Assert.IsTrue(_quotes.LastMinuteCandle.IsNil);
            Assert.AreEqual(121, _quotes.Minutely.Count);
            for(int i=0; i<119; i++)
                Assert.IsTrue(_quotes.Minutely.CandleAt(i).IsNil);
            y = _quotes.Minutely.CandleAt(_quotes.Minutely.FindExactly(new BTime(10, 59, 0).AsInt()));
            Assert.AreEqual(new BTime(10, 59, 0).AsInt(), y.Time);
            Assert.AreEqual(102, y.Close);
            Assert.AreEqual(4000, y.Volume);
            y = _quotes.Minutely.LastCandle;
            Assert.AreEqual(new BTime(12, 30, 0).AsInt(), y.Time);
            Assert.AreEqual(110, y.Close);
            Assert.AreEqual(10000, y.Volume);

            //ꂩ̃X^[g
            ResetDirectTimeAndSales(new BTime(12, 45, 0), "12:40:00 101 1000");
            Assert.IsTrue(_quotes.LastMinuteCandle.IsNil);
            Assert.AreEqual(135, _quotes.Minutely.Count);
            y = _quotes.Minutely.LastCandle;
            Assert.AreEqual(101, y.Close);
            Assert.AreEqual(0, y.Volume);
        }

        [Test]
        public void DirectTimeAndSales3() {
            Candle y;
            //ŐVS{l邱ƂmF
            ResetDirectTimeAndSales(new BTime(9, 1, 0), "9:00:00 100 1000\n9:01:00 101 1000\n9:01:30 102 1000");
            Assert.IsFalse(_quotes.LastMinuteCandle.IsNil);
            y = _quotes.LastMinuteCandle;
            Assert.AreEqual(new BTime(9, 1, 0).AsInt(), y.Time);
            Assert.AreEqual(101, y.Open);
            Assert.AreEqual(102, y.Close);
            Assert.AreEqual(2000, y.Volume);

        }

        [Test]
        public void ReverseOrder() {
            BTime time = new BTime(9, 1, 0);
            ResetTimeAndSales(time, "9:00:00 100 1000\n9:00:30 101 1000\n9:01:00 102 1000");
            //9:01܂ł̃f[^ʏ̌`łłĂ邱ƂmF
            Assert.AreEqual(3, _quotes.TimeAndSales.Count); 
            Assert.AreEqual(1, _quotes.Minutely.Count); 
            Assert.AreEqual(time.AsInt(), _quotes.LastMinuteCandle.Time);
            Candle c = _quotes.Minutely.CandleAt(0);
            Assert.AreEqual(101, c.Close);
            Assert.AreEqual(2000, c.Volume);
            Assert.AreEqual(1, _quotes.FixedLength);
            
            //ł̂ڂ9:00TickƂ
            _quotes.UpdateBySingleTick(new TickData(new BTime(9, 0, 40).AsInt(), 102, 1000, TickItaRelation.Unknown));
            Assert.AreEqual(4, _quotes.TimeAndSales.Count);
            Assert.AreEqual(1, _quotes.Minutely.Count);
            c = _quotes.Minutely.CandleAt(0);
            Assert.AreEqual(102, c.Close); //Ă͂
            Assert.AreEqual(3000, c.Volume);
            Assert.AreEqual(1, _quotes.FixedLength);

            //Ǝ߂...
            time.AddMin(9);
            SetTime(time);
            Assert.AreEqual(10, _quotes.Minutely.Count); //9:10ɂȂ̂10{łĂ͂
            c = _quotes.Minutely.CandleAt(8);
            _quotes.UpdateBySingleTick(new TickData(new BTime(9, 8, 30).AsInt(), 104, 1000, TickItaRelation.Unknown));
            Assert.AreEqual(1000, c.Volume);
            Assert.AreEqual(104, c.Open);
            Assert.AreEqual(new BTime(9, 8, 0).AsInt(), c.Time);
            
        }
    }

#endif
}
