/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Drawing/DrawingSettings.cs#4 $
 * $DateTime: 2008/02/22 14:44:40 $
 * 
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Diagnostics;

using Poderosa.Preferences;
using Poderosa;
using Bellagio.Environment;

//PoderosaPreference@\ɂ̂ƂADrawingSettingsn̊{@\

namespace Bellagio.Drawing {
    //ݒ\ڂ̊{
    public abstract class DSItem {
        protected IPreferenceItem _pref;
        protected bool _dirty;
        public DSItem(IPreferenceItem pref) {
            _pref = pref;
            _dirty = true;
        }
        public abstract void Refresh(); //Preferenceeڂւ̔f
        public abstract void Dispose();
        public IPreferenceItem UnderlyingItem {
            get {
                return _pref;
            }
        }
    }

    public abstract class DSGDIItem : DSItem {
        protected IStringPreferenceItem _strPref;
        public DSGDIItem(IStringPreferenceItem sp)
            : base(sp) {
            _strPref = sp;
        }
        public IStringPreferenceItem PreferenceItem {
            get {
                return _strPref;
            }
        }
    }

    //ݒ\ȐFƁAɊÂuV
    public class DSBrush : DSGDIItem {
        private SmartColor _color;

        public DSBrush(IStringPreferenceItem pref)
            : base(pref) {
        }

        public Brush B {
            get {
                if(_dirty) Refresh();
                return _color.B;
            }
        }
        public Color col {
            get {
                if(_dirty) Refresh();
                return _color.col;
            }
        }

        public override void Refresh() {
            Color c = Color.Empty;
            DrawingUtil.TryParseColor(_strPref.Value, out c);
            _color.Set(c);
            _dirty = false;
        }

        public override void Dispose() {
            _color.Clear();
        }
    }

    public class DSFont : DSGDIItem {
        private SmartFont _font;
        public DSFont(IStringPreferenceItem pref)
            : base(pref) {
        }

        public Font F {
            get {
                if(_dirty) Refresh();
                return _font.F;
            }
        }
        public SmartFont SmartFont {
            get {
                return _font;
            }
        }
        public override void Refresh() {
            SmartFont nv = SmartFont.Parse(_strPref.Value);
            _font.Set(nv);

            _dirty = false;
        }
        public override void Dispose() {
            _font.Clear();
        }

        //悭g
        public const string GenericFontName = "lr SVbN";
    }

    public class DSPenStyle : DSGDIItem {
        private BPenStyle _penStyle;
        public DSPenStyle(IStringPreferenceItem pref)
            : base(pref) {
        }
        public Color Color {
            get {
                if(_dirty) Refresh();
                return _penStyle.color;
            }
        }
        public int Width {
            get {
                if(_dirty) Refresh();
                return _penStyle.width;
            }
        }

        public Pen P {
            get {
                if(_dirty) Refresh();
                return _penStyle.Pen;
            }
        }
        public override void Refresh() {
            if(_dirty) _penStyle.Clear();
            _penStyle = BPenStyle.Parse(_strPref.Value);
            _dirty = false;
        }
        public override void Dispose() {
            _penStyle.Clear();
        }
    }

    public class DSInt : DSItem {
        private int _value;
        private IIntPreferenceItem _intPref;
        public DSInt(IIntPreferenceItem pref) : base(pref) {
            _intPref = pref;
        }

        public int V {
            get {
                if(_dirty) Refresh();
                return _intPref.Value;
            }
        }
        public override void Refresh() {
            _value = _intPref.Value;
            _dirty = false;
        }
        public override void Dispose() {
            
        }

    }

    public abstract class DrawingSettingsBase : IPreferenceSupplier, IPreferenceChangeListener {
        private List<DSItem> _prefElements;
        private string _preferenceID;
        private IPreferenceBuilder _builder;
        private IPreferenceFolder _folder;

        public DrawingSettingsBase(string id) {
            _prefElements = new List<DSItem>();
            _preferenceID = id;
        }

        protected abstract void CreateElements();

        public virtual void Refresh() {
            foreach(DSItem item in _prefElements) item.Refresh();
        }
        public virtual void Dispose() {
            foreach(DSItem item in _prefElements) item.Dispose();
        }

        protected DSBrush DefineBrush(string name, Color initial) {
            Debug.Assert(!String.IsNullOrEmpty(name));
            DSBrush v = new DSBrush(_builder.DefineStringValue(_folder, name, DrawingUtil.ColorToString(initial), _colorValidator));
            _prefElements.Add(v);
            return v;
        }
        protected DSFont DefineFont(string name, string fontName, float fontSize) {
            Debug.Assert(!String.IsNullOrEmpty(name));
            DSFont v = new DSFont(_builder.DefineStringValue(_folder, name, new StringBuilder().Append(fontName).Append(';').Append(fontSize).ToString(), null));
            _prefElements.Add(v);
            return v;
        }
        protected DSFont DefineFont(string name, string fontName, float fontSize, FontStyle style) {
            Debug.Assert(!String.IsNullOrEmpty(name));
            StringBuilder bld = new StringBuilder().Append(fontName).Append(';').Append(fontSize);
            if((style & FontStyle.Bold)!=FontStyle.Regular) bld.Append(';').Append("bold");
            if((style & FontStyle.Italic)!=FontStyle.Regular) bld.Append(';').Append("italic");
            DSFont v = new DSFont(_builder.DefineStringValue(_folder, name, bld.ToString(), _fontValidator));
            _prefElements.Add(v);
            return v;
        }
        protected DSInt DefineInt(string name, int initial) {
            Debug.Assert(!String.IsNullOrEmpty(name));
            DSInt v = new DSInt(_builder.DefineIntValue(_folder, name, initial, null));
            _prefElements.Add(v);
            return v;
        }
        protected DSInt DefinePositiveInt(string name, int initial) {
            Debug.Assert(!String.IsNullOrEmpty(name));
            DSInt v = new DSInt(_builder.DefineIntValue(_folder, name, initial,
                delegate(int value, IPreferenceValidationResult result) {
                    if(value<=0) result.ErrorMessage = "̒l͂Ă";
                }));
            _prefElements.Add(v);
            return v;
        }
        protected DSPenStyle DefinePenStyle(string name, Color col, BPenStyle.DashStyle dash, int width) {
            Debug.Assert(!String.IsNullOrEmpty(name));
            DSPenStyle v = new DSPenStyle(_builder.DefineStringValue(_folder, name, new BPenStyle(col, dash, width).ToString(), _penValidator));
            _prefElements.Add(v);
            return v;

        }

        protected IPreferenceFolder GetFolder() {
            return _folder;
        }

        public string PreferenceID {
            get {
                return _preferenceID;
            }
        }
        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        public void InitializePreference(IPreferenceBuilder builder, IPreferenceFolder folder) {
            _builder = builder;
            _folder = folder;
            folder.AddChangeListener(this);
            CreateElements();
        }

        public object QueryAdapter(IPreferenceFolder folder, Type type) {
            return null;
        }

        public void ValidateFolder(IPreferenceFolder folder, IPreferenceValidationResult output) {
        }


        public void OnPreferenceImport(IPreferenceFolder oldvalues, IPreferenceFolder newvalues) {
        }

        public virtual void OnPreferenceImportFinish() {
            this.Refresh();
        }

        //Validator
        private static PreferenceItemValidator<string> _colorValidator = new PreferenceItemValidator<string>(ValidateColor);
        private static PreferenceItemValidator<string> _fontValidator = new PreferenceItemValidator<string>(ValidateFont);
        private static PreferenceItemValidator<string> _penValidator = new PreferenceItemValidator<string>(ValidatePen);

        private static void ValidateColor(string value, IPreferenceValidationResult result) {
            Color t;
            if(!DrawingUtil.TryParseColor(value, out t)) {
                result.ErrorMessage = "F̎w肪܂";
            }
        }
        private static void ValidateFont(string value, IPreferenceValidationResult result) {
            SmartFont f;
            if(!SmartFont.TryParse(value, out f)) {
                result.ErrorMessage = "tHg̎w肪܂";
            }
        }
        private static void ValidatePen(string value, IPreferenceValidationResult result) {
            BPenStyle ps;
            if(!BPenStyle.TryParse(value, out ps)) {
                result.ErrorMessage = "y̎w肪܂";
            }
        }

    }
}
