﻿/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id$
 * $DateTime$
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using System.Drawing;
using System.Xml;

using Poderosa;
using Poderosa.Plugins;
using Poderosa.Commands;
using Poderosa.Forms;
using Poderosa.Sessions;
using Poderosa.View;
using Poderosa.Preferences;

using Bellagio.Environment;
using Bellagio.Data;
using Bellagio.Values;

[assembly: PluginDeclaration(typeof(Bellagio.ExternalData.ExternalDataPlugin))]

/*
ExternalDataは、外部から取得するデータの基礎部分を提供するプラグイン
MarketWebからデータを取得する拡張の件がきっかけで作成。
ホスト側ではオブジェクトレベルでのデータの検索、追加・削除のサービスと、シリアライズの基礎を扱う。
ゲスト側では外部データの実際の取得、要素のシリアライズ、スクリーニングからアクセスするための関数登録関係を行う
*/

namespace Bellagio.ExternalData {
    [PluginInfo(ID=ExternalDataPlugin.PLUGIN_ID, Version=BellagioVersionInfo.PLUGIN_VERSION, Author=BellagioVersionInfo.AUTHOR_NAME, Dependencies=BellagioPlugin.PLUGIN_ID)]
    public class ExternalDataPlugin : PluginBase, IMainWindowEventHandler, IExternalDataManager {
        public const string PLUGIN_ID = "jp.co.lagarto.tactico.externalData";
        private static ExternalDataPlugin _instance;

        private IExtensionPoint _externalDataDailyDataProvider;
        private ICoreServices _coreServices;
        private ExternalDataCommands _commands;
        private List<ExternalDataHostBase> _providerHosts;

        public override void InitializePlugin(IPoderosaWorld poderosa) {
            base.InitializePlugin(poderosa);

            _instance = this;

            ICoreServices cs = (ICoreServices)poderosa.GetAdapter(typeof(ICoreServices));
            _coreServices = cs;
            ICommandCategory cat = cs.CommandManager.CommandCategories.Dialogs;

            //_preferences = new ExternalDataPreferences();
            //_coreServices.PreferenceExtensionPoint.RegisterExtension(_preferences);

            _externalDataDailyDataProvider = poderosa.PluginManager.CreateExtensionPoint("tactico.externalData.providers", typeof(IExternalDataProvider), this);
            poderosa.PluginManager.FindExtensionPoint(WindowManagerConstants.MAINWINDOWEVENTHANDLER_ID).RegisterExtension(this);

            _commands = new ExternalDataCommands();
            _commands.Init(cs, poderosa.PluginManager.FindExtensionPoint("org.poderosa.menu.tool"));
        }
        public static ExternalDataPlugin Instance {
            get {
                return _instance;
            }
        }
        public ICommandManager CommandManager {
            get {
                return _coreServices.CommandManager;
            }
        }

        public override void TerminatePlugin() {
            base.TerminatePlugin();
        }

        public List<ExternalDataHostBase> ProviderHosts {
            get {
                return _providerHosts;
            }
        }

        //IMainWindowEventHandler
        public void OnFirstMainWindowLoaded(IPoderosaMainWindow window) {
            _providerHosts = new List<ExternalDataHostBase>();
            IExternalDataProvider[] providers = (IExternalDataProvider[])_externalDataDailyDataProvider.GetExtensions();
            try {
                foreach(IExternalDataProvider prov in providers) {
                    ExternalDataHostBase host;
                    if(prov.KeyType==ExternalDataKeyType.Date)
                        host = new DateBasedExternalDataHost(prov);
                    else
                        host = new DateAndStockBasedExternalDataHost(prov);

                    host.Initialize();
                    host.LoadFromAppFile(); //遅延ロードという手もあるけども
                    _providerHosts.Add(host);
                }
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

        public void OnLastMainWindowUnloaded(IPoderosaMainWindow window) {
            foreach(ExternalDataHostBase host in _providerHosts)
                host.SaveToAppFile();
        }

        //コマンド実行による手動フルリセット
        public void ResetAll(NotifyLoadStatusDelegate notifier) {
            try {
                _notifier = notifier;
                int today = BDate.DateTimeToInt(DateTime.Now);
                foreach(ExternalDataHostBase host in _providerHosts) {
                    host.ResetAll(today);
                }
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
            _notifier = null;
        }

        //ResetAll中の通知
        public delegate void NotifyLoadStatusDelegate(ExternalDataHostBase dh, string msg);
        private NotifyLoadStatusDelegate _notifier;

        public void NotifyLoadStatus(ExternalDataHostBase dh, string msg) {
            if(_notifier!=null) _notifier(dh, msg);
        }

        #region unused
        public void OnMainWindowLoaded(IPoderosaMainWindow window) {
        }

        public void OnMainWindowUnloaded(IPoderosaMainWindow window) {
        }
        public bool AskCancelClosingLastWindow(IPoderosaMainWindow window) {
            return false;
        }
        #endregion


        public void DailyUpdate(int[] dates) {
            try {
                foreach(ExternalDataHostBase host in _providerHosts) {
                    host.DailyUpdate(dates[0], dates[dates.Length-1]);
                }
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

    }

    //ExternalDataはあくまで補助的なデータについてなので、UI上はメインデータでの更新において連動的更新をかける
    public interface IExternalDataManager : IAdaptable {
        void DailyUpdate(int[] dates); //日付（複数個もある）指定で更新　種類ごとにステータステキストを出してもよいかも
    }

    //データの種類ごとにProvider-Hostペアがつくられる
    //データは、日付をキーとしたコレクションで収録されるものと、日付・銘柄ペアをキーとしたものとに分かれる

    public enum ExternalDataKeyType {
        Date,
        DateAndStock
    }

    public interface IExternalDataProvider {
        Type ElementType { get; }
        ExternalDataKeyType KeyType { get; }
        string ProviderId { get; } //ファイル名を指定
        string FunctionPostfix { get; } //要素オブジェクトを探すための関数のpostfix  findXXX系が作成される
        string Description { get; }
        int AvailableMonthCount { get; }

        //この中で関数登録系の動作を呼び出す
        void InitializeProvider(IExternalDataHost host);

        //指定日付までのすべてをロード
        void ImportDataUntil(int startdate, int enddate);

        //読み出し部分はOriginReportableTextを使うことを検討したが、XMLを人間が書くわけではないこと、要素の型が固定しづらいことからやめた
        void SerializeElement(XmlWriter writer, object value); //書き出し。writerはStartElementの直後の状態で引き渡される。EndElementの直前までの動作をする
        object DeserializeElement(XmlReader reader); //読み込み。readerはReadStartElementのあとのdateアトリビュートを読んだ状態で引き渡される。ReadEndElementの直前までの操作をここで。
    }

    //関数のデリゲート
    public delegate double ExternalData_DoubleDelegate(object value);
    public delegate bool ExternalData_BoolDelegate(object value);


    public interface IExternalDataHost {
        void DeleteAll(); //全部削除
        void DeleteAfter(int date); //指定した日付以降のデータを削除
        void NotifyLoadStatus(string text); //ロード中の状況を通知

        //関数登録
        void RegisterElementFunction(string name, ExternalData_DoubleDelegate dg);
        void RegisterElementFunction(string name, ExternalData_BoolDelegate dg);

        //銘柄検索はわりとある、ショートカット
        AbstractStockProfile FindStock(string code);
    }
    public interface IDateBasedExternalDataHost : IExternalDataHost {
        void AppendElement(int date, object value);
    }
    public interface IDateAndStockBasedExternalDataHost : IExternalDataHost {
        void AppendElement(int date, Stock stock, object value);
    }

}
