﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using System.Drawing;
using System.Threading;

using Travis.IO;

using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Forms;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Screening {
    /*
      MemoryMappedFile上に作った日足データベース
      次の構造になっている
    　ヘッダ [ コード(8byte), データベース内オフセット(4byte), 長さ(4byte) ] * 銘柄数上限
      つづいてデータ本体

    　銘柄数は可変にはできないので、4096個で固定しておく。
    　コードは文字列化したもの。オフセットはデータの先頭からの位置で、バイト単位の長さ。
    　１銘柄あたりの長さは可変である。
    */
    public class MMFQuoteBase {
        public const int MAX_STOCK_COUNT = 4096;
        public const int STOCK_TAG_SIZE = 16;

        public class StockTag {
            public string code;
            public int offset;
            public uint sizeAndFlags; //上位１ビットを、ローカルの日足ファイルがすべて収録されているかどうかに使用

            public unsafe StockTag(byte* ptr) {
                int i = 7;
                while(*(ptr+i)==0 && i>=0) i--;
                code = new string((sbyte*)ptr, 0, i+1, Encoding.ASCII);
                offset = *(int*)(ptr + 8);
                sizeAndFlags   = *(uint*)(ptr + 12);
            }
            public StockTag() {
            }

            public void SetSizeAndFlags(int size, bool fully_read) {
                uint t = (uint)size;
                if(fully_read) t |= 0x80000000;
                sizeAndFlags = t;
            }
            public bool FullyRead {
                get {
                    return (sizeAndFlags & 0x80000000)!=0;
                }
            }
            public int Size {
                get {
                    return (int)(sizeAndFlags & 0x7FFFFFFF);
                }
            }

            public void FormatTo(byte[] buf) {
                byte[] name = Encoding.ASCII.GetBytes(code);
                Debug.Assert(name.Length <= 8);
                unsafe {
                    fixed(byte* p = buf) {
                        int i = 0;
                        while(i < name.Length) {
                            *(p+i) = name[i];
                            i++;
                        }
                        while(i < 8) *(p + i++) = 0;

                        *(int*)(p + 8)= offset;
                        *(uint*)(p + 12) = sizeAndFlags;
                    }
                }

            }

        }

        private Dictionary<string, StockTag> _tagMap;
        private MemoryMappedFile _memoryMappedFile;
        private IntPtr _baseAddress;

        //MMFを作成し、タグマップのみ作る
        public MMFQuoteBase(string path) {
            FileInfo fi = new FileInfo(path);

            _tagMap = new Dictionary<string, StockTag>();
            _memoryMappedFile = MemoryMappedFile.Create(path, MapProtection.PageReadOnly, fi.Length, "BellagioQuoteBase");
            IntPtr ptr = _memoryMappedFile.MapView(MapAccess.FileMapRead, 0, (int)fi.Length);

            unsafe {
                byte* p = (byte*)ptr.ToPointer();
                while(true) {
                    if(*p == 0) break; //null char
                    StockTag tag = new StockTag(p);

                    _tagMap.Add(tag.code, tag);
                    p += STOCK_TAG_SIZE;
                }
            }
            _baseAddress = ptr;
        }

        public int TagCount {
            get {
                return _tagMap.Count;
            }
        }

        public void Dispose() {
            _memoryMappedFile.Dispose();
        }

        //fully_readを返す
        public bool RequestMMFQuote(string code, out MMFQuote result) {
            StockTag tag;
            if(_tagMap.TryGetValue(code, out tag)) {
                result = new MMFQuote(new IntPtr(_baseAddress.ToInt32() + tag.offset), tag.Size / Candle.ELEMENT_SIZE_IN_MEMORY, Quote.QuoteUnit.Daily);
                return tag.FullyRead;
            }
            else {
                result = null;
               return false;
            }
        }

        public StockTag GetStockTag(string code) {
            return _tagMap[code];
        }
    }

    //１銘柄ずつ記入してデータを作る
    public class MMFQuoteBaseBuilder {
        private List<MMFQuoteBase.StockTag> _tags;
        private int _currentOffset;
        private int _maxCandleCount; //最大で何本分収録するか
        private FileStream _stream;
        private byte[] _stockBuffer; //１銘柄ぶんのバッファ
        private byte[] _fileBuffer; //ファイルバッファ
        private int _fileBufferOffset;
        private const int STOCKS_IN_FILEBUFFER = 1000;

        //メモ
        //これを使ってMMFQuoteを書きだすとき、通常の日足ダウンロードの読み込みと並行動作になることが多い。
        //１銘柄ごとに_streamに書いているとファイルアクセスが頻発（しかも多数のファイルへのアクセスになる）するので、高速化のため1000銘柄ごとにまとめて書き出す。
        //それでも、日足１個で32byte * 500(_maxCandleCountの標準的な値) * 1000 で16MB程度のメモリにしかならない。

        public MMFQuoteBaseBuilder(string path, int maxCandleCount) {
            _stream = new FileStream(path, FileMode.Create, FileAccess.Write);
            _maxCandleCount = maxCandleCount;
            _tags = new List<MMFQuoteBase.StockTag>();
            _currentOffset = MMFQuoteBase.STOCK_TAG_SIZE * MMFQuoteBase.MAX_STOCK_COUNT;
            _stream.SetLength(_currentOffset);
            _stream.Seek(_currentOffset, SeekOrigin.Begin); //先頭のヘッダ部分はクローズ前に埋める
            _stockBuffer = new byte[Candle.ELEMENT_SIZE_IN_MEMORY * _maxCandleCount]; //これだけあれば１銘柄分は収録できる
            _fileBuffer = new byte[_stockBuffer.Length * STOCKS_IN_FILEBUFFER];
            _fileBufferOffset = 0;
        }

        //銘柄コードと、オープン済みの個別銘柄日足ファイルを渡して１銘柄分を取得
        public void AppendQuote(BasicStockProfile prof, FileStream fs) {
            bool fully_read = fs.Length <= _maxCandleCount * Candle.ELEMENT_SIZE_IN_MEMORY;
            long seek_point = Math.Max(fs.Length -_maxCandleCount * Candle.ELEMENT_SIZE_IN_MEMORY, 0);
            long seek = fs.Seek(seek_point, SeekOrigin.Begin);
            int length = (int)(fs.Length - seek);
            int read = fs.Read(_stockBuffer, 0, length);
            Debug.Assert(read==length);
            AdjustDataInBuffer(prof, length);

            if(_fileBufferOffset + length >= _fileBuffer.Length) {
                _stream.Write(_fileBuffer, 0, _fileBufferOffset);
                _fileBufferOffset = 0;
            }
            Buffer.BlockCopy(_stockBuffer, 0, _fileBuffer, _fileBufferOffset, length);
            _fileBufferOffset += length;
            
            MMFQuoteBase.StockTag tag = new MMFQuoteBase.StockTag();
            tag.code = prof.Code;
            tag.offset = _currentOffset;
            tag.SetSizeAndFlags(length, fully_read);
            _tags.Add(tag);
            _currentOffset += length;
        }
        private void AdjustDataInBuffer(BasicStockProfile prof, int length) {
            //_buffer内のデータの出来高を調整する。出来高が０の場合、直前のデータのある日に基づいて価格を書き換えるべし。
            unsafe {
                int last_valid_price = 0;
                int candle_count = length / Candle.ELEMENT_SIZE_IN_MEMORY;

                fixed(byte* base_addr_b = _stockBuffer) {
                    int* base_addr = (int*)base_addr_b;
                    for(int i=0; i<candle_count; i++) {
                        int* h = base_addr + i * (Candle.ELEMENT_SIZE_IN_MEMORY>>2);

                        int vol  = *(h + Candle.VolumeDelegate.OffsetByIntPtr);

                        if(vol!=0 /*|| !volume_available*/) {
                            last_valid_price = *(h+Candle.CloseDelegate.OffsetByIntPtr);
                        }
                        else {
                            if(last_valid_price==0) { //読みだす先頭が値つかずの上場初日等で出来高０だと値段が決定できずいやらしい
                                continue; //nilflagがセットできたほうがよさそう
                            }
                            //スクリーニングではチャート表示とちがいこう処理しないといけない
                            *(h + Candle.OpenDelegate.OffsetByIntPtr)  = last_valid_price;
                            *(h + Candle.HighDelegate.OffsetByIntPtr)  = last_valid_price;
                            *(h + Candle.LowDelegate.OffsetByIntPtr)   = last_valid_price;
                            *(h + Candle.CloseDelegate.OffsetByIntPtr) = last_valid_price;
                        }
                    }
                }
            }

            //分割情報の反映：生ファイルは分割未反映と無条件に想定しているのでやや危険
            if(/*_provider.DefaultSplitReflection==SplitReflection.NotRefrected &&*/ prof.SplitInfo!=null && prof.SplitInfo.Length>0) {
                QuoteConverter.ReflectSplitInMemory(_stockBuffer, length, prof.SplitInfo, false);
            }

        }

        //ヘッダ部を書いてクローズ
        public void Close() {
            if(_fileBufferOffset > 0)
                _stream.Write(_fileBuffer, 0, _fileBufferOffset);

            int size = _currentOffset; //これがファイルサイズになるはず
            _stream.Seek(0, SeekOrigin.Begin);
            byte[] buf = new byte[MMFQuoteBase.STOCK_TAG_SIZE];
            foreach(MMFQuoteBase.StockTag tag in _tags) {
                tag.FormatTo(buf);
                _stream.Write(buf, 0, buf.Length);
            }
            //終端
            buf = new byte[MMFQuoteBase.STOCK_TAG_SIZE];
            _stream.Write(buf, 0, buf.Length);

            _stream.Close();
        }

    }

    //進行状況を示すダイアログ
    internal class MMFQuoteBaseBuildingDialog : Form {
        private ProgressBar _progBar;
        private Button _cancelButton;
        private Thread _thread;

        public MMFQuoteBaseBuildingDialog() {
            FormUtil.AdjustStyleForModalDialog(this);
            this.Text = "データ作成中";
            _progBar = new ProgressBar();
            _cancelButton = FormUtil.CreateCancelButton();
            _cancelButton.Click += new EventHandler(OnCancel);

            _progBar.Size = new Size(160, 19);
            _progBar.Location = new Point(4, 4);
            _cancelButton.Location = new Point(_progBar.Right + 4, _progBar.Top);
            this.ClientSize = new Size(_cancelButton.Right+4, 32);
            this.Controls.AddRange(new Control[] { _progBar, _cancelButton });
            _progressDelegate = new ProgressDelegate(OnProgress);
        }
        private void OnCancel(object sender, EventArgs arg) {
            if(_thread!=null) {
                _thread.Abort();
                _thread.Join();
                //ディスクをクリーン
                string path = ScreeningPlugin.Instance.DailyDataBasePath;
                if(File.Exists(path)) File.Delete(path);
            }
        }

        protected override void OnLoad(EventArgs e) {
            base.OnLoad(e);
            //フォームロードで即実行開始
            _progBar.Maximum = BellagioRoot.GlobalStockCollection.Count;
            _progBar.Value = 0;
            _thread = new Thread(new ThreadStart(BuildRoot));
            _thread.Start();
        }

        protected override void OnClosed(EventArgs e) {
            base.OnClosed(e);
            if(_thread!=null) _thread.Join();
        }

        //あまり良くないマナーだが、デリゲートを用意するんがめんどうなのでprog==-1で完了を示すことにする
        private delegate void ProgressDelegate(int prog);
        private ProgressDelegate _progressDelegate;
        private void OnProgress(int prog) {
            if(prog == -1)
                this.Close();
            else
                _progBar.Value = prog;
        }

        private void BuildRoot() {
            MMFQuoteBaseBuilder builder = null;
            FileStream fs = null;
            try {
                builder = new MMFQuoteBaseBuilder(ScreeningPlugin.Instance.DailyDataBasePath, ScreeningPlugin.Instance.Preferences.MaximumDailyDataLength);
                int count = 0;
                foreach(AbstractStockProfile prof in BellagioRoot.GlobalStockCollection) {
                //AbstractStockProfile prof = BellagioRoot.GlobalStockCollection.FindExact("4740");
                    BasicStockProfile bp = prof as BasicStockProfile;
                    if(bp==null) continue;

                    string quote_path = ScreeningPlugin.Instance.BellagioEnv.DataHomeDir + "daily\\" + bp.Code;

                    fs = new FileStream(quote_path, FileMode.Open, FileAccess.Read);
                    builder.AppendQuote(bp, fs);
                    fs.Close();
                    fs = null;

                    this.Invoke(_progressDelegate, count++);
                }

                this.Invoke(_progressDelegate, -1); //complete
            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
            }
            finally {
                builder.Close();
                if(fs!=null) fs.Close();
            }
        }

    }

#if UNITTEST
    [TestFixture]
    public class MMFQuoteBaseTest {
        [Test]
        public void TagTest() {
            //ヘッダ部分のみ適当に書き出し
            string path = Path.GetTempFileName();
            FileStream fs = new FileStream(path, FileMode.Create, FileAccess.Write);
            byte[] tag1 = new byte[MMFQuoteBase.STOCK_TAG_SIZE];
            unsafe {
                fixed(byte* p = tag1) {
                    byte zero = (byte)'0';
                    *(p) = zero;
                    *(p+1) = (byte)(zero+1);
                    *(p+2) = (byte)(zero+2);
                    *(p+3) = (byte)(zero+3);
                    *((int*)(p + 8)) = 3000;
                    *((int*)(p + 12)) = 2000;
                }
            }
            fs.Write(tag1, 0, tag1.Length);
            tag1 = new byte[MMFQuoteBase.STOCK_TAG_SIZE]; //終端
            fs.Write(tag1, 0, tag1.Length);
            fs.Close();

            MMFQuoteBase qb = new MMFQuoteBase(path);
            Assert.AreEqual(1, qb.TagCount);

            MMFQuoteBase.StockTag tag = qb.GetStockTag("0123");
            Assert.IsNotNull(tag);
            Assert.AreEqual("0123", tag.code);
            Assert.AreEqual(3000, tag.offset);
            Assert.AreEqual(2000, tag.Size);

            qb.Dispose();
            File.Delete(path);

        }
    }
#endif
}
