/*
 * Copyright (c) Daisuke OKAJIMA    All rights reserved.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/BValue.cs#10 $
 * $DateTime: 2007/12/28 19:23:38 $
 * 
 * ulṽIuWFNgQ
 */
using System;
using System.Text;
using System.Collections.Generic;
using System.Diagnostics;

using Bellagio.Script;
using Bellagio.Evaluators;

#if UNITTEST
using NUnit.Framework;
#endif

using Travis.ORT;

namespace Bellagio.Values
{
    /// ȑOEvalResultƂO悭gOȂ̂ŒZB'Bellagio Value'̈ӖB
    //TESTINFO ̃t@CNUnit BValueType.csƂ
 	public abstract class BV {
        public abstract BT BT { get; }
        public abstract void Format(BVFormatter formatter);
        public abstract void Let(BV value); //valuethisւ̑B^ႤƂ͗O

#if UNITTEST
        //Evaluator̋쓮BṼCX^X쐬ɗ͗}ĂB̂ƂmF邽߂̂
        public static int instanceCount;

        public BV() {
            instanceCount++;
        }
#endif

        //lBv~eBuł͗B̃CX^Xpӂ邪A^ɂĂ͓IɌ߂Ă悢
        public virtual bool IsNil {
            get {
                return false;
            }
        }
        public virtual BArray AsArray() {
            return null;
        }

        //łΎsAłȂΐVK쐬ĕԂ
        public static void Let(ref BV dest, BV source) {
            if(source==null)
                dest = null;
            else if(source.IsNil)
                dest = source.BT.Nil;
            else {
                if(dest==null || dest.IsNil || !dest.BT.IsConvertibleTo(source.BT)) dest = source.BT.CreateInstance();
                dest.Let(source);
            }
        }

        public static string DefaultFormat(BV value) {
            DefaultBVFormatter f = new DefaultBVFormatter();
            value.Format(f);
            return f.Result;
        }

        private static BV[] _emptyArray = new BV[0];
        public static BV[] EmptyArray {
            get {
                return _emptyArray;
            }
        }
	}

	public class BBoolean : BV {
		private bool _value;
		public BBoolean(bool v) {
			_value = v;
		}

		public bool Value {
			get {
				return _value;
			}
			set {
				_value = value;
			}
		}
        public override BT BT {
            get {
                return BT.Bool;
            }
        }
		public override void Format(BVFormatter formatter) {
            formatter.BBoolean(this);
		}
        public override void Let(BV value) {
            _value = ((BBoolean)value)._value;
        }
	}

	public class BDouble  : BV {
		private double _value;
		public BDouble(double v) {
			_value = v;
		}
		public double Value {
			get {
				return _value;
			}
			set {
				_value = value;
			}
		}
        public override BT BT {
            get {
                return BT.Double;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.BDouble(this);
        }
        public override void Let(BV value) {
            _value = ((BDouble)value)._value;
        }
    }

    public class BInt : BV {
        private int _value;
        public BInt(int v) {
            _value = v;
        }
        public int Value {
            get {
                return _value;
            }
            set {
                _value = value;
            }
        }
        public override BT BT {
            get {
                return BT.Int;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.BInt(this);
        }
        public override void Let(BV value) {
            _value = ((BInt)value)._value;
        }
    }


    public class BString : BV {
        private string _value;
        public BString() {
        }
        public BString(string value) {
            _value = value;
        }
        public override BT BT {
            get {
                return BT.String;
            }
        }
        public override void Let(BV value) {
            _value = ((BString)value)._value;
        }
        public override void Format(BVFormatter formatter) {
            formatter.BString(this);
        }
        public string Value {
            get {
                return _value;
            }
            set {
                _value = value;
            }
        }
    }

    public abstract class BArray : BV {
        protected BArrayT _arrayType;
        protected bool _nilFlag;

        public BArray(BArrayT arr) {
            _arrayType = arr;
        }
        public override BT BT {
            get {
                return _arrayType;
            }
        }
        public BArrayT BArrayT {
            get {
                return _arrayType;
            }
        }

        public override BArray AsArray() {
            return this;
        }

        public abstract int Count { get; }
        public abstract BV this[int index] { get; }
        public abstract void LetAt(int index, BV value);
        //V[g^Cv
        public virtual BFastenIntArray AsFastenIntArray() {
            return null;
        }
        public virtual BFastenDoubleArray AsFastenDoubleArray() {
            return null;
        }

        public override bool IsNil {
            get {
                return _nilFlag;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.BVArray(this);
        }

        //Arraył͎蓮nil\
        public BArray MakeNil() {
            _nilFlag = true;
            return this;
        }

        protected void IndexOutOfRange(int index) {
            throw new IndexOutOfRangeException(String.Format("invalid index {0} ( actual length = {1} )", index, this.Count));
        }
    }

    public sealed class BRegularArray : BArray {
        private BV[] _data;
        private int _length;

        public BRegularArray(BArrayT arr, int length) : base(arr) {
            if(length > 0)
                Extend(length);
        }
        /*
        public BRegularArray(BArrayT arr)
            : this(arr, arr.Length) {
        }
        */
        public void SetDirect(BV[] data) {
            _data = data;
            _length = data.Length;
        }
        public void SetDirect(BV[] data, int length) {
            Debug.Assert(data.Length >= length);
            _data = data;
            _length = length;
        }
        public override void Let(BV value) {
            BArray arr = (BArray)value;
            Debug.Assert(_arrayType.ElementType.IsConvertibleTo(arr.BArrayT.ElementType));
            
            Extend(arr.Count);
            _nilFlag = arr.IsNil;
            for(int i=0; i<_length; i++)
                _data[i].Let(arr[i]);
        }

        //z͕KvɉĊgBnew񐔂팸
        public void Extend(int length) {
            if(length > _length) { //sƂ
                BV[] newdata = new BV[length];
                if(_length > 0) Array.Copy(_data, newdata, _length); //Rs[
                //s𖄂߂
                for(int i = _length; i<length; i++)
                    newdata[i] = _arrayType.ElementType.CreateInstance();
                _data = newdata;
            }
            _length = length;
        }
        public override int Count {
            get {
                return _length;
            }
        }
        public int Capacity {
            get {
                return _data.Length;
            }
        }
        public override BV this[int index] {
            get {
                if(index<0 || index>=_length) IndexOutOfRange(index);
                return _data[index];
            }
        }
        public override void LetAt(int index, BV value) {
            if(index<0 || index>=_length) IndexOutOfRange(index);
            Debug.Assert(_arrayType.ElementType.IsConvertibleTo(value.BT));
            BV.Let(ref _data[index], value);
        }
        public void SetAt(int index, BV value) {
            if(index<0 || index>=_length) throw new IndexOutOfRangeException(String.Format("invalid index {0} ( actual length = {1} )", index, _length));
            _data[index] = value;
        }
    }

    //̔z̕o
    internal sealed class BSubArray : BArray {
        private BArray _content;
        private int _start;
        private int _length;
        public BSubArray(BArray content, int start, int length) : base(content.BArrayT) {
            Debug.Assert(content.Count >= start + length);
            Set(content, start, length);
        }
        public void Set(BArray content, int start, int length) {
            _content = content;
            _start = start;
            _length = length;
        }

        public override bool IsNil {
            get {
                return _content.IsNil;
            }
        }
        public override BT BT {
            get {
                return _content.BT;
            }
        }

        public override int Count {
            get {
                return _length;
            }
        }

        public override BV this[int index] {
            get {
                Debug.Assert(index < _length);
                return _content[_start + index];
            }
        }

        public override void LetAt(int index, BV value) {
            Debug.Assert(index < _length);
            _content.LetAt(_start + index, value);
        }

        public override void Let(BV value) {
            BArray arr = (BArray)value;
            Set(arr, 0, arr.Count);
        }
    }

    public enum ExecResult {
        OK,
        Nil
    }

    //֐^
    public abstract class BFunction : BV {
        protected BFunctionT _fType;

        public abstract ExecResult Execute(BV target, EvalContext context, BV[] args, BV result);

        public BFunctionT FType {
            get {
                return _fType;
            }
        }
        public override BT BT {
            get {
                return _fType;
            }
        }
    }
    public class BVNilFunction : BFunction {
        public BVNilFunction(BFunctionT t) {
            _fType = t;
        }
        public override bool IsNil {
            get {
                return true;
            }
        }
        public override ExecResult Execute(BV target, EvalContext context, BV[] args, BV result) {
            return ExecResult.Nil;
        }
        public override void Format(BVFormatter formatter) {
            formatter.Nil(this);
        }
        public override void Let(BV value) {
            throw new ScriptException("invalid operation: assign to static nil value");
        }
    }

    internal sealed class BMethodValue : BFunction {
        private BMethod _method;
        private BV[] _externalBoundValues;

        public BMethodValue(BMethod m) {
            _fType = m.FunctionT;
            _method = m;
            InitExternalBounds();
        }
        public BMethodValue(BFunctionT ft) {
            //^CreateInstanceō쐬邱Ƃ
            _fType = ft;
            _method = null;
        }
        public BMethod Method {
            get {
                return _method;
            }
        }

        public BV[] ExternalBoundValues {
            get {
                return _externalBoundValues;
            }
        }

        public override void Format(BVFormatter formatter) {
            formatter.Lambda(this);
        }

        public override void Let(BV value) {
            BMethodValue l = (BMethodValue)value;
            _method = l._method;
            _fType = _method.FunctionT;
            _externalBoundValues = l._externalBoundValues;
        }
        private void InitExternalBounds() {
            BMethod.ExternalBoundValueTag[] ext = _method.ExternalContextValues;
            if(ext.Length > 0) {
                _externalBoundValues = new BV[ext.Length];
                for(int i=0; i<ext.Length; i++)
                    _externalBoundValues[i] = ext[i].externalVar.BT.CreateInstance();
            }
        }

        public override ExecResult Execute(BV target, EvalContext context, BV[] args, BV result) {
            BStackFrame frame = context.AllocateStack(_method, args);

            //OQƂ̒l΃[JɃRs[ ̒lZbĝLambdaEvaluator̕]
            if(_externalBoundValues!=null) {
                BMethod.ExternalBoundValueTag[] ext = _method.ExternalContextValues;
                for(int i=0; i<ext.Length; i++) {
                    frame.GetAt(ext[i].internalVar.Index).Let(_externalBoundValues[i]);
                }
            }

            BV value = _method.Body.Eval(context);
            context.ReleaseStack(frame);

            if(value.IsNil)
                return ExecResult.Nil;
            else {
                result.Let(value);
                return ExecResult.OK;
            }
        }
    }

    public class BMethod {

        //OReLXgQƂ̂Ƃ̃uc
        public class ExternalBoundValueTag {
            public LocalVariable externalVar;
            public LocalVariable internalVar;

            public ExternalBoundValueTag(LocalVariable e, LocalVariable i) {
                externalVar = e;
                internalVar = i;
            }
        }

        private BFunctionT _functionT;
        private LocalVariable[] _params; //
        private LocalVariable[] _intermediateValues; //vZrŕKvɂȂꎞLϐ̌^@letgƖOtɂȂ
        private ExternalBoundValueTag[] _externalContextValues; //lambdae̕ϐƂȂǁB̊֐\zƂɕ]ׂl
        private int[] _ivokeArgumentSizes;
        private IEvaluator _body;

        public BMethod(LocalVariable[] parameters, IEvaluator body) {
            _params = parameters;
            _body = body;

            BT[] at = new BT[parameters.Length];
            for(int i=0; i<at.Length; i++)
                at[i] = parameters[i].BT;
            _functionT = new BFunctionT(null, at, _body.BT); //lambdał̓^[QbgȂƂB̂ɕύX邩
        }
        public void Fix(EvaluatorBuildContext ctx) {
            _intermediateValues = ctx.Intermediates;
            _externalContextValues = ctx.ExternalContextValues;
            _ivokeArgumentSizes = ctx.InvokeArgSizes;
        }
        public BFunctionT FunctionT {
            get {
                return _functionT;
            }
        }

        public LocalVariable[] Parameters {
            get {
                return _params;
            }
        }
        public LocalVariable[] IntermediateValues {
            get {
                return _intermediateValues;
            }
        }
        public ExternalBoundValueTag[] ExternalContextValues {
            get {
                return _externalContextValues;
            }
        }
        public int[] InvokeArgumentSizes {
            get {
                return _ivokeArgumentSizes;
            }
        }
        public IEvaluator Body {
            get {
                return _body;
            }
        }



    }

    //BV[]oRȂ
    public class BFastenIntArray : BArray {
        private int[] _data; //Ƀ_CNgȔzōo[WADataFarmǂރo[WAȂǂɕ̗\

        public BFastenIntArray(int[] data)
            : base(new BArrayT(BT.Int)) {
            _data = data;
        }

        public override BFastenIntArray AsFastenIntArray() {
            return this;
        }
        public override int Count {
            get {
                return _data.Length;
            }
        }

        public override BV this[int index] {
            get {
                return new BInt(_data[index]); 
            }
        }
        public override void LetAt(int index, BV value) {
            _data[index] = ((BInt)value).Value;
        }
        public override void Let(BV value) {
            _data = ((BFastenIntArray)value)._data;
        }

        //擾͂
        public int FastGet(int index) {
            return _data[index];
        }
    }

    public class BFastenDoubleArray : BArray {
        private double[] _data; //Ƀ_CNgȔzōo[WADataFarmǂރo[WAȂǂɕ̗\

        public BFastenDoubleArray(double[] data)
            : base(new BArrayT(BT.Double)) {
            _data = data;
        }

        public override BFastenDoubleArray AsFastenDoubleArray() {
            return this;
        }
        public override int Count {
            get {
                return _data.Length;
            }
        }

        public override BV this[int index] {
            get {
                return new BDouble(_data[index]); 
            }
        }
        public override void LetAt(int index, BV value) {
            _data[index] = ((BDouble)value).Value;
        }
        public override void Let(BV value) {
            _data = ((BFastenDoubleArray)value)._data;
        }

        //擾͂
        public double FastGet(int index) {
            return _data[index];
        }
    }

    //tH[}bg
    public abstract class BVFormatter {
        protected StringBuilder _bld;
        public BVFormatter() {
            _bld = new StringBuilder();
        }
        public void Clear() {
            _bld.Remove(0, _bld.Length);
        }
        public string Result {
            get {
                return _bld.ToString();
            }
        }
        public string Format(BV value) {
            Clear();
            value.Format(this);
            return _bld.ToString();
        }

        internal abstract void BBoolean(BBoolean value);
        internal abstract void BInt(BInt value);
        internal abstract void BDouble(BDouble value);
        internal abstract void BTime(BTime value);
        internal abstract void BDate(BDate value);
        internal abstract void BString(BString value);
        internal abstract void BVArray(BArray value);
        internal abstract void Nil(BV value);

        internal virtual void Lambda(BFunction lambda) {
            _bld.Append("lambda (").Append(lambda.FType.Name).Append(")");
        }
        //܂overrideɂȂ
        internal virtual void TimeAndSales(TimeAndSales value) {
            _bld.Append("ts[").Append(value.Count).Append("]");
        }
        internal virtual void Quote(Quote value) {
            _bld.Append("q[").Append(value.Count).Append("]");
        }
        internal virtual void TickData(TickData value) {
            _bld.Append("tick[");
            _bld.Append(value.Time); //ǂ
            _bld.Append(", ").Append(value.Price).Append(", ").Append(value.Volume).Append("]");
        }
        internal virtual void Candle(Candle value) {
            _bld.Append("<");
            if(value.IsNil)
                _bld.Append("nil");
            else {
                _bld.Append(value.Open).Append(",");
                _bld.Append(value.High).Append(",");
                _bld.Append(value.Low).Append(",");
                _bld.Append(value.Close).Append(",");
            }
            _bld.Append(">");
        }
        internal virtual void Stock(StockRef stock) {
            _bld.Append("[").Append(stock.IsNil? "nil" : stock.Stock.Profile.Code).Append("]");
        }
        internal virtual void IntraDayTrade(IntraDayTradeRef dt) {
            _bld.Append("[").Append(dt.IsNil? "nil" : dt.Value.Stock.Profile.Code).Append("]");
        }
        internal virtual void LightIntraDayTrade(LightIntraDayTradeRef lt) {
            _bld.Append("[").Append(lt.IsNil? "nil" : lt.Value.Stock.Profile.Code).Append("]");
        }
        public void AppendDirect(string text) {
            _bld.Append(text);
        }
    }
    public class DefaultBVFormatter : BVFormatter {
        internal override void BBoolean(BBoolean value) {
            _bld.Append(value.Value);
        }
        internal override void BInt(BInt value) {
            _bld.Append(value.Value);
        }
        internal override void BDouble(BDouble value) {
            _bld.Append(value.Value);
        }
        internal override void BTime(BTime value) {
            _bld.Append(String.Format("{0:D2}:{1:D2}:{2:D2}", value.Hour, value.Minute, value.Second));
        }
        internal override void BDate(BDate value) {
            _bld.Append("d").Append(value.AsInt()); //Ăʂ
        }
        internal override void BString(BString value) {
            _bld.Append(value.Value);
        }
        internal override void BVArray(BArray value) {
            _bld.Append("{");
            for(int i=0; i<value.Count; i++) {
                if(i>0) _bld.Append(", ");
                value[i].Format(this);
            }
            _bld.Append("}");
        }
        internal override void Nil(BV value) {
            _bld.Append("nil");
        }
    }
    public class NumberFormatter : DefaultBVFormatter {
        //[Uڂɂ̂͑͂ꂾ낤
        private bool _plusSign; //̒l̂Ƃ + 邩
        private bool _percent;  //100{ % 
        private string _body;   //.NET̏w蕶

        //+, %, .N̑gݍ킹Ŏw
        public NumberFormatter(string desc) {
            int n = 0;
            for(int i = 0; i<desc.Length; i++) {
                char ch = desc[i];
                if(ch=='+')
                    _plusSign = true;
                else if(ch=='%')
                    _percent = true;
                else if(ch=='.')
                    n = desc[++i] - '0';
                else
                    throw new ScriptException(String.Format("{0} ͏wƂĕsł", desc));
            }

            if(n > 0)
                _body = String.Format("F{0}", n);
            else
                _body = "";
        }

        //̂Q
        internal override void BInt(BInt value) {
            FormatMain(value.Value); //int=>doubleϊNĂ邪܂
        }
        internal override void BDouble(BDouble value) {
            FormatMain(value.Value);
        }
        private void FormatMain(double value) {
            if(_percent) value *= 100;
            if(_plusSign && value > 0) _bld.Append('+');
            if(_body.Length>0)
                _bld.Append(value.ToString(_body));
            else
                _bld.Append(value);
            if(_percent) _bld.Append('%');
        }

        public static NumberFormatter Parse(ORText text, NumberFormatter defaultvalue) {
            if(text.IsOmitted)
                return defaultvalue;
            else
                return new NumberFormatter(text.Value);
        }
    }
#if UNITTEST
    [TestFixture]
    public class BVFormatTests {
        [Test]
        public void IntFormat() {
            BInt i = new BInt(123);
            NumberFormatter f1 = new NumberFormatter("");
            Assert.AreEqual("123", f1.Format(i));
            NumberFormatter f2 = new NumberFormatter("%");
            Assert.AreEqual("12300%", f2.Format(i));
            NumberFormatter f3 = new NumberFormatter("+");
            Assert.AreEqual("+123", f3.Format(i));
            NumberFormatter f4 = new NumberFormatter("+%.2");
            Assert.AreEqual("+12300.00%", f4.Format(i));

            i = new BInt(-123);
            Assert.AreEqual("-123", f3.Format(i));
            Assert.AreEqual("-12300.00%", f4.Format(i));
        }
        [Test]
        public void DoubleFormat() {
            BDouble d = new BDouble(1.234);
            NumberFormatter f1 = new NumberFormatter("+%.2");
            Assert.AreEqual("+123.40%", f1.Format(d));
            NumberFormatter f2 = new NumberFormatter(".4");
            Assert.AreEqual("1.2340", f2.Format(d));
            NumberFormatter f3 = new NumberFormatter(".1");
            Assert.AreEqual("1.2", f3.Format(d));
            d = new BDouble(1.66);
            Assert.AreEqual("1.7", f3.Format(d)); //ľܓ
            d = new BDouble(-1.66);
            Assert.AreEqual("-1.7", f3.Format(d));
        }
    }
#endif
}
