/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/FunctionLibrary.cs#11 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * gݍ݊֐Cu
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using Travis.ORT;

using Bellagio.Evaluators;
using Bellagio.Script;

#if UNITTEST
using Bellagio.Data;
using NUnit.Framework;
#endif

namespace Bellagio.Values {
    //֐s{
    public delegate ExecResult BInternalExecution(BV target, EvalContext extra, BV[] args, BV result);

    //֐Cu
    public class FunctionLibrary {
        
        //e֐ɂ́ABellagioɃn[hR[hꂽ֐delegateŌĂяoHardCodedFunctionƁAgLbgɒ`ꂽ֐Ƃ̂Qނ
        public abstract class Definition : BFunction, INamedValue {
            protected string _name;

            public BV GetValue(EvalContext ctx) {
                return this;
            }

            public string Name {
                get {
                    return _name;
                }
            }
            public override void Format(BVFormatter formatter) {
                formatter.Lambda(this);
            }
            public override void Let(BV value) {
                throw new Exception("static function cannot be overwrite");
            }
        }
        public class HardCodedFunction : Definition {
            private BInternalExecution _body;
            public HardCodedFunction(string name, BT target, BT retType, BT[] args, BInternalExecution body) {
                _name = name;
                _fType = new BFunctionT(target, args, retType);
                _body = body;
            }
            public override ExecResult Execute(BV target, EvalContext extra, BV[] args, BV result) {
                ExecResult r = _body(target, extra, args, result);
                Debug.Assert(result.BT.IsConvertibleTo(_fType.ReturnType));
                return r;
            }
        }
        public class UserDefinedFunction : Definition {
            private string _nameSpace;
            private Expression _expression;
            private ParameterListDefinition _parameters;
            private ObjectOrigin _location;
 
            //x쐬
            private BMethod _method;

            public UserDefinedFunction(string name_space, string name, ParameterListDefinition parameters, TypeDesc retType, Expression expression, ObjectOrigin location) {
                _nameSpace = name_space;
                _name = name;
                _fType = _root.ParameterToFunctionType(null, parameters, retType);
                _parameters = parameters;
                _location = location;
                _expression = expression;
            }
            public void Compile() {
                Debug.Assert(_method==null); //P񂵂słȂ

                //֐{̂̍\z@ċÂƂɍȂ悤ɃtO𗧂Ă
                try {
                    EvaluatorBuildContext bc = new EvaluatorBuildContext(null);
                    bc.Initialize(null, _parameters);
                    _method = new EvaluatorBuilder(bc).BuildMethod(_expression);
                    _method.Fix(bc);
                }
                catch(Exception e) {
                    Debug.WriteLine(e.StackTrace);
                    throw new ScriptException(String.Format("֐{0}\n̓eɌ肪܂\n{1}", _name, e.Message));
                }
                if(!_method.FunctionT.ReturnType.IsConvertibleTo(_fType.ReturnType))
                    throw new ScriptException("return type Error / " + _name);
            }
            public override ExecResult Execute(BV target, EvalContext context, BV[] args, BV result) {
                BStackFrame frame = context.AllocateStack(_method, args);

                BV r = _method.Body.Eval(context);
                context.ReleaseStack(frame);

                if(r.IsNil)
                    return ExecResult.Nil;
                else {
                    result.Let(r);
                    return ExecResult.OK;
                }
            }
        }

        //I[o[[hǗ邽߂
        public class NameTag {
            private string _name;
            private List<Definition> _definitions;
            public NameTag(string name) {
                _name = name;
                _definitions = new List<Definition>();
            }
            public void Add(Definition d) {
                _definitions.Add(d);
                Debug.Assert(d.Name==_name);
            }
            public void Remove(Definition d) {
                _definitions.Remove(d);
            }
            public int Count {
                get {
                    return _definitions.Count;
                }
            }
            public Definition GetAt(int index) {
                return _definitions[index];
            }
            public Definition Find(BT targetType, BT[] argsTypes) {
                for(int i=0; i<_definitions.Count; i++) {
                    Definition d = _definitions[i];
                    if(d.FType.IsMatchArgs(targetType, argsTypes)) return d;
                }
                return null;
            }
            public NamedValueCollection Find(BT targetType) {
                Definition first = null;
                List<INamedValue> l = null;
                for(int i=0; i<_definitions.Count; i++) {
                    Definition d = _definitions[i];
                    BT ft = d.FType.TargetType;
                    bool match = ft==null? true : ft.IsConvertibleTo(targetType);
                    if(match) {
                        if(first==null)
                            first = d;
                        else { //2ڂ̌┭
                            if(l==null) {
                                l = new List<INamedValue>();
                                l.Add(first);
                            }
                            l.Add(d);
                        }
                    }
                }

                if(l!=null)
                    return new NamedValueCollection(l.ToArray());
                else if(first!=null)
                    return new NamedValueCollection(first);
                else
                    return null;

            }

        }

        //ȉDefaultFunctionLibrary̖{
        internal static ValueSystemRoot _root;
        private Dictionary<string, NameTag> _functions;

        public FunctionLibrary() {
            _functions = new Dictionary<string, NameTag>();
        }
        public Definition DefineInternalFunction(string name, BT target, BT retType, BT[] args, BInternalExecution body) {
            Definition d = new HardCodedFunction(name, target, retType, args, body);
            DefineFunction(d);
            return d;
        }
        public UserDefinedFunction DefineUserFunction(string name_space, string name, ParameterListDefinition parameters, TypeDesc retType, Expression expression, ObjectOrigin location) {
            UserDefinedFunction d = new UserDefinedFunction(name_space, name, parameters, retType, expression, location);
            DefineFunction(d);
            return d;
        }
        public void RemoveFunction(Definition value) {
            Debug.Assert(value!=null);
            NameTag tag = _functions[value.Name];
            if(tag==null) return;

            tag.Remove(value);
            if(tag.Count==0)
                _functions.Remove(value.Name);
        }

        public void DefineFunction(Definition d) {
            NameTag tag;
            if(!_functions.TryGetValue(d.Name, out tag)) {
                tag = new NameTag(d.Name);
                _functions.Add(d.Name, tag);
            }
            tag.Add(d);
        }
        internal NameTag GetOverloads(string name) {
            return _functions[name];
        }

        public NamedValueCollection Resolve(string name, BT targetType) {
            NameTag tag;
            if(_functions.TryGetValue(name, out tag)) {
                NamedValueCollection f = tag.Find(targetType);
                if(f!=null) return f; //Found
            }

            return null;
        }

        //ȒPTarget/`FbN
        public static bool CheckAnyArray(BT target) {
            return target!=null && target.AsArray()!=null;
        }
        public static bool CheckNoArg(BT[] args) {
            return args==null || args.Length==0;
        }
        public static bool CheckOneDoubleArrayArg(BT[] args) {
            if(args==null || args.Length!=1) return false;

            BArrayT array = args[0].AsArray();
            if(array==null) return false;

            BT elem = array.ElementType;
            return elem==BT.Double;
        }
        public static bool CheckOneIntArrayArg(BT[] args) {
            if(args==null || args.Length!=1) return false;

            BArrayT array = args[0].AsArray();
            if(array==null) return false;

            BT elem = array.ElementType;
            return elem==BT.Int;
        }
        public static bool CheckOneDoubleArg(BT[] args) {
            if(args==null || args.Length!=1) return false;
            return args[0]==BT.Double;
        }
        public static bool CheckTwoDoubleArgs(BT[] args) {
            if(args==null || args.Length!=2) return false;
            return args[0]==BT.Double && args[1]==BT.Double;
        }
        public static bool CheckOneIntArg(BT[] args) {
            if(args==null || args.Length!=1) return false;
            return args[0]==BT.Int;
        }
        public static bool CheckTwoIntArgs(BT[] args) {
            if(args==null || args.Length!=2) return false;
            return args[0]==BT.Int && args[1]==BT.Int;
        }

        //̑
        public static bool IsIntArray(BArray array) {
            return array!=null && array.BArrayT.ElementType==BT.Int;
        }
        public static bool IsDoubleArray(BArray array) {
            return array!=null && array.BArrayT.ElementType==BT.Double;
        }

        //悭delegate
        //publicǕύXႾ߂
        public static BT[] noArg = new BT[0];
        public static BT[] oneDoubleArg = new BT[] { BT.Double };
        public static BT[] twoDoubleArgs = new BT[] { BT.Double, BT.Double };
        public static BT[] oneDoubleArrayArg = new BT[] { BT.Double.ArrayType() };
        public static BT[] oneIntArg = new BT[] { BT.Int };
        public static BT[] twoIntArgs = new BT[] { BT.Int, BT.Int };
        public static BT[] oneIntArrayArg = new BT[] { BT.Int.ArrayType() };
        
        public static BT noTarget = null;
        public static BT intArrayTarget = BT.Int.ArrayType();
        public static BT doubleArrayTarget = BT.Double.ArrayType();

    }

    //WŎg֐Q
    public class BuiltInFunctionLibrary : FunctionLibrary {
        //int/doublěJԂZ
        private delegate int EachInt(int value, int context);
        private delegate double EachDouble(double value, double context);

        public BuiltInFunctionLibrary() {
            DefineDefaultFunctions();
        }

        private void DefineDefaultFunctions() {
            BT[] twoDouble = new BT[] { BT.Double, BT.Double };
            
            //z񂩂̌vZ
            DefineFunction(new HardCodedFunction("avg",  intArrayTarget, BT.Double, noArg, new BInternalExecution(IAAverage)));
            DefineFunction(new HardCodedFunction("avg",  doubleArrayTarget, BT.Double, noArg, new BInternalExecution(DAAverage)));
            DefineFunction(new HardCodedFunction("avg",  noTarget, BT.Double, oneIntArrayArg, new BInternalExecution(GLAverage)));
            DefineFunction(new HardCodedFunction("avg",  noTarget, BT.Double, oneDoubleArrayArg, new BInternalExecution(GLAverage)));
            DefineFunction(new HardCodedFunction("sdev", intArrayTarget, BT.Double, noArg, new BInternalExecution(IAStandardDeviation)));
            DefineFunction(new HardCodedFunction("sdev", doubleArrayTarget, BT.Double, noArg, new BInternalExecution(DAStandardDeviation)));
            DefineFunction(new HardCodedFunction("sdev", noTarget, BT.Double, oneIntArrayArg, new BInternalExecution(GLStandardDeviation)));
            DefineFunction(new HardCodedFunction("sdev", noTarget, BT.Double, oneDoubleArrayArg, new BInternalExecution(GLStandardDeviation)));
            DefineFunction(new HardCodedFunction("sum",  intArrayTarget, BT.Int, noArg, new BInternalExecution(IASum)));
            DefineFunction(new HardCodedFunction("sum",  doubleArrayTarget, BT.Double, noArg, new BInternalExecution(DASum)));
            DefineFunction(new HardCodedFunction("sum",  noTarget, BT.Int, oneIntArrayArg, new BInternalExecution(GLISum)));
            DefineFunction(new HardCodedFunction("sum",  noTarget, BT.Double,  oneDoubleArrayArg, new BInternalExecution(GLDSum)));
            DefineFunction(new HardCodedFunction("min",  intArrayTarget, BT.Int, noArg, new BInternalExecution(IAMin)));
            DefineFunction(new HardCodedFunction("min",  doubleArrayTarget, BT.Double, noArg, new BInternalExecution(DAMin)));
            DefineFunction(new HardCodedFunction("min",  noTarget, BT.Int, oneIntArrayArg, new BInternalExecution(GLIMin)));
            DefineFunction(new HardCodedFunction("min",  noTarget, BT.Double, oneDoubleArrayArg, new BInternalExecution(GLDMin)));
            DefineFunction(new HardCodedFunction("min", noTarget, BT.Double, twoDouble, new BInternalExecution(TwoDMin)));
            DefineFunction(new HardCodedFunction("max", intArrayTarget, BT.Int, noArg, new BInternalExecution(IAMax)));
            DefineFunction(new HardCodedFunction("max",  doubleArrayTarget, BT.Double, noArg, new BInternalExecution(DAMax)));
            DefineFunction(new HardCodedFunction("max",  noTarget, BT.Int, oneIntArrayArg, new BInternalExecution(GLIMax)));
            DefineFunction(new HardCodedFunction("max",  noTarget, BT.Double, oneDoubleArrayArg, new BInternalExecution(GLDMax)));
            DefineFunction(new HardCodedFunction("max", noTarget, BT.Double, twoDouble, new BInternalExecution(TwoDMax)));
            DefineFunction(new HardCodedFunction("series", noTarget, BT.Int.ArrayType(), oneIntArg, new BInternalExecution(IntSeries))); //z̍쐬
            DefineFunction(new HardCodedFunction("series", noTarget, BT.Int.ArrayType(), twoIntArgs, new BInternalExecution(IntSeries)));
            DefineFunction(new HardCodedFunction("rseries", noTarget, BT.Int.ArrayType(), oneIntArg, new BInternalExecution(IntRSeries)));
            DefineFunction(new HardCodedFunction("rseries", noTarget, BT.Int.ArrayType(), twoIntArgs, new BInternalExecution(IntRSeries)));


            //eNjJ͗p@ȂړI͌肳ĂȂ
            DefineFunction(new HardCodedFunction("rci", noTarget, BT.Double, oneDoubleArrayArg, new BInternalExecution(RCI)));

            //wn
            DefineFunction(new HardCodedFunction("sqrt", noTarget, BT.Double, oneDoubleArg, new BInternalExecution(Sqrt)));
            DefineFunction(new HardCodedFunction("pow",  noTarget, BT.Double, twoDoubleArgs, new BInternalExecution(Pow)));
            DefineFunction(new HardCodedFunction("log",  noTarget, BT.Double, oneDoubleArg, new BInternalExecution(Log))); //Rΐ
            DefineFunction(new HardCodedFunction("abs",  noTarget, BT.Int, oneIntArg, new BInternalExecution(IAbs)));
            DefineFunction(new HardCodedFunction("abs",  noTarget, BT.Double, oneDoubleArg, new BInternalExecution(DAbs)));
            DefineFunction(new HardCodedFunction("round", noTarget, BT.Int, oneDoubleArg, new BInternalExecution(DRound)));
            DefineFunction(new HardCodedFunction("floor", noTarget, BT.Int, oneDoubleArg, new BInternalExecution(DFloor)));
            DefineFunction(new HardCodedFunction("ceiling", noTarget, BT.Int, oneDoubleArg, new BInternalExecution(DCeiling)));
            DefineFunction(new HardCodedFunction("to_d", noTarget, BT.Double, oneIntArg, new BInternalExecution(ToD)));

            //nil
            DefineFunction(new HardCodedFunction("nil", noTarget, BT.Double, noArg, new BInternalExecution(RetNil)));

        }

#if UNITTEST
        //eXgp̃uc
        public void DefineTestFunctions() {
            BT[] noArg = new BT[0];
            BT noTarget = null;
            BArrayT i2 = new BArrayT(BT.Int); //int̒Q
            BArrayT d2 = new BArrayT(BT.Double); //
            BFastenDoubleArray da12 = new BFastenDoubleArray(new double[] { 1, 2 });
            DefineFunction(new HardCodedFunction("da12", noTarget, d2, noArg,
                delegate(BV target, EvalContext extra, BV[] args, BV result) { result.Let(da12); return ExecResult.OK;  }));
            BFastenIntArray ia12 = new BFastenIntArray(new int[] { 1, 2 });
            DefineFunction(new HardCodedFunction("ia12", noTarget, i2, noArg,
                delegate(BV target, EvalContext extra, BV[] args, BV result) { result.Let(ia12); return ExecResult.OK; }));

            BRegularArray nil = new BRegularArray(i2, 0);
            nil.MakeNil();
            DefineFunction(new HardCodedFunction("ianil", noTarget, i2, noArg,
                delegate(BV target, EvalContext extra, BV[] args, BV result) { return ExecResult.Nil;  }));

            ParameterListDefinition p = Expression.ParseParameterListDefinition("int x");
            DefineUserFunction("", "factorial", p, Expression.ParseTypeDesc("int"), Expression.ParseExpression("x==0? 1 : factorial(x-1) * x"), null).Compile();
            //tail recursion^Cv
            p = Expression.ParseParameterListDefinition("int x, int a");
            DefineUserFunction("", "factorial_tail_rec", p, Expression.ParseTypeDesc("int"), Expression.ParseExpression("x==0? a : factorial_tail_rec(x - 1, a * x)"), null).Compile();

            //o[WtB{ib`
            DefineUserFunction("", "fib", Expression.ParseParameterListDefinition("int x"), Expression.ParseTypeDesc("int"), Expression.ParseExpression("x<=1? 1 : (fib(x-2) + fib(x-1))"), null).Compile();
        }
#endif

        //ȉʊ֐
        private static ExecResult ArrayLength(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            ((BInt)result).Value = array.Count;
            return ExecResult.OK;
        }

        private static ExecResult Sqrt(BV target, EvalContext extra, BV[] args, BV result) {
            BDouble v = (BDouble)args[0];
            ((BDouble)result).Value = Math.Sqrt(v.Value);
            return ExecResult.OK;
        }
        private static ExecResult Pow(BV target, EvalContext extra, BV[] args, BV result) {
            BDouble a = (BDouble)args[0];
            BDouble b = (BDouble)args[1];
            ((BDouble)result).Value = Math.Pow(a.Value, b.Value);
            return ExecResult.OK;
        }
        private static ExecResult Log(BV target, EvalContext extra, BV[] args, BV result) {
            BDouble v = (BDouble)args[0];
            ((BDouble)result).Value = Math.Log(v.Value);
            return ExecResult.OK;
        }
        private static ExecResult DAbs(BV target, EvalContext extra, BV[] args, BV result) {
            BDouble v = (BDouble)args[0];
            ((BDouble)result).Value = Math.Abs(v.Value);
            return ExecResult.OK;
        }
        private static ExecResult IAbs(BV target, EvalContext extra, BV[] args, BV result) {
            BInt v = (BInt)args[0];
            ((BInt)result).Value = Math.Abs(v.Value);
            return ExecResult.OK;
        }
        private static ExecResult DRound(BV target, EvalContext extra, BV[] args, BV result) {
            BDouble v = (BDouble)args[0];
            ((BInt)result).Value = (int)Math.Round(v.Value);
            return ExecResult.OK;
        }
        private static ExecResult DFloor(BV target, EvalContext extra, BV[] args, BV result) {
            BDouble v = (BDouble)args[0];
            ((BInt)result).Value = (int)Math.Floor(v.Value);
            return ExecResult.OK;
        }
        private static ExecResult DCeiling(BV target, EvalContext extra, BV[] args, BV result) {
            BDouble v = (BDouble)args[0];
            ((BInt)result).Value = (int)Math.Ceiling(v.Value);
            return ExecResult.OK;
        }
        private static ExecResult ToD(BV target, EvalContext extra, BV[] args, BV result) {
            BInt v = (BInt)args[0];
            ((BDouble)result).Value = (double)v.Value;
            return ExecResult.OK;
        }
        //Nil`FbN͈ʂɕsvBInvokeEvaluator`FbNĂ

        //z֌W
        private static ExecResult IAMin(BV target, EvalContext extra, BV[] args,  BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            IArrayOp(target.AsArray(), Int32.MaxValue, imin, (BInt)result);
            return ExecResult.OK;
        }
        private static ExecResult DAMin(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            DArrayOp(target.AsArray(), Double.MaxValue, dmin, (BDouble)result);
            return ExecResult.OK;
        }
        private static ExecResult GLIMin(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            IArrayOp(array, Int32.MaxValue, imin, (BInt)result);
            return ExecResult.OK;
        }
        private static ExecResult GLDMin(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            DArrayOp(array, Double.MaxValue, dmin, (BDouble)result);
            return ExecResult.OK;
        }
        private static EachInt imin = delegate(int value, int context) { return Math.Min(value, context); };
        private static EachDouble dmin = delegate(double value, double context) { return Math.Min(value, context); };

        private static ExecResult IAMax(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            IArrayOp(target.AsArray(), Int32.MinValue, imax, (BInt)result);
            return ExecResult.OK;
        }
        private static ExecResult DAMax(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            DArrayOp(target.AsArray(), Double.MinValue, dmax, (BDouble)result);
            return ExecResult.OK;
        }
        private static ExecResult GLIMax(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            IArrayOp(array, Int32.MinValue, imax, (BInt)result);
            return ExecResult.OK;
        }
        private static ExecResult GLDMax(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            DArrayOp(array, Double.MinValue, dmax, (BDouble)result);
            return ExecResult.OK;
        }
        private static EachInt imax = delegate(int value, int context) { return Math.Max(value, context); };
        private static EachDouble dmax = delegate(double value, double context) { return Math.Max(value, context); };

        private static ExecResult TwoDMax(BV target, EvalContext extra, BV[] args, BV result) {
            BV d1 = args[0];
            BV d2 = args[1];
            ((BDouble)result).Value = Math.Max(((BDouble)d1).Value, ((BDouble)d2).Value);
            return ExecResult.OK;
        }
        private static ExecResult TwoDMin(BV target, EvalContext extra, BV[] args, BV result) {
            BV d1 = args[0];
            BV d2 = args[1];
            ((BDouble)result).Value = Math.Min(((BDouble)d1).Value, ((BDouble)d2).Value);
            return ExecResult.OK;
        }

        private static ExecResult IASum(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            IArrayOp(target.AsArray(), 0, isum, (BInt)result);
            return ExecResult.OK;
        }
        private static ExecResult DASum(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            DArrayOp(target.AsArray(), 0, dsum, (BDouble)result);
            return ExecResult.OK;
        }
        private static ExecResult GLISum(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            IArrayOp(array, 0, isum, (BInt)result);
            return ExecResult.OK;
        }
        private static ExecResult GLDSum(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            DArrayOp(array, 0, dsum, (BDouble)result);
            return ExecResult.OK;
        }
        private static EachInt isum = delegate(int value, int context) { return value + context; };
        private static EachDouble dsum = delegate(double value, double context) { return value + context; };

        private static ExecResult IAAverage(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            else {
                BInt temp = new BInt(0); //NOTE ƃ[ᔽ
                IArrayOp(array, 0, isum, temp);
                ((BDouble)result).Value = ((double)temp.Value) / array.Count;
                return ExecResult.OK;
            }
        }
        private static ExecResult DAAverage(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            else {
                DArrayOp(target.AsArray(), 0, dsum, (BDouble)result); //tempȂł
                ((BDouble)result).Value = ((BDouble)result).Value / array.Count;
                return ExecResult.OK;
            }
        }
        private static ExecResult GLAverage(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(IsIntArray(array))
                return IAAverage(array, extra, null, result);
            else
                return DAAverage(array, extra, null, result);
        }

        //W΍ł͂ƃgĂ܂Ȃ낤
        private static ExecResult IAStandardDeviation(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            else {
                BDouble dr = (BDouble)result;
                BInt temp = new BInt(0);
                IArrayOp(array, 0, isum, temp);
                double avg = ((double)temp.Value) / array.Count;
                IArrayOp(array, 0, new StandardDeviationMemory(avg).EachDouble(), dr);
                dr.Value = Math.Sqrt(dr.Value / array.Count);
                return ExecResult.OK;
            }
        }
        private static ExecResult DAStandardDeviation(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = target.AsArray();
            if(array.Count==0)
                return ExecResult.Nil;
            else {
                BDouble dr = (BDouble)result;
                DArrayOp(array, 0, dsum, dr);
                double avg = dr.Value / array.Count;
                DArrayOp(array, 0, new StandardDeviationMemory(avg).EachDouble(), dr);
                dr.Value = Math.Sqrt(dr.Value / array.Count);
                return ExecResult.OK;
            }
        }
        private static ExecResult GLStandardDeviation(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            if(IsIntArray(array))
                return IAStandardDeviation(array, extra, null, result);
            else
                return DAStandardDeviation(array, extra, null, result);
        }
        private class StandardDeviationMemory {
            private double _avg;
            public StandardDeviationMemory(double avg) {
                _avg = avg;
            }
            public EachDouble EachDouble() {
                return new EachDouble(Collect);
            }
            private double Collect(double value, double context) {
                double x = value - _avg;
                return context + x*x;
            }
        }



        //JԂn
        private static void IArrayOp(BArray array, int initial_value, EachInt op, BInt result) {
            Debug.Assert(array!=null && array.BArrayT.ElementType==BT.Int);
            BFastenIntArray fi = array.AsFastenIntArray();
            int value = initial_value;
            int length = array.Count;
            if(fi!=null) { //Fast
                for(int i=0; i<length; i++) value = op(fi.FastGet(i), value);
            }
            else { //ʏ
                for(int i=0; i<length; i++) value = op(((BInt)array[i]).Value, value);
            }

            result.Value = value;
        }
        private static void IArrayOp(BArray array, double initial_value, EachDouble op, BDouble result) {
            Debug.Assert(array!=null && array.BArrayT.ElementType==BT.Int);
            BFastenIntArray fi = array.AsFastenIntArray();
            double value = initial_value;
            int length = array.Count;
            if(fi!=null) { //Fast
                for(int i=0; i<length; i++) value = op(fi.FastGet(i), value);
            }
            else { //ʏ
                for(int i=0; i<length; i++) value = op(((BInt)array[i]).Value, value);
            }

            result.Value = value;
        }
        private static void DArrayOp(BArray array, double initial_value, EachDouble op, BDouble result) {
            Debug.Assert(array!=null && array.BArrayT.ElementType==BT.Double);
            BFastenDoubleArray fd = array.AsFastenDoubleArray();
            double value = initial_value;
            int length = array.Count;
            if(fd!=null) { //Fast
                for(int i=0; i<length; i++) value = op(fd.FastGet(i), value);
            }
            else { //ʏ
                for(int i=0; i<length; i++) value = op(((BDouble)array[i]).Value, value);
            }

            result.Value = value;
        }


        //RCI
        private static ExecResult RCI(BV target, EvalContext extra, BV[] args, BV result) {
            BArray array = args[0].AsArray();
            int len = array.Count;
            if(len==0)
                return ExecResult.Nil;
            else if(len==1) {
                ((BDouble)result).Value = 1.0;
                return ExecResult.OK;
            }
            else { //QȏłȂƌvZłȂ
                //RNV͓CX^X̎g܂킵Ȃ̂ŃbN@Xbh[JɂقǊ撣ȂĂ낤
                ((BDouble)result).Value = RCIBody(array);
                return ExecResult.OK;
            }
        }
        //P̃eXg₷悤ɔo
        internal static double RCIBody(BArray array) {
            int len = array.Count;
            List<RCIEntry> rciEntries = new List<RCIEntry>();
            int i;
            for(i=0; i<len; i++) {
                rciEntries.Add(new RCIEntry(((BDouble)array[i]).Value, i+1));
            }
            rciEntries.Sort(delegate(RCIEntry a, RCIEntry b) { return a.value.CompareTo(b.value); });

            //LOvZ@l̂Ƃl̂ł₱
            i = 0;
            while(i < len) {
                double v = rciEntries[i].value;
                int count = GetRCICorrespondingEntryCount(rciEntries, v, i+1) + 1;
                double rank = (i+1) + count*(count-1)/2 / (double)count; //lʂ𕪔z
                for(int j=i; j<i+count; j++) {
                    RCIEntry e = rciEntries[j]; //struct̗񂾂߂ǂ
                    e.rank = rank;
                    rciEntries[j] = e;
                }
                i += count;
            }

            //ʂ̌vZ
            double corr = 0;
            for(i=0; i<len; i++) {
                RCIEntry e = rciEntries[i];
                double x = e.rank - e.index;
                corr += x*x;
            }

            return 1.0 - 6 * corr / ((len-1) * len * (len+1));
        }
        internal struct RCIEntry {
            public double value;
            public double rank; //l̂Ƃ̂intł͂Ȃ
            public int index;
            public RCIEntry(double v, int i) {
                value = v;
                index = i;
                rank = 0;
            }
        }
        //startȍ~ŁAlvaluełԂ
        private static int GetRCICorrespondingEntryCount(List<RCIEntry> rciEntries, double value, int start) {
            int index = start;
            while(index < rciEntries.Count) {
                if(value!=rciEntries[index].value) break;
                index++;
            }
            return index - start;
        }

        private static ExecResult RetNil(BV target, EvalContext extra, BV[] args, BV result) {
            return ExecResult.Nil;
        }

        //z쐬
        private static ExecResult IntSeries(BV target, EvalContext extra, BV[] args, BV result) {
            int start = 0, end = 0;

            if(args.Length==1) {
                start = 0;
                BV e = args[0];
                if(!e.IsNil)
                    end = ((BInt)e).Value;
            }
            else {
                BV e = args[0];
                if(!e.IsNil)
                    start = ((BInt)e).Value;
                e = args[1];
                if(!e.IsNil)
                    end = ((BInt)e).Value;
            }

            if(start < end) {
                BRegularArray arr = (BRegularArray)result;
                BInt x = new BInt(0);
                arr.Extend(end-start);
                for(int i=0; i<arr.Count; i++) {
                    BV v = arr[i];
                    x.Value = start + i;
                    BV.Let(ref v, x);
                    arr.SetAt(i, v);
                }
                return ExecResult.OK;
            }
            else
                return ExecResult.Nil;

        }
        //zt쐬
        private static ExecResult IntRSeries(BV target, EvalContext extra, BV[] args, BV result) {
            int start = 0, end = 0;

            if(args.Length==1) {
                start = 0;
                BV e = args[0];
                if(!e.IsNil)
                    end = ((BInt)e).Value;
            }
            else {
                BV e = args[0];
                if(!e.IsNil)
                    start = ((BInt)e).Value;
                e = args[1];
                if(!e.IsNil)
                    end = ((BInt)e).Value;
            }

            if(start < end) {
                BRegularArray arr = (BRegularArray)result;
                BInt x = new BInt(0);
                arr.Extend(end-start);
                for(int i=0; i<arr.Count; i++) {
                    BV v = arr[i];
                    x.Value = end-1 - i;
                    BV.Let(ref v, x);
                    arr.SetAt(i, v);
                }
                return ExecResult.OK;
            }
            else
                return ExecResult.Nil;

        }
    }

    //ArrayIteration̓^Cv֐
    public class SystemFunctionLibrary {
        private Dictionary<string, IBSystemFunction> _functions;

        public SystemFunctionLibrary() {
            _functions = new Dictionary<string, IBSystemFunction>();
            DefineBuiltInFunctions();
        }
        private void DefineBuiltInFunctions() {
            //z̓֐
            _functions.Add("length", new LengthFunction());
            _functions.Add("empty", new EmptyFunction());
            _functions.Add("count", new CountFunction());
            _functions.Add("all", new AllFunction());
            _functions.Add("any", new AnyFunction());
            _functions.Add("map", new MapFunction());
            _functions.Add("car", new CarFunction()); 
            _functions.Add("cdr", new CdrFunction());

            //nil~
            _functions.Add("nilProof", new NilProofFunction());

            //ԃVtg
            //_functions.Add("minshift", new MinShiftFunction(MinShiftFunction.Behavior.Stock));
            //_functions.Add("minshiftABS", new MinShiftFunction(MinShiftFunction.Behavior.Regular));

            //ReLXg̎擾
            _functions.Add("now", new CurrentTimeFunction());
            _functions.Add("today", new CurrentDateFunction());
            _functions.Add("stock", new CurrentStockFunction());
        }
        public void Register(string name, IBSystemFunction function) {
            _functions.Add(name, function);
        }

        public IBSystemFunction Resolve(string name, BT targetType) {
            //TODO targetTypel悤ɕύX
            IBSystemFunction r = null;
            _functions.TryGetValue(name, out r);
            return r;
        }

        private class LengthFunction : IBSystemFunction {
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                EvaluatorBuildContext context = builder.Context;
                BT targetType = context.ContextType;

                if(targetType!=null && targetType.AsArray()!=null && expr.ArgCount==0) {
                    return new BSystemFunctionArg(BT.Int, new Evaluator[0]);
                }
                else
                    throw new ScriptException("Arg Type error in length()"); //G[bZ[W܂Ƃ
            }

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                BArray arr = context.ContextValue.AsArray();
                ((BInt)result).Value = arr.Count;
                return ExecResult.OK;
            }
        }
        private class EmptyFunction : IBSystemFunction {
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                EvaluatorBuildContext context = builder.Context;
                BT targetType = context.ContextType;

                if(targetType!=null && targetType.AsArray()!=null && expr.ArgCount==0) {
                    return new BSystemFunctionArg(BT.Bool, new Evaluator[0]);
                }
                else
                    throw new ScriptException("Arg Type error in length()"); //G[bZ[W܂Ƃ
            }

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                BArray arr = context.ContextValue.AsArray();
                ((BBoolean)result).Value = (arr.Count==0);
                return ExecResult.OK;
            }
        }

        private abstract class BoolIterationFunction : IBSystemFunction {

            public abstract BT ReturnType();
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                EvaluatorBuildContext context = builder.Context;
                BT targetType = context.ContextType;

                BArrayT array = null;
                if(targetType!=null && targetType.AsArray()!=null && expr.ArgCount==1)  {
                    array = targetType.AsArray();

                    BT save = context.ContextType;
                    context.ContextType = array.ElementType; //ElementTypêƂŕ]
                    Evaluator v = expr[0].Apply<Evaluator>(builder);
                    context.ContextType = save; //ɖ߂

                    //ElementContextTypêƂboolԂA܂͒ڂ̊֐^ȂOK
                    BFunctionT ft = new BFunctionT(null, new BT[] { array.ElementType }, BT.Bool);

                    if(v.BT.IsConvertibleTo(BT.Bool) || v.BT.IsConvertibleTo(ft)) {
                        return new BSystemFunctionArg(ReturnType(), new Evaluator[] { v });
                    }
                }
                throw new ScriptException("Arg Type error in array iteration"); //G[bZ[W܂Ƃ
            }
            public abstract ExecResult InternalExecuteShortStyle(BArray array, EvalContext context, Evaluator evaluator, BV Result);
            public abstract ExecResult InternalExecuteFunctionStyle(BArray array, EvalContext context, BFunction function, BV Result);

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                BV target = context.ContextValue;
                Debug.Assert(target!=null);
                Debug.Assert(target.AsArray()!=null);
                Debug.Assert(evaluators.Length==1);
                Evaluator ev = evaluators[0];

                ExecResult r;
                if(ev.BT.IsConvertibleTo(BT.Bool))
                    r = InternalExecuteShortStyle(target.AsArray(), context, ev, result);
                else { //̂ǂ炩ł邱ƂCreateArgEvaluatorsŃ`FbNς
                    BFunction f = (BFunction)ev.Eval(context);
                    r = InternalExecuteFunctionStyle(target.AsArray(), context, f, result);
                }

                context.ContextValue = target;
                return r;
            }
        }

        private class CountFunction : BoolIterationFunction {
            public override BT ReturnType() {
                return BT.Int;
            }
            public override ExecResult InternalExecuteShortStyle(BArray array, EvalContext context, Evaluator evaluator, BV result) {
                int array_len = array.Count;
                int count = 0;
                for(int i=0; i<array_len; i++) {
                    //ʂ̕]
                    context.ContextValue = array[i];
                    BBoolean t = (BBoolean)evaluator.Eval(context);
                    if(t.IsNil) {
                        return ExecResult.Nil;
                    }
                    if(t.Value) count++;
                }

                ((BInt)result).Value = count; //
                return ExecResult.OK;
            }
            public override ExecResult InternalExecuteFunctionStyle(BArray array, EvalContext context, BFunction function, BV result) {
                int array_len = array.Count;
                int count = 0;
                BBoolean tempResult = new BBoolean(false);
                BV[] args = new BV[1];
                for(int i=0; i<array_len; i++) {
                    //ʂ̕]
                    args[0] = array[i];
                    ExecResult r = function.Execute(null, context, args, tempResult);
                    Debug.Assert(tempResult!=null && tempResult.BT==BT.Bool);
                    if(r==ExecResult.Nil) {
                        return ExecResult.Nil;
                    }
                    if(((BBoolean)tempResult).Value) count++;
                }

                ((BInt)result).Value = count; //
                return ExecResult.OK;
            }
        }
        private class AllFunction : BoolIterationFunction {
            public override BT ReturnType() {
                return BT.Bool;
            }
            public override ExecResult InternalExecuteShortStyle(BArray array, EvalContext context, Evaluator evaluator, BV result) {
                int array_len = array.Count;
                for(int i=0; i<array_len; i++) {
                    //ʂ̕]
                    context.ContextValue = array[i];
                    BBoolean t = (BBoolean)evaluator.Eval(context);
                    if(t.IsNil) {
                        return ExecResult.Nil;
                    }
                    if(!t.Value) {
                        ((BBoolean)result).Value = false; //łfalseȂSfalse
                        return ExecResult.OK;
                    }
                }
                ((BBoolean)result).Value = true;
                return ExecResult.OK;

            }
            public override ExecResult InternalExecuteFunctionStyle(BArray array, EvalContext context, BFunction function, BV result) {
                int array_len = array.Count;
                BBoolean tempResult = new BBoolean(false);
                BV[] args = new BV[1];
                for(int i=0; i<array_len; i++) {
                    //ʂ̕]
                    args[0] = array[i];
                    ExecResult r = function.Execute(null, context, args, tempResult);
                    Debug.Assert(tempResult!=null && tempResult.BT==BT.Bool);
                    if(r==ExecResult.Nil) {
                        return ExecResult.Nil;
                    }
                    if(!((BBoolean)tempResult).Value) {
                        ((BBoolean)result).Value = false; //łfalseȂSfalse
                        return ExecResult.OK;
                    }
                }
                ((BBoolean)result).Value = true;
                return ExecResult.OK;

            }
        }
        private class AnyFunction : BoolIterationFunction {
            public override BT ReturnType() {
                return BT.Bool;
            }
            public override ExecResult InternalExecuteShortStyle(BArray array, EvalContext context, Evaluator evaluator, BV result) {
                int array_len = array.Count;
                for(int i=0; i<array_len; i++) {
                    //ʂ̕]
                    context.ContextValue = array[i];
                    BBoolean t = (BBoolean)evaluator.Eval(context);
                    if(t.IsNil) {
                        return ExecResult.Nil;
                    }
                    if(t.Value) {
                        ((BBoolean)result).Value = true; //łtrueȂStrue
                        return ExecResult.OK;
                    }
                }
                ((BBoolean)result).Value = false;
                return ExecResult.OK;

            }
            public override ExecResult InternalExecuteFunctionStyle(BArray array, EvalContext context, BFunction function, BV result) {
                int array_len = array.Count;
                BBoolean tempResult = new BBoolean(false);
                BV[] args = new BV[1];
                for(int i=0; i<array_len; i++) {
                    //ʂ̕]
                    args[0] = array[i];
                    ExecResult r = function.Execute(null, context, args, tempResult);
                    Debug.Assert(tempResult!=null && tempResult.BT==BT.Bool);
                    if(r==ExecResult.Nil) {
                        return ExecResult.Nil;
                    }
                    if(((BBoolean)tempResult).Value) {
                        ((BBoolean)result).Value = true; //łtrueȂStrue
                        return ExecResult.OK;
                    }
                }
                ((BBoolean)result).Value = false;
                return ExecResult.OK;
            }
        }
        private class MapFunction : IBSystemFunction {
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                EvaluatorBuildContext context = builder.Context;
                BT targetType = context.ContextType;

                //TODOQmap
                if(targetType!=null && targetType.AsArray()!=null && expr.ArgCount==1) {
                    BArrayT array = targetType.AsArray();

                    context.ContextType = array.ElementType; //ElementTypêƂŕ]
                    Evaluator v = expr[0].Apply<Evaluator>(builder);
                    context.ContextType = targetType; //ɖ߂

                    //arg_evals = new Evaluator[] { v }; //OK!
                    BFunctionT ft = v.BT as BFunctionT;
                    if(ft!=null) { //֐łȂ΁AElementTypeP^łȂ΂ȂȂ
                        if(ft.IsMatchArgs(null, new BT[] { array.ElementType }))
                            return new BSystemFunctionArg(ft.ReturnType.ArrayType(), new Evaluator[] { v });
                        else
                            throw new ScriptException("Arg Type error in array iteration"); //G[bZ[W܂Ƃ
                    }
                    else
                        return new BSystemFunctionArg(v.BT.ArrayType(), new Evaluator[] { v });
                }
                else
                    throw new ScriptException("Arg Type error in array iteration"); //G[bZ[W܂Ƃ
            }

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                BArray target = context.ContextValue.AsArray();
                BRegularArray arr = (BRegularArray)result; //TODO ^xŁAꂪRegularArrayɂȂ邱Ƃۏ؂ׂ
                int length = target.Count;
                arr.Extend(length); //^[QbgƓ̔zɂȂ

                Evaluator ev = evaluators[0];
                BFunctionT ft = ev.BT as BFunctionT;
                if(ft!=null) { //֐X^C
                    BFunction f = (BFunction)ev.Eval(context);
                    BV[] args = new BV[1];
                    for(int i=0; i<length; i++) {
                        args[0] = target[i];
                        BV x = arr[i];
                        ExecResult r = f.Execute(null, context, args, x);
                        if(r==ExecResult.Nil) return r;
                        arr.LetAt(i, x);
                    }
                    return ExecResult.OK;
                }
                else {
                    for(int i=0; i<length; i++) {
                        context.ContextValue = target[i];
                        BV x = ev.Eval(context);
                        if(x.IsNil) return ExecResult.Nil;
                        arr.SetAt(i, x);
                    }
                    context.ContextValue = target;
                    return ExecResult.OK;
                }
            }
        }

        //TODO 1
        private class CarFunction : IBSystemFunction {
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                EvaluatorBuildContext context = builder.Context;
                BT targetType = context.ContextType;

                if(targetType!=null && targetType.AsArray()!=null && expr.ArgCount==0) {
                    BArrayT array = targetType.AsArray();
                    return new BSystemFunctionArg(array.ElementType,  new Evaluator[0]);
                }
                else
                    throw new ScriptException("Arg Type error in array iteration"); //G[bZ[W܂Ƃ
            }

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                BArray target = context.ContextValue.AsArray();
                if(target.Count==0)
                    return ExecResult.Nil;
                result.Let(target[target.Count-1]); //z̍Ōオcar
                return ExecResult.OK;
            }
        }
        private class CdrFunction : IBSystemFunction {
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                EvaluatorBuildContext context = builder.Context;
                BT targetType = context.ContextType;

                if(targetType!=null && targetType.AsArray()!=null && expr.ArgCount==0) {
                    BArrayT array = targetType.AsArray();
                    return new BSystemFunctionArg(array, new Evaluator[0]);
                }
                else
                    throw new ScriptException("Arg Type error in array iteration"); //G[bZ[W܂Ƃ
            }

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                BArray target = context.ContextValue.AsArray();
                if(target.Count==0)
                    return ExecResult.Nil;
                else {
                    BRegularArray arr = result as BRegularArray;
                    arr.Extend(target.Count-1);
                    for(int i=0; i<target.Count-1; i++)
                        arr.SetAt(i, target[i]);
                    return ExecResult.OK;
                }
            }
        }

        //nilProof(expr, value) expr]nilȂvalue, łȂ΂̒l
        private class NilProofFunction : IBSystemFunction {
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                if(expr.ArgCount==2) {
                    Evaluator body = expr[0].Apply<Evaluator>(builder);
                    Evaluator nil  = expr[1].Apply<Evaluator>(builder);
                    if(body.BT!=nil.BT)
                        throw new ScriptException("nilProof type mismatch error");
                    return new BSystemFunctionArg(body.BT, new Evaluator[2] { body, nil });
                }
                throw new ScriptException("nilProof arg count error");
            }

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                BV value = evaluators[0].Eval(context);
                if(value.IsNil) {
                    value = evaluators[1].Eval(context);
                    if(value.IsNil)
                        return ExecResult.Nil;
                }
                result.Let(value);
                return ExecResult.OK;
            }
        }

        //ԃVtg
        /*
    private class MinShiftFunction : IBSystemFunction {
        public enum Behavior {
            Regular, Stock
        }
        private Behavior _behavior; //x݂̍l邩ǂBStock̂Ƃ͂
            
        public MinShiftFunction(Behavior b) {
            _behavior = b;
        }

        public void CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr, out Evaluator[] arg_evals, out BT retType) {
            if(expr.ArgCount==2) {
                Evaluator a = expr[0].Apply<Evaluator>(builder);
                if(a.BT.IsConvertibleTo(BT.Int)) {
                    Evaluator v = expr[1].Apply<Evaluator>(builder);
                    arg_evals = new Evaluator[] { a, v }; //OK!
                    retType = v.BT;
                    return;
                }
            }
            throw new ScriptException("Arg Type error in system function"); //G[bZ[W܂Ƃ
        }
        public void Execute(EvalContext context, Evaluator[] evaluators, ref BV result) {
            BTime time = context.CurrentTime;
            Debug.Assert(time!=null);
            int save = time.AsInt();

            BInt shift = (BInt)evaluators[0].Eval(context);

            if(_behavior==Behavior.Stock) //x݂l^Cv̂Ƃ...
                AdjustLunchBreak(context, time,  -shift.Value * 60);
            else
                time.LetInt(save - shift.Value*60); //Pʈړ


            result = evaluators[1].Eval(context);

            time.LetInt(save);
        }
        private void AdjustLunchBreak(EvalContext context, BTime time, int add_seconds) {
            StockExchange m = context.CurrentStock==null? StockExchange.T : context.CurrentStock.Stock.Market;
            int bs = MarketUtil.GetZenbaCloseTime(m).AsInt();
            int be = MarketUtil.GetGobaOpenTime(m).AsInt();
            int t = time.AsInt();
            int s = t + add_seconds; //ʂ̌vZ
            if(add_seconds > 0 && t <= bs) {
                if(bs <= s)
                    s += be - bs;
            }
            else if(add_seconds < 0 && t >= be) {
                if(be > s)
                    s -= be - bs;
            }

            time.LetInt(s);

        }
    }
        */


        private abstract class ContextObtainingFunction : IBSystemFunction {
            protected string _name;
            protected BT _retType;
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr) {
                if(expr.ArgCount!=0)
                    throw new ScriptException(String.Format("{0}ɂ͈͕svł", _name));
                return new BSystemFunctionArg(_retType, new Evaluator[0]);
            }

            public abstract ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result);
        }
        private class CurrentTimeFunction : ContextObtainingFunction {
            public CurrentTimeFunction() {
                _name = "now";
                _retType = BT.Time;
            }
            public override ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                if(context.CurrentTime==null)
                    return ExecResult.Nil;
                ((BTime)result).LetInt(context.CurrentTime.AsInt());
                return ExecResult.OK;
            }
        }
        private class CurrentDateFunction : ContextObtainingFunction {
            public CurrentDateFunction() {
                _name = "today";
                _retType = BT.Date;
            }
            public override ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                if(context.CurrentTime==null)
                    return ExecResult.Nil;
                ((BDate)result).LetInt(context.CurrentDate.AsInt());
                return ExecResult.OK;
            }
        }
        private class CurrentStockFunction : ContextObtainingFunction {
            public CurrentStockFunction() {
                _name = "stock";
                _retType = StockType.instance;
            }
            public override ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                if(context.CurrentStock==null)
                    return ExecResult.Nil;
                ((StockRef)result).Stock = context.CurrentStock.Stock;
                return ExecResult.OK;
            }
        }
    }

#if UNITTEST
    public class EvaluatorAndContext {
        public Evaluator evaluator;
        public BStackFrame initialStack;
        public Stock contextStock;
        public EvaluatorAndContext(Evaluator e, EvaluatorBuildContext c) {
            evaluator = e;
            initialStack = new BStackFrame(c, new BV[0]);
        }
        public BV Eval() {
            EvalContext c = new EvalContext();
            c.InitializeStack(initialStack);
            if(contextStock!=null)
                c.CurrentStock = new StockRef(contextStock);
            return evaluator.Eval(c);
        }

        public static EvaluatorAndContext CreateEvaluator(string expression) {
            Expression expr = Expression.ParseExpression(expression);
            EvaluatorBuildContext c  = new EvaluatorBuildContext(null);
            c.Initialize(null, LocalVariable.EmptyArray);

            EvaluatorBuilder bld = new EvaluatorBuilder(c);
            Evaluator ev = bld.Build(expr);

            return new EvaluatorAndContext(ev, c);
        }
    }


    [TestFixture]
    public class BuiltInFunctionLibraryTests {
        private ValueSystemRoot _root;

        [TestFixtureSetUp]
        public void InitLibrary() {
            _root = new ValueSystemRoot();
            EvaluatorBuildContext._root = _root;
            FunctionLibrary._root = _root;

            _root.BuiltIn.DefineTestFunctions();
            
        }
        private EvaluatorAndContext CreateEvaluator(string expression) {
            return EvaluatorAndContext.CreateEvaluator(expression);
        }

        [Test]
        public void Sum() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().sum()");
            EvaluatorAndContext ev2 = CreateEvaluator("sum(ia12())");
            EvaluatorAndContext ev3 = CreateEvaluator("da12().sum()");
            EvaluatorAndContext ev4 = CreateEvaluator("sum(da12())");

            int count = BV.instanceCount;
            BInt r1 = (BInt)ev1.Eval();
            Assert.AreEqual(3, r1.Value);

            BInt r2 = (BInt)ev2.Eval();
            Assert.AreEqual(3, r2.Value);

            BDouble r3 = (BDouble)ev3.Eval();
            Assert.AreEqual(3.0, r3.Value);

            BDouble r4 = (BDouble)ev4.Eval();
            Assert.AreEqual(3.0, r4.Value);

            EvaluatorAndContext ev5 = CreateEvaluator("ianil().sum()");
            BV r5 = (BV)ev5.Eval();
            Assert.IsTrue(r5.IsNil); //NilABInt̃CX^Xł邱Ƃɒ
        }

        [Test]
        public void Min() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().min()");
            EvaluatorAndContext ev2 = CreateEvaluator("min(ia12())");
            EvaluatorAndContext ev3 = CreateEvaluator("da12().min()");
            EvaluatorAndContext ev4 = CreateEvaluator("min(da12())");

            int count = BV.instanceCount;
            BInt r1 = (BInt)ev1.Eval();
            Assert.AreEqual(1, r1.Value);

            BInt r2 = (BInt)ev2.Eval();
            Assert.AreEqual(1, r2.Value);

            BDouble r3 = (BDouble)ev3.Eval();
            Assert.AreEqual(1.0, r3.Value);

            BDouble r4 = (BDouble)ev4.Eval();
            Assert.AreEqual(1.0, r4.Value);

            //Assert.AreEqual(count, BV.instanceCount);

            EvaluatorAndContext ev5 = CreateEvaluator("ianil().min()");
            BV r5 = (BV)ev5.Eval();
            Assert.IsTrue(r5.IsNil); 
        }

        [Test]
        public void Max() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().max()");
            EvaluatorAndContext ev2 = CreateEvaluator("max(ia12())");
            EvaluatorAndContext ev3 = CreateEvaluator("da12().max()");
            EvaluatorAndContext ev4 = CreateEvaluator("max(da12())");

            int count = BV.instanceCount;
            BInt r1 = (BInt)ev1.Eval();
            Assert.AreEqual(2, r1.Value);

            BInt r2 = (BInt)ev2.Eval();
            Assert.AreEqual(2, r2.Value);

            BDouble r3 = (BDouble)ev3.Eval();
            Assert.AreEqual(2.0, r3.Value);

            BDouble r4 = (BDouble)ev4.Eval();
            Assert.AreEqual(2.0, r4.Value);

            //͖̏ȂȂ
            //Assert.AreEqual(count, BV.instanceCount);

            EvaluatorAndContext ev5 = CreateEvaluator("ianil().max()");
            BV r5 = ev5.Eval();
            Assert.IsTrue(r5.IsNil);
        }

        [Test]
        public void Average() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().avg()");
            EvaluatorAndContext ev2 = CreateEvaluator("avg(ia12())");
            EvaluatorAndContext ev3 = CreateEvaluator("da12().avg()");
            EvaluatorAndContext ev4 = CreateEvaluator("avg(da12())");

            int count = BV.instanceCount;
            BDouble r1 = (BDouble)ev1.Eval();
            Assert.AreEqual(1.5, r1.Value);

            BDouble r2 = (BDouble)ev2.Eval();
            Assert.AreEqual(1.5, r2.Value);

            BDouble r3 = (BDouble)ev3.Eval();
            Assert.AreEqual(1.5, r3.Value);

            BDouble r4 = (BDouble)ev4.Eval();
            Assert.AreEqual(1.5, r4.Value);

            //Assert.AreEqual(count, BV.instanceCount); ςł͍̎łBVĂ܂Ƃ

            EvaluatorAndContext ev5 = CreateEvaluator("ianil().avg()");
            BV r5 = (BV)ev5.Eval();
            Assert.IsTrue(r5.IsNil);
        }

        [Test]
        public void SD() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().sdev()");
            EvaluatorAndContext ev2 = CreateEvaluator("sdev(ia12())");
            EvaluatorAndContext ev3 = CreateEvaluator("da12().sdev()");
            EvaluatorAndContext ev4 = CreateEvaluator("sdev(da12())");

            int count = BV.instanceCount;
            BDouble r1 = (BDouble)ev1.Eval();
            Assert.AreEqual(0.5, r1.Value);

            BDouble r2 = (BDouble)ev2.Eval();
            Assert.AreEqual(0.5, r2.Value);

            BDouble r3 = (BDouble)ev3.Eval();
            Assert.AreEqual(0.5, r3.Value);

            BDouble r4 = (BDouble)ev4.Eval();
            Assert.AreEqual(0.5, r4.Value);

            //Assert.AreEqual(count, BV.instanceCount); W΍ł̓CX^XĂ܂

            EvaluatorAndContext ev5 = CreateEvaluator("ianil().sdev()");
            BV r5 = (BV)ev5.Eval();
            Assert.IsTrue(r5.IsNil);
        }

        [Test]
        public void Array() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().length()");
            EvaluatorAndContext ev2 = CreateEvaluator("da12().length()");

            BInt r1 = (BInt)ev1.Eval();
            Assert.AreEqual(2, r1.Value);

            BInt r2 = (BInt)ev2.Eval();
            Assert.AreEqual(2, r2.Value);

            //zvfANZXFfunctionLibƂ͂܂֌WȂȒPɃANZXłz񂪂ȂƃeXgP[XÂ炢̂
            EvaluatorAndContext ev3 = CreateEvaluator("ia12()[1]");
            EvaluatorAndContext ev4 = CreateEvaluator("da12()[0]");
            BInt r3 = (BInt)ev3.Eval();
            Assert.AreEqual(2, r3.Value);
            BDouble r4 = (BDouble)ev4.Eval();
            Assert.AreEqual(1.0, r4.Value);
        }

        [Test]
        public void ArrayArith() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12() + ia12()");
            BArray r1 = (BArray)ev1.Eval();
            Assert.AreEqual(2, r1.Count);
            Assert.AreEqual(2, ((BInt)r1[0]).Value);
            Assert.AreEqual(4, ((BInt)r1[1]).Value);

            EvaluatorAndContext ev2 = CreateEvaluator("da12() - da12()");
            BArray r2 = (BArray)ev2.Eval();
            Assert.AreEqual(2, r2.Count);
            Assert.AreEqual(0, ((BDouble)r2[0]).Value);
            Assert.AreEqual(0, ((BDouble)r2[1]).Value);
        }

        [Test]
        public void Abs() {
            EvaluatorAndContext ev1 = CreateEvaluator("abs(-5)");
            EvaluatorAndContext ev2 = CreateEvaluator("abs(3.0)");

            BInt r1 = (BInt)ev1.Eval();
            Assert.AreEqual(5, r1.Value); 
            BDouble r2 = (BDouble)ev2.Eval();
            Assert.AreEqual(3.0, r2.Value);

            EvaluatorAndContext ev3 = CreateEvaluator("abs(ianil().avg())");
            BV r3 = (BV)ev3.Eval();
            Assert.IsTrue(r3.IsNil);
        }

        [Test]
        public void Sqrt() {
            EvaluatorAndContext ev1 = CreateEvaluator("sqrt(256.0)");
            BDouble r1 = (BDouble)ev1.Eval();
            Assert.AreEqual(16, r1.Value);

            EvaluatorAndContext ev2 = CreateEvaluator("sqrt(ianil().avg())");
            BV r2 = (BV)ev2.Eval();
            Assert.IsTrue(r2.IsNil);
        }
        [Test]
        public void Pow() {
            EvaluatorAndContext ev1 = CreateEvaluator("pow(3.0, 4.0)");
            BDouble r1 = (BDouble)ev1.Eval();
            Assert.AreEqual(81, r1.Value);

            EvaluatorAndContext ev2 = CreateEvaluator("pow(ianil().avg(), 2.0)");
            BV r2 = (BV)ev2.Eval();
            Assert.IsTrue(r2.IsNil);
        }
        [Test]
        public void Double2Int() {
            //round͎ľܓł͂ȂBQ̐̒Ԃł͋Ɋۂ܂
            //Ȃ݂ɁAdoubleintփLXgƐ̂ƂFloorÂƂCeilingɂȂ

            EvaluatorAndContext ev1 = CreateEvaluator("round(1.5)");
            BInt r1 = (BInt)ev1.Eval();
            Assert.AreEqual(2, r1.Value);

            EvaluatorAndContext ev2 = CreateEvaluator("round(2.5)");
            BInt r2 = (BInt)ev2.Eval();
            Assert.AreEqual(2, r2.Value);

            EvaluatorAndContext ev3 = CreateEvaluator("round(3.5)");
            BInt r3 = (BInt)ev3.Eval();
            Assert.AreEqual(4, r3.Value);

            EvaluatorAndContext ev4 = CreateEvaluator("floor(2.5)");
            BInt r4 = (BInt)ev4.Eval();
            Assert.AreEqual(2, r4.Value);

            EvaluatorAndContext ev5 = CreateEvaluator("ceiling(2.5)");
            BInt r5 = (BInt)ev5.Eval();
            Assert.AreEqual(3, r5.Value);
        }

        [Test]
        public void Log() {
            EvaluatorAndContext ev1 = CreateEvaluator("log(100.0)");
            BDouble r1 = (BDouble)ev1.Eval();
            Assert.AreEqual(Math.Log(100), r1.Value);

            EvaluatorAndContext ev2 = CreateEvaluator("log(ianil().avg())");
            BV r2 = (BV)ev2.Eval();
            Assert.IsTrue(r2.IsNil);
        }

        [Test]
        public void Nil() {
            //nilo镁ʂ̉ZƂƂnil`d̂mF
            EvaluatorAndContext ev1 = CreateEvaluator("3.0 < 2 * ianil().avg() && (ia12().length() > 3)");
            BV r1 = (BV)ev1.Eval();
            Assert.IsTrue(r1.IsNil);
        }

        [Test]
        public void Let1() {
            EvaluatorAndContext ev1 = CreateEvaluator("let ((qqq=3) qqq)");
            BV r1 = (BV)ev1.Eval();
            Assert.AreSame(ev1.evaluator.BT, BT.Int);
            Assert.AreSame(r1.BT, BT.Int);
            Assert.AreEqual(3, ((BInt)r1).Value);
        }

        [Test]
        public void Let2() {
            EvaluatorAndContext ev1 = CreateEvaluator("let ((qqq = 3, sss = 2) qqq + sss)");
            BV r1 = (BV)ev1.Eval();
            Assert.AreSame(ev1.evaluator.BT, BT.Int);
            Assert.AreSame(r1.BT, BT.Int);
            Assert.AreEqual(5, ((BInt)r1).Value);
        }
        [Test]
        public void Let3() {
            //QԖڂł͑O̕ϐg
            EvaluatorAndContext ev1 = CreateEvaluator("let ((qqq = 3, sss = qqq*3) sss + 1)");
            BV r1 = (BV)ev1.Eval();
            Assert.AreSame(ev1.evaluator.BT, BT.Int);
            Assert.AreSame(r1.BT, BT.Int);
            Assert.AreEqual(10, ((BInt)r1).Value);
        }
        [Test]
        public void Lambda1() {
            EvaluatorAndContext ev1 = CreateEvaluator("let((f = lambda ((int x)  x*3 )) f(5))");
            BV r1 = (BV)ev1.Eval();
            Assert.AreSame(ev1.evaluator.BT, BT.Int);
            Assert.AreSame(r1.BT, BT.Int);
            Assert.AreEqual(15, ((BInt)r1).Value);

            EvaluatorAndContext ev2 = CreateEvaluator("let((f = lambda ((double x, double y)  x*y )) f(2.0, 4.0))");
            BV r2 = (BV)ev2.Eval();
            Assert.AreSame(ev2.evaluator.BT, BT.Double);
            Assert.AreSame(r2.BT, BT.Double);
            Assert.AreEqual(8.0, ((BDouble)r2).Value);
        }
        [Test]
        public void Lambda2() {
            //invokeł悤ɂȂ
            EvaluatorAndContext ev1 = CreateEvaluator("lambda((int x) x * 2)(3)");
            BV value = ev1.Eval();
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(6, ((BInt)value).Value);
        }
        [Test]
        public void Lambda3() {
            //Oϐ
            EvaluatorAndContext ev1 = CreateEvaluator("let((a = 2*4, f = lambda((int x) x * a)) f(3))");
            BV value = ev1.Eval();
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(24, ((BInt)value).Value);
        }

        [Test]
        public void BooleanSystem() {
            //ReLXgύX^CvTradingObjectsŃeXg
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().count(lambda((int a) a>1))");
            BV r1 = (BV)ev1.Eval();
            Assert.AreEqual(1, ((BInt)r1).Value);

            EvaluatorAndContext ev2 = CreateEvaluator("ia12().any(lambda((int a) a>1))");
            BV r2 = (BV)ev2.Eval();
            Assert.AreEqual(true, ((BBoolean)r2).Value);

            EvaluatorAndContext ev3 = CreateEvaluator("ia12().all(lambda((int a) a>1))");
            BV r3 = (BV)ev3.Eval();
            Assert.AreEqual(false, ((BBoolean)r3).Value);
        }
        [Test]
        public void MapSystem() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().map(lambda((int a) a*2))");
            BV r1 = (BV)ev1.Eval();
            BArrayT r1t = r1.BT.AsArray();
            Assert.IsTrue(r1t!=null);
            Assert.IsTrue(r1t.ElementType==BT.Int);
            Assert.AreEqual(2, r1.AsArray().Count);
            Assert.AreEqual(2, ((BInt)(r1.AsArray()[0])).Value);
            Assert.AreEqual(4, ((BInt)(r1.AsArray()[1])).Value);
        }
        [Test]
        public void CarCdr() {
            EvaluatorAndContext ev1 = CreateEvaluator("ia12().car()");
            BV r1 = ev1.Eval();
            Assert.AreEqual(2, ((BInt)r1).Value);

            EvaluatorAndContext ev2 = CreateEvaluator("ia12().cdr()");
            BV r2 = ev2.Eval();
            Assert.AreEqual(1, r2.AsArray().Count);
            Assert.AreEqual(1, ((BInt)(r2.AsArray()[0])).Value);

            EvaluatorAndContext ev3 = CreateEvaluator("ia12().cdr().car()");
            BV r3 = ev3.Eval();
            Assert.AreEqual(1, ((BInt)r3).Value);

            Assert.AreEqual(1, r2.AsArray().Count);

            EvaluatorAndContext ev4 = CreateEvaluator("ia12().cdr().cdr()");
            BV r4 = ev4.Eval();
            Assert.AreEqual(0, r4.AsArray().Count);

            EvaluatorAndContext ev5 = CreateEvaluator("ia12().cdr().cdr().cdr()");
            BV r5 = ev5.Eval();
            Assert.IsTrue(r5.IsNil);
        }
        [Test]
        public void Recursive() {
            //`ς݊KgčċAĂ݂
            EvaluatorAndContext ev1 = CreateEvaluator("factorial(5)");
            BV r1 = ev1.Eval();
            Assert.AreEqual(120, ((BInt)r1).Value);

            EvaluatorAndContext ev2 = CreateEvaluator("factorial_tail_rec(6, 1)");
            BV r2 = ev2.Eval();
            Assert.AreEqual(720, ((BInt)r2).Value);

            EvaluatorAndContext ev3 = CreateEvaluator("fib(6)");
            BV r3 = ev3.Eval();
            Assert.AreEqual(13, ((BInt)r3).Value);
        }
        /*
        [Test]
        public void NowAndMinShift() {
            EvaluatorAndContext ev1 = CreateEvaluator("now()");
            EvalContext ec = ;
            ec.CurrentTime = new BTime(10, 0, 0);
            BV r = (BV)ev1.Eval(ec);
            Assert.AreSame(r.BT, BT.Time);
            Assert.AreEqual(new BTime(10, 0, 0).AsInt(), ((BTime)r).AsInt());

            //ԃVtg
            EvaluatorAndContext ev_abs   = CreateEvaluator("minshiftABS(-90, now())");
            EvaluatorAndContext ev_lunch = CreateEvaluator("minshift(-90, now())");
            r = (BV)ev_abs.Eval(ec);
            Assert.AreEqual(new BTime(11, 30, 0).AsInt(), ((BTime)r).AsInt());
            r = (BV)ev_lunch.Eval(ec);
            Assert.AreEqual(new BTime(13, 0, 0).AsInt(), ((BTime)r).AsInt()); //x30Ԃ

            //t
            ec.CurrentTime = new BTime(13, 0, 0);
            ev_abs   = CreateEvaluator("minshiftABS(60, now())");
            ev_lunch = CreateEvaluator("minshift(60, now())");
            r = (BV)ev_abs.Eval(ec);
            Assert.AreEqual(new BTime(12, 0, 0).AsInt(), ((BTime)r).AsInt());
            r = (BV)ev_lunch.Eval(ec);
            Assert.AreEqual(new BTime(10, 30, 0).AsInt(), ((BTime)r).AsInt()); //x30Ԃ

            //x݂܂Ȃminshift
            ec.CurrentTime = new BTime(10, 0, 0);
            ev_lunch = CreateEvaluator("minshift(10, now())");
            r = (BV)ev_lunch.Eval(ec);
            Assert.AreEqual(new BTime(9, 50, 0).AsInt(), ((BTime)r).AsInt());

            ec.CurrentTime = new BTime(13, 0, 0);
            ev_lunch = CreateEvaluator("minshift(-10, now())");
            r = (BV)ev_lunch.Eval(ec);
            Assert.AreEqual(new BTime(13, 10, 0).AsInt(), ((BTime)r).AsInt());

            //TODO Stock, Date
        }
        */
        [Test]
        public void RCI() {
            EvaluatorAndContext ev1 = CreateEvaluator("rci(da12())");
            BV r1 = (BV)ev1.Eval();
            Assert.AreEqual(1.0, ((BDouble)r1).Value);

            double v1 = BuiltInFunctionLibrary.RCIBody(new BFastenDoubleArray(new double[] { 150, 140, 130, 120, 110 }));
            Assert.AreEqual(-1.0, v1);
            double v2 = BuiltInFunctionLibrary.RCIBody(new BFastenDoubleArray(new double[] { 100, 110, 110 }));
            Assert.AreEqual(0.875, v2); //vZƂ̏ƍ
        }

        [Test]
        public void Series() {
            EvaluatorAndContext ev1 = CreateEvaluator("series(3)");
            BV r1 = ev1.Eval();
            Assert.IsTrue(r1 is BRegularArray);
            BRegularArray arr = (BRegularArray)r1;
            Assert.AreEqual(BT.Int, arr.BArrayT.ElementType);
            Assert.AreEqual(3, arr.Count);
            Assert.AreEqual(1, ((BInt)arr[1]).Value);

            ev1 = CreateEvaluator("series(3, 8)");
            BV r2 = ev1.Eval();
            arr = (BRegularArray)r2;
            Assert.AreEqual(5, arr.Count);
            Assert.AreEqual(5, ((BInt)arr[2]).Value);
        }
        [Test]
        public void RSeries() {
            EvaluatorAndContext ev1 = CreateEvaluator("rseries(3)");
            BV r1 = ev1.Eval();
            Assert.IsTrue(r1 is BRegularArray);
            BRegularArray arr = (BRegularArray)r1;
            Assert.AreEqual(BT.Int, arr.BArrayT.ElementType);
            Assert.AreEqual(3, arr.Count);
            Assert.AreEqual(1, ((BInt)arr[1]).Value);

            ev1 = CreateEvaluator("rseries(3, 8)");
            BV r2 = ev1.Eval();
            arr = (BRegularArray)r2;
            Assert.AreEqual(5, arr.Count);
            Assert.AreEqual(3, ((BInt)arr[4]).Value);
        }
        [Test]
        public void NilProof() {
            EvaluatorAndContext ev1 = CreateEvaluator("nilProof(1 / 0, 3)");
            BV r1 = ev1.Eval();
            Assert.AreEqual(BT.Int, r1.BT);
            Assert.AreEqual(3, ((BInt)r1).Value);
        }

    }
#endif
}
