/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Personal/IndexFile.cs#3 $
 * $DateTime$
 * 
 * ̃CfbNXt@C̃[h
 */
using System;
using System.Diagnostics;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Threading;
using System.Net;

using Poderosa;
using Bellagio.Data;
using Bellagio.Environment;

namespace Bellagio.Personal {
    internal class IndexFile : IBellagioBootProcedure {

        public void Load(string filename, IStockListConstructor collection, IBellagioLoadingLog log) {
            //indext@Cǂō\z
            using(StreamReader sr = new StreamReader(filename, Encoding.Default)) {
                string line = sr.ReadLine();
                if(line.StartsWith("date=")) _version = line.Substring(5);

                line = sr.ReadLine(); 
                int linenum = 2;
                List<SplitInfo> splits = new List<SplitInfo>();

                while(line!=null) {
                    string[] d = line.Split(',');
                    //
                    //9984,\tgoN,T1,100,TA,R:NK225,S:19950328:1.4,S:19950926:1.4,S:19960326:1.4,S:19960925:1.4,S:19970326:1.3,S:20000425:3,S:20051228:3

                    if(d.Length<5) throw new IOException("index file format error: line"+ linenum);
                    StockExchange exch;
                    StockExchangeSubType subtype;
                    try {
                        BasicStockProfile br = new BasicStockProfile(d[1], d[0], StockProfileFlags.None, Int32.Parse(d[3]));
                        ParseMarket(d[2], out exch, out subtype);
                        br.CreatePrimary(exch, subtype, d[4]=="TA"? StockFlags.Taisyaku: StockFlags.None);
                        collection.AddStock(br);
                        for(int i=5; i<d.Length; i++) {
                            string col = d[i];
                            if(col.Length>0 && col[0]=='S') splits.Add(new SplitInfo(d[i]));
                            if(col=="R:NK225")  br.Flags |= StockProfileFlags.Nikkei225;
                        }
                        if(splits.Count>0) {
                            br.SplitInfo = splits.ToArray();
                            splits.Clear();
                        }
                    }
                    catch(Exception ex) {
                        string msg = String.Format("{0} s{1} {2}", ex.Message, linenum, line);
                        log.Add(msg);
                    }
                    line = sr.ReadLine();
                    linenum++;
                }
            }
        }

        private ManualResetEvent _event;
        private IStockListConstructor _constructor;
        private IBellagioLoadingLog _log;
        private string _version;

        public void RunBootProcedure(IAdaptable server, IBellagioLoadingLog log) {
            _constructor = (IStockListConstructor)server.GetAdapter(typeof(IStockListConstructor));
            _log = log;
            Debug.Assert(_constructor!=null);
            _event = new ManualResetEvent(false);
            ThreadPool.QueueUserWorkItem(new WaitCallback(AsyncLoad));
        }

        public string VersionInfo {
            get {
                return _version;
            }
        }

        private void AsyncLoad(object state) {
            try {
                //wo^
                _constructor.AddStock(new MarketIndex("o", MarketIndex.Builtins.NK225, "0101"));
                _constructor.AddStock(new MarketIndex("TOPIX", MarketIndex.Builtins.TOPIX, "0201"));
                AddGyoshubetsu();

                string filename = PersonalPlugin.Instance.BellagioEnvironment.DataHomeDir + "index.txt";
                if(!File.Exists(filename))
                    DownloadIndexFile(filename);
                Load(filename, _constructor, _log);
            }
            catch(Exception ex) {
                _log.Add(ex.Message);
            }
            finally {
                _event.Set();
            }
        }
        private void DownloadIndexFile(string dest_file) {
            string url = String.Format("http://www.lagarto.co.jp/tactico/startup/dataget.cgi?ac=newbie&cv={0}", BellagioVersionInfo.BELLAGIO_VERSION);
            HttpWebRequest req = (HttpWebRequest)WebRequest.Create(url);
            req.AllowAutoRedirect = true;
            HttpWebResponse response = (HttpWebResponse)req.GetResponse();
            if(response.StatusCode==HttpStatusCode.OK) {
                //CfbNXXV
                FileStream fs = new FileStream(dest_file, FileMode.Create, FileAccess.Write);
                PUtil.CopyStream(response.GetResponseStream(), fs, 0x4000);
                fs.Close();
                response.Close();
                Debug.WriteLine("IndexFile Initialize");
            }
        }
        private void AddGyoshubetsu() {
            //Ǝʎw
            _constructor.AddStock(new MarketIndex("Y(Ǝ)", "I101", "I101"));
            _constructor.AddStock(new MarketIndex("z(Ǝ)", "I102", "I102"));
            _constructor.AddStock(new MarketIndex("(Ǝ)", "I103", "I103"));
            _constructor.AddStock(new MarketIndex("Hi(Ǝ)", "I104", "I104"));
            _constructor.AddStock(new MarketIndex("@(Ǝ)", "I105", "I105"));
            _constructor.AddStock(new MarketIndex("pvE(Ǝ)", "I106", "I106"));
            _constructor.AddStock(new MarketIndex("w(Ǝ)", "I107", "I107"));
            _constructor.AddStock(new MarketIndex("i(Ǝ)", "I108", "I108"));
            _constructor.AddStock(new MarketIndex("Ζ(Ǝ)", "I109", "I109"));
            _constructor.AddStock(new MarketIndex("S(Ǝ)", "I110", "I110"));
            _constructor.AddStock(new MarketIndex("q(Ǝ)", "I111", "I111"));
            _constructor.AddStock(new MarketIndex("S|(Ǝ)", "I112", "I112"));
            _constructor.AddStock(new MarketIndex("SE(Ǝ)", "I113", "I113"));
            _constructor.AddStock(new MarketIndex("@B(Ǝ)", "I114", "I114"));
            _constructor.AddStock(new MarketIndex("dC@(Ǝ)", "I115", "I115"));
            _constructor.AddStock(new MarketIndex("D(Ǝ)", "I116", "I116"));
            _constructor.AddStock(new MarketIndex("(Ǝ)", "I117", "I117"));
            _constructor.AddStock(new MarketIndex("Ap@(Ǝ)", "I118", "I118"));
            _constructor.AddStock(new MarketIndex("@(Ǝ)", "I119", "I119"));
            _constructor.AddStock(new MarketIndex("̑(Ǝ)", "I120", "I120"));
            _constructor.AddStock(new MarketIndex("(Ǝ)", "I121", "I121"));
            _constructor.AddStock(new MarketIndex("(Ǝ)", "I122", "I122"));
            _constructor.AddStock(new MarketIndex("s(Ǝ)", "I123", "I123"));
            _constructor.AddStock(new MarketIndex("̑Z(Ǝ)", "I124", "I124"));
            _constructor.AddStock(new MarketIndex("،(Ǝ)", "I125", "I125"));
            _constructor.AddStock(new MarketIndex("ی(Ǝ)", "I126", "I126"));
            _constructor.AddStock(new MarketIndex("sY(Ǝ)", "I127", "I127"));
            _constructor.AddStock(new MarketIndex("SEoX(Ǝ)", "I128", "I128"));
            _constructor.AddStock(new MarketIndex("^(Ǝ)", "I129", "I129"));
            _constructor.AddStock(new MarketIndex("C^(Ǝ)", "I130", "I130"));
            _constructor.AddStock(new MarketIndex("^(Ǝ)", "I131", "I131"));
            _constructor.AddStock(new MarketIndex("q(Ǝ)", "I132", "I132"));
            _constructor.AddStock(new MarketIndex("ʐM(Ǝ)", "I133", "I133"));
            _constructor.AddStock(new MarketIndex("d(Ǝ)", "I134", "I134"));
            _constructor.AddStock(new MarketIndex("KX(Ǝ)", "I135", "I135"));
            _constructor.AddStock(new MarketIndex("T[rX(Ǝ)", "I136", "I136"));
        }

        public void Join() {
            _event.WaitOne();
            _event.Close();
        }

        private static void ParseMarket(string code, out StockExchange exch, out StockExchangeSubType subtype) {
            char e = code[0];
            char s = code.Length>1? code[1] : '\0';
            if(e=='T') {
                exch = StockExchange.T;
                if(s=='1')
                    subtype = StockExchangeSubType.Ichibu;
                else if(s=='2')
                    subtype = StockExchangeSubType.Nibu;
                else
                    subtype = StockExchangeSubType.None;
            }
            else if(e=='O') {
                exch = StockExchange.O;
                if(s=='1')
                    subtype = StockExchangeSubType.Ichibu;
                else if(s=='2')
                    subtype = StockExchangeSubType.Nibu;
                else
                    subtype = StockExchangeSubType.None;
            }
            else if(e=='J') {
                exch = StockExchange.J;
                subtype = StockExchangeSubType.None;
            }
            else if(e=='H') {
                exch = StockExchange.Hercules;
                subtype = StockExchangeSubType.None;
            }
            else if(e=='M') {
                exch = StockExchange.T;
                subtype = StockExchangeSubType.Mothers;
            }
            else if(e=='N') {
                exch = StockExchange.Nagoya;
                subtype = StockExchangeSubType.None;
            }
            else
                throw new FormatException("Unknown market " + code);
        }
    }
}
