/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.file.dao.standard;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import jp.terasoluna.fw.file.annotation.FileFormat;
import jp.terasoluna.fw.file.annotation.OutputFileColumn;
import jp.terasoluna.fw.file.annotation.StringConverter;
import jp.terasoluna.fw.file.dao.FileException;
import jp.terasoluna.fw.file.dao.FileLineException;
import jp.terasoluna.fw.file.dao.FileLineWriter;

/**
 * t@CANZX(f[^)p̋ʃNXB
 * 
 * <p>
 * t@CANZX(f[^)s3̃NX(CSVAŒ蒷Aϒ) ɋʂ鏈܂Ƃ߂ۃNXB
 * t@C̎ނɑΉTuNXsB<br>
 * gp{@link jp.terasoluna.fw.file.dao.FileLineWriter}QƂ̂ƁB
 * </p>
 * <p>
 * t@C擾͉L̎菇ŌĂяo悤Ɏ邱ƁB
 * <ul>
 * <li>(init()ACX^XKPsȂ)</li>
 * <li>wb_擾(printHeaderLine())</li>
 * <li>f[^擾(printDataLine())</li>
 * <li>gC擾(printTrailerLine())</li>
 * </ul>
 * L̏Ԃł̂ݐmɏo͂łB<br>
 * 
 * </p>
 * 
 * @see jp.terasoluna.fw.file.dao.FileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.CSVFileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.FixedFileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.VariableFileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.PlainFileLineWriter
 * @param <T> t@CsIuWFNgB
 */
public abstract class AbstractFileLineWriter<T>
        implements FileLineWriter<T> {

    /**
     * t@CANZXiójp̕Xg[B
     */
    private Writer writer = null;

    /**
     * t@CANZXst@CB
     */
    private String fileName = null;

    /**
     * t@CGR[fBOB
     */
    private String fileEncoding = System.getProperty("file.encoding");

    /**
     * p[^NX̃NXB
     */
    private Class<T> clazz = null;

    /**
     * s؂蕶B
     */
    private String lineFeedChar = System.getProperty("line.separator");

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐB
     */
    private Field[] fields = null;

    /**
     * \bhIuWFNg
     */
    private Method[] methods = null;

    /**
     * JtH[}bg(t@Cji[}bvB
     */
    private Map<String, ColumnFormatter> columnFormatterMap = null;

    /**
     * f[^o͊mFptOB
     */
    private boolean writeData = false;

    /**
     * gCo͊mFptOB
     */
    private boolean writeTrailer = false;

    /**
     * ݏς݃f[^̍sB
     */
    private int currentLineCount = 0;

    /**
     * RXgN^B
     * 
     * @param fileName t@C
     * @param clazz p[^NX
     * @param columnFormatterMap eLXg擾[
     */
    public AbstractFileLineWriter(String fileName, Class<T> clazz,
            Map<String, ColumnFormatter> columnFormatterMap) {

        this.fileName = fileName;
        this.clazz = clazz;
        this.columnFormatterMap = columnFormatterMap;
    }

    /**
     * B
     */
    protected void init() {
        // t@CtH[}bg擾B
        FileFormat fileFormat = clazz.getAnnotation(FileFormat.class);

        // @FileFormatꍇAOX[B
        if (fileFormat == null) {
            throw new FileException("FileFormat annotation is not found.",
                    new IllegalStateException(), fileName);
        }

        // R[hB
        if (fileFormat.fileEncoding() != null
                && !"".equals(fileFormat.fileEncoding())) {
            this.fileEncoding = fileFormat.fileEncoding();
        }

        // s؂蕶B
        if (fileFormat.lineFeedChar() != null
                && !"".equals(fileFormat.lineFeedChar())) {
            this.lineFeedChar = fileFormat.lineFeedChar();
        }

        // s؂蕶3ȏ̏ꍇAOX[B
        if (lineFeedChar.length() > 2) {
            throw new FileException(new IllegalStateException(
                    "lineFeedChar length must be 1 or 2. but: "
                            + lineFeedChar.length()), fileName);
        }

        buildFields();

        // ㏑tOmF
        if (fileFormat.overWriteFlg()) {
            File file = new File(fileName);
            file.delete();
        }

        // t@CI[v
        try {
            writer = new BufferedWriter(new OutputStreamWriter(
                    (new FileOutputStream(fileName, true)), fileEncoding));
        } catch (UnsupportedEncodingException e) {
            throw new FileException(e, fileName);
        } catch (FileNotFoundException e) {
            throw new FileException(e, fileName);
        }
    }

    /**
     * t@CsIuWFNg̑̃tB[hIuWFNg̔z𐶐B
     */
    private void buildFields() {
        // tB[hIuWFNg𐶐
        fields = clazz.getDeclaredFields();

        Field[] fieldArray = new Field[fields.length];

        for (Field field : getFields()) {
            OutputFileColumn outputFileColumn = field
                    .getAnnotation(OutputFileColumn.class);
            if (outputFileColumn != null) {
                if (fieldArray[outputFileColumn.columnIndex()] == null) {
                    fieldArray[outputFileColumn.columnIndex()] = field;
                } else {
                    throw new FileException("Column Index is duplicate : "
                            + outputFileColumn.columnIndex(), fileName);
                }
            }
        }
        buildMethods();
    }

    /**
     * t@CsIuWFNg̑getter\bh̃\bhIuWFNg̔z𐶐B
     */
    private void buildMethods() {
        // \bhIuWFNg𐶐
        List<Method> methodList = new ArrayList<Method>();
        List<Field> fieldList = new ArrayList<Field>();
        StringBuilder getterName = new StringBuilder();
        for (Field field : fields) {
            if (field != null
                    && field.getAnnotation(OutputFileColumn.class) != null) {

                // JavaBean珈̑ΏۂƂȂ鑮̑擾B
                String fieldName = field.getName();

                // ɁAsetter\bh̖O𐶐B
                getterName.setLength(0);
                getterName.append("get");
                getterName.append(StringUtils.upperCase(fieldName.substring(0,
                        1)));
                getterName.append(fieldName.substring(1, fieldName.length()));

                // setter̃tNVIuWFNg擾B
                // fields[i].getType()ň̌^w肵ĂB
                try {
                    methodList.add(clazz.getMethod(getterName.toString()));
                    fieldList.add(field);
                } catch (NoSuchMethodException e) {
                    throw new FileException(e, fileName);
                }
            }
        }
        methods = methodList.toArray(new Method[methodList.size()]);
        fields = fieldList.toArray(new Field[fieldList.size()]);
    }

    /**
     * wb_ւ̏ݏB
     * 
     * @param headerLine wb_֏ޕ̃Xg
     */
    public void printHeaderLine(List<String> headerLine) {
        if (writeData || writeTrailer) {
            throw new FileException("Header part should be called before "
                    + "data part or trailer part.",
                    new IllegalStateException(), fileName);
        }
        printList(headerLine);
    }

    /**
     * f[^ւ̏ݏB
     * 
     * @param t f[^֏ރt@CsIuWFNg
     */
    public void printDataLine(T t) {
        checkWriteTrailer();
        // t@C݂̏
        StringBuilder fileLineBuilder = new StringBuilder();
        
        // Œ蒷t@C̏ꍇ
        // (؂蕶A͂ݕȂꍇ͌Œ蒷t@CƔfB)
        if (getDelimiter() == Character.MIN_VALUE
                && getEncloseChar() == Character.MIN_VALUE) {
            for (int i = 0; i < getFields().length; i++) {
                fileLineBuilder.append(getColumn(t, i));
            }
        } else {
            for (int i = 0; i < getFields().length; i++) {
                // ͂ݕA؂蕶̒ǉB
                if (getEncloseChar() != Character.MIN_VALUE) {
                    fileLineBuilder.append(getEncloseChar());
                    fileLineBuilder.append(getColumn(t, i));
                    fileLineBuilder.append(getEncloseChar());
                } else {
                    fileLineBuilder.append(getColumn(t, i));
                }
                fileLineBuilder.append(getDelimiter());
            }
            // ԍŌ̋؂蕶폜B
            if (fileLineBuilder.length() > 0) {
                fileLineBuilder.deleteCharAt(fileLineBuilder.length() - 1);
            }
        }

        // s؂蕶ǉB
        fileLineBuilder.append(getLineFeedChar());

        // t@Cւ̏ݏB
        try {
            getWriter().write(fileLineBuilder.toString());
        } catch (IOException e) {
            throw new FileException(e, fileName);
        }
        currentLineCount++;
        setWriteData(true);
    }

    /**
     * gCւ̏ݏB
     * 
     * @param trailerLine gC֏ޕ̃Xg
     */
    public void printTrailerLine(List<String> trailerLine) {
        printList(trailerLine);
        writeTrailer = true;
    }

    /**
     * wb_AgC̏ݗp̋ʃ\bhB
     * 
     * @param stringList ̃Xg
     */
    private void printList(List<String> stringList) {
        for (String header : stringList) {
            try {
                writer.write(header);
                writer.write(lineFeedChar);
            } catch (IOException e) {
                throw new FileException(e, fileName);
            }
        }
    }

    /**
     * t@CN[YB
     */
    public void closeFile() {
        try {
            writer.flush();
            writer.close();
        } catch (IOException e) {
            throw new FileException(e, fileName);
        }
    }

    /**
     * <p>
     * t@CsIuWFNgJCfbNXƈv鑮̒l擾B
     * </p>
     * 
     * <p>
     * 擾ہAt@CsIuWFNg̃Ame[V̋Lqɂ<br>
     * <li>pfBO<br>
     * <li>g<br>
     * <li>ϊsB<br>
     * <br>
     * t@CsIuWFNg̃Ame[VŃJ̃oCgw肳ĂꍇA<br>
     * ԋp镶񂪃oCgƈvĂ邩mFB
     * </p>
     * 
     * @param t t@CsIuWFNg
     * @param index J̃CfbNX
     * @return J̕
     */
    protected String getColumn(T t, int index) {
        // t@CɏރJ̕B
        String columnString = null;

        OutputFileColumn outputFileColumn = fields[index]
                .getAnnotation(OutputFileColumn.class);

        // t@CsIuWFNg(t)JCfbNXƈv鑮̒l擾B
        ColumnFormatter columnFormatter = columnFormatterMap.get(methods[index]
                .getReturnType().getName());
        try {
            columnString = columnFormatter.format(t, methods[index],
                    outputFileColumn.columnFormat());
        } catch (IllegalArgumentException e) {
            throw new FileLineException(e, fileName, currentLineCount + 1,
                    fields[index].getName(),
                    outputFileColumn.columnIndex());
        } catch (IllegalAccessException e) {
            throw new FileLineException(e, fileName, currentLineCount + 1,
                    fields[index].getName(),
                    outputFileColumn.columnIndex());
        } catch (InvocationTargetException e) {
            throw new FileLineException(e, fileName, currentLineCount + 1,
                    fields[index].getName(),
                    outputFileColumn.columnIndex());
        }

        if (columnString == null) {
            columnString = "";
        }

        // g
        columnString = FileDAOUtility.trim(columnString, fileEncoding,
                outputFileColumn.trimChar(), outputFileColumn.trimType());

        // pfBO
        columnString = FileDAOUtility.padding(columnString, fileEncoding,
                outputFileColumn.bytes(), outputFileColumn.paddingChar(),
                outputFileColumn.paddingType());

        // 啶ϊAϊB
        // outputFileColumn.textTransform()̓eɂ菈U蕪B
        try {
            StringConverter stringConverter = outputFileColumn
                    .stringConverter().newInstance();
            columnString = stringConverter.convert(columnString);
        } catch (InstantiationException e) {
            throw new FileLineException(e, fileName, currentLineCount + 1, 
                    fields[index].getName(),
                    outputFileColumn.columnIndex());
        } catch (IllegalAccessException e) {
            throw new FileLineException(e, fileName, currentLineCount + 1, 
                    fields[index].getName(),
                    outputFileColumn.columnIndex());
        }

        // J̃oCg`FbNB
        if (isCheckByte(outputFileColumn)) {
            try {
                //Œ蒷o͎ABytesl̐ݒ肪̗O
                if (0 >= outputFileColumn.bytes()) {
                    throw new FileLineException("bytes is not set "
                            + "or a number equal to or less than 0 is set.",
                            new IllegalStateException(), getFileName(),
                            currentLineCount + 1, fields[index].getName(),
                            outputFileColumn.columnIndex());
                }
                //ݒ肳ꂽByteslƃf[^̃TCYႤꍇ͗O
                if (columnString.getBytes(fileEncoding).length
                        != outputFileColumn.bytes()) {
                    throw new FileLineException(
                            "The data size is different from bytes value of "
                            + "the set value of the column .",
                            new IllegalStateException(), fileName,
                            currentLineCount + 1, fields[index].getName(),
                            outputFileColumn.columnIndex());
                }
            } catch (UnsupportedEncodingException e) {
                throw new FileException(e, fileName);
            }
        }
        return columnString;
    }
    
    /**
     * t@C擾B
     * 
     * @return fileName t@C
     */
    public String getFileName() {
        return fileName;
    }

    /**
     * s؂蕶ݒ肷B
     * 
     * @return lineFeedChar s؂蕶
     */
    protected String getLineFeedChar() {
        return lineFeedChar;
    }

    /**
     * JtH[}bg(t@Cji[}bv擾B
     * 
     * @param columnFormatterMap
     *            JtH[}bg(t@Cji[}bv
     */
    public void setColumnFormatterMap(
            Map<String, ColumnFormatter> columnFormatterMap) {
        this.columnFormatterMap = columnFormatterMap;
    }

    /**
     * t@CANZXiójp̕Xg[擾B
     * 
     * @return bufferedWriter t@CANZXiójp̕Xg[
     */
    protected Writer getWriter() {
        return writer;
    }

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐ擾B
     * 
     * @return fields t@CsIuWFNgFieldiAnnotationji[ϐ
     */
    protected Field[] getFields() {
        return fields;
    }

    /**
     * t@CsIuWFNgFieldɑΉgetter\bhi[ϐ擾B
     * 
     * @return methods t@CsIuWFNgFieldɑΉgetter\bhi[ϐ
     */
    protected Method[] getMethods() {
        return methods;
    }

    /**
     * f[^̏o͂JnĂ邩ǂ𔻒肷tOB
     * 
     * @param writeData tO
     */
    protected void setWriteData(boolean writeData) {
        this.writeData = writeData;
    }

    /**
     * gC̏IĂ邩ǂ𔻒肷B<br>
     * ĂꍇAOX[B
     */
    protected void checkWriteTrailer() {
        if (writeTrailer) {
            throw new FileException("Header part or data part should be "
                    + "called before TrailerPart",
                    new IllegalStateException(), fileName);
        }
    }

    /**
     * ؂蕶擾B
     * 
     * @return ؂蕶
     */
    public abstract char getDelimiter();

    /**
     * ͂ݕ擾B
     * 
     * @return ͂ݕ
     */
    public abstract char getEncloseChar();


    /**
     * oCg`FbNsǂB
     * @param outputFileColumn o̓J
     * @return oCgݒ肳Ă(1oCgȏ)ꍇtrueB
     */
    protected boolean isCheckByte(OutputFileColumn outputFileColumn) {

        if (0 < outputFileColumn.bytes()) {
            return true;
        }

        return false;
    }
}
