/*
 * Copyright (c) 2010-2012 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _CORE_BITMAP_H
#define _CORE_BITMAP_H

#include <core/types.h>
#include <core/string.h>
#include "asm.h"

#define BITS_PER_ULONG (sizeof(ulong) * 8)
#define ELEMENTS_COUNT(size) (((size) + sizeof(ulong) - 1) / sizeof(ulong))

#define BITMAP_DECLARE(type_name, size) \
	typedef struct { \
		ulong e[ELEMENTS_COUNT(size)]; \
	} type_name

#define BITMAP_INIT(bitmap) \
do { \
	memset ((bitmap), 0, sizeof(*(bitmap))); \
} while (0)

#define BITMAP_SET(bitmap, bit) \
	(asm_lock_test_and_set_ulong ((bitmap)->e + (bit) / BITS_PER_ULONG, \
				    (bit) % BITS_PER_ULONG))

#define BITMAP_CLEAR(bitmap, bit) \
	(asm_lock_test_and_clear_ulong ((bitmap)->e + (bit) / BITS_PER_ULONG, \
				      (bit) % BITS_PER_ULONG))

#define BITMAP_TEST(bitmap, bit) \
	(asm_lock_test_ulong ((bitmap)->e + (bit) / BITS_PER_ULONG, \
			      ((bit) % BITS_PER_ULONG) ))

static inline long
bitmap_bsr (ulong e[], int size)
{
	int i;
	int bit = -1;
	for (i = (ELEMENTS_COUNT(size) - 1); i >= 0; i--) {
		bit = asm_bsr_ulong (e + i);
		if (bit >= 0) {
			return i * BITS_PER_ULONG + bit;
		}
	}
	return bit;
}

#define BITMAP_BSR(bitmap) \
	(bitmap_bsr ((bitmap)->e, sizeof (*(bitmap))))

static inline long
bitmap_bsf (ulong e[], int size)
{
	int i;
	int bit;
	for (i = 0; i < ELEMENTS_COUNT(size); i++) {
		bit = asm_bsf_ulong (e + i);
		if (bit >= 0) {
			return i * BITS_PER_ULONG + bit;
		}
	}
	return bit; 
}

#define BITMAP_BSF(bitmap) \
	(bitmap_bsf ((bitmap)->e, sizeof (*(bitmap))))

#endif
