/*
 * Copyright (c) 2012 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <core/acpi.h>
#include <core/cpu.h>
#include <core/types.h>
#include <core/initfunc.h>
#include <core/printf.h>
#include "current.h"
#include "vcpu.h"

#define ACPI_MADT_SIGNATURE			"APIC"
#define APIC_STRUCT_TYPE_LOCAL_APIC		0
#define LOCAL_APIC_FLGAS_ENABLED		0x1

struct acpi_madt {
	struct acpi_desc_header	header;
	u32			local_apic_addr;
	u32			flags;
} __attribute__ ((packed));

struct apic_header {
	u8			type;
	u8			length;
} __attribute__ ((packed));

struct local_apic {
	struct apic_header	header;
	u8			acpi_processor_id;
	u8			apic_id;
	u32			flags;
} __attribute__ ((packed));


static bool
check_vm0(apic_id_t apic_id)
{
	struct vcpu	*vcpu = NULL;
	struct vm	*vm;

	vm = current->vm;

	while ((vcpu = vcpu_next(vm, vcpu)) != NULL) {
		if (vcpu->apic_id == apic_id) {
			return true;
		}
	}
	return false;
}

static acpi_parser_t parse_madt;
static void
parse_madt(void *table, u32 length)
{
	struct acpi_madt	*acpi_madt = (struct acpi_madt *)table;
	struct apic_header	*apic_header;
	struct local_apic	*local_apic;

	for (apic_header = (struct apic_header *)(acpi_madt + 1);
	     (ulong)apic_header < (ulong)acpi_madt + length;
	     apic_header = (struct apic_header *)((ulong)apic_header + apic_header->length)) {
		if (apic_header->type != APIC_STRUCT_TYPE_LOCAL_APIC) {
			continue;
		}
		local_apic = (struct local_apic *)apic_header;
		if (check_vm0(local_apic->apic_id)) {
			continue;
		}
		local_apic->flags &= ~LOCAL_APIC_FLGAS_ENABLED;
	}
}

static void
acpi_madt_overwrite(void)
{
	if (!cpu_is_bsp()) {
		return;
	}
	acpi_register_parser(ACPI_MADT_SIGNATURE, parse_madt);
}

INITFUNC("passvm4", acpi_madt_overwrite);
