/* ----------------------------------------------------------------------  
* Copyright (C) 2010 ARM Limited. All rights reserved.  
*  
* $Date:        29. November 2010  
* $Revision: 	V1.0.3  
*  
* Project: 	    CMSIS DSP Library  
* Title:	    arm_rfft_init_q31.c  
*  
* Description:	RFFT & RIFFT Q31 initialisation function  
*  
* Target Processor: Cortex-M4/Cortex-M3
*  
* Version 1.0.3 2010/11/29 
*    Re-organized the CMSIS folders and updated documentation.  
*   
* Version 1.0.2 2010/11/11  
*    Documentation updated.   
*  
* Version 1.0.1 2010/10/05   
*    Production release and review comments incorporated.  
*  
* Version 1.0.0 2010/09/20   
*    Production release and review comments incorporated  
*  
* Version 0.0.7  2010/06/10   
*    Misra-C changes done  
* -------------------------------------------------------------------- */ 
 
#include "arm_math.h" 
 
/**  
 * @ingroup groupTransforms  
 */ 
 
/**  
 * @addtogroup RFFT_RIFFT  
 * @{  
 */ 
 
/**  
* \par  
* Generation floating point realCoefAQ31 array:  
* \par  
* n = 1024  
* <pre>for (i = 0; i < n; i++)  
* {  
*    pATable[2 * i] = 0.5 * (1.0 - sin (2 * PI / (double) (2 * n) * (double) i));  
*    pATable[2 * i + 1] = 0.5 * (-1.0 * cos (2 * PI / (double) (2 * n) * (double) i));  
* }</pre>  
* \par  
* Convert to fixed point Q31 format  
*     round(pATable[i] * pow(2, 31))  
*/ 
 
 
static const q31_t realCoefAQ31[2048] = { 
  0x40000000, 0xc0000000, 0x3fcdbc0e, 0xc00013be, 0x3f9b783c, 0xc0004ef5, 
  0x3f6934a7, 0xc000b1a7, 
  0x3f36f170, 0xc0013bd3, 0x3f04aeb4, 0xc001ed79, 0x3ed26c94, 0xc002c698, 
  0x3ea02b2d, 0xc003c730, 
  0x3e6deaa0, 0xc004ef3f, 0x3e3bab0b, 0xc0063ec7, 0x3e096c8c, 0xc007b5c5, 
  0x3dd72f44, 0xc0095438, 
  0x3da4f351, 0xc00b1a21, 0x3d72b8d2, 0xc00d077c, 0x3d407fe5, 0xc00f1c4b, 
  0x3d0e48ab, 0xc011588a, 
  0x3cdc1341, 0xc013bc3a, 0x3ca9dfc8, 0xc0164757, 0x3c77ae5d, 0xc018f9e1, 
  0x3c457f20, 0xc01bd3d7, 
  0x3c135230, 0xc01ed535, 0x3be127ac, 0xc021fdfb, 0x3baeffb2, 0xc0254e27, 
  0x3b7cda62, 0xc028c5b6, 
  0x3b4ab7db, 0xc02c64a6, 0x3b18983b, 0xc0302af6, 0x3ae67ba1, 0xc03418a2, 
  0x3ab4622d, 0xc0382da9, 
  0x3a824bfd, 0xc03c6a07, 0x3a503930, 0xc040cdbb, 0x3a1e29e5, 0xc04558c1, 
  0x39ec1e3b, 0xc04a0b16, 
  0x39ba1650, 0xc04ee4b9, 0x39881245, 0xc053e5a5, 0x39561236, 0xc0590dd8, 
  0x39241644, 0xc05e5d4f, 
  0x38f21e8e, 0xc063d406, 0x38c02b31, 0xc06971f9, 0x388e3c4d, 0xc06f3727, 
  0x385c5200, 0xc075238a, 
  0x382a6c6a, 0xc07b371f, 0x37f88ba9, 0xc08171e2, 0x37c6afdc, 0xc087d3d1, 
  0x3794d921, 0xc08e5ce6, 
  0x37630798, 0xc0950d1d, 0x37313b5f, 0xc09be473, 0x36ff7495, 0xc0a2e2e4, 
  0x36cdb359, 0xc0aa086a, 
  0x369bf7c8, 0xc0b15502, 0x366a4203, 0xc0b8c8a8, 0x36389227, 0xc0c06355, 
  0x3606e854, 0xc0c82507, 
  0x35d544a7, 0xc0d00db7, 0x35a3a740, 0xc0d81d61, 0x3572103d, 0xc0e05401, 
  0x35407fbc, 0xc0e8b191, 
  0x350ef5dd, 0xc0f1360c, 0x34dd72be, 0xc0f9e16c, 0x34abf67d, 0xc102b3ad, 
  0x347a8139, 0xc10bacc8, 
  0x34491310, 0xc114ccb9, 0x3417ac22, 0xc11e137a, 0x33e64c8b, 0xc1278105, 
  0x33b4f46c, 0xc1311553, 
  0x3383a3e1, 0xc13ad061, 0x33525b0b, 0xc144b226, 0x33211a06, 0xc14eba9e, 
  0x32efe0f2, 0xc158e9c1, 
  0x32beafed, 0xc1633f8a, 0x328d8715, 0xc16dbbf3, 0x325c6688, 0xc1785ef5, 
  0x322b4e65, 0xc1832888, 
  0x31fa3eca, 0xc18e18a8, 0x31c937d6, 0xc1992f4c, 0x319839a6, 0xc1a46c6e, 
  0x31674459, 0xc1afd008, 
  0x3136580d, 0xc1bb5a12, 0x310574e0, 0xc1c70a84, 0x30d49af0, 0xc1d2e159, 
  0x30a3ca5c, 0xc1dede87, 
  0x30730342, 0xc1eb0209, 0x304245bf, 0xc1f74bd7, 0x301191f2, 0xc203bbe8, 
  0x2fe0e7f9, 0xc2105236, 
  0x2fb047f1, 0xc21d0eb9, 0x2f7fb1fa, 0xc229f168, 0x2f4f2630, 0xc236fa3c, 
  0x2f1ea4b1, 0xc244292c, 
  0x2eee2d9d, 0xc2517e32, 0x2ebdc110, 0xc25ef943, 0x2e8d5f28, 0xc26c9a59, 
  0x2e5d0804, 0xc27a616a, 
  0x2e2cbbc0, 0xc2884e6f, 0x2dfc7a7c, 0xc296615e, 0x2dcc4454, 0xc2a49a2f, 
  0x2d9c1966, 0xc2b2f8d9, 
  0x2d6bf9d1, 0xc2c17d53, 0x2d3be5b1, 0xc2d02794, 0x2d0bdd25, 0xc2def794, 
  0x2cdbe049, 0xc2eded49, 
  0x2cabef3d, 0xc2fd08aa, 0x2c7c0a1c, 0xc30c49ad, 0x2c4c3105, 0xc31bb04a, 
  0x2c1c6416, 0xc32b3c76, 
  0x2beca36b, 0xc33aee28, 0x2bbcef23, 0xc34ac556, 0x2b8d475a, 0xc35ac1f8, 
  0x2b5dac2e, 0xc36ae401, 
  0x2b2e1dbd, 0xc37b2b6a, 0x2afe9c23, 0xc38b9828, 0x2acf277f, 0xc39c2a30, 
  0x2a9fbfed, 0xc3ace178, 
  0x2a70658a, 0xc3bdbdf7, 0x2a411874, 0xc3cebfa1, 0x2a11d8c8, 0xc3dfe66c, 
  0x29e2a6a3, 0xc3f1324e, 
  0x29b38222, 0xc402a33c, 0x29846b63, 0xc414392b, 0x29556282, 0xc425f411, 
  0x2926679c, 0xc437d3e2, 
  0x28f77acf, 0xc449d893, 0x28c89c36, 0xc45c0219, 0x2899cbf0, 0xc46e506a, 
  0x286b0a19, 0xc480c379, 
  0x283c56ce, 0xc4935b3c, 0x280db22c, 0xc4a617a7, 0x27df1c4f, 0xc4b8f8ae, 
  0x27b09555, 0xc4cbfe45, 
  0x27821d59, 0xc4df2862, 0x2753b479, 0xc4f276f8, 0x27255ad1, 0xc505e9fb, 
  0x26f7107e, 0xc519815f, 
  0x26c8d59c, 0xc52d3d19, 0x269aaa48, 0xc5411d1b, 0x266c8e9e, 0xc555215b, 
  0x263e82bb, 0xc56949ca, 
  0x261086bc, 0xc57d965e, 0x25e29abc, 0xc5920709, 0x25b4bed8, 0xc5a69bbf, 
  0x2586f32c, 0xc5bb5473, 
  0x255937d4, 0xc5d03118, 0x252b8ced, 0xc5e531a2, 0x24fdf293, 0xc5fa5603, 
  0x24d068e2, 0xc60f9e2f, 
  0x24a2eff6, 0xc6250a18, 0x247587eb, 0xc63a99b1, 0x244830dc, 0xc6504ced, 
  0x241aeae8, 0xc66623bf, 
  0x23edb627, 0xc67c1e19, 0x23c092b8, 0xc6923bec, 0x239380b6, 0xc6a87d2d, 
  0x2366803c, 0xc6bee1cd, 
  0x23399166, 0xc6d569be, 0x230cb451, 0xc6ec14f2, 0x22dfe917, 0xc702e35c, 
  0x22b32fd4, 0xc719d4ed, 
  0x228688a4, 0xc730e997, 0x2259f3a3, 0xc748214d, 0x222d70eb, 0xc75f7bfe, 
  0x22010099, 0xc776f99e, 
  0x21d4a2c7, 0xc78e9a1d, 0x21a85792, 0xc7a65d6e, 0x217c1f15, 0xc7be4381, 
  0x214ff96a, 0xc7d64c48, 
  0x2123e6ad, 0xc7ee77b4, 0x20f7e6f9, 0xc806c5b5, 0x20cbfa69, 0xc81f363e, 
  0x20a02119, 0xc837c93e, 
  0x20745b24, 0xc8507ea8, 0x2048a8a3, 0xc869566a, 0x201d09b4, 0xc8825077, 
  0x1ff17e6f, 0xc89b6cbf, 
  0x1fc606f1, 0xc8b4ab32, 0x1f9aa354, 0xc8ce0bc1, 0x1f6f53b2, 0xc8e78e5c, 
  0x1f441827, 0xc90132f3, 
  0x1f18f0cd, 0xc91af976, 0x1eedddbf, 0xc934e1d7, 0x1ec2df17, 0xc94eec03, 
  0x1e97f4f0, 0xc96917ed, 
  0x1e6d1f65, 0xc9836583, 0x1e425e8f, 0xc99dd4b5, 0x1e17b289, 0xc9b86572, 
  0x1ded1b6e, 0xc9d317ac, 
  0x1dc29957, 0xc9edeb50, 0x1d982c60, 0xca08e04f, 0x1d6dd4a1, 0xca23f698, 
  0x1d439236, 0xca3f2e1a, 
  0x1d196538, 0xca5a86c4, 0x1cef4dc1, 0xca760086, 0x1cc54bec, 0xca919b4e, 
  0x1c9b5fd1, 0xcaad570c, 
  0x1c71898c, 0xcac933ae, 0x1c47c936, 0xcae53124, 0x1c1e1ee8, 0xcb014f5b, 
  0x1bf48abd, 0xcb1d8e43, 
  0x1bcb0ccd, 0xcb39edca, 0x1ba1a533, 0xcb566ddf, 0x1b785408, 0xcb730e70, 
  0x1b4f1966, 0xcb8fcf6c, 
  0x1b25f566, 0xcbacb0c0, 0x1afce821, 0xcbc9b25b, 0x1ad3f1b1, 0xcbe6d42b, 
  0x1aab122f, 0xcc04161e, 
  0x1a8249b4, 0xcc217822, 0x1a599859, 0xcc3efa25, 0x1a30fe38, 0xcc5c9c15, 
  0x1a087b69, 0xcc7a5ddf, 
  0x19e01006, 0xcc983f71, 0x19b7bc27, 0xccb640b8, 0x198f7fe5, 0xccd461a3, 
  0x19675b5a, 0xccf2a21e, 
  0x193f4e9d, 0xcd110217, 0x191759c8, 0xcd2f817b, 0x18ef7cf4, 0xcd4e2037, 
  0x18c7b838, 0xcd6cde39, 
  0x18a00bad, 0xcd8bbb6d, 0x1878776c, 0xcdaab7c1, 0x1850fb8e, 0xcdc9d321, 
  0x1829982a, 0xcde90d7a, 
  0x18024d59, 0xce0866b9, 0x17db1b33, 0xce27deca, 0x17b401d0, 0xce47759a, 
  0x178d0149, 0xce672b16, 
  0x176619b5, 0xce86ff2a, 0x173f4b2d, 0xcea6f1c3, 0x171895c8, 0xcec702cc, 
  0x16f1f99f, 0xcee73232, 
  0x16cb76c8, 0xcf077fe1, 0x16a50d5d, 0xcf27ebc5, 0x167ebd74, 0xcf4875cb, 
  0x16588725, 0xcf691ddd, 
  0x16326a88, 0xcf89e3e9, 0x160c67b4, 0xcfaac7d9, 0x15e67ec1, 0xcfcbc999, 
  0x15c0afc6, 0xcfece916, 
  0x159afada, 0xd00e263a, 0x15756015, 0xd02f80f1, 0x154fdf8e, 0xd050f927, 
  0x152a795c, 0xd0728ec7, 
  0x15052d96, 0xd09441bc, 0x14dffc54, 0xd0b611f1, 0x14bae5ab, 0xd0d7ff52, 
  0x1495e9b3, 0xd0fa09c9, 
  0x14710883, 0xd11c3142, 0x144c4231, 0xd13e75a8, 0x142796d4, 0xd160d6e5, 
  0x14030684, 0xd18354e4, 
  0x13de9155, 0xd1a5ef91, 0x13ba3760, 0xd1c8a6d4, 0x1395f8b9, 0xd1eb7a9a, 
  0x1371d579, 0xd20e6acd, 
  0x134dcdb4, 0xd2317757, 0x1329e181, 0xd254a022, 0x130610f6, 0xd277e519, 
  0x12e25c2a, 0xd29b4626, 
  0x12bec333, 0xd2bec334, 0x129b4625, 0xd2e25c2b, 0x1277e518, 0xd30610f7, 
  0x1254a021, 0xd329e182, 
  0x12317756, 0xd34dcdb5, 0x120e6acc, 0xd371d57a, 0x11eb7a99, 0xd395f8ba, 
  0x11c8a6d3, 0xd3ba3761, 
  0x11a5ef90, 0xd3de9156, 0x118354e3, 0xd4030685, 0x1160d6e4, 0xd42796d5, 
  0x113e75a7, 0xd44c4232, 
  0x111c3141, 0xd4710884, 0x10fa09c8, 0xd495e9b4, 0x10d7ff51, 0xd4bae5ac, 
  0x10b611f0, 0xd4dffc55, 
  0x109441bb, 0xd5052d97, 0x10728ec6, 0xd52a795d, 0x1050f926, 0xd54fdf8f, 
  0x102f80f0, 0xd5756016, 
  0x100e2639, 0xd59afadb, 0xfece915, 0xd5c0afc7, 0xfcbc998, 0xd5e67ec2, 
  0xfaac7d8, 0xd60c67b5, 
  0xf89e3e8, 0xd6326a89, 0xf691ddc, 0xd6588726, 0xf4875ca, 0xd67ebd75, 
  0xf27ebc4, 0xd6a50d5e, 
  0xf077fe0, 0xd6cb76c9, 0xee73231, 0xd6f1f9a0, 0xec702cb, 0xd71895c9, 
  0xea6f1c2, 0xd73f4b2e, 
  0xe86ff29, 0xd76619b6, 0xe672b15, 0xd78d014a, 0xe477599, 0xd7b401d1, 
  0xe27dec9, 0xd7db1b34, 
  0xe0866b8, 0xd8024d5a, 0xde90d79, 0xd829982b, 0xdc9d320, 0xd850fb8f, 
  0xdaab7c0, 0xd878776d, 
  0xd8bbb6c, 0xd8a00bae, 0xd6cde38, 0xd8c7b839, 0xd4e2036, 0xd8ef7cf5, 
  0xd2f817a, 0xd91759c9, 
  0xd110216, 0xd93f4e9e, 0xcf2a21d, 0xd9675b5b, 0xcd461a2, 0xd98f7fe6, 
  0xcb640b7, 0xd9b7bc28, 
  0xc983f70, 0xd9e01007, 0xc7a5dde, 0xda087b6a, 0xc5c9c14, 0xda30fe39, 
  0xc3efa24, 0xda59985a, 
  0xc217821, 0xda8249b5, 0xc04161d, 0xdaab1230, 0xbe6d42a, 0xdad3f1b2, 
  0xbc9b25a, 0xdafce822, 
  0xbacb0bf, 0xdb25f567, 0xb8fcf6b, 0xdb4f1967, 0xb730e6f, 0xdb785409, 
  0xb566dde, 0xdba1a534, 
  0xb39edc9, 0xdbcb0cce, 0xb1d8e42, 0xdbf48abe, 0xb014f5a, 0xdc1e1ee9, 
  0xae53123, 0xdc47c937, 
  0xac933ad, 0xdc71898d, 0xaad570b, 0xdc9b5fd2, 0xa919b4d, 0xdcc54bed, 
  0xa760085, 0xdcef4dc2, 
  0xa5a86c3, 0xdd196539, 0xa3f2e19, 0xdd439237, 0xa23f697, 0xdd6dd4a2, 
  0xa08e04e, 0xdd982c61, 
  0x9edeb4f, 0xddc29958, 0x9d317ab, 0xdded1b6f, 0x9b86571, 0xde17b28a, 
  0x99dd4b4, 0xde425e90, 
  0x9836582, 0xde6d1f66, 0x96917ec, 0xde97f4f1, 0x94eec02, 0xdec2df18, 
  0x934e1d6, 0xdeedddc0, 
  0x91af975, 0xdf18f0ce, 0x90132f2, 0xdf441828, 0x8e78e5b, 0xdf6f53b3, 
  0x8ce0bc0, 0xdf9aa355, 
  0x8b4ab31, 0xdfc606f2, 0x89b6cbe, 0xdff17e70, 0x8825076, 0xe01d09b5, 
  0x8695669, 0xe048a8a4, 
  0x8507ea7, 0xe0745b25, 0x837c93d, 0xe0a0211a, 0x81f363d, 0xe0cbfa6a, 
  0x806c5b4, 0xe0f7e6fa, 
  0x7ee77b3, 0xe123e6ae, 0x7d64c47, 0xe14ff96b, 0x7be4380, 0xe17c1f16, 
  0x7a65d6d, 0xe1a85793, 
  0x78e9a1c, 0xe1d4a2c8, 0x776f99d, 0xe201009a, 0x75f7bfd, 0xe22d70ec, 
  0x748214c, 0xe259f3a4, 
  0x730e996, 0xe28688a5, 0x719d4ec, 0xe2b32fd5, 0x702e35b, 0xe2dfe918, 
  0x6ec14f1, 0xe30cb452, 
  0x6d569bd, 0xe3399167, 0x6bee1cc, 0xe366803d, 0x6a87d2c, 0xe39380b7, 
  0x6923beb, 0xe3c092b9, 
  0x67c1e18, 0xe3edb628, 0x66623be, 0xe41aeae9, 0x6504cec, 0xe44830dd, 
  0x63a99b0, 0xe47587ec, 
  0x6250a17, 0xe4a2eff7, 0x60f9e2e, 0xe4d068e3, 0x5fa5602, 0xe4fdf294, 
  0x5e531a1, 0xe52b8cee, 
  0x5d03117, 0xe55937d5, 0x5bb5472, 0xe586f32d, 0x5a69bbe, 0xe5b4bed9, 
  0x5920708, 0xe5e29abd, 
  0x57d965d, 0xe61086bd, 0x56949c9, 0xe63e82bc, 0x555215a, 0xe66c8e9f, 
  0x5411d1a, 0xe69aaa49, 
  0x52d3d18, 0xe6c8d59d, 0x519815e, 0xe6f7107f, 0x505e9fa, 0xe7255ad2, 
  0x4f276f7, 0xe753b47a, 
  0x4df2861, 0xe7821d5a, 0x4cbfe44, 0xe7b09556, 0x4b8f8ad, 0xe7df1c50, 
  0x4a617a6, 0xe80db22d, 
  0x4935b3b, 0xe83c56cf, 0x480c378, 0xe86b0a1a, 0x46e5069, 0xe899cbf1, 
  0x45c0218, 0xe8c89c37, 
  0x449d892, 0xe8f77ad0, 0x437d3e1, 0xe926679d, 0x425f410, 0xe9556283, 
  0x414392a, 0xe9846b64, 
  0x402a33b, 0xe9b38223, 0x3f1324d, 0xe9e2a6a4, 0x3dfe66b, 0xea11d8c9, 
  0x3cebfa0, 0xea411875, 
  0x3bdbdf6, 0xea70658b, 0x3ace177, 0xea9fbfee, 0x39c2a2f, 0xeacf2780, 
  0x38b9827, 0xeafe9c24, 
  0x37b2b69, 0xeb2e1dbe, 0x36ae400, 0xeb5dac2f, 0x35ac1f7, 0xeb8d475b, 
  0x34ac555, 0xebbcef24, 
  0x33aee27, 0xebeca36c, 0x32b3c75, 0xec1c6417, 0x31bb049, 0xec4c3106, 
  0x30c49ac, 0xec7c0a1d, 
  0x2fd08a9, 0xecabef3e, 0x2eded48, 0xecdbe04a, 0x2def793, 0xed0bdd26, 
  0x2d02793, 0xed3be5b2, 
  0x2c17d52, 0xed6bf9d2, 0x2b2f8d8, 0xed9c1967, 0x2a49a2e, 0xedcc4455, 
  0x296615d, 0xedfc7a7d, 
  0x2884e6e, 0xee2cbbc1, 0x27a6169, 0xee5d0805, 0x26c9a58, 0xee8d5f29, 
  0x25ef942, 0xeebdc111, 
  0x2517e31, 0xeeee2d9e, 0x244292b, 0xef1ea4b2, 0x236fa3b, 0xef4f2631, 
  0x229f167, 0xef7fb1fb, 
  0x21d0eb8, 0xefb047f2, 0x2105235, 0xefe0e7fa, 0x203bbe7, 0xf01191f3, 
  0x1f74bd6, 0xf04245c0, 
  0x1eb0208, 0xf0730343, 0x1dede86, 0xf0a3ca5d, 0x1d2e158, 0xf0d49af1, 
  0x1c70a83, 0xf10574e1, 
  0x1bb5a11, 0xf136580e, 0x1afd007, 0xf167445a, 0x1a46c6d, 0xf19839a7, 
  0x1992f4b, 0xf1c937d7, 
  0x18e18a7, 0xf1fa3ecb, 0x1832887, 0xf22b4e66, 0x1785ef4, 0xf25c6689, 
  0x16dbbf2, 0xf28d8716, 
  0x1633f89, 0xf2beafee, 0x158e9c0, 0xf2efe0f3, 0x14eba9d, 0xf3211a07, 
  0x144b225, 0xf3525b0c, 
  0x13ad060, 0xf383a3e2, 0x1311552, 0xf3b4f46d, 0x1278104, 0xf3e64c8c, 
  0x11e1379, 0xf417ac23, 
  0x114ccb8, 0xf4491311, 0x10bacc7, 0xf47a813a, 0x102b3ac, 0xf4abf67e, 
  0xf9e16b, 0xf4dd72bf, 
  0xf1360b, 0xf50ef5de, 0xe8b190, 0xf5407fbd, 0xe05400, 0xf572103e, 0xd81d60, 
  0xf5a3a741, 
  0xd00db6, 0xf5d544a8, 0xc82506, 0xf606e855, 0xc06354, 0xf6389228, 0xb8c8a7, 
  0xf66a4204, 
  0xb15501, 0xf69bf7c9, 0xaa0869, 0xf6cdb35a, 0xa2e2e3, 0xf6ff7496, 0x9be472, 
  0xf7313b60, 
  0x950d1c, 0xf7630799, 0x8e5ce5, 0xf794d922, 0x87d3d0, 0xf7c6afdd, 0x8171e1, 
  0xf7f88baa, 
  0x7b371e, 0xf82a6c6b, 0x752389, 0xf85c5201, 0x6f3726, 0xf88e3c4e, 0x6971f8, 
  0xf8c02b32, 
  0x63d405, 0xf8f21e8f, 0x5e5d4e, 0xf9241645, 0x590dd7, 0xf9561237, 0x53e5a4, 
  0xf9881246, 
  0x4ee4b8, 0xf9ba1651, 0x4a0b15, 0xf9ec1e3c, 0x4558c0, 0xfa1e29e6, 0x40cdba, 
  0xfa503931, 
  0x3c6a06, 0xfa824bfe, 0x382da8, 0xfab4622e, 0x3418a1, 0xfae67ba2, 0x302af5, 
  0xfb18983c, 
  0x2c64a5, 0xfb4ab7dc, 0x28c5b5, 0xfb7cda63, 0x254e26, 0xfbaeffb3, 0x21fdfa, 
  0xfbe127ad, 
  0x1ed534, 0xfc135231, 0x1bd3d6, 0xfc457f21, 0x18f9e0, 0xfc77ae5e, 0x164756, 
  0xfca9dfc9, 
  0x13bc39, 0xfcdc1342, 0x115889, 0xfd0e48ac, 0xf1c4a, 0xfd407fe6, 0xd077b, 
  0xfd72b8d3, 
  0xb1a20, 0xfda4f352, 0x95437, 0xfdd72f45, 0x7b5c4, 0xfe096c8d, 0x63ec6, 
  0xfe3bab0c, 
  0x4ef3e, 0xfe6deaa1, 0x3c72f, 0xfea02b2e, 0x2c697, 0xfed26c95, 0x1ed78, 
  0xff04aeb5, 
  0x13bd2, 0xff36f171, 0xb1a6, 0xff6934a8, 0x4ef4, 0xff9b783d, 0x13bd, 
  0xffcdbc0f, 
  0x0, 0x0, 0x13bd, 0x3243f1, 0x4ef4, 0x6487c3, 0xb1a6, 0x96cb58, 
  0x13bd2, 0xc90e8f, 0x1ed78, 0xfb514b, 0x2c697, 0x12d936b, 0x3c72f, 
  0x15fd4d2, 
  0x4ef3e, 0x192155f, 0x63ec6, 0x1c454f4, 0x7b5c4, 0x1f69373, 0x95437, 
  0x228d0bb, 
  0xb1a20, 0x25b0cae, 0xd077b, 0x28d472d, 0xf1c4a, 0x2bf801a, 0x115889, 
  0x2f1b754, 
  0x13bc39, 0x323ecbe, 0x164756, 0x3562037, 0x18f9e0, 0x38851a2, 0x1bd3d6, 
  0x3ba80df, 
  0x1ed534, 0x3ecadcf, 0x21fdfa, 0x41ed853, 0x254e26, 0x451004d, 0x28c5b5, 
  0x483259d, 
  0x2c64a5, 0x4b54824, 0x302af5, 0x4e767c4, 0x3418a1, 0x519845e, 0x382da8, 
  0x54b9dd2, 
  0x3c6a06, 0x57db402, 0x40cdba, 0x5afc6cf, 0x4558c0, 0x5e1d61a, 0x4a0b15, 
  0x613e1c4, 
  0x4ee4b8, 0x645e9af, 0x53e5a4, 0x677edba, 0x590dd7, 0x6a9edc9, 0x5e5d4e, 
  0x6dbe9bb, 
  0x63d405, 0x70de171, 0x6971f8, 0x73fd4ce, 0x6f3726, 0x771c3b2, 0x752389, 
  0x7a3adff, 
  0x7b371e, 0x7d59395, 0x8171e1, 0x8077456, 0x87d3d0, 0x8395023, 0x8e5ce5, 
  0x86b26de, 
  0x950d1c, 0x89cf867, 0x9be472, 0x8cec4a0, 0xa2e2e3, 0x9008b6a, 0xaa0869, 
  0x9324ca6, 
  0xb15501, 0x9640837, 0xb8c8a7, 0x995bdfc, 0xc06354, 0x9c76dd8, 0xc82506, 
  0x9f917ab, 
  0xd00db6, 0xa2abb58, 0xd81d60, 0xa5c58bf, 0xe05400, 0xa8defc2, 0xe8b190, 
  0xabf8043, 
  0xf1360b, 0xaf10a22, 0xf9e16b, 0xb228d41, 0x102b3ac, 0xb540982, 0x10bacc7, 
  0xb857ec6, 
  0x114ccb8, 0xbb6ecef, 0x11e1379, 0xbe853dd, 0x1278104, 0xc19b374, 0x1311552, 
  0xc4b0b93, 
  0x13ad060, 0xc7c5c1e, 0x144b225, 0xcada4f4, 0x14eba9d, 0xcdee5f9, 0x158e9c0, 
  0xd101f0d, 
  0x1633f89, 0xd415012, 0x16dbbf2, 0xd7278ea, 0x1785ef4, 0xda39977, 0x1832887, 
  0xdd4b19a, 
  0x18e18a7, 0xe05c135, 0x1992f4b, 0xe36c829, 0x1a46c6d, 0xe67c659, 0x1afd007, 
  0xe98bba6, 
  0x1bb5a11, 0xec9a7f2, 0x1c70a83, 0xefa8b1f, 0x1d2e158, 0xf2b650f, 0x1dede86, 
  0xf5c35a3, 
  0x1eb0208, 0xf8cfcbd, 0x1f74bd6, 0xfbdba40, 0x203bbe7, 0xfee6e0d, 0x2105235, 
  0x101f1806, 
  0x21d0eb8, 0x104fb80e, 0x229f167, 0x10804e05, 0x236fa3b, 0x10b0d9cf, 
  0x244292b, 0x10e15b4e, 
  0x2517e31, 0x1111d262, 0x25ef942, 0x11423eef, 0x26c9a58, 0x1172a0d7, 
  0x27a6169, 0x11a2f7fb, 
  0x2884e6e, 0x11d3443f, 0x296615d, 0x12038583, 0x2a49a2e, 0x1233bbab, 
  0x2b2f8d8, 0x1263e699, 
  0x2c17d52, 0x1294062e, 0x2d02793, 0x12c41a4e, 0x2def793, 0x12f422da, 
  0x2eded48, 0x13241fb6, 
  0x2fd08a9, 0x135410c2, 0x30c49ac, 0x1383f5e3, 0x31bb049, 0x13b3cefa, 
  0x32b3c75, 0x13e39be9, 
  0x33aee27, 0x14135c94, 0x34ac555, 0x144310dc, 0x35ac1f7, 0x1472b8a5, 
  0x36ae400, 0x14a253d1, 
  0x37b2b69, 0x14d1e242, 0x38b9827, 0x150163dc, 0x39c2a2f, 0x1530d880, 
  0x3ace177, 0x15604012, 
  0x3bdbdf6, 0x158f9a75, 0x3cebfa0, 0x15bee78b, 0x3dfe66b, 0x15ee2737, 
  0x3f1324d, 0x161d595c, 
  0x402a33b, 0x164c7ddd, 0x414392a, 0x167b949c, 0x425f410, 0x16aa9d7d, 
  0x437d3e1, 0x16d99863, 
  0x449d892, 0x17088530, 0x45c0218, 0x173763c9, 0x46e5069, 0x1766340f, 
  0x480c378, 0x1794f5e6, 
  0x4935b3b, 0x17c3a931, 0x4a617a6, 0x17f24dd3, 0x4b8f8ad, 0x1820e3b0, 
  0x4cbfe44, 0x184f6aaa, 
  0x4df2861, 0x187de2a6, 0x4f276f7, 0x18ac4b86, 0x505e9fa, 0x18daa52e, 
  0x519815e, 0x1908ef81, 
  0x52d3d18, 0x19372a63, 0x5411d1a, 0x196555b7, 0x555215a, 0x19937161, 
  0x56949c9, 0x19c17d44, 
  0x57d965d, 0x19ef7943, 0x5920708, 0x1a1d6543, 0x5a69bbe, 0x1a4b4127, 
  0x5bb5472, 0x1a790cd3, 
  0x5d03117, 0x1aa6c82b, 0x5e531a1, 0x1ad47312, 0x5fa5602, 0x1b020d6c, 
  0x60f9e2e, 0x1b2f971d, 
  0x6250a17, 0x1b5d1009, 0x63a99b0, 0x1b8a7814, 0x6504cec, 0x1bb7cf23, 
  0x66623be, 0x1be51517, 
  0x67c1e18, 0x1c1249d8, 0x6923beb, 0x1c3f6d47, 0x6a87d2c, 0x1c6c7f49, 
  0x6bee1cc, 0x1c997fc3, 
  0x6d569bd, 0x1cc66e99, 0x6ec14f1, 0x1cf34bae, 0x702e35b, 0x1d2016e8, 
  0x719d4ec, 0x1d4cd02b, 
  0x730e996, 0x1d79775b, 0x748214c, 0x1da60c5c, 0x75f7bfd, 0x1dd28f14, 
  0x776f99d, 0x1dfeff66, 
  0x78e9a1c, 0x1e2b5d38, 0x7a65d6d, 0x1e57a86d, 0x7be4380, 0x1e83e0ea, 
  0x7d64c47, 0x1eb00695, 
  0x7ee77b3, 0x1edc1952, 0x806c5b4, 0x1f081906, 0x81f363d, 0x1f340596, 
  0x837c93d, 0x1f5fdee6, 
  0x8507ea7, 0x1f8ba4db, 0x8695669, 0x1fb7575c, 0x8825076, 0x1fe2f64b, 
  0x89b6cbe, 0x200e8190, 
  0x8b4ab31, 0x2039f90e, 0x8ce0bc0, 0x20655cab, 0x8e78e5b, 0x2090ac4d, 
  0x90132f2, 0x20bbe7d8, 
  0x91af975, 0x20e70f32, 0x934e1d6, 0x21122240, 0x94eec02, 0x213d20e8, 
  0x96917ec, 0x21680b0f, 
  0x9836582, 0x2192e09a, 0x99dd4b4, 0x21bda170, 0x9b86571, 0x21e84d76, 
  0x9d317ab, 0x2212e491, 
  0x9edeb4f, 0x223d66a8, 0xa08e04e, 0x2267d39f, 0xa23f697, 0x22922b5e, 
  0xa3f2e19, 0x22bc6dc9, 
  0xa5a86c3, 0x22e69ac7, 0xa760085, 0x2310b23e, 0xa919b4d, 0x233ab413, 
  0xaad570b, 0x2364a02e, 
  0xac933ad, 0x238e7673, 0xae53123, 0x23b836c9, 0xb014f5a, 0x23e1e117, 
  0xb1d8e42, 0x240b7542, 
  0xb39edc9, 0x2434f332, 0xb566dde, 0x245e5acc, 0xb730e6f, 0x2487abf7, 
  0xb8fcf6b, 0x24b0e699, 
  0xbacb0bf, 0x24da0a99, 0xbc9b25a, 0x250317de, 0xbe6d42a, 0x252c0e4e, 
  0xc04161d, 0x2554edd0, 
  0xc217821, 0x257db64b, 0xc3efa24, 0x25a667a6, 0xc5c9c14, 0x25cf01c7, 
  0xc7a5dde, 0x25f78496, 
  0xc983f70, 0x261feff9, 0xcb640b7, 0x264843d8, 0xcd461a2, 0x2670801a, 
  0xcf2a21d, 0x2698a4a5, 
  0xd110216, 0x26c0b162, 0xd2f817a, 0x26e8a637, 0xd4e2036, 0x2710830b, 
  0xd6cde38, 0x273847c7, 
  0xd8bbb6c, 0x275ff452, 0xdaab7c0, 0x27878893, 0xdc9d320, 0x27af0471, 
  0xde90d79, 0x27d667d5, 
  0xe0866b8, 0x27fdb2a6, 0xe27dec9, 0x2824e4cc, 0xe477599, 0x284bfe2f, 
  0xe672b15, 0x2872feb6, 
  0xe86ff29, 0x2899e64a, 0xea6f1c2, 0x28c0b4d2, 0xec702cb, 0x28e76a37, 
  0xee73231, 0x290e0660, 
  0xf077fe0, 0x29348937, 0xf27ebc4, 0x295af2a2, 0xf4875ca, 0x2981428b, 
  0xf691ddc, 0x29a778da, 
  0xf89e3e8, 0x29cd9577, 0xfaac7d8, 0x29f3984b, 0xfcbc998, 0x2a19813e, 
  0xfece915, 0x2a3f5039, 
  0x100e2639, 0x2a650525, 0x102f80f0, 0x2a8a9fea, 0x1050f926, 0x2ab02071, 
  0x10728ec6, 0x2ad586a3, 
  0x109441bb, 0x2afad269, 0x10b611f0, 0x2b2003ab, 0x10d7ff51, 0x2b451a54, 
  0x10fa09c8, 0x2b6a164c, 
  0x111c3141, 0x2b8ef77c, 0x113e75a7, 0x2bb3bdce, 0x1160d6e4, 0x2bd8692b, 
  0x118354e3, 0x2bfcf97b, 
  0x11a5ef90, 0x2c216eaa, 0x11c8a6d3, 0x2c45c89f, 0x11eb7a99, 0x2c6a0746, 
  0x120e6acc, 0x2c8e2a86, 
  0x12317756, 0x2cb2324b, 0x1254a021, 0x2cd61e7e, 0x1277e518, 0x2cf9ef09, 
  0x129b4625, 0x2d1da3d5, 
  0x12bec333, 0x2d413ccc, 0x12e25c2a, 0x2d64b9da, 0x130610f6, 0x2d881ae7, 
  0x1329e181, 0x2dab5fde, 
  0x134dcdb4, 0x2dce88a9, 0x1371d579, 0x2df19533, 0x1395f8b9, 0x2e148566, 
  0x13ba3760, 0x2e37592c, 
  0x13de9155, 0x2e5a106f, 0x14030684, 0x2e7cab1c, 0x142796d4, 0x2e9f291b, 
  0x144c4231, 0x2ec18a58, 
  0x14710883, 0x2ee3cebe, 0x1495e9b3, 0x2f05f637, 0x14bae5ab, 0x2f2800ae, 
  0x14dffc54, 0x2f49ee0f, 
  0x15052d96, 0x2f6bbe44, 0x152a795c, 0x2f8d7139, 0x154fdf8e, 0x2faf06d9, 
  0x15756015, 0x2fd07f0f, 
  0x159afada, 0x2ff1d9c6, 0x15c0afc6, 0x301316ea, 0x15e67ec1, 0x30343667, 
  0x160c67b4, 0x30553827, 
  0x16326a88, 0x30761c17, 0x16588725, 0x3096e223, 0x167ebd74, 0x30b78a35, 
  0x16a50d5d, 0x30d8143b, 
  0x16cb76c8, 0x30f8801f, 0x16f1f99f, 0x3118cdce, 0x171895c8, 0x3138fd34, 
  0x173f4b2d, 0x31590e3d, 
  0x176619b5, 0x317900d6, 0x178d0149, 0x3198d4ea, 0x17b401d0, 0x31b88a66, 
  0x17db1b33, 0x31d82136, 
  0x18024d59, 0x31f79947, 0x1829982a, 0x3216f286, 0x1850fb8e, 0x32362cdf, 
  0x1878776c, 0x3255483f, 
  0x18a00bad, 0x32744493, 0x18c7b838, 0x329321c7, 0x18ef7cf4, 0x32b1dfc9, 
  0x191759c8, 0x32d07e85, 
  0x193f4e9d, 0x32eefde9, 0x19675b5a, 0x330d5de2, 0x198f7fe5, 0x332b9e5d, 
  0x19b7bc27, 0x3349bf48, 
  0x19e01006, 0x3367c08f, 0x1a087b69, 0x3385a221, 0x1a30fe38, 0x33a363eb, 
  0x1a599859, 0x33c105db, 
  0x1a8249b4, 0x33de87de, 0x1aab122f, 0x33fbe9e2, 0x1ad3f1b1, 0x34192bd5, 
  0x1afce821, 0x34364da5, 
  0x1b25f566, 0x34534f40, 0x1b4f1966, 0x34703094, 0x1b785408, 0x348cf190, 
  0x1ba1a533, 0x34a99221, 
  0x1bcb0ccd, 0x34c61236, 0x1bf48abd, 0x34e271bd, 0x1c1e1ee8, 0x34feb0a5, 
  0x1c47c936, 0x351acedc, 
  0x1c71898c, 0x3536cc52, 0x1c9b5fd1, 0x3552a8f4, 0x1cc54bec, 0x356e64b2, 
  0x1cef4dc1, 0x3589ff7a, 
  0x1d196538, 0x35a5793c, 0x1d439236, 0x35c0d1e6, 0x1d6dd4a1, 0x35dc0968, 
  0x1d982c60, 0x35f71fb1, 
  0x1dc29957, 0x361214b0, 0x1ded1b6e, 0x362ce854, 0x1e17b289, 0x36479a8e, 
  0x1e425e8f, 0x36622b4b, 
  0x1e6d1f65, 0x367c9a7d, 0x1e97f4f0, 0x3696e813, 0x1ec2df17, 0x36b113fd, 
  0x1eedddbf, 0x36cb1e29, 
  0x1f18f0cd, 0x36e5068a, 0x1f441827, 0x36fecd0d, 0x1f6f53b2, 0x371871a4, 
  0x1f9aa354, 0x3731f43f, 
  0x1fc606f1, 0x374b54ce, 0x1ff17e6f, 0x37649341, 0x201d09b4, 0x377daf89, 
  0x2048a8a3, 0x3796a996, 
  0x20745b24, 0x37af8158, 0x20a02119, 0x37c836c2, 0x20cbfa69, 0x37e0c9c2, 
  0x20f7e6f9, 0x37f93a4b, 
  0x2123e6ad, 0x3811884c, 0x214ff96a, 0x3829b3b8, 0x217c1f15, 0x3841bc7f, 
  0x21a85792, 0x3859a292, 
  0x21d4a2c7, 0x387165e3, 0x22010099, 0x38890662, 0x222d70eb, 0x38a08402, 
  0x2259f3a3, 0x38b7deb3, 
  0x228688a4, 0x38cf1669, 0x22b32fd4, 0x38e62b13, 0x22dfe917, 0x38fd1ca4, 
  0x230cb451, 0x3913eb0e, 
  0x23399166, 0x392a9642, 0x2366803c, 0x39411e33, 0x239380b6, 0x395782d3, 
  0x23c092b8, 0x396dc414, 
  0x23edb627, 0x3983e1e7, 0x241aeae8, 0x3999dc41, 0x244830dc, 0x39afb313, 
  0x247587eb, 0x39c5664f, 
  0x24a2eff6, 0x39daf5e8, 0x24d068e2, 0x39f061d1, 0x24fdf293, 0x3a05a9fd, 
  0x252b8ced, 0x3a1ace5e, 
  0x255937d4, 0x3a2fcee8, 0x2586f32c, 0x3a44ab8d, 0x25b4bed8, 0x3a596441, 
  0x25e29abc, 0x3a6df8f7, 
  0x261086bc, 0x3a8269a2, 0x263e82bb, 0x3a96b636, 0x266c8e9e, 0x3aaadea5, 
  0x269aaa48, 0x3abee2e5, 
  0x26c8d59c, 0x3ad2c2e7, 0x26f7107e, 0x3ae67ea1, 0x27255ad1, 0x3afa1605, 
  0x2753b479, 0x3b0d8908, 
  0x27821d59, 0x3b20d79e, 0x27b09555, 0x3b3401bb, 0x27df1c4f, 0x3b470752, 
  0x280db22c, 0x3b59e859, 
  0x283c56ce, 0x3b6ca4c4, 0x286b0a19, 0x3b7f3c87, 0x2899cbf0, 0x3b91af96, 
  0x28c89c36, 0x3ba3fde7, 
  0x28f77acf, 0x3bb6276d, 0x2926679c, 0x3bc82c1e, 0x29556282, 0x3bda0bef, 
  0x29846b63, 0x3bebc6d5, 
  0x29b38222, 0x3bfd5cc4, 0x29e2a6a3, 0x3c0ecdb2, 0x2a11d8c8, 0x3c201994, 
  0x2a411874, 0x3c31405f, 
  0x2a70658a, 0x3c424209, 0x2a9fbfed, 0x3c531e88, 0x2acf277f, 0x3c63d5d0, 
  0x2afe9c23, 0x3c7467d8, 
  0x2b2e1dbd, 0x3c84d496, 0x2b5dac2e, 0x3c951bff, 0x2b8d475a, 0x3ca53e08, 
  0x2bbcef23, 0x3cb53aaa, 
  0x2beca36b, 0x3cc511d8, 0x2c1c6416, 0x3cd4c38a, 0x2c4c3105, 0x3ce44fb6, 
  0x2c7c0a1c, 0x3cf3b653, 
  0x2cabef3d, 0x3d02f756, 0x2cdbe049, 0x3d1212b7, 0x2d0bdd25, 0x3d21086c, 
  0x2d3be5b1, 0x3d2fd86c, 
  0x2d6bf9d1, 0x3d3e82ad, 0x2d9c1966, 0x3d4d0727, 0x2dcc4454, 0x3d5b65d1, 
  0x2dfc7a7c, 0x3d699ea2, 
  0x2e2cbbc0, 0x3d77b191, 0x2e5d0804, 0x3d859e96, 0x2e8d5f28, 0x3d9365a7, 
  0x2ebdc110, 0x3da106bd, 
  0x2eee2d9d, 0x3dae81ce, 0x2f1ea4b1, 0x3dbbd6d4, 0x2f4f2630, 0x3dc905c4, 
  0x2f7fb1fa, 0x3dd60e98, 
  0x2fb047f1, 0x3de2f147, 0x2fe0e7f9, 0x3defadca, 0x301191f2, 0x3dfc4418, 
  0x304245bf, 0x3e08b429, 
  0x30730342, 0x3e14fdf7, 0x30a3ca5c, 0x3e212179, 0x30d49af0, 0x3e2d1ea7, 
  0x310574e0, 0x3e38f57c, 
  0x3136580d, 0x3e44a5ee, 0x31674459, 0x3e502ff8, 0x319839a6, 0x3e5b9392, 
  0x31c937d6, 0x3e66d0b4, 
  0x31fa3eca, 0x3e71e758, 0x322b4e65, 0x3e7cd778, 0x325c6688, 0x3e87a10b, 
  0x328d8715, 0x3e92440d, 
  0x32beafed, 0x3e9cc076, 0x32efe0f2, 0x3ea7163f, 0x33211a06, 0x3eb14562, 
  0x33525b0b, 0x3ebb4dda, 
  0x3383a3e1, 0x3ec52f9f, 0x33b4f46c, 0x3eceeaad, 0x33e64c8b, 0x3ed87efb, 
  0x3417ac22, 0x3ee1ec86, 
  0x34491310, 0x3eeb3347, 0x347a8139, 0x3ef45338, 0x34abf67d, 0x3efd4c53, 
  0x34dd72be, 0x3f061e94, 
  0x350ef5dd, 0x3f0ec9f4, 0x35407fbc, 0x3f174e6f, 0x3572103d, 0x3f1fabff, 
  0x35a3a740, 0x3f27e29f, 
  0x35d544a7, 0x3f2ff249, 0x3606e854, 0x3f37daf9, 0x36389227, 0x3f3f9cab, 
  0x366a4203, 0x3f473758, 
  0x369bf7c8, 0x3f4eaafe, 0x36cdb359, 0x3f55f796, 0x36ff7495, 0x3f5d1d1c, 
  0x37313b5f, 0x3f641b8d, 
  0x37630798, 0x3f6af2e3, 0x3794d921, 0x3f71a31a, 0x37c6afdc, 0x3f782c2f, 
  0x37f88ba9, 0x3f7e8e1e, 
  0x382a6c6a, 0x3f84c8e1, 0x385c5200, 0x3f8adc76, 0x388e3c4d, 0x3f90c8d9, 
  0x38c02b31, 0x3f968e07, 
  0x38f21e8e, 0x3f9c2bfa, 0x39241644, 0x3fa1a2b1, 0x39561236, 0x3fa6f228, 
  0x39881245, 0x3fac1a5b, 
  0x39ba1650, 0x3fb11b47, 0x39ec1e3b, 0x3fb5f4ea, 0x3a1e29e5, 0x3fbaa73f, 
  0x3a503930, 0x3fbf3245, 
  0x3a824bfd, 0x3fc395f9, 0x3ab4622d, 0x3fc7d257, 0x3ae67ba1, 0x3fcbe75e, 
  0x3b18983b, 0x3fcfd50a, 
  0x3b4ab7db, 0x3fd39b5a, 0x3b7cda62, 0x3fd73a4a, 0x3baeffb2, 0x3fdab1d9, 
  0x3be127ac, 0x3fde0205, 
  0x3c135230, 0x3fe12acb, 0x3c457f20, 0x3fe42c29, 0x3c77ae5d, 0x3fe7061f, 
  0x3ca9dfc8, 0x3fe9b8a9, 
  0x3cdc1341, 0x3fec43c6, 0x3d0e48ab, 0x3feea776, 0x3d407fe5, 0x3ff0e3b5, 
  0x3d72b8d2, 0x3ff2f884, 
  0x3da4f351, 0x3ff4e5df, 0x3dd72f44, 0x3ff6abc8, 0x3e096c8c, 0x3ff84a3b, 
  0x3e3bab0b, 0x3ff9c139, 
  0x3e6deaa0, 0x3ffb10c1, 0x3ea02b2d, 0x3ffc38d0, 0x3ed26c94, 0x3ffd3968, 
  0x3f04aeb4, 0x3ffe1287, 
  0x3f36f170, 0x3ffec42d, 0x3f6934a7, 0x3fff4e59, 0x3f9b783c, 0x3fffb10b, 
  0x3fcdbc0e, 0x3fffec42 
}; 
 
 
/**  
* \par 
* Generation of realCoefBQ31 array:  
* \par  
*  n = 1024  
* <pre>for (i = 0; i < n; i++)  
* {  
*    pBTable[2 * i] = 0.5 * (1.0 + sin (2 * PI / (double) (2 * n) * (double) i));  
*    pBTable[2 * i + 1] = 0.5 * (1.0 * cos (2 * PI / (double) (2 * n) * (double) i));  
* } </pre>  
* \par  
* Convert to fixed point Q31 format  
*     round(pBTable[i] * pow(2, 31))  
*  
*/ 
 
static const q31_t realCoefBQ31[2048] = { 
  0x40000000, 0x40000000, 0x403243f1, 0x3fffec42, 0x406487c3, 0x3fffb10b, 
  0x4096cb58, 0x3fff4e59, 
  0x40c90e8f, 0x3ffec42d, 0x40fb514b, 0x3ffe1287, 0x412d936b, 0x3ffd3968, 
  0x415fd4d2, 0x3ffc38d0, 
  0x4192155f, 0x3ffb10c1, 0x41c454f4, 0x3ff9c139, 0x41f69373, 0x3ff84a3b, 
  0x4228d0bb, 0x3ff6abc8, 
  0x425b0cae, 0x3ff4e5df, 0x428d472d, 0x3ff2f884, 0x42bf801a, 0x3ff0e3b5, 
  0x42f1b754, 0x3feea776, 
  0x4323ecbe, 0x3fec43c6, 0x43562037, 0x3fe9b8a9, 0x438851a2, 0x3fe7061f, 
  0x43ba80df, 0x3fe42c29, 
  0x43ecadcf, 0x3fe12acb, 0x441ed853, 0x3fde0205, 0x4451004d, 0x3fdab1d9, 
  0x4483259d, 0x3fd73a4a, 
  0x44b54824, 0x3fd39b5a, 0x44e767c4, 0x3fcfd50a, 0x4519845e, 0x3fcbe75e, 
  0x454b9dd2, 0x3fc7d257, 
  0x457db402, 0x3fc395f9, 0x45afc6cf, 0x3fbf3245, 0x45e1d61a, 0x3fbaa73f, 
  0x4613e1c4, 0x3fb5f4ea, 
  0x4645e9af, 0x3fb11b47, 0x4677edba, 0x3fac1a5b, 0x46a9edc9, 0x3fa6f228, 
  0x46dbe9bb, 0x3fa1a2b1, 
  0x470de171, 0x3f9c2bfa, 0x473fd4ce, 0x3f968e07, 0x4771c3b2, 0x3f90c8d9, 
  0x47a3adff, 0x3f8adc76, 
  0x47d59395, 0x3f84c8e1, 0x48077456, 0x3f7e8e1e, 0x48395023, 0x3f782c2f, 
  0x486b26de, 0x3f71a31a, 
  0x489cf867, 0x3f6af2e3, 0x48cec4a0, 0x3f641b8d, 0x49008b6a, 0x3f5d1d1c, 
  0x49324ca6, 0x3f55f796, 
  0x49640837, 0x3f4eaafe, 0x4995bdfc, 0x3f473758, 0x49c76dd8, 0x3f3f9cab, 
  0x49f917ab, 0x3f37daf9, 
  0x4a2abb58, 0x3f2ff249, 0x4a5c58bf, 0x3f27e29f, 0x4a8defc2, 0x3f1fabff, 
  0x4abf8043, 0x3f174e6f, 
  0x4af10a22, 0x3f0ec9f4, 0x4b228d41, 0x3f061e94, 0x4b540982, 0x3efd4c53, 
  0x4b857ec6, 0x3ef45338, 
  0x4bb6ecef, 0x3eeb3347, 0x4be853dd, 0x3ee1ec86, 0x4c19b374, 0x3ed87efb, 
  0x4c4b0b93, 0x3eceeaad, 
  0x4c7c5c1e, 0x3ec52f9f, 0x4cada4f4, 0x3ebb4dda, 0x4cdee5f9, 0x3eb14562, 
  0x4d101f0d, 0x3ea7163f, 
  0x4d415012, 0x3e9cc076, 0x4d7278ea, 0x3e92440d, 0x4da39977, 0x3e87a10b, 
  0x4dd4b19a, 0x3e7cd778, 
  0x4e05c135, 0x3e71e758, 0x4e36c829, 0x3e66d0b4, 0x4e67c659, 0x3e5b9392, 
  0x4e98bba6, 0x3e502ff8, 
  0x4ec9a7f2, 0x3e44a5ee, 0x4efa8b1f, 0x3e38f57c, 0x4f2b650f, 0x3e2d1ea7, 
  0x4f5c35a3, 0x3e212179, 
  0x4f8cfcbd, 0x3e14fdf7, 0x4fbdba40, 0x3e08b429, 0x4fee6e0d, 0x3dfc4418, 
  0x501f1806, 0x3defadca, 
  0x504fb80e, 0x3de2f147, 0x50804e05, 0x3dd60e98, 0x50b0d9cf, 0x3dc905c4, 
  0x50e15b4e, 0x3dbbd6d4, 
  0x5111d262, 0x3dae81ce, 0x51423eef, 0x3da106bd, 0x5172a0d7, 0x3d9365a7, 
  0x51a2f7fb, 0x3d859e96, 
  0x51d3443f, 0x3d77b191, 0x52038583, 0x3d699ea2, 0x5233bbab, 0x3d5b65d1, 
  0x5263e699, 0x3d4d0727, 
  0x5294062e, 0x3d3e82ad, 0x52c41a4e, 0x3d2fd86c, 0x52f422da, 0x3d21086c, 
  0x53241fb6, 0x3d1212b7, 
  0x535410c2, 0x3d02f756, 0x5383f5e3, 0x3cf3b653, 0x53b3cefa, 0x3ce44fb6, 
  0x53e39be9, 0x3cd4c38a, 
  0x54135c94, 0x3cc511d8, 0x544310dc, 0x3cb53aaa, 0x5472b8a5, 0x3ca53e08, 
  0x54a253d1, 0x3c951bff, 
  0x54d1e242, 0x3c84d496, 0x550163dc, 0x3c7467d8, 0x5530d880, 0x3c63d5d0, 
  0x55604012, 0x3c531e88, 
  0x558f9a75, 0x3c424209, 0x55bee78b, 0x3c31405f, 0x55ee2737, 0x3c201994, 
  0x561d595c, 0x3c0ecdb2, 
  0x564c7ddd, 0x3bfd5cc4, 0x567b949c, 0x3bebc6d5, 0x56aa9d7d, 0x3bda0bef, 
  0x56d99863, 0x3bc82c1e, 
  0x57088530, 0x3bb6276d, 0x573763c9, 0x3ba3fde7, 0x5766340f, 0x3b91af96, 
  0x5794f5e6, 0x3b7f3c87, 
  0x57c3a931, 0x3b6ca4c4, 0x57f24dd3, 0x3b59e859, 0x5820e3b0, 0x3b470752, 
  0x584f6aaa, 0x3b3401bb, 
  0x587de2a6, 0x3b20d79e, 0x58ac4b86, 0x3b0d8908, 0x58daa52e, 0x3afa1605, 
  0x5908ef81, 0x3ae67ea1, 
  0x59372a63, 0x3ad2c2e7, 0x596555b7, 0x3abee2e5, 0x59937161, 0x3aaadea5, 
  0x59c17d44, 0x3a96b636, 
  0x59ef7943, 0x3a8269a2, 0x5a1d6543, 0x3a6df8f7, 0x5a4b4127, 0x3a596441, 
  0x5a790cd3, 0x3a44ab8d, 
  0x5aa6c82b, 0x3a2fcee8, 0x5ad47312, 0x3a1ace5e, 0x5b020d6c, 0x3a05a9fd, 
  0x5b2f971d, 0x39f061d1, 
  0x5b5d1009, 0x39daf5e8, 0x5b8a7814, 0x39c5664f, 0x5bb7cf23, 0x39afb313, 
  0x5be51517, 0x3999dc41, 
  0x5c1249d8, 0x3983e1e7, 0x5c3f6d47, 0x396dc414, 0x5c6c7f49, 0x395782d3, 
  0x5c997fc3, 0x39411e33, 
  0x5cc66e99, 0x392a9642, 0x5cf34bae, 0x3913eb0e, 0x5d2016e8, 0x38fd1ca4, 
  0x5d4cd02b, 0x38e62b13, 
  0x5d79775b, 0x38cf1669, 0x5da60c5c, 0x38b7deb3, 0x5dd28f14, 0x38a08402, 
  0x5dfeff66, 0x38890662, 
  0x5e2b5d38, 0x387165e3, 0x5e57a86d, 0x3859a292, 0x5e83e0ea, 0x3841bc7f, 
  0x5eb00695, 0x3829b3b8, 
  0x5edc1952, 0x3811884c, 0x5f081906, 0x37f93a4b, 0x5f340596, 0x37e0c9c2, 
  0x5f5fdee6, 0x37c836c2, 
  0x5f8ba4db, 0x37af8158, 0x5fb7575c, 0x3796a996, 0x5fe2f64b, 0x377daf89, 
  0x600e8190, 0x37649341, 
  0x6039f90e, 0x374b54ce, 0x60655cab, 0x3731f43f, 0x6090ac4d, 0x371871a4, 
  0x60bbe7d8, 0x36fecd0d, 
  0x60e70f32, 0x36e5068a, 0x61122240, 0x36cb1e29, 0x613d20e8, 0x36b113fd, 
  0x61680b0f, 0x3696e813, 
  0x6192e09a, 0x367c9a7d, 0x61bda170, 0x36622b4b, 0x61e84d76, 0x36479a8e, 
  0x6212e491, 0x362ce854, 
  0x623d66a8, 0x361214b0, 0x6267d39f, 0x35f71fb1, 0x62922b5e, 0x35dc0968, 
  0x62bc6dc9, 0x35c0d1e6, 
  0x62e69ac7, 0x35a5793c, 0x6310b23e, 0x3589ff7a, 0x633ab413, 0x356e64b2, 
  0x6364a02e, 0x3552a8f4, 
  0x638e7673, 0x3536cc52, 0x63b836c9, 0x351acedc, 0x63e1e117, 0x34feb0a5, 
  0x640b7542, 0x34e271bd, 
  0x6434f332, 0x34c61236, 0x645e5acc, 0x34a99221, 0x6487abf7, 0x348cf190, 
  0x64b0e699, 0x34703094, 
  0x64da0a99, 0x34534f40, 0x650317de, 0x34364da5, 0x652c0e4e, 0x34192bd5, 
  0x6554edd0, 0x33fbe9e2, 
  0x657db64b, 0x33de87de, 0x65a667a6, 0x33c105db, 0x65cf01c7, 0x33a363eb, 
  0x65f78496, 0x3385a221, 
  0x661feff9, 0x3367c08f, 0x664843d8, 0x3349bf48, 0x6670801a, 0x332b9e5d, 
  0x6698a4a5, 0x330d5de2, 
  0x66c0b162, 0x32eefde9, 0x66e8a637, 0x32d07e85, 0x6710830b, 0x32b1dfc9, 
  0x673847c7, 0x329321c7, 
  0x675ff452, 0x32744493, 0x67878893, 0x3255483f, 0x67af0471, 0x32362cdf, 
  0x67d667d5, 0x3216f286, 
  0x67fdb2a6, 0x31f79947, 0x6824e4cc, 0x31d82136, 0x684bfe2f, 0x31b88a66, 
  0x6872feb6, 0x3198d4ea, 
  0x6899e64a, 0x317900d6, 0x68c0b4d2, 0x31590e3d, 0x68e76a37, 0x3138fd34, 
  0x690e0660, 0x3118cdce, 
  0x69348937, 0x30f8801f, 0x695af2a2, 0x30d8143b, 0x6981428b, 0x30b78a35, 
  0x69a778da, 0x3096e223, 
  0x69cd9577, 0x30761c17, 0x69f3984b, 0x30553827, 0x6a19813e, 0x30343667, 
  0x6a3f5039, 0x301316ea, 
  0x6a650525, 0x2ff1d9c6, 0x6a8a9fea, 0x2fd07f0f, 0x6ab02071, 0x2faf06d9, 
  0x6ad586a3, 0x2f8d7139, 
  0x6afad269, 0x2f6bbe44, 0x6b2003ab, 0x2f49ee0f, 0x6b451a54, 0x2f2800ae, 
  0x6b6a164c, 0x2f05f637, 
  0x6b8ef77c, 0x2ee3cebe, 0x6bb3bdce, 0x2ec18a58, 0x6bd8692b, 0x2e9f291b, 
  0x6bfcf97b, 0x2e7cab1c, 
  0x6c216eaa, 0x2e5a106f, 0x6c45c89f, 0x2e37592c, 0x6c6a0746, 0x2e148566, 
  0x6c8e2a86, 0x2df19533, 
  0x6cb2324b, 0x2dce88a9, 0x6cd61e7e, 0x2dab5fde, 0x6cf9ef09, 0x2d881ae7, 
  0x6d1da3d5, 0x2d64b9da, 
  0x6d413ccc, 0x2d413ccc, 0x6d64b9da, 0x2d1da3d5, 0x6d881ae7, 0x2cf9ef09, 
  0x6dab5fde, 0x2cd61e7e, 
  0x6dce88a9, 0x2cb2324b, 0x6df19533, 0x2c8e2a86, 0x6e148566, 0x2c6a0746, 
  0x6e37592c, 0x2c45c89f, 
  0x6e5a106f, 0x2c216eaa, 0x6e7cab1c, 0x2bfcf97b, 0x6e9f291b, 0x2bd8692b, 
  0x6ec18a58, 0x2bb3bdce, 
  0x6ee3cebe, 0x2b8ef77c, 0x6f05f637, 0x2b6a164c, 0x6f2800ae, 0x2b451a54, 
  0x6f49ee0f, 0x2b2003ab, 
  0x6f6bbe44, 0x2afad269, 0x6f8d7139, 0x2ad586a3, 0x6faf06d9, 0x2ab02071, 
  0x6fd07f0f, 0x2a8a9fea, 
  0x6ff1d9c6, 0x2a650525, 0x701316ea, 0x2a3f5039, 0x70343667, 0x2a19813e, 
  0x70553827, 0x29f3984b, 
  0x70761c17, 0x29cd9577, 0x7096e223, 0x29a778da, 0x70b78a35, 0x2981428b, 
  0x70d8143b, 0x295af2a2, 
  0x70f8801f, 0x29348937, 0x7118cdce, 0x290e0660, 0x7138fd34, 0x28e76a37, 
  0x71590e3d, 0x28c0b4d2, 
  0x717900d6, 0x2899e64a, 0x7198d4ea, 0x2872feb6, 0x71b88a66, 0x284bfe2f, 
  0x71d82136, 0x2824e4cc, 
  0x71f79947, 0x27fdb2a6, 0x7216f286, 0x27d667d5, 0x72362cdf, 0x27af0471, 
  0x7255483f, 0x27878893, 
  0x72744493, 0x275ff452, 0x729321c7, 0x273847c7, 0x72b1dfc9, 0x2710830b, 
  0x72d07e85, 0x26e8a637, 
  0x72eefde9, 0x26c0b162, 0x730d5de2, 0x2698a4a5, 0x732b9e5d, 0x2670801a, 
  0x7349bf48, 0x264843d8, 
  0x7367c08f, 0x261feff9, 0x7385a221, 0x25f78496, 0x73a363eb, 0x25cf01c7, 
  0x73c105db, 0x25a667a6, 
  0x73de87de, 0x257db64b, 0x73fbe9e2, 0x2554edd0, 0x74192bd5, 0x252c0e4e, 
  0x74364da5, 0x250317de, 
  0x74534f40, 0x24da0a99, 0x74703094, 0x24b0e699, 0x748cf190, 0x2487abf7, 
  0x74a99221, 0x245e5acc, 
  0x74c61236, 0x2434f332, 0x74e271bd, 0x240b7542, 0x74feb0a5, 0x23e1e117, 
  0x751acedc, 0x23b836c9, 
  0x7536cc52, 0x238e7673, 0x7552a8f4, 0x2364a02e, 0x756e64b2, 0x233ab413, 
  0x7589ff7a, 0x2310b23e, 
  0x75a5793c, 0x22e69ac7, 0x75c0d1e6, 0x22bc6dc9, 0x75dc0968, 0x22922b5e, 
  0x75f71fb1, 0x2267d39f, 
  0x761214b0, 0x223d66a8, 0x762ce854, 0x2212e491, 0x76479a8e, 0x21e84d76, 
  0x76622b4b, 0x21bda170, 
  0x767c9a7d, 0x2192e09a, 0x7696e813, 0x21680b0f, 0x76b113fd, 0x213d20e8, 
  0x76cb1e29, 0x21122240, 
  0x76e5068a, 0x20e70f32, 0x76fecd0d, 0x20bbe7d8, 0x771871a4, 0x2090ac4d, 
  0x7731f43f, 0x20655cab, 
  0x774b54ce, 0x2039f90e, 0x77649341, 0x200e8190, 0x777daf89, 0x1fe2f64b, 
  0x7796a996, 0x1fb7575c, 
  0x77af8158, 0x1f8ba4db, 0x77c836c2, 0x1f5fdee6, 0x77e0c9c2, 0x1f340596, 
  0x77f93a4b, 0x1f081906, 
  0x7811884c, 0x1edc1952, 0x7829b3b8, 0x1eb00695, 0x7841bc7f, 0x1e83e0ea, 
  0x7859a292, 0x1e57a86d, 
  0x787165e3, 0x1e2b5d38, 0x78890662, 0x1dfeff66, 0x78a08402, 0x1dd28f14, 
  0x78b7deb3, 0x1da60c5c, 
  0x78cf1669, 0x1d79775b, 0x78e62b13, 0x1d4cd02b, 0x78fd1ca4, 0x1d2016e8, 
  0x7913eb0e, 0x1cf34bae, 
  0x792a9642, 0x1cc66e99, 0x79411e33, 0x1c997fc3, 0x795782d3, 0x1c6c7f49, 
  0x796dc414, 0x1c3f6d47, 
  0x7983e1e7, 0x1c1249d8, 0x7999dc41, 0x1be51517, 0x79afb313, 0x1bb7cf23, 
  0x79c5664f, 0x1b8a7814, 
  0x79daf5e8, 0x1b5d1009, 0x79f061d1, 0x1b2f971d, 0x7a05a9fd, 0x1b020d6c, 
  0x7a1ace5e, 0x1ad47312, 
  0x7a2fcee8, 0x1aa6c82b, 0x7a44ab8d, 0x1a790cd3, 0x7a596441, 0x1a4b4127, 
  0x7a6df8f7, 0x1a1d6543, 
  0x7a8269a2, 0x19ef7943, 0x7a96b636, 0x19c17d44, 0x7aaadea5, 0x19937161, 
  0x7abee2e5, 0x196555b7, 
  0x7ad2c2e7, 0x19372a63, 0x7ae67ea1, 0x1908ef81, 0x7afa1605, 0x18daa52e, 
  0x7b0d8908, 0x18ac4b86, 
  0x7b20d79e, 0x187de2a6, 0x7b3401bb, 0x184f6aaa, 0x7b470752, 0x1820e3b0, 
  0x7b59e859, 0x17f24dd3, 
  0x7b6ca4c4, 0x17c3a931, 0x7b7f3c87, 0x1794f5e6, 0x7b91af96, 0x1766340f, 
  0x7ba3fde7, 0x173763c9, 
  0x7bb6276d, 0x17088530, 0x7bc82c1e, 0x16d99863, 0x7bda0bef, 0x16aa9d7d, 
  0x7bebc6d5, 0x167b949c, 
  0x7bfd5cc4, 0x164c7ddd, 0x7c0ecdb2, 0x161d595c, 0x7c201994, 0x15ee2737, 
  0x7c31405f, 0x15bee78b, 
  0x7c424209, 0x158f9a75, 0x7c531e88, 0x15604012, 0x7c63d5d0, 0x1530d880, 
  0x7c7467d8, 0x150163dc, 
  0x7c84d496, 0x14d1e242, 0x7c951bff, 0x14a253d1, 0x7ca53e08, 0x1472b8a5, 
  0x7cb53aaa, 0x144310dc, 
  0x7cc511d8, 0x14135c94, 0x7cd4c38a, 0x13e39be9, 0x7ce44fb6, 0x13b3cefa, 
  0x7cf3b653, 0x1383f5e3, 
  0x7d02f756, 0x135410c2, 0x7d1212b7, 0x13241fb6, 0x7d21086c, 0x12f422da, 
  0x7d2fd86c, 0x12c41a4e, 
  0x7d3e82ad, 0x1294062e, 0x7d4d0727, 0x1263e699, 0x7d5b65d1, 0x1233bbab, 
  0x7d699ea2, 0x12038583, 
  0x7d77b191, 0x11d3443f, 0x7d859e96, 0x11a2f7fb, 0x7d9365a7, 0x1172a0d7, 
  0x7da106bd, 0x11423eef, 
  0x7dae81ce, 0x1111d262, 0x7dbbd6d4, 0x10e15b4e, 0x7dc905c4, 0x10b0d9cf, 
  0x7dd60e98, 0x10804e05, 
  0x7de2f147, 0x104fb80e, 0x7defadca, 0x101f1806, 0x7dfc4418, 0xfee6e0d, 
  0x7e08b429, 0xfbdba40, 
  0x7e14fdf7, 0xf8cfcbd, 0x7e212179, 0xf5c35a3, 0x7e2d1ea7, 0xf2b650f, 
  0x7e38f57c, 0xefa8b1f, 
  0x7e44a5ee, 0xec9a7f2, 0x7e502ff8, 0xe98bba6, 0x7e5b9392, 0xe67c659, 
  0x7e66d0b4, 0xe36c829, 
  0x7e71e758, 0xe05c135, 0x7e7cd778, 0xdd4b19a, 0x7e87a10b, 0xda39977, 
  0x7e92440d, 0xd7278ea, 
  0x7e9cc076, 0xd415012, 0x7ea7163f, 0xd101f0d, 0x7eb14562, 0xcdee5f9, 
  0x7ebb4dda, 0xcada4f4, 
  0x7ec52f9f, 0xc7c5c1e, 0x7eceeaad, 0xc4b0b93, 0x7ed87efb, 0xc19b374, 
  0x7ee1ec86, 0xbe853dd, 
  0x7eeb3347, 0xbb6ecef, 0x7ef45338, 0xb857ec6, 0x7efd4c53, 0xb540982, 
  0x7f061e94, 0xb228d41, 
  0x7f0ec9f4, 0xaf10a22, 0x7f174e6f, 0xabf8043, 0x7f1fabff, 0xa8defc2, 
  0x7f27e29f, 0xa5c58bf, 
  0x7f2ff249, 0xa2abb58, 0x7f37daf9, 0x9f917ab, 0x7f3f9cab, 0x9c76dd8, 
  0x7f473758, 0x995bdfc, 
  0x7f4eaafe, 0x9640837, 0x7f55f796, 0x9324ca6, 0x7f5d1d1c, 0x9008b6a, 
  0x7f641b8d, 0x8cec4a0, 
  0x7f6af2e3, 0x89cf867, 0x7f71a31a, 0x86b26de, 0x7f782c2f, 0x8395023, 
  0x7f7e8e1e, 0x8077456, 
  0x7f84c8e1, 0x7d59395, 0x7f8adc76, 0x7a3adff, 0x7f90c8d9, 0x771c3b2, 
  0x7f968e07, 0x73fd4ce, 
  0x7f9c2bfa, 0x70de171, 0x7fa1a2b1, 0x6dbe9bb, 0x7fa6f228, 0x6a9edc9, 
  0x7fac1a5b, 0x677edba, 
  0x7fb11b47, 0x645e9af, 0x7fb5f4ea, 0x613e1c4, 0x7fbaa73f, 0x5e1d61a, 
  0x7fbf3245, 0x5afc6cf, 
  0x7fc395f9, 0x57db402, 0x7fc7d257, 0x54b9dd2, 0x7fcbe75e, 0x519845e, 
  0x7fcfd50a, 0x4e767c4, 
  0x7fd39b5a, 0x4b54824, 0x7fd73a4a, 0x483259d, 0x7fdab1d9, 0x451004d, 
  0x7fde0205, 0x41ed853, 
  0x7fe12acb, 0x3ecadcf, 0x7fe42c29, 0x3ba80df, 0x7fe7061f, 0x38851a2, 
  0x7fe9b8a9, 0x3562037, 
  0x7fec43c6, 0x323ecbe, 0x7feea776, 0x2f1b754, 0x7ff0e3b5, 0x2bf801a, 
  0x7ff2f884, 0x28d472d, 
  0x7ff4e5df, 0x25b0cae, 0x7ff6abc8, 0x228d0bb, 0x7ff84a3b, 0x1f69373, 
  0x7ff9c139, 0x1c454f4, 
  0x7ffb10c1, 0x192155f, 0x7ffc38d0, 0x15fd4d2, 0x7ffd3968, 0x12d936b, 
  0x7ffe1287, 0xfb514b, 
  0x7ffec42d, 0xc90e8f, 0x7fff4e59, 0x96cb58, 0x7fffb10b, 0x6487c3, 
  0x7fffec42, 0x3243f1, 
  0x7fffffff, 0x0, 0x7fffec42, 0xffcdbc0f, 0x7fffb10b, 0xff9b783d, 0x7fff4e59, 
  0xff6934a8, 
  0x7ffec42d, 0xff36f171, 0x7ffe1287, 0xff04aeb5, 0x7ffd3968, 0xfed26c95, 
  0x7ffc38d0, 0xfea02b2e, 
  0x7ffb10c1, 0xfe6deaa1, 0x7ff9c139, 0xfe3bab0c, 0x7ff84a3b, 0xfe096c8d, 
  0x7ff6abc8, 0xfdd72f45, 
  0x7ff4e5df, 0xfda4f352, 0x7ff2f884, 0xfd72b8d3, 0x7ff0e3b5, 0xfd407fe6, 
  0x7feea776, 0xfd0e48ac, 
  0x7fec43c6, 0xfcdc1342, 0x7fe9b8a9, 0xfca9dfc9, 0x7fe7061f, 0xfc77ae5e, 
  0x7fe42c29, 0xfc457f21, 
  0x7fe12acb, 0xfc135231, 0x7fde0205, 0xfbe127ad, 0x7fdab1d9, 0xfbaeffb3, 
  0x7fd73a4a, 0xfb7cda63, 
  0x7fd39b5a, 0xfb4ab7dc, 0x7fcfd50a, 0xfb18983c, 0x7fcbe75e, 0xfae67ba2, 
  0x7fc7d257, 0xfab4622e, 
  0x7fc395f9, 0xfa824bfe, 0x7fbf3245, 0xfa503931, 0x7fbaa73f, 0xfa1e29e6, 
  0x7fb5f4ea, 0xf9ec1e3c, 
  0x7fb11b47, 0xf9ba1651, 0x7fac1a5b, 0xf9881246, 0x7fa6f228, 0xf9561237, 
  0x7fa1a2b1, 0xf9241645, 
  0x7f9c2bfa, 0xf8f21e8f, 0x7f968e07, 0xf8c02b32, 0x7f90c8d9, 0xf88e3c4e, 
  0x7f8adc76, 0xf85c5201, 
  0x7f84c8e1, 0xf82a6c6b, 0x7f7e8e1e, 0xf7f88baa, 0x7f782c2f, 0xf7c6afdd, 
  0x7f71a31a, 0xf794d922, 
  0x7f6af2e3, 0xf7630799, 0x7f641b8d, 0xf7313b60, 0x7f5d1d1c, 0xf6ff7496, 
  0x7f55f796, 0xf6cdb35a, 
  0x7f4eaafe, 0xf69bf7c9, 0x7f473758, 0xf66a4204, 0x7f3f9cab, 0xf6389228, 
  0x7f37daf9, 0xf606e855, 
  0x7f2ff249, 0xf5d544a8, 0x7f27e29f, 0xf5a3a741, 0x7f1fabff, 0xf572103e, 
  0x7f174e6f, 0xf5407fbd, 
  0x7f0ec9f4, 0xf50ef5de, 0x7f061e94, 0xf4dd72bf, 0x7efd4c53, 0xf4abf67e, 
  0x7ef45338, 0xf47a813a, 
  0x7eeb3347, 0xf4491311, 0x7ee1ec86, 0xf417ac23, 0x7ed87efb, 0xf3e64c8c, 
  0x7eceeaad, 0xf3b4f46d, 
  0x7ec52f9f, 0xf383a3e2, 0x7ebb4dda, 0xf3525b0c, 0x7eb14562, 0xf3211a07, 
  0x7ea7163f, 0xf2efe0f3, 
  0x7e9cc076, 0xf2beafee, 0x7e92440d, 0xf28d8716, 0x7e87a10b, 0xf25c6689, 
  0x7e7cd778, 0xf22b4e66, 
  0x7e71e758, 0xf1fa3ecb, 0x7e66d0b4, 0xf1c937d7, 0x7e5b9392, 0xf19839a7, 
  0x7e502ff8, 0xf167445a, 
  0x7e44a5ee, 0xf136580e, 0x7e38f57c, 0xf10574e1, 0x7e2d1ea7, 0xf0d49af1, 
  0x7e212179, 0xf0a3ca5d, 
  0x7e14fdf7, 0xf0730343, 0x7e08b429, 0xf04245c0, 0x7dfc4418, 0xf01191f3, 
  0x7defadca, 0xefe0e7fa, 
  0x7de2f147, 0xefb047f2, 0x7dd60e98, 0xef7fb1fb, 0x7dc905c4, 0xef4f2631, 
  0x7dbbd6d4, 0xef1ea4b2, 
  0x7dae81ce, 0xeeee2d9e, 0x7da106bd, 0xeebdc111, 0x7d9365a7, 0xee8d5f29, 
  0x7d859e96, 0xee5d0805, 
  0x7d77b191, 0xee2cbbc1, 0x7d699ea2, 0xedfc7a7d, 0x7d5b65d1, 0xedcc4455, 
  0x7d4d0727, 0xed9c1967, 
  0x7d3e82ad, 0xed6bf9d2, 0x7d2fd86c, 0xed3be5b2, 0x7d21086c, 0xed0bdd26, 
  0x7d1212b7, 0xecdbe04a, 
  0x7d02f756, 0xecabef3e, 0x7cf3b653, 0xec7c0a1d, 0x7ce44fb6, 0xec4c3106, 
  0x7cd4c38a, 0xec1c6417, 
  0x7cc511d8, 0xebeca36c, 0x7cb53aaa, 0xebbcef24, 0x7ca53e08, 0xeb8d475b, 
  0x7c951bff, 0xeb5dac2f, 
  0x7c84d496, 0xeb2e1dbe, 0x7c7467d8, 0xeafe9c24, 0x7c63d5d0, 0xeacf2780, 
  0x7c531e88, 0xea9fbfee, 
  0x7c424209, 0xea70658b, 0x7c31405f, 0xea411875, 0x7c201994, 0xea11d8c9, 
  0x7c0ecdb2, 0xe9e2a6a4, 
  0x7bfd5cc4, 0xe9b38223, 0x7bebc6d5, 0xe9846b64, 0x7bda0bef, 0xe9556283, 
  0x7bc82c1e, 0xe926679d, 
  0x7bb6276d, 0xe8f77ad0, 0x7ba3fde7, 0xe8c89c37, 0x7b91af96, 0xe899cbf1, 
  0x7b7f3c87, 0xe86b0a1a, 
  0x7b6ca4c4, 0xe83c56cf, 0x7b59e859, 0xe80db22d, 0x7b470752, 0xe7df1c50, 
  0x7b3401bb, 0xe7b09556, 
  0x7b20d79e, 0xe7821d5a, 0x7b0d8908, 0xe753b47a, 0x7afa1605, 0xe7255ad2, 
  0x7ae67ea1, 0xe6f7107f, 
  0x7ad2c2e7, 0xe6c8d59d, 0x7abee2e5, 0xe69aaa49, 0x7aaadea5, 0xe66c8e9f, 
  0x7a96b636, 0xe63e82bc, 
  0x7a8269a2, 0xe61086bd, 0x7a6df8f7, 0xe5e29abd, 0x7a596441, 0xe5b4bed9, 
  0x7a44ab8d, 0xe586f32d, 
  0x7a2fcee8, 0xe55937d5, 0x7a1ace5e, 0xe52b8cee, 0x7a05a9fd, 0xe4fdf294, 
  0x79f061d1, 0xe4d068e3, 
  0x79daf5e8, 0xe4a2eff7, 0x79c5664f, 0xe47587ec, 0x79afb313, 0xe44830dd, 
  0x7999dc41, 0xe41aeae9, 
  0x7983e1e7, 0xe3edb628, 0x796dc414, 0xe3c092b9, 0x795782d3, 0xe39380b7, 
  0x79411e33, 0xe366803d, 
  0x792a9642, 0xe3399167, 0x7913eb0e, 0xe30cb452, 0x78fd1ca4, 0xe2dfe918, 
  0x78e62b13, 0xe2b32fd5, 
  0x78cf1669, 0xe28688a5, 0x78b7deb3, 0xe259f3a4, 0x78a08402, 0xe22d70ec, 
  0x78890662, 0xe201009a, 
  0x787165e3, 0xe1d4a2c8, 0x7859a292, 0xe1a85793, 0x7841bc7f, 0xe17c1f16, 
  0x7829b3b8, 0xe14ff96b, 
  0x7811884c, 0xe123e6ae, 0x77f93a4b, 0xe0f7e6fa, 0x77e0c9c2, 0xe0cbfa6a, 
  0x77c836c2, 0xe0a0211a, 
  0x77af8158, 0xe0745b25, 0x7796a996, 0xe048a8a4, 0x777daf89, 0xe01d09b5, 
  0x77649341, 0xdff17e70, 
  0x774b54ce, 0xdfc606f2, 0x7731f43f, 0xdf9aa355, 0x771871a4, 0xdf6f53b3, 
  0x76fecd0d, 0xdf441828, 
  0x76e5068a, 0xdf18f0ce, 0x76cb1e29, 0xdeedddc0, 0x76b113fd, 0xdec2df18, 
  0x7696e813, 0xde97f4f1, 
  0x767c9a7d, 0xde6d1f66, 0x76622b4b, 0xde425e90, 0x76479a8e, 0xde17b28a, 
  0x762ce854, 0xdded1b6f, 
  0x761214b0, 0xddc29958, 0x75f71fb1, 0xdd982c61, 0x75dc0968, 0xdd6dd4a2, 
  0x75c0d1e6, 0xdd439237, 
  0x75a5793c, 0xdd196539, 0x7589ff7a, 0xdcef4dc2, 0x756e64b2, 0xdcc54bed, 
  0x7552a8f4, 0xdc9b5fd2, 
  0x7536cc52, 0xdc71898d, 0x751acedc, 0xdc47c937, 0x74feb0a5, 0xdc1e1ee9, 
  0x74e271bd, 0xdbf48abe, 
  0x74c61236, 0xdbcb0cce, 0x74a99221, 0xdba1a534, 0x748cf190, 0xdb785409, 
  0x74703094, 0xdb4f1967, 
  0x74534f40, 0xdb25f567, 0x74364da5, 0xdafce822, 0x74192bd5, 0xdad3f1b2, 
  0x73fbe9e2, 0xdaab1230, 
  0x73de87de, 0xda8249b5, 0x73c105db, 0xda59985a, 0x73a363eb, 0xda30fe39, 
  0x7385a221, 0xda087b6a, 
  0x7367c08f, 0xd9e01007, 0x7349bf48, 0xd9b7bc28, 0x732b9e5d, 0xd98f7fe6, 
  0x730d5de2, 0xd9675b5b, 
  0x72eefde9, 0xd93f4e9e, 0x72d07e85, 0xd91759c9, 0x72b1dfc9, 0xd8ef7cf5, 
  0x729321c7, 0xd8c7b839, 
  0x72744493, 0xd8a00bae, 0x7255483f, 0xd878776d, 0x72362cdf, 0xd850fb8f, 
  0x7216f286, 0xd829982b, 
  0x71f79947, 0xd8024d5a, 0x71d82136, 0xd7db1b34, 0x71b88a66, 0xd7b401d1, 
  0x7198d4ea, 0xd78d014a, 
  0x717900d6, 0xd76619b6, 0x71590e3d, 0xd73f4b2e, 0x7138fd34, 0xd71895c9, 
  0x7118cdce, 0xd6f1f9a0, 
  0x70f8801f, 0xd6cb76c9, 0x70d8143b, 0xd6a50d5e, 0x70b78a35, 0xd67ebd75, 
  0x7096e223, 0xd6588726, 
  0x70761c17, 0xd6326a89, 0x70553827, 0xd60c67b5, 0x70343667, 0xd5e67ec2, 
  0x701316ea, 0xd5c0afc7, 
  0x6ff1d9c6, 0xd59afadb, 0x6fd07f0f, 0xd5756016, 0x6faf06d9, 0xd54fdf8f, 
  0x6f8d7139, 0xd52a795d, 
  0x6f6bbe44, 0xd5052d97, 0x6f49ee0f, 0xd4dffc55, 0x6f2800ae, 0xd4bae5ac, 
  0x6f05f637, 0xd495e9b4, 
  0x6ee3cebe, 0xd4710884, 0x6ec18a58, 0xd44c4232, 0x6e9f291b, 0xd42796d5, 
  0x6e7cab1c, 0xd4030685, 
  0x6e5a106f, 0xd3de9156, 0x6e37592c, 0xd3ba3761, 0x6e148566, 0xd395f8ba, 
  0x6df19533, 0xd371d57a, 
  0x6dce88a9, 0xd34dcdb5, 0x6dab5fde, 0xd329e182, 0x6d881ae7, 0xd30610f7, 
  0x6d64b9da, 0xd2e25c2b, 
  0x6d413ccc, 0xd2bec334, 0x6d1da3d5, 0xd29b4626, 0x6cf9ef09, 0xd277e519, 
  0x6cd61e7e, 0xd254a022, 
  0x6cb2324b, 0xd2317757, 0x6c8e2a86, 0xd20e6acd, 0x6c6a0746, 0xd1eb7a9a, 
  0x6c45c89f, 0xd1c8a6d4, 
  0x6c216eaa, 0xd1a5ef91, 0x6bfcf97b, 0xd18354e4, 0x6bd8692b, 0xd160d6e5, 
  0x6bb3bdce, 0xd13e75a8, 
  0x6b8ef77c, 0xd11c3142, 0x6b6a164c, 0xd0fa09c9, 0x6b451a54, 0xd0d7ff52, 
  0x6b2003ab, 0xd0b611f1, 
  0x6afad269, 0xd09441bc, 0x6ad586a3, 0xd0728ec7, 0x6ab02071, 0xd050f927, 
  0x6a8a9fea, 0xd02f80f1, 
  0x6a650525, 0xd00e263a, 0x6a3f5039, 0xcfece916, 0x6a19813e, 0xcfcbc999, 
  0x69f3984b, 0xcfaac7d9, 
  0x69cd9577, 0xcf89e3e9, 0x69a778da, 0xcf691ddd, 0x6981428b, 0xcf4875cb, 
  0x695af2a2, 0xcf27ebc5, 
  0x69348937, 0xcf077fe1, 0x690e0660, 0xcee73232, 0x68e76a37, 0xcec702cc, 
  0x68c0b4d2, 0xcea6f1c3, 
  0x6899e64a, 0xce86ff2a, 0x6872feb6, 0xce672b16, 0x684bfe2f, 0xce47759a, 
  0x6824e4cc, 0xce27deca, 
  0x67fdb2a6, 0xce0866b9, 0x67d667d5, 0xcde90d7a, 0x67af0471, 0xcdc9d321, 
  0x67878893, 0xcdaab7c1, 
  0x675ff452, 0xcd8bbb6d, 0x673847c7, 0xcd6cde39, 0x6710830b, 0xcd4e2037, 
  0x66e8a637, 0xcd2f817b, 
  0x66c0b162, 0xcd110217, 0x6698a4a5, 0xccf2a21e, 0x6670801a, 0xccd461a3, 
  0x664843d8, 0xccb640b8, 
  0x661feff9, 0xcc983f71, 0x65f78496, 0xcc7a5ddf, 0x65cf01c7, 0xcc5c9c15, 
  0x65a667a6, 0xcc3efa25, 
  0x657db64b, 0xcc217822, 0x6554edd0, 0xcc04161e, 0x652c0e4e, 0xcbe6d42b, 
  0x650317de, 0xcbc9b25b, 
  0x64da0a99, 0xcbacb0c0, 0x64b0e699, 0xcb8fcf6c, 0x6487abf7, 0xcb730e70, 
  0x645e5acc, 0xcb566ddf, 
  0x6434f332, 0xcb39edca, 0x640b7542, 0xcb1d8e43, 0x63e1e117, 0xcb014f5b, 
  0x63b836c9, 0xcae53124, 
  0x638e7673, 0xcac933ae, 0x6364a02e, 0xcaad570c, 0x633ab413, 0xca919b4e, 
  0x6310b23e, 0xca760086, 
  0x62e69ac7, 0xca5a86c4, 0x62bc6dc9, 0xca3f2e1a, 0x62922b5e, 0xca23f698, 
  0x6267d39f, 0xca08e04f, 
  0x623d66a8, 0xc9edeb50, 0x6212e491, 0xc9d317ac, 0x61e84d76, 0xc9b86572, 
  0x61bda170, 0xc99dd4b5, 
  0x6192e09a, 0xc9836583, 0x61680b0f, 0xc96917ed, 0x613d20e8, 0xc94eec03, 
  0x61122240, 0xc934e1d7, 
  0x60e70f32, 0xc91af976, 0x60bbe7d8, 0xc90132f3, 0x6090ac4d, 0xc8e78e5c, 
  0x60655cab, 0xc8ce0bc1, 
  0x6039f90e, 0xc8b4ab32, 0x600e8190, 0xc89b6cbf, 0x5fe2f64b, 0xc8825077, 
  0x5fb7575c, 0xc869566a, 
  0x5f8ba4db, 0xc8507ea8, 0x5f5fdee6, 0xc837c93e, 0x5f340596, 0xc81f363e, 
  0x5f081906, 0xc806c5b5, 
  0x5edc1952, 0xc7ee77b4, 0x5eb00695, 0xc7d64c48, 0x5e83e0ea, 0xc7be4381, 
  0x5e57a86d, 0xc7a65d6e, 
  0x5e2b5d38, 0xc78e9a1d, 0x5dfeff66, 0xc776f99e, 0x5dd28f14, 0xc75f7bfe, 
  0x5da60c5c, 0xc748214d, 
  0x5d79775b, 0xc730e997, 0x5d4cd02b, 0xc719d4ed, 0x5d2016e8, 0xc702e35c, 
  0x5cf34bae, 0xc6ec14f2, 
  0x5cc66e99, 0xc6d569be, 0x5c997fc3, 0xc6bee1cd, 0x5c6c7f49, 0xc6a87d2d, 
  0x5c3f6d47, 0xc6923bec, 
  0x5c1249d8, 0xc67c1e19, 0x5be51517, 0xc66623bf, 0x5bb7cf23, 0xc6504ced, 
  0x5b8a7814, 0xc63a99b1, 
  0x5b5d1009, 0xc6250a18, 0x5b2f971d, 0xc60f9e2f, 0x5b020d6c, 0xc5fa5603, 
  0x5ad47312, 0xc5e531a2, 
  0x5aa6c82b, 0xc5d03118, 0x5a790cd3, 0xc5bb5473, 0x5a4b4127, 0xc5a69bbf, 
  0x5a1d6543, 0xc5920709, 
  0x59ef7943, 0xc57d965e, 0x59c17d44, 0xc56949ca, 0x59937161, 0xc555215b, 
  0x596555b7, 0xc5411d1b, 
  0x59372a63, 0xc52d3d19, 0x5908ef81, 0xc519815f, 0x58daa52e, 0xc505e9fb, 
  0x58ac4b86, 0xc4f276f8, 
  0x587de2a6, 0xc4df2862, 0x584f6aaa, 0xc4cbfe45, 0x5820e3b0, 0xc4b8f8ae, 
  0x57f24dd3, 0xc4a617a7, 
  0x57c3a931, 0xc4935b3c, 0x5794f5e6, 0xc480c379, 0x5766340f, 0xc46e506a, 
  0x573763c9, 0xc45c0219, 
  0x57088530, 0xc449d893, 0x56d99863, 0xc437d3e2, 0x56aa9d7d, 0xc425f411, 
  0x567b949c, 0xc414392b, 
  0x564c7ddd, 0xc402a33c, 0x561d595c, 0xc3f1324e, 0x55ee2737, 0xc3dfe66c, 
  0x55bee78b, 0xc3cebfa1, 
  0x558f9a75, 0xc3bdbdf7, 0x55604012, 0xc3ace178, 0x5530d880, 0xc39c2a30, 
  0x550163dc, 0xc38b9828, 
  0x54d1e242, 0xc37b2b6a, 0x54a253d1, 0xc36ae401, 0x5472b8a5, 0xc35ac1f8, 
  0x544310dc, 0xc34ac556, 
  0x54135c94, 0xc33aee28, 0x53e39be9, 0xc32b3c76, 0x53b3cefa, 0xc31bb04a, 
  0x5383f5e3, 0xc30c49ad, 
  0x535410c2, 0xc2fd08aa, 0x53241fb6, 0xc2eded49, 0x52f422da, 0xc2def794, 
  0x52c41a4e, 0xc2d02794, 
  0x5294062e, 0xc2c17d53, 0x5263e699, 0xc2b2f8d9, 0x5233bbab, 0xc2a49a2f, 
  0x52038583, 0xc296615e, 
  0x51d3443f, 0xc2884e6f, 0x51a2f7fb, 0xc27a616a, 0x5172a0d7, 0xc26c9a59, 
  0x51423eef, 0xc25ef943, 
  0x5111d262, 0xc2517e32, 0x50e15b4e, 0xc244292c, 0x50b0d9cf, 0xc236fa3c, 
  0x50804e05, 0xc229f168, 
  0x504fb80e, 0xc21d0eb9, 0x501f1806, 0xc2105236, 0x4fee6e0d, 0xc203bbe8, 
  0x4fbdba40, 0xc1f74bd7, 
  0x4f8cfcbd, 0xc1eb0209, 0x4f5c35a3, 0xc1dede87, 0x4f2b650f, 0xc1d2e159, 
  0x4efa8b1f, 0xc1c70a84, 
  0x4ec9a7f2, 0xc1bb5a12, 0x4e98bba6, 0xc1afd008, 0x4e67c659, 0xc1a46c6e, 
  0x4e36c829, 0xc1992f4c, 
  0x4e05c135, 0xc18e18a8, 0x4dd4b19a, 0xc1832888, 0x4da39977, 0xc1785ef5, 
  0x4d7278ea, 0xc16dbbf3, 
  0x4d415012, 0xc1633f8a, 0x4d101f0d, 0xc158e9c1, 0x4cdee5f9, 0xc14eba9e, 
  0x4cada4f4, 0xc144b226, 
  0x4c7c5c1e, 0xc13ad061, 0x4c4b0b93, 0xc1311553, 0x4c19b374, 0xc1278105, 
  0x4be853dd, 0xc11e137a, 
  0x4bb6ecef, 0xc114ccb9, 0x4b857ec6, 0xc10bacc8, 0x4b540982, 0xc102b3ad, 
  0x4b228d41, 0xc0f9e16c, 
  0x4af10a22, 0xc0f1360c, 0x4abf8043, 0xc0e8b191, 0x4a8defc2, 0xc0e05401, 
  0x4a5c58bf, 0xc0d81d61, 
  0x4a2abb58, 0xc0d00db7, 0x49f917ab, 0xc0c82507, 0x49c76dd8, 0xc0c06355, 
  0x4995bdfc, 0xc0b8c8a8, 
  0x49640837, 0xc0b15502, 0x49324ca6, 0xc0aa086a, 0x49008b6a, 0xc0a2e2e4, 
  0x48cec4a0, 0xc09be473, 
  0x489cf867, 0xc0950d1d, 0x486b26de, 0xc08e5ce6, 0x48395023, 0xc087d3d1, 
  0x48077456, 0xc08171e2, 
  0x47d59395, 0xc07b371f, 0x47a3adff, 0xc075238a, 0x4771c3b2, 0xc06f3727, 
  0x473fd4ce, 0xc06971f9, 
  0x470de171, 0xc063d406, 0x46dbe9bb, 0xc05e5d4f, 0x46a9edc9, 0xc0590dd8, 
  0x4677edba, 0xc053e5a5, 
  0x4645e9af, 0xc04ee4b9, 0x4613e1c4, 0xc04a0b16, 0x45e1d61a, 0xc04558c1, 
  0x45afc6cf, 0xc040cdbb, 
  0x457db402, 0xc03c6a07, 0x454b9dd2, 0xc0382da9, 0x4519845e, 0xc03418a2, 
  0x44e767c4, 0xc0302af6, 
  0x44b54824, 0xc02c64a6, 0x4483259d, 0xc028c5b6, 0x4451004d, 0xc0254e27, 
  0x441ed853, 0xc021fdfb, 
  0x43ecadcf, 0xc01ed535, 0x43ba80df, 0xc01bd3d7, 0x438851a2, 0xc018f9e1, 
  0x43562037, 0xc0164757, 
  0x4323ecbe, 0xc013bc3a, 0x42f1b754, 0xc011588a, 0x42bf801a, 0xc00f1c4b, 
  0x428d472d, 0xc00d077c, 
  0x425b0cae, 0xc00b1a21, 0x4228d0bb, 0xc0095438, 0x41f69373, 0xc007b5c5, 
  0x41c454f4, 0xc0063ec7, 
  0x4192155f, 0xc004ef3f, 0x415fd4d2, 0xc003c730, 0x412d936b, 0xc002c698, 
  0x40fb514b, 0xc001ed79, 
  0x40c90e8f, 0xc0013bd3, 0x4096cb58, 0xc000b1a7, 0x406487c3, 0xc0004ef5, 
  0x403243f1, 0xc00013be 
}; 
 
/**  
* @brief  Initialization function for the Q31 RFFT/RIFFT. 
* @param[in, out] *S             points to an instance of the Q31 RFFT/RIFFT structure. 
* @param[in, out] *S_CFFT        points to an instance of the Q31 CFFT/CIFFT structure. 
* @param[in]      fftLenReal     length of the FFT. 
* @param[in]      ifftFlagR      flag that selects forward (ifftFlagR=0) or inverse (ifftFlagR=1) transform. 
* @param[in]      bitReverseFlag flag that enables (bitReverseFlag=1) or disables (bitReverseFlag=0) bit reversal of output. 
* @return		The function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>fftLenReal</code> is not a supported value. 
*  
* \par Description: 
* \par 
* The parameter <code>fftLenReal</code>	Specifies length of RFFT/RIFFT Process. Supported FFT Lengths are 128, 512, 2048.  
* \par  
* The parameter <code>ifftFlagR</code> controls whether a forward or inverse transform is computed.  
* Set(=1) ifftFlagR to calculate RIFFT, otherwise RFFT is calculated.  
* \par  
* The parameter <code>bitReverseFlag</code> controls whether output is in normal order or bit reversed order.  
* Set(=1) bitReverseFlag for output to be in normal order otherwise output is in bit reversed order. 
* \par  
* This function also initializes Twiddle factor table.  
*/ 
 
arm_status arm_rfft_init_q31( 
  arm_rfft_instance_q31 * S, 
  arm_cfft_radix4_instance_q31 * S_CFFT, 
  uint32_t fftLenReal, 
  uint32_t ifftFlagR, 
  uint32_t bitReverseFlag) 
{ 
  /*  Initialise the default arm status */ 
  arm_status status = ARM_MATH_SUCCESS; 
 
  /*  Initialize the Real FFT length */ 
  S->fftLenReal = (uint16_t) fftLenReal; 
 
  /*  Initialize the Complex FFT length */ 
  S->fftLenBy2 = (uint16_t) fftLenReal / 2u; 
 
  /*  Initialize the Twiddle coefficientA pointer */ 
  S->pTwiddleAReal = (q31_t *) realCoefAQ31; 
 
  /*  Initialize the Twiddle coefficientB pointer */ 
  S->pTwiddleBReal = (q31_t *) realCoefBQ31; 
 
  /*  Initialize the Flag for selection of RFFT or RIFFT */ 
  S->ifftFlagR = (uint8_t) ifftFlagR; 
 
  /*  Initialize the Flag for calculation Bit reversal or not */ 
  S->bitReverseFlagR = (uint8_t) bitReverseFlag; 
 
  /*  Initialization of coef modifier depending on the FFT length */ 
  switch (S->fftLenReal) 
  { 
  case 2048u: 
    S->twidCoefRModifier = 1u; 
    break; 
  case 512u: 
    S->twidCoefRModifier = 4u; 
    break; 
  case 128u: 
    S->twidCoefRModifier = 16u; 
    break; 
  default: 
    /*  Reporting argument error if rfftSize is not valid value */ 
    status = ARM_MATH_ARGUMENT_ERROR; 
    break; 
  } 
 
  /* Init Complex FFT Instance */ 
  S->pCfft = S_CFFT; 
 
  if(S->ifftFlagR) 
  { 
    /* Initializes the CIFFT Module for fftLenreal/2 length */ 
    arm_cfft_radix4_init_q31(S->pCfft, (uint16_t) S->fftLenBy2, 1u, 1u); 
  } 
  else 
  { 
    /* Initializes the CFFT Module for fftLenreal/2 length */ 
    arm_cfft_radix4_init_q31(S->pCfft, (uint16_t) S->fftLenBy2, 0u, 1u); 
  } 
 
  /* return the status of RFFT Init function */ 
  return (status); 
 
} 
 
  /**  
   * @} end of RFFT_RIFFT group  
   */ 
