﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Diagnostics;
using System.Reflection;

namespace TouchToKey
{
    public partial class Form1 : Form
    {
        private Config config;
        private Point startPoint;
        private Point prevPoint;
        private bool doing = false;
        private bool moving = false;

        RamGecTools.MouseHook mouseHook = new RamGecTools.MouseHook();

        public Form1()
        {
            InitializeComponent();
        }

        private void contextMenuStrip1_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            // 簡易2重起動防止
            if (Process.GetProcessesByName("TouchToKey").Length > 1) this.Close();

            // グローバルフックのコールバックを設定(1回だけ)
            mouseHook.LeftButtonDown += new RamGecTools.MouseHook.MouseHookCallback(this.LeftButtonDown);
            mouseHook.MouseMove += new RamGecTools.MouseHook.MouseHookCallback(this.MouseMove);
            mouseHook.LeftButtonUp += new RamGecTools.MouseHook.MouseHookCallback(this.LeftButtonUp);

            // グローバルフック開始
            this.startHook();
        }

        // グローバルフック開始
        private void startHook()
        {
            // 設定ファイルを(再)読み込み
            config = new Config();
            if (config.areas == null) this.Close();

            // グローバルフック開始
            mouseHook.Install();
        }

        // グローバルフック終了
        private void stopHook()
        {
            mouseHook.Uninstall();
        }

        // ドラッグ開始
        private void LeftButtonDown(RamGecTools.MouseHook.MSLLHOOKSTRUCT mouseStruct)
        {
            startPoint = new Point(mouseStruct.pt.x, mouseStruct.pt.y);

            prevPoint = new Point(startPoint.X, startPoint.Y);

            if (DoAction(prevPoint, startPoint))
            {
                doing = true;
            }
        }
        // ドラッグ中
        private new void MouseMove(RamGecTools.MouseHook.MSLLHOOKSTRUCT mouseStruct)
        {
            if (doing == false) return;

            moving = true;

            Point curPoint = new Point(mouseStruct.pt.x, mouseStruct.pt.y);

            if (DoAction(prevPoint, curPoint))
            {
                // アクションを実行したなら移動開始点をリセット
                prevPoint = new Point(curPoint.X, curPoint.Y);
            }
        }

        // ドラッグ終了
        private void LeftButtonUp(RamGecTools.MouseHook.MSLLHOOKSTRUCT mouseStruct)
        {
            if (doing == false) return;

            moving = false;

            Point endPoint = new Point(mouseStruct.pt.x, mouseStruct.pt.y);

            DoAction(startPoint, endPoint);

            doing = false;
        }

        // 設定に従いアクションを実行する
        // アクションを実行した場合はtrueを返す
        private bool DoAction(Point point1, Point point2)
        {
            Window window = new Window(Window.GetForegroundWindowHandle());

            if (!moving)
                Debug.WriteLine("DoAction(): processName=" + window.info.processName + ";" +
                                              "className=" + window.info.className);

            Size delta = new Size(point2.X - point1.X, point2.Y - point1.Y);

            if (!moving)
                Debug.WriteLine("DoAction(): (x1=" + point1.X + ",y1=" + point1.Y + ")," +
                                            "(x2=" + point2.X + ",y2=" + point2.Y + ")," +
                                            "(w=" + delta.Width + ",h=" + delta.Height + ")");

            foreach (Config.AREA area in config.areas)
            {
                // 対象プログラムがフォアグラウンドウィンドウか判定する
                if (area.processName != "*" && area.processName != window.info.processName) continue;
                if (area.className.Length > 0 && area.className != window.info.className) continue;

                // 操作がウィンドウ領域内か判定する
                // (開始ポイント)
                if (!IsTargetArea(area, window.info.rect, point1)) continue;
                // 開始ポイントでイベントを中断したなら終了ポイントもセットで中断する
                if (area.breakEvent == "TRUE") mouseHook.breakEvent = true;
                // (終了ポイント)
                if (!IsTargetArea(area, window.info.rect, point2)) continue;

                // 操作開始判定
                if (doing == false) return true;

                // 操作が対象アクションか判定する
                if (!IsTargetAction(config, area.action, window.info.rect, delta)) continue;

                // キーを送信する
                try
                {
                    SendKeys.Send(area.key);
                    Debug.WriteLine("DoAction(): Send Key=" + area.key);
                    return true;
                }
                catch (ArgumentException ex)
                {
                    Debug.WriteLine(ex.Message);
                }
            }
            return false;
        }

        // 操作が領域内か判定する
        private bool IsTargetArea(Config.AREA area, Window.RECT rect, Point point)
        {
            Window.RECT areaRect;
            areaRect.left = Window.ToScreenX(area.left, rect.left, rect.right);
            areaRect.right = Window.ToScreenX(area.right, rect.left, rect.right);
            areaRect.top = Window.ToScreenY(area.top, rect.top, rect.bottom);
            areaRect.bottom = Window.ToScreenY(area.bottom, rect.top, rect.bottom);
            if (point.X < areaRect.left || point.X > areaRect.right) return false;
            if (point.Y < areaRect.top || point.Y > areaRect.bottom) return false;
            return true;
        }
        // 対象の操作か判定する
        private bool IsTargetAction(Config config, string action, Window.RECT rect, Size delta)
        {
            Size absDelta = new Size(Math.Abs(delta.Width), Math.Abs(delta.Height));

            // 斜めドラッグは長い方向のみ有効にする。
            if (absDelta.Width > absDelta.Height) delta.Height = 0;
            else if (absDelta.Width < absDelta.Height) delta.Width = 0;

            Size baseSize = new Size(rect.right - rect.left, rect.bottom - rect.top);

            Size min, max;

            // クリック
            action = action.ToUpper().Replace("TAP", "CLICK");
            if (moving == false && action == "CLICK")
            {
                min = new Size(Window.ToPixcel(config.minClickLength, baseSize.Width),
                               Window.ToPixcel(config.minClickLength, baseSize.Height));
                max = new Size(Window.ToPixcel(config.maxClickLength, baseSize.Width),
                               Window.ToPixcel(config.maxClickLength, baseSize.Height));
                return (absDelta.Width >= min.Width && absDelta.Width <= max.Width &&
                        absDelta.Height >= min.Height && absDelta.Height <= max.Height);
            }

            // ドラッグ(&ドラッグ中)
            action = action.ToUpper().Replace("SWIPE", "DRAG");
            action = action.ToUpper().Replace("SWIPING", "DRAGGING");
            // 方向と操作を分離
            string direction = "";
            foreach (string key in new String[] { "LONGDRAG", "DRAGGING", "DRAG" })
            {
                if (action.Length < key.Length) continue;
                if (action.Substring(action.Length - key.Length) != key) continue;
                direction = action.Substring(0, action.Length - key.Length);
                action = key;
                break;
            }
            if (moving == false && action == "LONGDRAG")
            {
                min = new Size(Window.ToPixcel(config.minLongDragLength, baseSize.Width),
                               Window.ToPixcel(config.minLongDragLength, baseSize.Height));
                max = new Size(Window.ToPixcel(config.maxLongDragLength, baseSize.Width),
                               Window.ToPixcel(config.maxLongDragLength, baseSize.Height));
            }
            else if (moving == false && action == "DRAG")
            {
                min = new Size(Window.ToPixcel(config.minDragLength, baseSize.Width),
                               Window.ToPixcel(config.minDragLength, baseSize.Height));
                max = new Size(Window.ToPixcel(config.maxDragLength, baseSize.Width),
                               Window.ToPixcel(config.maxDragLength, baseSize.Height));
            }
            else if (moving == true && action == "DRAGGING")
            {
                min = new Size(Window.ToPixcel(config.swipingStepLength, baseSize.Width),
                               Window.ToPixcel(config.swipingStepLength, baseSize.Height));
                max = new Size(baseSize.Width, baseSize.Height);
            }
            else
            {
                return false;
            }
            bool isDragX = (absDelta.Width >= min.Width && absDelta.Width <= max.Width);
            bool isDragY = (absDelta.Height >= min.Height && absDelta.Height <= max.Height);
            if (direction.Length == 0) return (isDragX || isDragY);
            else if (direction == "RIGHT" && delta.Width > 0) return isDragX;
            else if (direction == "LEFT" && delta.Width < 0) return isDragX;
            else if (direction == "UP" && delta.Height < 0) return isDragY;
            else if (direction == "DOWN" && delta.Height > 0) return isDragY;
            return false;
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            stopHook();
        }

        private void notifyIcon1_MouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                // 機能の有効・無効切り替え
                if (notifyIcon1.Text == "TouchToKey")
                {
                    notifyIcon1.Text = "TouchToKey (無効)";
                    notifyIcon1.Icon = Properties.Resources.IconX16x16;
                    stopHook();
                }
                else
                {
                    notifyIcon1.Text = "TouchToKey";
                    notifyIcon1.Icon = Properties.Resources.Icon16x16;
                    startHook();
                }
            }
        }
    }


}
