# -*- coding: utf-8 -*-
module ActiveRecord

  # 作成時および更新時の情報を記録するモジュール。
  module ContextMonitor

    def self.included(base) # :ndoc:
      base.extend(ClassMethods)
    end

    module ClassMethods
      #
      # モデルクラスで呼び出すことにより指定した対象の状況を記録するようになる。
      #
      # <tt>monitor</tt> : 記録する対象のモデルクラスの名前を指定する。
      #                    デフォルトは <tt>User</tt>
      #                    ここで使用するモデルクラスには current_id というクラスメソッドが
      # 　　　　　　　　　 定義されている必要がある。
      #
      def context_monitor(target_model_name = :user, options = {})
        model_name  = target_model_name.to_s.capitalize
        suffix = (options[:suffix] || 'by').to_s
        created = 'created_' + suffix
        updated = 'updated_' + suffix

        class_eval do
          callbacks = Callbacks.new(model_name, created, updated)
          before_create callbacks
          before_update callbacks
          belongs_to created, :class_name => model_name, :foreign_key => "#{created}_id"
          belongs_to updated, :class_name => model_name, :foreign_key => "#{updated}_id"
        end
      end
      alias monitor context_monitor
    end

    class Callbacks # :ndoc: all
      def initialize(model_name, created, updated)
        @model = model_name.constantize
        @created_id = "#{created}_id"
        @updated_id = "#{updated}_id"
      end
      def before_create(record)
        record.write_attribute(@created_id, @model.current_id) if include_column_and_nil?(record, @created_id)
        before_update record
      end
      def before_update(record)
        record.write_attribute(@updated_id, @model.current_id) if include_column?(record, @updated_id)
      end

      private
      def column_names(record)
        @column_names ||= record.class.column_names
      end

      def include_column?(record, column)
        column_names(record).include? column
      end

      def include_column_and_nil?(record, column)
        include_column?(record, column) && record.read_attribute(column).nil?
      end
    end
  end
end

ActiveRecord::Base.__send__(:include, ActiveRecord::ContextMonitor)

