# -*- coding: utf-8 -*-

require File.join(File.dirname(__FILE__), 'spec_helper')

module CurrentMethods
  def current=(user)
    @current = user
  end
  def current
    @current
  end
  def current_id
    current.id if current
  end
end

describe ActiveRecord::ContextMonitor do
  module ActiveRecord::ContextMonitor
    describe ClassMethods do
      it 'defines context_monitor' do
        ClassMethods.instance_methods.should be_include('context_monitor')
      end

      it 'defines alias monitor' do
        ClassMethods.instance_methods.should be_include('monitor')
      end
    end

    describe Callbacks do
      before do
        @created_by = 'created_by'
        @updated_by = 'updated_by'
        @callbacks = Callbacks.new('Object', @created_by, @updated_by)

        @a, @b, @c = 'a', 'b', 'c'
        @record = mock('record', :null_object => true)
        @record.stub(:class => @record)
      end

      it 'sets model and needed ids' do
        created_by, updated_by = @created_by, @updated_by
        @callbacks.instance_eval do
          @model.should == Object
          @created_id.should == "#{created_by}_id"
          @updated_id.should == "#{updated_by}_id"
        end
      end

      it 'can list all column names' do
        @record.should_receive(:column_names).and_return([@a, @b, @c])
        column_names = @callbacks.__send__(:column_names, @record)
        column_names.should == [@a, @b, @c]
      end

      it 'can test does record has column' do
        @record.stub(:column_names).and_return([@a, @b, @c])
        [@a, @b, @c].each do |target|
          @callbacks.__send__(:include_column?, @record, target).should be_true
        end
        [nil, :hoge, 'test', 1].each do |target|
          @callbacks.__send__(:include_column?, @record, target).should be_false
        end
      end

      it 'can test does record has column and it value is nil' do
        column = :test
        @callbacks.should_receive(:include_column?).
          with(@record, column).and_return(true)
        @record.should_receive(:read_attribute).with(column).and_return(nil)
        @callbacks.__send__(:include_column_and_nil?, @record, column).should be_true
      end
    end
  end

  describe 'Default user' do
    before(:all) do
      class User < ActiveRecord::Base
        extend CurrentMethods
      end

      class Article < ActiveRecord::Base
        context_monitor :user
      end
    end

    describe 'create' do
      before do
        @user = User.create(:name => 'test')
        User.current = @user
        @article = Article.create!(:title => 'title',
                                   :body  => 'body' * 100)
      end
      after do
        @user = nil
        User.current = nil
      end
      it 'has same id' do
        @article.created_by_id.should == @user.id
        @article.updated_by_id.should == @user.id
      end
      it 'has same object' do
        @article.created_by.should == @user
        @article.updated_by.should == @user
      end
    end

    describe 'update' do
      before do
        @user1 = ::User.create(:name => 'alice')
        User.current = @user1
        @article = ::Article.create!(:title => 'title',
                                     :body  => 'body' * 100)
        @article.title = 'change title'
        @user2 = ::User.create(:name => 'bob')
        User.current = @user2
        @article.save!
      end
      after do
        @user1 = nil
        @user2 = nil
        User.current = nil
      end
      it 'has collect id' do
        @article.created_by_id.should == @user1.id
        @article.updated_by_id.should == @user2.id
      end
      it 'has collect object' do
        @article.created_by.should == @user1
        @article.updated_by.should == @user2
      end
    end

    describe "no current user" do
      before do
        @user = User.create(:name => 'good')
        @article = Article.create!(:title => 'title',
                                   :body  => 'body' * 100)
      end
      after do
        @user = nil
        User.current = nil
      end
      it 'all is nil' do
        @article.created_by_id.should be_nil
        @article.updated_by_id.should be_nil
        @article.created_by.should be_nil
        @article.updated_by.should be_nil
      end
    end
  end

  describe 'Set Class and Suffix' do
    before(:all) do
      class Country < ActiveRecord::Base
        extend CurrentMethods
      end

      class Car < ActiveRecord::Base
        context_monitor :country, :suffix => :in
      end
    end

    before do
      @country1 = Country.create(:name => 'Japan')
      Country.current = @country1
      @car = Car.create!(:name => 'XXX', :price => 100)
      @country2 = Country.create(:name => 'America')
    end

    after do
      @country1 = nil
      Country.current = nil
    end

    it 'has same id' do
      @car.created_in_id.should == @country1.id
      @car.updated_in_id.should == @country1.id
    end

    it 'has same object' do
      @car.created_in.should == @country1
      @car.updated_in.should == @country1
    end

    it 'has collect id and object when update' do
      Country.current = @country2
      @car.price = 200
      @car.save!

      @car.created_in_id.should == @country1.id
      @car.updated_in_id.should == @country2.id
      @car.created_in.should == @country1
      @car.updated_in.should == @country2
    end
  end
end

