package org.ultramonkey.l7.model;

import java.net.*;
import java.util.HashSet;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.ultramonkey.l7.controller.DirectorController;
import org.ultramonkey.l7.controller.DirectorData;
import org.ultramonkey.l7.controller.L7vsConfController;
import org.ultramonkey.l7.controller.L7vsConfData;
import org.ultramonkey.l7.controller.LogSet;

/**
 * <p>class L7SyncManager</p>
 * <p>Copyright(c) NTT COMWARE 2008</p>
 * @author momose
 */
public class L7SyncManager {
    protected Logger debugLogger = Logger.getLogger(LogCategorySet.SYNC_SYNC);

    protected Logger debugDirectorLogger = Logger
                    .getLogger(LogCategorySet.SYNC_SYNC_DIRECTOR_DATA);

    protected Logger debugL7vsdConfLogger = Logger
                    .getLogger(LogCategorySet.SYNC_SYNC_L7VSDCONF_DATA);

    protected Logger debugUserLogger = Logger
                    .getLogger(LogCategorySet.SYNC_SYNC_USER_DATA);

    protected String categoryName;

    protected String actHost;

    protected String sbyHost;

    protected int sbyPort;

    protected Logger logger = Logger.getLogger(LogCategorySet.SYNC_SYNC);

    protected Logger directorLogger = Logger
                    .getLogger(LogCategorySet.SYNC_SYNC_DIRECTOR_DATA);

    protected Logger l7vsdConfLogger = Logger
                    .getLogger(LogCategorySet.SYNC_SYNC_L7VSDCONF_DATA);

    protected Logger userLogger = Logger.getLogger(LogCategorySet.SYNC_SYNC_USER_DATA);

    protected enum SyncDataType {
        DIRECTORDATA, L7VSCONFDATA, USERDATA
    };

    /**
     * Constructor gets data from Parameter class
     *
     */
    public L7SyncManager() {
        //      debug log(constructor)
        if (debugLogger.isDebugEnabled()) {
            debugLogger.debug("11058 class L7SyncManager created.");
        }

        Parameter parameter = Parameter.getInstance();
        actHost = parameter.getValue("actHost");
        sbyHost = parameter.getValue("sbyHost");
        sbyPort = Integer.parseInt(parameter.getValue("sbyPort"));
    }

    /**
     * <p> Active server sends synchronous data from active server to stand-by server. It recieves results from stand-by server.</p>
     *
     * @param categoryIdSet category IDs of synchronous data
     * @throws MessageView
     */
    public void syncAct(HashSet<Integer> categoryIdSet) throws MessageView {
        //      debug log(in method)
        if (debugLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::syncAct(HashSet<Integer> categoryIdSet) throws MessageView in ");
            buf.append("categoryIdSet= " + categoryIdSet.toString());
            debugLogger.debug("11059 " + buf.toString());
        }

        Vector<Payload> payloadVec = new Vector<Payload>();
        Payload payload = null;
        L7SyncSocket l7SyncSocket = new L7SyncSocket();

        for (int categoryId : categoryIdSet) {
            switch (getDataTypeFromId(categoryId)) {
            case DIRECTORDATA:
                try {
                    DirectorController directorController = new DirectorController();
                    payload = new Payload(categoryId, directorController
                                    .getDirectorData());
                } catch (Exception e) {
                    logger.error("41007 " + Message.SYNC_SET_ERROR + "DirectorData : " + e);
                    throw new MessageView(40012, LogCategorySet.SYNC_START_STOP,
                                    Message.SYNC_SET_ERROR + " DirectorData :" + e);
                }
                break;

            case L7VSCONFDATA:
                L7vsConfController l7vsConfController = new L7vsConfController();
                try {
                    payload = new Payload(categoryId, l7vsConfController.getConfData());
                } catch (FileNotReadException e) {
                    logger.error("41008 " + Message.SYNC_SET_ERROR + "L7vsConfData : " + e);
                    throw new MessageView(40013, LogCategorySet.SYNC_SYNC,
                                    Message.SYNC_GET_ERROR + " L7vsConfData :" + e);
                }
                break;

            case USERDATA:
                try {
                    UserManager userManager = new UserManager();
                    payload = new Payload(categoryId, userManager.getUserList());
                } catch (FileNotReadException e) {
                    logger.error("41009 " + Message.SYNC_SET_ERROR + "L7vsdConfData : " + e);
                    throw new MessageView(40030, LogCategorySet.SYNC_SYNC,
                                    Message.SYNC_GET_ERROR + " UserData :" + e);
                }
                break;

            default:
                throw new MessageView(40027, LogCategorySet.SYNC_START_STOP,
                                Message.SYNC_INVALID_CATEGORY_ID);
            }
            payloadVec.add(payload);
        }
        try {
        	if (sbyHost == null)
        		throw new ConnectException();
            Socket socket = new Socket(sbyHost, sbyPort);
            l7SyncSocket
                            .openConnection(socket.getInputStream(), socket
                                            .getOutputStream());
        } catch (Exception e) {
            logger.error("41010 " + Message.SYNC_OPEN_ERROR + " : " + e);
            throw new MessageView(40014, LogCategorySet.SYNC_START_STOP,
                            Message.SYNC_OPEN_ERROR);
        }
        sendRecv(l7SyncSocket, payloadVec);

        try {
            l7SyncSocket.closeConnection();
        } catch (MessageView m) {
            logger.error("41011 " + m.message);
            throw m;
        }

        //      debug log(out method)
        if (debugLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::syncAct(HashSet<Integer> categoryIdSet) throws MessageView out ");
            debugLogger.debug("11060 " + buf.toString());
        }

    }

    /**
     * <p> Stand-by server recieves synchronous data from active server. It compares synchronous data. It updates stand-by data if there are any differences between active server settings and stand-by server settings.</p>
     *
     * @throws MessageView
     */
    public void syncSby() throws MessageView {
        //      debug log(in method)
        if (debugLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf.append("L7SyncManager::syncSby() throws MessageView in ");
            debugLogger.debug("11061 " + buf.toString());
        }

        L7SyncSocket l7SyncSocket = new L7SyncSocket();
        Payload recvPayload = null;
        Payload sendPayload;
        String[] recvData;
        Object sbyData;

        try {
            l7SyncSocket.openConnection(System.in, System.out);
        } catch (MessageView m) {
            logger.error("41012 " + m.message);
            throw m;
        }

        while (null != (recvData = l7SyncSocket.recvData())) {
            try {
                recvPayload = new Payload(recvData);
                switch (getDataTypeFromId(recvPayload.categoryId)) {
                case DIRECTORDATA:
                    DirectorController directorController = new DirectorController();
                    try {
                        DirectorData sbyDirectorData = compareDirectorData(recvPayload,
                                        directorController.getDirectorData());
                        if (null != sbyDirectorData) {
                            directorController.setDirectorData(sbyDirectorData);
                        }
                    } catch (Exception e) {
                        throw new MessageView(40015, LogCategorySet.SYNC_START_STOP,
                                        Message.SYNC_GET_ERROR + " DirectorData");
                    }
                    break;

                case L7VSCONFDATA:
                    L7vsConfController l7vsConfController = new L7vsConfController();
                    try {
                        L7vsConfData sbyL7vsConfData = compareL7vsConfData(recvPayload,
                                        l7vsConfController.getConfData());
                        if (null != sbyL7vsConfData) {
                            l7vsConfController.setConfData(sbyL7vsConfData);
                        }
                    } catch (FileNotReadException e) {
                        throw new MessageView(40016, LogCategorySet.SYNC_SYNC,
                                        Message.SYNC_GET_ERROR + " L7vsConfData :" + e);
                    } catch (FileNotWriteException e) {
                        throw new MessageView(40017, LogCategorySet.SYNC_SYNC,
                                        Message.SYNC_SET_ERROR + " L7vsConfData :" + e);
                    }
                    break;

                case USERDATA:
                    try {
                        UserManager userManager = new UserManager();
                        Vector<UserData> sbyUserVec = compareUserData(recvPayload,
                                        userManager.getUserList());
                        if (null != sbyUserVec) {
                            userManager.setUserList(sbyUserVec);
                        }
                    } catch (Exception e) {
                        throw new MessageView(40018, LogCategorySet.SYNC_START_STOP,
                                        Message.SYNC_GET_ERROR + " userdata");
                    }
                    break;
                default:
                    throw new MessageView(40028, LogCategorySet.SYNC_START_STOP,
                                    Message.SYNC_INVALID_CATEGORY_ID);
                }
                sendPayload = new Payload(SyncCategoryIdSet.SYNC_OK, new DirectorData());
                logger.info("21001 " + SyncCategoryIdSet.toCategoryName(recvPayload.categoryId)
                                + Message.SYNC_SUCCESS);

            } catch (MessageView m) {
                logger.error("41013 " + m.message);
                sendPayload = new Payload(SyncCategoryIdSet.SYNC_NG, new DirectorData());
            }

            l7SyncSocket.sendData(sendPayload.getMessage());
        }
        //      debug log(out method)
        if (debugLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf.append("L7SyncManager::syncSby() throws MessageView out ");
            debugLogger.debug("11062 " + buf.toString());
        }

    }

    /**
     * <p>  Sends synchronous data and recieves results from stand-by server. </p>
     *
     * @param l7SyncSocket
     * @param payloadVec synchronous data
     * @throws MessageView
     */
    protected void sendRecv(L7SyncSocket l7SyncSocket, Vector<Payload> payloadVec)
                    throws MessageView {
        //      debug log(in method)
        if (debugLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::sendRecv(L7SyncSocket l7SyncSocket, Vector<Payload> payloadVec) throws MessageView in ");
            buf.append("l7SyncSocket= " + l7SyncSocket.toString() + ", ");
            buf.append("payloadVec= " + payloadVec.toString());
            debugLogger.debug("11063 " + buf.toString());
        }

        Payload recvPayload;
        for (Payload payload : payloadVec) {
            l7SyncSocket.sendData(payload.getMessage());
            recvPayload = new Payload(l7SyncSocket.recvData());
            switch (recvPayload.categoryId) {
            case SyncCategoryIdSet.SYNC_NG:
                logger.error("41014 " + Message.SYNC_FAIL + " : "
                                + SyncCategoryIdSet.toCategoryName(payload.categoryId));
                throw new MessageView(
                                40019,
                                LogCategorySet.SYNC_CONNECTION,
                                Message.SYNC_FAIL
                                                + " : "
                                                + SyncCategoryIdSet
                                                                .toCategoryName(payload.categoryId));
            case SyncCategoryIdSet.SYNC_OK:
                logger.info("21002 " + SyncCategoryIdSet.toCategoryName(payload.categoryId)
                                + Message.SYNC_SUCCESS);
                break;
            default:
                throw new MessageView(40029, LogCategorySet.SYNC_START_STOP,
                                Message.SYNC_INVALID_CATEGORY_ID);
            }
        }
        //      debug log(out method)
        if (debugLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::sendRecv(L7SyncSocket l7SyncSocket, Vector<Payload> payloadVec) throws MessageView out ");
            debugLogger.debug("11064 " + buf.toString());
        }

    }

    /**
     * <p> Compares DirectorData settings in active server with those in stand-by server. Updates DirectorData in stand-by server if there are any differences between them.</p>
     *
     * @param categoryId category ID of synchronous data
     * @param recvData recieve data from active server
     * @throws MessageView
     */
    protected DirectorData compareDirectorData(Payload recvData,
                    DirectorData sbyDirectorData) throws MessageView {
        //      debug log(in method)
        if (debugDirectorLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::compareDirectorData(Payload recvData, DirectorData sbyDirectorData) throws MessageView in ");
            buf.append("recvData= " + recvData.toString() + ", ");
            buf.append("sbyDirectorData= " + sbyDirectorData.toString());
            debugDirectorLogger.debug("11065 " + buf.toString());
        }

        DirectorData actDirectorData = recvData.getDirectorData();
        boolean needSync = false;

        switch (recvData.categoryId) {
        case SyncCategoryIdSet.MONITOR_ID:
            if (actDirectorData.checkTimeOut != sbyDirectorData.checkTimeOut) {
                directorLogger.info("21003 " + Message.SYNC_DIFFERENT_DATA + " : checkTimeOut ");
                sbyDirectorData.checkTimeOut = actDirectorData.checkTimeOut;
                needSync = true;
            }
            if (actDirectorData.negotiateTimeOut != sbyDirectorData.negotiateTimeOut) {
                directorLogger.info("21004 " + Message.SYNC_DIFFERENT_DATA + " : negotiateTimeOut ");
                sbyDirectorData.negotiateTimeOut = actDirectorData.negotiateTimeOut;
                needSync = true;
            }
            if (actDirectorData.checkInterval != sbyDirectorData.checkInterval) {
                directorLogger.info("21005 " + Message.SYNC_DIFFERENT_DATA + " : checkInterval ");
                sbyDirectorData.checkInterval = actDirectorData.checkInterval;
                needSync = true;
            }
            if (actDirectorData.checkCount != sbyDirectorData.checkCount) {
                directorLogger.info("21006 " + Message.SYNC_DIFFERENT_DATA + " : checkCount ");
                sbyDirectorData.checkCount = actDirectorData.checkCount;
                needSync = true;
            }
            if (actDirectorData.autoreload != sbyDirectorData.autoreload) {
                directorLogger.info("21007 " + Message.SYNC_DIFFERENT_DATA + " : autoreload ");
                sbyDirectorData.autoreload = actDirectorData.autoreload;
                needSync = true;
            }
            if (!actDirectorData.callback.equals(sbyDirectorData.callback)) {
                directorLogger.info("21008 " + Message.SYNC_DIFFERENT_DATA + " : callback ");
                sbyDirectorData.callback = actDirectorData.callback;
                needSync = true;
            }
            if (!actDirectorData.fallback.equals(sbyDirectorData.fallback)) {
                directorLogger.info("21009 " + Message.SYNC_DIFFERENT_DATA + " : fallback ");
                sbyDirectorData.fallback = actDirectorData.fallback;
                needSync = true;
            }
            if (!actDirectorData.logfile.equals(sbyDirectorData.logfile)) {
                directorLogger.info("21010 " + Message.SYNC_DIFFERENT_DATA + " : logfile ");
                sbyDirectorData.logfile = actDirectorData.logfile;
                needSync = true;
            }
            if (!actDirectorData.execute.equals(sbyDirectorData.execute)) {
                directorLogger.info("21011 " + Message.SYNC_DIFFERENT_DATA + " : execute ");
                sbyDirectorData.execute = actDirectorData.execute;
                needSync = true;
            }
            if (!actDirectorData.supervised.equals(sbyDirectorData.supervised)) {
                directorLogger.info("21012 " + Message.SYNC_DIFFERENT_DATA + " : supervised ");
                sbyDirectorData.supervised = actDirectorData.supervised;
                needSync = true;
            }
            if (!actDirectorData.quiescent.equals(sbyDirectorData.quiescent)) {
                directorLogger.info("21013 " + Message.SYNC_DIFFERENT_DATA + " : quiescent ");
                sbyDirectorData.quiescent = actDirectorData.quiescent;
                needSync = true;
            }
            break;

        case SyncCategoryIdSet.SERVICE_ID:
            if (actDirectorData.virtualSettings.size() != sbyDirectorData.virtualSettings
                            .size()) {
                sbyDirectorData.virtualSettings = actDirectorData.virtualSettings;
                directorLogger.info("21014 " + Message.SYNC_DIFFERENT_DATA + " : virtualSettings ");
                needSync = true;
            } else {
                if (!actDirectorData.virtualSettings
                                .equals(sbyDirectorData.virtualSettings)) {
                    sbyDirectorData.virtualSettings = actDirectorData.virtualSettings;
                    needSync = true;
                }
            }
            break;

        default:
            throw new MessageView(40020, LogCategorySet.SYNC_SYNC_DIRECTOR_DATA,
                            Message.SYNC_INVALID_CATEGORY_ID);
        }
        if (!needSync) {
            directorLogger.info("21015 " + Message.SYNC_SAME_DATA + " : "
                            + SyncCategoryIdSet.toCategoryName(recvData.categoryId));

            //      debug log(out method)
            if (debugDirectorLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::compareDirectorData(Payload recvData, DirectorData sbyDirectorData) throws MessageView out ");
                buf.append("return= null");
                debugDirectorLogger.debug("11066 " + buf.toString());
            }
            return null;
        }

        //      debug log(out method)
        if (debugDirectorLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::compareDirectorData(Payload recvData, DirectorData sbyDirectorData) throws MessageView out ");
            buf.append("return= " + sbyDirectorData.toString());
            debugDirectorLogger.debug("11067 " + buf.toString());
        }

        return sbyDirectorData;
    }

    /**
     * <p> Compares UserData settings in active server with those in stand-by server. Updates UserData in stand-by server if there are any differences between them.</p>
     *
     * @param categoryId category id of synchronous data
     * @param recvData recieve data from active server
     * @throws MessageView
     */
    protected Vector<UserData> compareUserData(Payload recvData,
                    Vector<UserData> sbyUserVec) throws MessageView {
        //      debug log(in method)
        if (debugUserLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::compareUserData(Payload recvData, Vector<UserData> sbyUserVec) throws MessageView in ");
            buf.append("recvData= " + recvData.toString() + ", ");
            buf.append("sbyUserVec= " + sbyUserVec.toString());
            debugUserLogger.debug("11068 " + buf.toString());
        }

        if (SyncCategoryIdSet.USERDATA_ID != recvData.categoryId) {
            throw new MessageView(40021, LogCategorySet.SYNC_SYNC_USER_DATA,
                            Message.SYNC_INVALID_CATEGORY_ID);
        }

        Vector<UserData> actUserVec = recvData.getUserData();

        if (actUserVec.size() != sbyUserVec.size() || !actUserVec.equals(sbyUserVec)) {
            userLogger.info("21016 " + Message.SYNC_DIFFERENT_DATA + " : "
                            + SyncCategoryIdSet.toCategoryName(recvData.categoryId));

            // debug log(out method)
            if (debugUserLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::compareUserData(Payload recvData, Vector<UserData> sbyUserVec) throws MessageView out ");
                buf.append("return= " + actUserVec.toString());
                debugUserLogger.debug("11069 " + buf.toString());
            }

            return actUserVec;
        } else {
            userLogger.info("21017 " + Message.SYNC_SAME_DATA + " : "
                            + SyncCategoryIdSet.toCategoryName(recvData.categoryId));

            // debug log(out method)
            if (debugUserLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::compareUserData(Payload recvData, Vector<UserData> sbyUserVec) throws MessageView out ");
                buf.append("return= null");
                debugUserLogger.debug("11070 " + buf.toString());
            }

            return null;
        }
    }

    /**
     * <p> Compares L7vsConfData settings in active server with those in stand-by server. Updates L7vsConfData in stand-by server if there are any differences between them.</p>
     *
     * @param categoryId category ID of synchronous data
     * @param recvData recieve data from active server
     * @throws MessageView
     */
    protected L7vsConfData compareL7vsConfData(Payload recvData,
                    L7vsConfData sbyL7vsdConfData) throws MessageView {
        //      debug log(in method)
        if (debugL7vsdConfLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::compareL7vsConfData(Payload recvData, L7vsConfData sbyL7vsdConfData) throws MessageView in ");
            buf.append("recvData= " + recvData.toString() + ", ");
            buf.append("sbyL7vsdConfData= " + sbyL7vsdConfData.toString());
            debugL7vsdConfLogger.debug("11071 " + buf.toString());
        }

        L7vsConfData actL7vsdConfData = recvData.getL7vsdConfData();
        boolean needSync = false;
        L7vsConfController confController = new L7vsConfController();

        switch (recvData.categoryId) {
        case SyncCategoryIdSet.L7VSD_LOG_ID:
            needSync = compareLogSettings(actL7vsdConfData.logger.lbLog,
                            sbyL7vsdConfData.logger.lbLog);
            if (needSync) {
                sbyL7vsdConfData.logger.lbLog = actL7vsdConfData.logger.lbLog;
            }
            break;

        case SyncCategoryIdSet.ADM_LOG_ID:
            needSync = compareLogSettings(actL7vsdConfData.logger.admLog,
                            sbyL7vsdConfData.logger.admLog);
            if (needSync) {
                sbyL7vsdConfData.logger.admLog = actL7vsdConfData.logger.admLog;
            }
            break;

        case SyncCategoryIdSet.SNMP_LOG_ID:
            needSync = compareLogSettings(actL7vsdConfData.logger.snmpLog,
                            sbyL7vsdConfData.logger.snmpLog);
            if (needSync) {
                sbyL7vsdConfData.logger.snmpLog = actL7vsdConfData.logger.snmpLog;
            }
            break;

        case SyncCategoryIdSet.LOG_ALL_ID:
            boolean lbLogFlg = compareLogSettings(actL7vsdConfData.logger.lbLog,
                            sbyL7vsdConfData.logger.lbLog);
            boolean admLogFlg = compareLogSettings(actL7vsdConfData.logger.admLog,
                            sbyL7vsdConfData.logger.admLog);
            boolean snmpLogFlg = compareLogSettings(actL7vsdConfData.logger.snmpLog,
                            sbyL7vsdConfData.logger.snmpLog);

            needSync = lbLogFlg || admLogFlg || snmpLogFlg;

            if (needSync) {
                sbyL7vsdConfData.logger.lbLog = actL7vsdConfData.logger.lbLog;
                sbyL7vsdConfData.logger.admLog = actL7vsdConfData.logger.admLog;
                sbyL7vsdConfData.logger.snmpLog = actL7vsdConfData.logger.snmpLog;
            }
            break;

        case SyncCategoryIdSet.REPLICATION_INFO_ID:
            if (actL7vsdConfData.replicationInterval != sbyL7vsdConfData.replicationInterval) {
                logger.info("21018 " + Message.SYNC_DIFFERENT_DATA + ": Replication");
                sbyL7vsdConfData.replicationInterval = actL7vsdConfData.replicationInterval;
                needSync = true;
            }
            break;

        case SyncCategoryIdSet.SNMP_INFO_ID:
            if (actL7vsdConfData.snmpInterval != sbyL7vsdConfData.snmpInterval) {
                logger.info("21019 " + Message.SYNC_DIFFERENT_DATA + ": snmpInterval");
                sbyL7vsdConfData.snmpInterval = actL7vsdConfData.snmpInterval;
                needSync = true;
            }
            break;

        case SyncCategoryIdSet.ADM_INFO_ID:
            if (actL7vsdConfData.admCmdInterval != sbyL7vsdConfData.admCmdInterval) {
                logger.info("21020 " + Message.SYNC_DIFFERENT_DATA + ": admCmdInterval");
                sbyL7vsdConfData.admCmdInterval = actL7vsdConfData.admCmdInterval;
                needSync = true;
            }
            if (actL7vsdConfData.admCmdCount != sbyL7vsdConfData.admCmdCount) {
                logger.info("21021 " + Message.SYNC_DIFFERENT_DATA + ": admCmdCount");
                sbyL7vsdConfData.admCmdCount = actL7vsdConfData.admCmdCount;
                needSync = true;
            }
            if (actL7vsdConfData.admConInterval != sbyL7vsdConfData.admConInterval) {
                logger.info("21022 " + Message.SYNC_DIFFERENT_DATA + ": admConInterval");
                sbyL7vsdConfData.admConInterval = actL7vsdConfData.admConInterval;
                needSync = true;
            }
            if (actL7vsdConfData.admConCount != sbyL7vsdConfData.admConCount) {
                logger.info("21023 " + Message.SYNC_DIFFERENT_DATA + ": admConCount");
                sbyL7vsdConfData.admConCount = actL7vsdConfData.admConCount;
                needSync = true;
            }
            break;

        default:
            l7vsdConfLogger.error("41015 " + Message.SYNC_INVALID_CATEGORY_ID);
            throw new MessageView(40022, LogCategorySet.SYNC_SYNC_L7VSDCONF_DATA,
                            Message.SYNC_INVALID_CATEGORY_ID);
        }

        if (!needSync) {
            l7vsdConfLogger.info("21024 " + Message.SYNC_SAME_DATA + " : "
                            + SyncCategoryIdSet.toCategoryName(recvData.categoryId));
            if (debugL7vsdConfLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::compareL7vsConfData(Payload recvData, L7vsConfData sbyL7vsdConfData) throws MessageView out ");
                buf.append("return= null");
                debugL7vsdConfLogger.debug("11072 " + buf.toString());
            }
            return null;

        } else {
            //          debug log(out method)
            if (debugL7vsdConfLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::compareL7vsConfData(Payload recvData, L7vsConfData sbyL7vsdConfData) throws MessageView out ");
                buf.append("return= " + sbyL7vsdConfData.toString());
                debugL7vsdConfLogger.debug("11073 " + buf.toString());
            }

            return sbyL7vsdConfData;
        }
    }

    /**
     * <p> Gets synchronous data class from category ID.</p>
     *
     * @param categoryId category ID of synchronous data
     * @return data class name of synchronous data
     * @throws MessageView
     */
    protected SyncDataType getDataTypeFromId(int categoryId) throws MessageView {
        //      debug log(in method)
        if (debugLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::getDataTypeFromId(int categoryId) throws MessageView in ");
            buf.append("categoryId=" + categoryId);
            debugLogger.debug("11074 " + buf.toString());
        }

        switch (categoryId) {
        case SyncCategoryIdSet.MONITOR_ID:
        case SyncCategoryIdSet.SERVICE_ID:
            //          debug log(out method)
            if (debugLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::getDataTypeFromId(int categoryId) throws MessageView out ");
                buf.append("return= SyncDataType.DIRECTORDATA");
                debugLogger.debug("11075 " + buf.toString());
            }
            return SyncDataType.DIRECTORDATA;

        case SyncCategoryIdSet.L7VSD_LOG_ID:
        case SyncCategoryIdSet.ADM_LOG_ID:
        case SyncCategoryIdSet.SNMP_LOG_ID:
        case SyncCategoryIdSet.REPLICATION_INFO_ID:
        case SyncCategoryIdSet.ADM_INFO_ID:
        case SyncCategoryIdSet.SNMP_INFO_ID:
        case SyncCategoryIdSet.LOG_ALL_ID:
            //          debug log(out method)
            if (debugLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::getDataTypeFromId(int categoryId) throws MessageView out ");
                buf.append("return= SyncDataType.L7VSCONFDATA");
                debugLogger.debug("11076 " + buf.toString());
            }
            return SyncDataType.L7VSCONFDATA;

        case SyncCategoryIdSet.USERDATA_ID:
            //          debug log(out method)
            if (debugLogger.isDebugEnabled()) {
                StringBuffer buf = new StringBuffer();
                buf
                                .append("L7SyncManager::getDataTypeFromId(int categoryId) throws MessageView out ");
                buf.append("return= SyncDataType.USERDATA");
                debugLogger.debug("11077 " + buf.toString());
            }
            return SyncDataType.USERDATA;

        default:
            throw new MessageView(40023, LogCategorySet.SYNC_SYNC,
                            Message.SYNC_INVALID_CATEGORY_ID);
        }
    }

    /**
     * <p> Compares log settings in active server with those in stand-by server. </p>
     *
     * @param actLogger LogSet of active server
     * @param sbyLogger LogSet of stand-by server
     * @return sync flag
     */
    protected boolean compareLogSettings(LogSet actLogger, LogSet sbyLogger) {
        //      debug log(in method)
        if (debugL7vsdConfLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::compareLogSettings(LogSet actLogger, LogSet sbyLogger) in ");
            buf.append("actLogger= " + actLogger.toString() + ", ");
            buf.append("sbyLogger= " + sbyLogger.toString());
            debugL7vsdConfLogger.debug("11078 " + buf.toString());
        }

        boolean syncFlag = false;
        if (!actLogger.logLevel.equals(sbyLogger.logLevel)) {
            l7vsdConfLogger.info("21025 " + Message.SYNC_DIFFERENT_DATA + " : LogLevel ");
            syncFlag = true;
        }
        if (!actLogger.maxFileSize.equals(sbyLogger.maxFileSize)) {
            l7vsdConfLogger.info("21026 " + Message.SYNC_DIFFERENT_DATA + " : maxFileSize ");
            syncFlag = true;
        }
        if (actLogger.maxBackupIndex != sbyLogger.maxBackupIndex) {
            l7vsdConfLogger.info("21027 " + Message.SYNC_DIFFERENT_DATA + " : maxBackupIndex ");
            syncFlag = true;
        }
        if (!actLogger.fileName.equals(sbyLogger.fileName)) {
            l7vsdConfLogger.info("21028 " + Message.SYNC_DIFFERENT_DATA + " : fileName ");
            syncFlag = true;
        }
        if (!actLogger.rotation.equals(sbyLogger.rotation)) {
            l7vsdConfLogger.info("21029 " + Message.SYNC_DIFFERENT_DATA + " : rotation ");
            syncFlag = true;
        }
        //      debug log(out method)
        if (debugL7vsdConfLogger.isDebugEnabled()) {
            StringBuffer buf = new StringBuffer();
            buf
                            .append("L7SyncManager::compareLogSettings(LogSet actLogger, LogSet sbyLogger) out ");
            buf.append("return= " + syncFlag);
            debugL7vsdConfLogger.debug("11079 " + buf.toString());
        }

        return syncFlag;
    }
}
