package org.ultramonkey.l7.model;

import java.io.*;
import java.util.*;
import java.util.regex.*;

import org.apache.log4j.Logger;
import org.ultramonkey.l7.controller.*;

/**
 * 
 * <p>
 * class L7guiFramework
 * </p>
 * <p>
 * Copyright(c) NTT COMWARE 2008
 * </p>
 * 
 * @author tanuma
 */
public class L7guiFramework {

	private Logger ioCommandLogger = Logger.getLogger(LogCategorySet.GUI_IO_COMMAND);
	private Logger ioFileLogger = Logger.getLogger(LogCategorySet.GUI_IO_FILE);
	private Logger saveLogger = Logger.getLogger(LogCategorySet.GUI_SAVE);
	private Logger syncLogger = Logger.getLogger(LogCategorySet.GUI_SYNC);
	private Logger repLogger = Logger.getLogger(LogCategorySet.GUI_REPLICATION);
	private Logger snmpLogger = Logger.getLogger(LogCategorySet.GUI_SNMP);
	
	/**
	 * 
	 * <p>
	 * addVirtualSetting method
	 * </p>
	 * 
	 * @param vs
	 *            virtual service to be added to l7vsd.
	 * @return result of l7vsadm add virtual service command command.
	 */
	public boolean addVirtualSetting(VirtualSetting vs) {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11101 L7guiFramework::addVirtualSetting(VirtualSetting vs) in vs=(" + vs + ")");
		}
		// --- debug log (in method) ---

		L7vsAdmController admController = new L7vsAdmController();
		boolean ret = admController.addVirtualService(vs);
		
		// --- debug log (out method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11102 L7guiFramework::addVirtualSetting(VirtualSetting vs) out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * delVirtualSetting method
	 * </p>
	 * 
	 * @param vs
	 *            virtual service to be removed from l7vsd.
	 * @return result of l7vsadm delete virtual service command.
	 */
	public boolean delVirtualSetting(VirtualSetting vs) {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11103 L7guiFramework::delVirtualSetting(VirtualSetting vs) in vs=(" + vs + ")");
		}
		// --- debug log (in method) ---

		L7vsAdmController admController = new L7vsAdmController();
		boolean ret = admController.delVirtualService(vs);
		
		// --- debug log (out method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11104 L7guiFramework::delVirtualSetting(VirtualSetting vs) out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * getAdmData method
	 * </p>
	 * 
	 * @return result of l7vsadm list command, including virtual service
	 *         setting.
	 */
	public L7vsAdmData getAdmData() {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11105 L7guiFramework::getAdmData() in");
		}
		// --- debug log (in method) ---

		L7vsAdmController admController = new L7vsAdmController();
		L7vsAdmData ret = admController.getData();
		
		// --- debug log (out method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11106 L7guiFramework::getAdmData() out return=(" + ret + ")");
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * reloadAdmData method
	 * </p>
	 * 
	 * @param category
	 * @return result of l7vsadm reload config.
	 */
	public boolean reloadAdmData(L7vsConfCategory category) {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11107 L7guiFramework::reloadAdmData() in category=" + category);
		}
		// --- debug log (in method) ---

		L7vsAdmController admController = new L7vsAdmController();
		boolean ret = admController.reloadConf(category);
		
		// --- debug log (out method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11108 L7guiFramework::reloadAdmData() out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * getClusterData method
	 * </p>
	 * 
	 * @return cluster data.
	 */
	public ClusterData getClusterData() {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11109 L7guiFramework::getClusterData() in");
		}
		// --- debug log (in method) ---

		ClusterController cluster = new ClusterController();
		ClusterData ret = cluster.getData();
		
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11110 L7guiFramework::getClusterData() out return=(" + ret + ")");
		}
		// --- debug log (in method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * setClusterData method
	 * </p>
	 * 
	 * @param c
	 */
	public void setClusterData(ClusterData c) {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11111 L7guiFramework::setClusterData(ClusterData c) in c=(" + c + ")");
		}
		// --- debug log (in method) ---

		ClusterController cluster = new ClusterController();
		cluster.setData(c);

		// --- debug log (out method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11112 L7guiFramework::setClusterData(ClusterData c) out");
		}
		// --- debug log (out method) ---
	}

	/**
	 * 
	 * <p>
	 * setClusterMode method
	 * </p>
	 * 
	 * @param mode
	 * @return result of switch over.
	 */
	public boolean setClusterMode() {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11113 L7guiFramework::setClusterMode() in");
		}
		// --- debug log (in method) ---

		ClusterController cluster = new ClusterController();
		boolean ret = cluster.changeClusterMode();
		
		// --- debug log (out method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11114 L7guiFramework::setClusterMode() out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * getDirectorData method
	 * </p>
	 * 
	 * @return l7directord.cf data.
	 */
	public DirectorData getDirectorData() {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11115 L7guiFramework::getDirectorData() in");
		}
		// --- debug log (in method) ---

		DirectorData d = null;
		DirectorController directord = new DirectorController();
		try {
			d = directord.getDirectorData();
		} catch (Exception e) {
			ioFileLogger.error("41016 Exception occured: " + e.getMessage());
		}

		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11116 L7guiFramework::getDirectorData() out return=(" + d + ")");
		}
		// --- debug log (out method) ---
		return d;
	}

	/**
	 * 
	 * <p>
	 * setDirectorData method
	 * </p>
	 * 
	 * @param d
	 * @return result of save l7directord.cf.
	 */
	public boolean setDirectorData(DirectorData d) {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11117 L7guiFramework::setDirectorData(DirectorData d) in d=(" + d + ")");
		}
		// --- debug log (in method) ---
		
		boolean ret = true;

		DirectorController directord = new DirectorController();
		try {
			directord.setDirectorData(d);
		} catch (FileNotWriteException e) {
			ioFileLogger.error("41017 Exception occured: " + e.getMessage());
			ret = false;
		} catch (Exception e) {
			ioFileLogger.error("41018 Exception occured: " + e.getMessage());
			ret = false;
		}

		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11118 L7guiFramework::setDirectorData() out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * checkDirectorData method
	 * </p>
	 * 
	 * @param d
	 * @return result of check l7directord.cf.
	 */
	public String checkDirectorData(DirectorData d) {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11119 L7guiFramework::checkDirectorData(DirectorData d) in d=(" + d + ")");
		}
		// --- debug log (in method) ---
		
		String result = null;
		DirectorController directord = new DirectorController();
		try {
			result = directord.checkDirectorData(d);
		} catch (FileNotWriteException e) {
			ioFileLogger.error("41019 Exception occured: " + e.getMessage());
		} catch (Exception e) {
			ioFileLogger.error("41020 Exception occured: " + e.getMessage());
		}

		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11120 L7guiFramework::setDirectorData() out return=\"" + result + "\"");
		}
		// --- debug log (out method) ---
		return result;
	}

	/**
	 * 
	 * <p>
	 * getReplicationData method
	 * </p>
	 * 
	 * @return replication setting data.
	 */
	public ReplicationData getReplicationData() {
		// --- debug log (in method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11121 L7guiFramework::getReplicationData() in");
		}
		// --- debug log (in method) ---

		ReplicationData replicationData = new ReplicationData();
		
		L7vsAdmController admController = new L7vsAdmController();
		L7vsAdmData admData = admController.getData();
		L7vsConfController confController = new L7vsConfController();
		try {
			L7vsConfData confData = confController.getConfData();

			replicationData.interval = confData.replicationInterval;
			replicationData.otherNode = confData.replicationOtherNode;
			replicationData.component = confData.component;
			replicationData.nic = confData.replicationNic;
			replicationData.replicationMode = admData.replicationMode;
		} catch (FileNotReadException e) {
			ioFileLogger.error("41021 Exception occured: " + e.getMessage());
			replicationData = null;
		} catch (NullPointerException e) {
			ioFileLogger.error("41022 Exception occured: " + e.getMessage());
			replicationData = null;
		} catch (Exception e) {
			ioFileLogger.error("41023 Exception occured: " + e.getMessage());
			replicationData = null;
		}
		
		// --- debug log (out method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11122 L7guiFramework::getReplicationData() out return=(" + replicationData + ")");
		}
		// --- debug log (out method) ---
		return replicationData;
	}

	/**
	 * 
	 * <p>
	 * setReplicationData method
	 * </p>
	 * 
	 * @param r
	 * @return result of save l7vs.cf.
	 */
	public boolean setReplicationData(ReplicationData r) {
		// --- debug log (in method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11123 L7guiFramework::setReplicationData(ReplicationData r) in r=(" + r + ")");
		}
		// --- debug log (in method) ---

		boolean ret = true;
		L7vsConfController confController = new L7vsConfController();
		try {
			L7vsConfData cf = confController.getConfData();
			cf.replicationInterval = r.interval;
			confController.setConfData(cf);
		} catch (FileNotReadException e) {
			ioFileLogger.error("41024 Exception occured: " + e.getMessage());
			ret = false;
		} catch (FileNotWriteException e) {
			ioFileLogger.error("41025 Exception occured: " + e.getMessage());
			ret = false;
		} catch (NullPointerException e) {
			ioFileLogger.error("41026 Exception occured: " + e.getMessage());
			ret = false;
		} catch (Exception e) {
			ioFileLogger.error("41027 Exception occured: " + e.getMessage());
			ret = false;
		}

		// --- debug log (out method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11124 L7guiFramework::setReplicationData(ReplicationData r) out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * setReplicationMode method
	 * </p>
	 * 
	 * @param mode
	 * @return result of l7vsadm change replication mode command.
	 */
	public boolean setReplicationMode(ReplicationStatus mode) {
		// --- debug log (in method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11125 L7guiFramework::setReplicationMode(ReplicationStatus mode) in mode=" + mode);
		}
		// --- debug log (in method) ---

		L7vsAdmController admController = new L7vsAdmController();
		boolean ret = admController.setReplicationMode(mode);
		
		// --- debug log (out method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11126 L7guiFramework::setReplicationMode(ReplicationStatus mode) out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * dumpReplicationData method
	 * </p>
	 * 
	 * @return result of l7vsadm replication dump command.
	 */
	public boolean dumpReplicationData() {
		// --- debug log (in method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11127 L7guiFramework::dumpReplicationData() in");
		}
		// --- debug log (in method) ---

		L7vsAdmController admController = new L7vsAdmController();
		boolean ret = admController.dumpReplicationData();

		// --- debug log (out method) ---
		if (repLogger.isDebugEnabled()) {
            repLogger.debug("11128 L7guiFramework::dumpReplicationData() out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * getSnmpAgentData method
	 * </p>
	 * 
	 * @return SNMP agent data.
	 */
	public SnmpAgentData getSnmpAgentData() {
		// --- debug log (in method) ---
		if (snmpLogger.isDebugEnabled()) {
            snmpLogger.debug("11129 L7guiFramework::getSnmpAgentData() in");
		}
		// --- debug log (in method) ---

		SnmpAgentData snmpData = new SnmpAgentData();

		L7vsAdmController admController = new L7vsAdmController();
		L7vsAdmData admData = admController.getData();
		L7vsConfController confController = new L7vsConfController();
		try {
			snmpData.status = admData.snmpStatus;

			L7vsConfData confData = confController.getConfData();
			snmpData.interval = confData.snmpInterval;
		} catch (FileNotReadException e) {
			ioFileLogger.error("41028 Exception occured: " + e.getMessage());
			snmpData = null;
		} catch (NullPointerException e) {
			ioFileLogger.error("41029 Exception occured: " + e.getMessage());
			snmpData = null;
		} catch (Exception e) {
			ioFileLogger.error("41030 Exception occured: " + e.getMessage());
			snmpData = null;
		}

		// --- debug log (out method) ---
		if (snmpLogger.isDebugEnabled()) {
            snmpLogger.debug("11130 L7guiFramework::getSnmpAgentData() out return=(" + snmpData + ")");
		}
		// --- debug log (out method) ---
		return snmpData;
	}

	/**
	 * 
	 * <p>
	 * setSnmpAgentData method
	 * </p>
	 * 
	 * @param s
	 * @return result of save l7vs.cf
	 */
	public boolean setSnmpAgentData(SnmpAgentData s) {
		// --- debug log (in method) ---
		if (snmpLogger.isDebugEnabled()) {
            snmpLogger.debug("11131 L7guiFramework::setSnmpAgentData(SnmpAgentData s) in s=(" + s + ")");
		}
		// --- debug log (in method) ---

		boolean ret = true;
		L7vsConfController confController = new L7vsConfController();
		try {
			L7vsConfData cf = confController.getConfData();
			cf.snmpInterval = s.interval;
			confController.setConfData(cf);
		} catch (FileNotReadException e) {
			ioFileLogger.error("41031 Exception occured: " + e.getMessage());
			ret = false;
		} catch (FileNotWriteException e) {
			ioFileLogger.error("41032 Exception occured: " + e.getMessage());
			ret = false;
		} catch (NullPointerException e) {
			ioFileLogger.error("41033 Exception occured: " + e.getMessage());
			ret = false;
		} catch (Exception e) {
			ioFileLogger.error("41034 Exception occured: " + e.getMessage());
			ret = false;
		}

		// --- debug log (out method) ---
		if (snmpLogger.isDebugEnabled()) {
            snmpLogger.debug("11132 L7guiFramework::setSnmpAgentData(SnmpAgentData s) out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * getL7vsConfData method
	 * </p>
	 * 
	 * @return l7vs.cf data.
	 */
	public L7vsConfData getL7vsConfData() {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11133 L7guiFramework::getL7vsConfData() in");
		}
		// --- debug log (in method) ---

		L7vsConfController confController = new L7vsConfController();
		L7vsConfData ret = null;
		try {
			ret = confController.getConfData();
		} catch (FileNotReadException e) {
			ioFileLogger.error("41035 Exception occured: " + e.getMessage());
			ret = null;
		} catch (Exception e) {
			ioFileLogger.error("41036 Exception occured: " + e.getMessage());
			ret = null;
		}
		
		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11134 L7guiFramework::getL7vsConfData() out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * setL7vsConfData method
	 * </p>
	 * 
	 * @param cf
	 * @return result of save l7vs.cf
	 */
	public boolean setL7vsConfData(L7vsConfData cf) {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11135 L7guiFramework::setL7vsConfData(L7vsConfData cf) in cf=(" + cf + ")");
		}
		// --- debug log (in method) ---

		boolean ret = true;
		L7vsConfController confController = new L7vsConfController();
		try {
			confController.setConfData(cf);
		} catch (FileNotWriteException e) {
			ioFileLogger.error("41037 Exception occured: " + e.getMessage());
			ret = false;
		} catch (Exception e) {
			ioFileLogger.error("41038 Exception occured: " + e.getMessage());
			ret = false;
		}

		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11136 L7guiFramework::setL7vsConfData(L7vsConfData cf) out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * getLogData method
	 * </p>
	 * 
	 * @return log setting data.
	 */
	public LogData getLogData() {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11137 L7guiFramework::getLogData() in");
		}
		// --- debug log (in method) ---

		L7vsLogController logController = new L7vsLogController();
		LogData ret = logController.getData();
		
		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11138 L7guiFramework::getLogData() out return=(" + ret + ")");
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * saveExecute method
	 * </p>
	 * 
	 * @param h
	 * @return
	 */
	public boolean saveExecute(HashSet<SaveCategoryIdSet> h) {
		// --- debug log (in method) ---
		if (saveLogger.isDebugEnabled()) {
            saveLogger.debug("11139 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) in h=(" + h + ")");
		}
		// --- debug log (in method) ---

		if (h == null) {
			saveLogger.warn("31001 Invalid null argument.");
			// --- debug log (out method) ---
			if (saveLogger.isDebugEnabled()) {
	            saveLogger.debug("11140 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
			}
			// --- debug log (out method) ---
			return false;
		}
		
		L7vsAdmController admController = new L7vsAdmController();
		L7vsConfController confController = new L7vsConfController();
		Iterator i = h.iterator();
		while (i.hasNext()) {
			SaveCategoryIdSet s = (SaveCategoryIdSet) i.next();
			switch (s) {
			case LOG_ID:
				L7vsAdmData admData = admController.getData();
				try {
					L7vsConfData cf = confController.getConfData();
					if (admData.l7vsdLogLevel.equals(cf.logger.lbLog.logLevel)
							&& admData.snmpLogLevel
									.equals(cf.logger.snmpLog.logLevel)) {
						saveLogger.info("21030 All log level settings have not changed.");
						// --- debug log (out method) ---
						if (saveLogger.isDebugEnabled()) {
				            saveLogger.debug("11141 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
						}
						// --- debug log (out method) ---
						return true;
					}
					cf.logger.lbLog.logLevel = admData.l7vsdLogLevel;
					cf.logger.snmpLog.logLevel = admData.snmpLogLevel;
					
					confController.setConfData(cf);
				} catch (FileNotReadException e) {
					ioFileLogger.error("41039 Exception occured: " + e.getMessage());
					// --- debug log (out method) ---
					if (saveLogger.isDebugEnabled()) {
			            saveLogger.debug("11142 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
					}
					// --- debug log (out method) ---
					return false;
				} catch (FileNotWriteException e) {
					ioFileLogger.error("41040 Exception occured: " + e.getMessage());
					// --- debug log (out method) ---
					if (saveLogger.isDebugEnabled()) {
			            saveLogger.debug("11143 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
					}
					// --- debug log (out method) ---
					return false;
				} catch (Exception e) {
					ioFileLogger.error("41041 Exception occured: " + e.getMessage());
					// --- debug log (out method) ---
					if (saveLogger.isDebugEnabled()) {
			            saveLogger.debug("11144 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
					}
					// --- debug log (out method) ---
					return false;
				}
				break;
			case VS_ID:
				admData = admController.getData();
				DirectorController directord = new DirectorController();
				try {
					DirectorData directorData = directord.getDirectorData();
					boolean sameFlag = true;
					int admVsCount = admData.virtualSettings.size();
					Vector<VirtualSetting> delVsList = new Vector<VirtualSetting>();
					for (VirtualSetting directorVs : directorData.virtualSettings) {
						boolean found = false;
						for (VirtualSetting admVs : admData.virtualSettings) {
							if (admVs.isSameService(directorVs)) {
								if (! admVs.equalsAdm(directorVs)) {
									directorVs.sched = admVs.sched;
									directorVs.qosclient = admVs.qosclient;
									directorVs.qosservice = admVs.qosservice;
									directorVs.maxconn = admVs.maxconn;
									directorVs.sorryserver = admVs.sorryserver;
									directorVs.real = admVs.real;
									sameFlag = false;
								}
								admData.virtualSettings.remove(admVs);
								found = true;
								break;
							}
						}
						if (found == false)
							delVsList.add(directorVs);
					}
					for (VirtualSetting admVs : admData.virtualSettings) {
						directorData.virtualSettings.add(admVs);
						sameFlag = false;
					}
					for (VirtualSetting delVs : delVsList) {
						directorData.virtualSettings.remove(delVs);
						sameFlag = false;
					}
					if (sameFlag) {
						saveLogger.info("21031 All Virtual Service setting have not changed.");
						// --- debug log (out method) ---
						if (saveLogger.isDebugEnabled()) {
				            saveLogger.debug("11145 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
						}
						// --- debug log (out method) ---
						return true;
					}
					directord.setDirectorData(directorData);
				} catch (FileNotReadException e) {
					ioFileLogger.error("41042 Exception occured: " + e.getMessage());
					// --- debug log (out method) ---
					if (saveLogger.isDebugEnabled()) {
			            saveLogger.debug("11146 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
					}
					// --- debug log (out method) ---
					return false;
				} catch (FileNotWriteException e) {
					ioFileLogger.error("41043 Exception occured: " + e.getMessage());
					// --- debug log (out method) ---
					if (saveLogger.isDebugEnabled()) {
			            saveLogger.debug("11147 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
					}
					// --- debug log (out method) ---
					return false;
				} catch (Exception e) {
					ioFileLogger.error("41044 Exception occured: " + e.getMessage());
					// --- debug log (out method) ---
					if (saveLogger.isDebugEnabled()) {
			            saveLogger.debug("11148 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
					}
					// --- debug log (out method) ---
					return false;
				}
				break;
			default:
				saveLogger.warn("31002 Unknown save category was specified.");
				// --- debug log (out method) ---
				if (saveLogger.isDebugEnabled()) {
		            saveLogger.debug("11149 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=false");
				}
				// --- debug log (out method) ---
				return false;
			}
		}
		
		// --- debug log (out method) ---
		if (saveLogger.isDebugEnabled()) {
            saveLogger.debug("11150 L7guiFramework::saveExecute(HashSet<SaveCategoryIdSet> h) out return=true");
		}
		// --- debug log (out method) ---
		return true;
	}

	/**
	 * 
	 * <p>
	 * syncExecute method
	 * </p>
	 * 
	 * @param h
	 * @return result of sync file.
	 */
	public boolean syncExecute(HashSet<Integer> h) {
		// --- debug log (out method) ---
		if (syncLogger.isDebugEnabled()) {
            syncLogger.debug("11151 L7guiFramework::syncExecute(HashSet<Integer> h) in h=(" + h + ")");
		}
		// --- debug log (out method) ---

		L7SyncManager sync = new L7SyncManager();
		try {
			sync.syncAct(h);
		} catch (MessageView mv) {
			syncLogger.error("41045 Exception occured: " + mv.getMessage());
			// --- debug log (out method) ---
			if (syncLogger.isDebugEnabled()) {
	            syncLogger.debug("11152 L7guiFramework::syncExecute(HashSet<Integer> h) out return=false");
			}
			// --- debug log (out method) ---
			return false;
		}

		// --- debug log (out method) ---
		if (syncLogger.isDebugEnabled()) {
            syncLogger.debug("11153 L7guiFramework::syncExecute(HashSet<Integer> h) out return=true");
		}
		// --- debug log (out method) ---
		return true;
	}

	/**
	 * 
	 * <p>
	 * getL7vsdProcessStatus method
	 * </p>
	 * 
	 * @return existence of l7vsd process.
	 */
	public boolean getL7vsdProcessStatus() {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11154 L7guiFramework::getL7vsdProcessStatus() in");
		}
		// --- debug log (in method) ---

		boolean ret = processCheck("l7vsd");
		
		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11155 L7guiFramework::getL7vsdProcessStatus() out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * getL7directordProcessStatus method
	 * </p>
	 * 
	 * @return existence of l7directord process.
	 */
	public boolean getL7directordProcessStatus() {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11156 L7guiFramework::getL7directordProcessStatus() in");
		}
		// --- debug log (in method) ---

		boolean ret = processCheck("l7directord");
		
		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11157 L7guiFramework::getL7directordProcessStatus() out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * setLogLevel method
	 * </p>
	 * 
	 * @param category
	 * @return result of l7vsadm set log level command.
	 */
	public boolean setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category) {
		// --- debug log (in method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11158 L7guiFramework::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category in category=(" + category + ")");
		}
		// --- debug log (in method) ---

		L7vsAdmController admController = new L7vsAdmController();
		boolean ret = admController.setLogLevel(category);
		
		// --- debug log (out method) ---
		if (ioCommandLogger.isDebugEnabled()) {
            ioCommandLogger.debug("11159 L7guiFramework::setLogLevel(HashMap<LogData.LogCategory, LogData.LogLevel> category out return=" + ret);
		}
		// --- debug log (out method) ---
		return ret;
	}

	/**
	 * 
	 * <p>
	 * processCheck method
	 * </p>
	 * 
	 * @param regex
	 * @return existence of regex process.
	 */
	protected boolean processCheck(String regex) {
		// --- debug log (in method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11160 L7guiFramework::processCheck(String regex) in regex=\"" + regex + "\"");
		}
		// --- debug log (in method) ---
		
		boolean found = false;
		String proc = "/proc";
		Pattern p = Pattern.compile("^\\d+$");
		Pattern l7vsd = Pattern.compile("^Name:\t" + regex + "$");

		try {
			File dir = new File(proc);
			File[] procdir = dir.listFiles();
			for (int i = 0; i < procdir.length; i++) {
				Matcher m = p.matcher(procdir[i].getName());
				if (m.matches()) {
					BufferedReader f = null;
					try {
						f = new BufferedReader(new FileReader("/proc/"
								+ procdir[i].getName() + "/status"));
						String line;
						while ((line = f.readLine()) != null) {
							Matcher mt = l7vsd.matcher(line);
							if (mt.matches()) {
								found = true;
								break;
							}
						}
					} catch (FileNotFoundException e) {
						ioFileLogger.error("41046 Exception occured: " + e.getMessage());
					} catch (IOException e) {
						ioFileLogger.error("41047 Exception occured: " + e.getMessage());
					} finally {
						try {
							if (f != null) {
								f.close();
							}
						} catch (Exception e) {
							ioFileLogger.error("41048 Exception occured: " + e.getMessage());
						}
					}
				}
			}
		} catch (Exception e) {
			ioFileLogger.error("41049 Exception occured: " + e.getMessage());
		}

		// --- debug log (out method) ---
		if (ioFileLogger.isDebugEnabled()) {
            ioFileLogger.debug("11161 L7guiFramework::processCheck(String regex) out return=" + found);
		}
		// --- debug log (out method) ---
		return found;
	}
}
