/*
 * @file  protomod_sslid.c
 * @brief protocol module of HTTPS(SSL/TLS).
 * @brief this module provide session persistence by SSL session ID.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 * Copyright (C) 2009  Shinya TAKEBAYASHI
 * Copyright (C) 2009  NTT Resonant Inc. O.Nakayama, T.Motoda.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define	__STDC_LIMIT_MACROS
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <getopt.h>
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_lsock.h"
#include "l7vs_dest.h"
#include "l7vs_replication.h"
#include "l7vs_module.h"
#include "module_http.h"
#include "module_sslid_hash.h"
#include "protomod_sslid.h"

static void  fini(void);
static int   create(void*, handle_t);
static void* create_sa(struct l7vs_service_arg*);
static int   compare(handle_t, handle_t);
static int   select_dest(struct l7vs_service*, struct l7vs_conn*,
                        char*, size_t*, struct l7vs_dest**);
static int   analyze_cldata(struct l7vs_service*, struct l7vs_conn*,
                        char*, size_t*);
static int   analyze_rsdata(struct l7vs_service*, struct l7vs_conn*,
                        char*, size_t*);
static int   destroy(handle_t);
static void  destroy_sa(void**);
static int   service_arg(struct l7vs_service_arg_multi*, handle_t);
static int   parse(void*, int, char**);

static struct l7vs_sslid_service* l7vs_protomod_sslid_search_service(handle_t);
static struct l7vs_sslid_service* l7vs_protomod_sslid_create_service();
static struct l7vs_sslid_service* l7vs_protomod_sslid_create_temp_service();

static void l7vs_protomod_sslid_read_replication_data(struct l7vs_sslid_service*, struct l7vs_service*);
static void l7vs_protomod_sslid_write_replication_data(struct l7vs_sslid_service*);

static void l7vs_sslid_service_c_str(char*, struct l7vs_sslid_service*);
static void l7vs_sslid_service_arg_c_str(char*, struct l7vs_sslid_service_arg*);
static void ssl_session_c_str(char*, struct ssl_session*);
static void replication_header_c_str(char*, struct replication_header*);
static void id_c_str(char*, char*, int);    /* add session id length param 2009.4.8 T.Motoda@NTTR */

struct l7vs_sslid_service* sslid_service_list[SSLID_SERVICE_NUMBER];

static struct l7vs_protomod sslid_protomod = {
    NULL,           /* handle */
    "sslid",        /* modname */
    0,              /* refcnt */
    0,              /* fast schedule */
    create,         /* create function */
    compare,        /* compare function */
    select_dest,    /* select_dest function */
    analyze_cldata, /* analyze_cldata function */
    analyze_rsdata, /* analyze_rsdata function */
    destroy,        /* destroy function */
    fini,           /* fini function */
    create_sa,      /* create_sa function */
    service_arg,    /* service_arg function */
    parse,          /* parse function */
    destroy_sa,     /* destroy_sa function */
    NULL,           /* initialize function */
    NULL,           /* finalize function */
    NULL,           /* get_log_level function */
    NULL,           /* put_log_debug function */
    NULL,           /* put_log_info function */
    NULL,           /* put_log_warn function */
    NULL,           /* put_log_error function */
    NULL,           /* put_log_fatal function */
    NULL            /* replication_pay_memory function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param[in] handle dlopen's handle
 * @return l7vs_protomod struct
 */
extern "C" struct l7vs_protomod*
init(void* handle)
{
	struct l7vs_protomod* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,92,
	            "in_function: struct l7vs_protomod* init(void* handle): handle=%p", handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (handle == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,79, "Arg(handle) is NULL pointer.");
		goto init_out;
	}

	/* initialize sslid service list */
	memset(sslid_service_list, 0, sizeof(struct l7vs_sslid_service*) * SSLID_SERVICE_NUMBER);
	/* set dlopen's handle */
	sslid_protomod.handle = handle;

	return_value = &sslid_protomod;

init_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char protomod_str[DEBUG_STR_LEN] = {0};
		l7vs_protomod_c_str(protomod_str, &sslid_protomod);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,93,
		    "out_function: struct l7vs_protomod* init(void* handle): return=&(%s)", protomod_str);
	}
	/*------ DEBUG LOG END ------*/
	return return_value;
}

/*!
 * Protocol module finalize function. free all sslid service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
	/* sslid service list counter */
	int service_number = 0;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,94, "in_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/

	/* check all sslid service list */
	for (service_number = 0; service_number < SSLID_SERVICE_NUMBER; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (sslid_service_list[service_number] != NULL) {
			/* free and points NULL */
			if (sslid_service_list[service_number]->session != NULL) {

				/*-------- DEBUG LOG --------*/
				if (sslid_protomod.get_log_level != NULL &&
				    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
					PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,25, "free: %p",
					    sslid_service_list[service_number]->session);
				}
				/*------ DEBUG LOG END ------*/

				free(sslid_service_list[service_number]->session);
				sslid_service_list[service_number]->session = NULL;
				hash_setPointer(sslid_service_list[service_number]->hash_map, sslid_service_list[service_number]->hash_list, sslid_service_list[service_number]->maxlist);
				hash_destroy();
			}

			/*-------- DEBUG LOG --------*/
			if (sslid_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,26, "free: %p",
				    sslid_service_list[service_number]);
			}
			/*------ DEBUG LOG END ------*/

			free(sslid_service_list[service_number]);
			sslid_service_list[service_number] = NULL;
		}
	}

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,95, "out_function: void fini(void)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create sslid service struct.
 * @param[in] sslid_arg      sslid service argument struct
 * @param[in] service_handle a unique service ID
 * @retval 0  successfully create sslid service.
 * @retval -1 some errors occur.
 */
static int
create(void* sslid_arg, handle_t service_handle)
{
	struct l7vs_sslid_service* sslid_service;
	struct l7vs_sslid_service_arg* sslid_service_arg;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_arg_c_str(sslid_arg_str, (struct l7vs_sslid_service_arg*) sslid_arg);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,96,
		    "in_function: int create(void* sslid_arg, handle_t service_handle):sslid_arg=&(%s), "
		    "service_handle=%d", sslid_arg_str, service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sslid_arg == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,80, "Arg(sslid_arg) is NULL pointer.");
		return_value = -1;
		goto create_out;
	}

	if (service_handle != TEMP_SERVICEHANDLE) {
		/* search empty sslid service list and create sslid service */
		sslid_service = l7vs_protomod_sslid_create_service();
	}
	else {
		/* create temporary sslid service */
		sslid_service = l7vs_protomod_sslid_create_temp_service();
	}

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(sslid_str, sslid_service);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,97, "pointer assign: sslid_service=&(%s)",
		    sslid_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sslid_service == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,81, "Could not make sslid service.");
		return_value = -1;
		goto create_out;
	}

	sslid_service_arg = (struct l7vs_sslid_service_arg*) sslid_arg;

	/* set service handle */
	sslid_service->service_handle = service_handle;

	/* set option value */
	sslid_service->timeout = sslid_service_arg->timeout;
	sslid_service->maxlist = sslid_service_arg->maxlist;
	sslid_service->replication_addr = NULL;
	sslid_service->reschedule = sslid_service_arg->reschedule;

	if (service_handle != TEMP_SERVICEHANDLE) {
		/* create session area */
		sslid_service->session = (struct ssl_session*) calloc(sizeof(struct ssl_session), sslid_service->maxlist);
		hash_allocate(sslid_service_arg->maxlist);
		sslid_service->hash_map = hash_getIDMAP();
		sslid_service->hash_list = hash_getIDLIST();

		/*-------- DEBUG LOG --------*/
		if (sslid_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
			PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,27, "calloc: addr=%p, size=%ld",
			    sslid_service->session, (unsigned long int) sizeof(struct ssl_session) * sslid_service->maxlist);
		}
		/*------ DEBUG LOG END ------*/

		if (sslid_service->session == NULL) {
			PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,22, "Could not allocate memory.");
			if (destroy(service_handle)) {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,23,
				    "Could not destroy SSL service. (handle=%d)", service_handle);
			}
			return_value = -1;
			goto create_out;
		}
	}

create_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,98,
		    "out_function: int create(void* sslid_arg, handle_t service_handle):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create sslid service argument struct.
 * @param[out] srv_arg service argument struct
 * @return sslid service argument struct
 */
static void*
create_sa(struct l7vs_service_arg* srv_arg)
{
	struct l7vs_sslid_service_arg* sslid_service_arg;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char service_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_service_arg_c_str(service_arg_str, srv_arg);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,99,
		    "in_function: void* create_sa(struct l7vs_service_arg* srv_arg):srv_arg=&(%s)",
		    service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,82, "Arg(srv_arg) is NULL pointer.");
		sslid_service_arg = NULL;
		goto create_sa_out;
	}

	/* create sslid service argument struct */
	sslid_service_arg = (struct l7vs_sslid_service_arg*) calloc(1, sizeof(struct l7vs_sslid_service_arg));

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,28, "calloc: addr=%p, size=%ld",
		    sslid_service_arg, (unsigned long int) sizeof(struct l7vs_sslid_service_arg));
	}
	/*------ DEBUG LOG END ------*/

	if (sslid_service_arg == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,24, "Could not allocate memory.");
		goto create_sa_out;
	}

	/* set sslid service argument size and protomod name "sslid" */
	srv_arg->len = sizeof(struct l7vs_sslid_service_arg);
	strcpy(srv_arg->protomod, sslid_protomod.modname);

create_sa_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_service_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_arg_c_str(sslid_service_arg_str, sslid_service_arg);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,100,
		    "out_function: void* create_sa(struct l7vs_service_arg* srv_arg):return_value=&(%s)",
		    sslid_service_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	return (void*) sslid_service_arg;
}

/*!
 * Always match 'cause SSL-Session-ID module does not have module key.
 * @param[in] srv_handle1 one of a unique service ID
 * @param[in] srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(handle_t srv_handle1, handle_t srv_handle2)
{
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,101,
		    "in_function: int compare(handle_t srv_handle1, handle_t srv_handle2):"
		    "srv_handle1=%u, srv_handle2=%u", srv_handle1, srv_handle2);
	}
	/*------ DEBUG LOG END ------*/

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,102,
		    "out_function: int compare(handle_t srv_handle1, handle_t srv_handle2):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Check and modify request packet.
 * @param[in]  srv     service struct include service handle, protocol module and schedule module.
 * @param[in]  conn    connection data.
 * @param[in]  request packet data from client
 * @param[in]  len     length of packet data
 * @param[out] dest    destination (real server) list
 * @param[out] tcps    TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
select_dest(struct l7vs_service* srv, struct l7vs_conn* conn,
      char* request, size_t* len, struct l7vs_dest** dest)
{
	struct l7vs_sslid_service* sslid_service;
	int i;
	int ret;
	int return_value = 0;
	time_t now;
	char id_str[DEBUG_STR_LEN] = {0};
	struct l7vs_dest *tmpdest = NULL;
	int searchret = -1;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN] = {0};
		char conn_str[DEBUG_STR_LEN] = {0};
		char dest_str[DEBUG_STR_LEN] = {0};
		char len_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (dest != NULL) {
			l7vs_dest_c_str(dest_str, *dest);
		}
		else {
			strncpy(dest_str, "NULL", DEBUG_STR_LEN);
		}
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,103,
		    "in_function: int select_dest(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest):srv=&(%s), conn=&(%s), "
		    "request=\"%s\", len=&(%s), dest=&(&(%s))",
		    srv_str, conn_str, request, len_str, dest_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,83, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto select_dest_out;
	}
	if (srv->pm == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,84, "Arg(srv->pm) is NULL pointer.");
		return_value = -1;
		goto select_dest_out;
	}
	if (request == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,85, "Arg(request) is NULL pointer.");
		return_value = -1;
		goto select_dest_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,86, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto select_dest_out;
	}
	if (dest == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,87, "Arg(dest) is NULL pointer.");
		return_value = -1;
		goto select_dest_out;
	}

	/* search service that has such a service ID */
	sslid_service = l7vs_protomod_sslid_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(sslid_str, sslid_service);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,104, "pointer assign: sslid_service=&(%s)",
		    sslid_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sslid_service == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,89, "Could not find such service handle's sslid service.");
		return_value = -1;
		goto select_dest_out;
	}
	if (sslid_service->session == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,90,
		    "Service has NULL pointer session.");
		return_value = -1;
		goto select_dest_out;
	}

	/* read replication data */
	if (sslid_service->replication_addr) {
		l7vs_protomod_sslid_read_replication_data(sslid_service, srv);
		hash_rebuild_sessionlist(sslid_service);
	}

	/* initialize protocol module ... clear destination list */
	ret = srv->pm->initialize(srv, conn, request, *len, dest);
	if (ret != 0){
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,91, "Could not initialize protomod.");
		return_value = -1;
		goto select_dest_out;
	}

	/* check payload */
	if (
	    *len > 44 &&	// Check if minimum length	2009.4.8 O.Nakayama@NTTR and T.Motoda@NTTR
	    (
             (request[1] == 0x03 && request[2] == 0x00 && request[9] == 0x03 && request [10] == 0x00) || // SSL v3
             (request[1] == 0x03 && request[2] == 0x01 && request[9] == 0x03 && request [10] == 0x01)    // TLS v1
	    ) &&
	    request[5] == 0x01 && // Client Hello
	    (request[43] >= 1 && request[43] <= SSLID_LENGTH && *len > (43 + request[43]))	// Session ID Length (variable length from 1 to SSLID_LENGTH)	2009.4.8 O.Nakayama and T.Motoda@NTTR
	   ) {
		/*-------- DEBUG LOG --------*/
		if (sslid_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
			char id_str[DEBUG_STR_LEN] = {0};
			id_c_str(id_str, &request[44], request[43]);	// Add length parameter 2009.4.8 T.Motoda@NTTR
			PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,105,
			    "Client Hello/SessionID=%s", id_str);
		}
		/*------ DEBUG LOG END ------*/

		hash_setPointer(sslid_service->hash_map, sslid_service->hash_list, sslid_service->maxlist);
		id_c_str(id_str, &request[44], request[43]);	// Add length parameter 2009.4.8 T.Motoda@NTTR
		searchret = hash_search(id_str, &tmpdest);
		
		if (searchret == 0) {
			*dest = tmpdest;
		}
		
	}

	/* finalize */
	ret = srv->pm->finalize(srv, conn, request, *len, dest, sslid_service->reschedule);
	if (ret != 0){
		PUT_LOG_INFO(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,11, "Could not finalize protomod. (Realserver decision failure)");
		return_value = -1;
		goto select_dest_out;
	}

select_dest_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,107,
		    "out_function: int select_dest(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len, struct l7vs_dest** dest, int* tcps):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Do nothing
 * @param[in]  srv     service struct include service handle, protocol module and schedule module.
 * @param[in]  conn    connection data.
 * @param[in]  request packet data from client
 * @param[in]  len     length of packet data
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
analyze_cldata(struct l7vs_service* srv, struct l7vs_conn* conn,
      char* request, size_t* len)
{
	struct l7vs_sslid_service* sslid_service;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN] = {0};
		char conn_str[DEBUG_STR_LEN] = {0};
		char dest_str[DEBUG_STR_LEN] = {0};
		char len_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,325,
		    "in_function: int analyze_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len):srv=&(%s), conn=&(%s), "
		    "request=\"%s\", len=&(%s)",
		    srv_str, conn_str, request, len_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,310, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto analyze_cldata_out;
	}
	if (srv->pm == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,311, "Arg(srv->pm) is NULL pointer.");
		return_value = -1;
		goto analyze_cldata_out;
	}
	if (request == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,312, "Arg(request) is NULL pointer.");
		return_value = -1;
		goto analyze_cldata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,313, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto analyze_cldata_out;
	}

	/* search service that has such a service ID */
	sslid_service = l7vs_protomod_sslid_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(sslid_str, sslid_service);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,326, "pointer assign: sslid_service=&(%s)",
		    sslid_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sslid_service == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,314, "Could not find such service handle's sslid service.");
		return_value = -1;
		goto analyze_cldata_out;
	}
	if (sslid_service->session == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,315,
		    "Service has NULL pointer session.");
		return_value = -1;
		goto analyze_cldata_out;
	}

analyze_cldata_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,327,
		    "out_function: int analyze_cldata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* request, size_t* len):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Check and modify response packet.
 * @param[in] srv      service struct include service handle, protocol module and schedule module.
 * @param[in] conn     connection data.
 * @param[in] response packet data from real server
 * @param[in] len      length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn,
	char* response, size_t* len)
{
	struct l7vs_sslid_service* sslid_service;
	int exist = 0;
	int return_value = 0;
	int i;
	int pick;
	time_t now, oldest = 0;
	char session_full = 1;
	char id_str[DEBUG_STR_LEN] = {0};

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_str[DEBUG_STR_LEN] = {0};
		char conn_str[DEBUG_STR_LEN] = {0};
		char len_str[DEBUG_STR_LEN] = {0};
		l7vs_service_c_str(srv_str, srv);
		l7vs_conn_c_str(conn_str, conn);
		if (len != NULL) {
			snprintf(len_str, DEBUG_STR_LEN, "%lu", (unsigned long int) *len);
		}
		else {
			strncpy(len_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,108,
		    "in_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):srv=&(%s), conn=&(%s), response=\"%s\", len=&(%s)",
		    srv_str, conn_str, response, len_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,93, "Arg(srv) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,94, "Arg(conn) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (conn->dest == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,95, "Arg(conn->dest) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (response == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,96, "Arg(response) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (len == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,97, "Arg(len) is NULL pointer.");
		return_value = -1;
		goto analyze_rsdata_out;
	}

	/* sorry flag check */
	if (conn->sorry_conn_flag == 1) {
		/*-------- DEBUG LOG --------*/
		if (sslid_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
			PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,109, "Response from sorry server.");
		}
		/*------ DEBUG LOG END ------*/

		goto analyze_rsdata_out;
	}

	/* search service that has such a service ID */
	sslid_service = l7vs_protomod_sslid_search_service(srv->handle);

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(sslid_str, sslid_service);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,110, "pointer assign: sslid_service=&(%s)",
		    sslid_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sslid_service == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,98,
		    "Could not find such service handle's sslid service.");
		return_value = -1;
		goto analyze_rsdata_out;
	}
	if (sslid_service->session == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,99,
		    "Service has NULL pointer session.");
		return_value = -1;
		goto analyze_rsdata_out;
	}

	/* check payload */
	if (
	    *len > 44 &&	// Check if minimum length	2009.4.8 O.Nakayama and T.Motoda@NTTR
	    (
             (response[1] == 0x03 && response[2] == 0x00 && response[9] == 0x03 && response [10] == 0x00) || // SSL v3
             (response[1] == 0x03 && response[2] == 0x01 && response[9] == 0x03 && response [10] == 0x01)    // TLS v1
	    ) &&
	    response[5]  == 0x02 && // Server Hello
	    (response[43] >= 1 && response[43] <= SSLID_LENGTH && *len > (43 + response[43]))	// Session ID Length (variable length from 1 to SSLID_LENGTH)	2009.4.8 O.Nakayama and T.Motoda@NTTR
	   ) {
		/*-------- DEBUG LOG --------*/
		if (sslid_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
			char id_str[DEBUG_STR_LEN] = {0};
			id_c_str(id_str, &response[44], response[43]);	// Add length parameter 2009.4.8 T.Motoda@NTTR
			PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,111,
			    "Server Hello/SessionID=%s", id_str);
		}
		/*------ DEBUG LOG END ------*/

		hash_setPointer(sslid_service->hash_map, sslid_service->hash_list, sslid_service->maxlist);
		id_c_str(id_str, &response[44], response[43]);	// Add length parameter 2009.4.8 T.Motoda@NTTR
		hash_add(id_str, *conn->dest);
		if (sslid_service->replication_addr) {
			hash_construct_sessionlist(sslid_service);
			l7vs_protomod_sslid_write_replication_data(sslid_service);
		}
	}

analyze_rsdata_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,114,
		    "out_function: int analyze_rsdata(struct l7vs_service* srv, struct l7vs_conn* conn, "
		    "char* response, size_t* len):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy sslid service
 * @param[in] srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(handle_t srv_handle)
{
	/* sslid service list counter */
	int service_number = 0;
	int free_flag = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,115,
		    "in_function: int destroy(handle_t srv_handle):srv_handle=%u",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all sslid service list */
	for (service_number = 0; service_number < SSLID_SERVICE_NUMBER; ++service_number) {
		/* found sslid service that has srv_handle */
		if (sslid_service_list[service_number] != NULL && 
		    sslid_service_list[service_number]->service_handle == srv_handle) {

			/* free and NULL */
			if (sslid_service_list[service_number]->session != NULL) {

				/*-------- DEBUG LOG --------*/
				if (sslid_protomod.get_log_level != NULL &&
				    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
					PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,29, "free: %p",
					    sslid_service_list[service_number]->session);
				}
				/*------ DEBUG LOG END ------*/

				free(sslid_service_list[service_number]->session);
				sslid_service_list[service_number]->session = NULL;
				hash_setPointer(sslid_service_list[service_number]->hash_map, sslid_service_list[service_number]->hash_list, sslid_service_list[service_number]->maxlist);
				hash_destroy();
			}

			/*-------- DEBUG LOG --------*/
			if (sslid_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,30, "free: %p",
				    sslid_service_list[service_number]);
			}
			/*------ DEBUG LOG END ------*/

			free(sslid_service_list[service_number]);
			sslid_service_list[service_number] = NULL;

			free_flag = 1;
			break;
		}
	}
	
	/* sslid service was not found */
	if (free_flag == 0) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,100, "Could not find such service handle's sslid service.");
		return_value = -1;
		goto destroy_out;
	}

destroy_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,116,
		    "out_function: int destroy(handle_t srv_handle):return_value=%d",
		    srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Destroy sslid service argument
 * @param[in] sslid_arg sslid service argument
 * @return void
 */
static void
destroy_sa(void** sslid_arg)
{
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_arg_str[DEBUG_STR_LEN] = {0};
		if (sslid_arg != NULL) {
			l7vs_sslid_service_arg_c_str(sslid_arg_str, (struct l7vs_sslid_service_arg*) *sslid_arg);
		}
		else {
			strncpy(sslid_arg_str, "NULL", DEBUG_STR_LEN);
		}
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,117,
		    "in_function: void destroy_sa(void** sslid_arg):sslid_arg=&(&(%s))",
		    sslid_arg_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sslid_arg == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,101, "Arg(sslid_arg) is NULL pointer.");
	}
	else if (*sslid_arg == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,102, "Arg(*sslid_arg) is NULL pointer.");
	}
	else {
		/*-------- DEBUG LOG --------*/
		if (sslid_protomod.get_log_level != NULL &&
		    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
			PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,31, "free: %p",
			    *sslid_arg);
		}
		/*------ DEBUG LOG END ------*/

		/* free and NULL */
		free((struct l7vs_sslid_service_arg*) *sslid_arg);
		*sslid_arg = NULL;
	}

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,118,
		    "out_function: void destroy_sa(void** sslid_arg)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create strings for service list of l7vsadm
 * @param[out] srv_arg_mt service argument struct
 * @param[in]  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi* srv_arg_mt, handle_t srv_handle)
{
	struct l7vs_sslid_service* sslid_service;
	struct l7vs_sslid_service_arg s_sarg;
	char sslid_argument[SERVICE_ARG_MAXSIZE];
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char srv_arg_mt_str[DEBUG_STR_LEN] = {0};
		l7vs_service_arg_multi_c_str(srv_arg_mt_str, srv_arg_mt);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,119,
		    "in_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):srv_arg_mt=&(%s), srv_handle=%u",
		    srv_arg_mt_str, srv_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (srv_arg_mt == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,103, "Arg(srv_arg_mt) is NULL pointer.");
		return_value = -1;
		goto service_arg_out;
	}

	/* search service that has such a service ID */
	sslid_service = l7vs_protomod_sslid_search_service(srv_handle);

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(sslid_str, sslid_service);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,120, "pointer assign: sslid_service=&(%s)",
		    sslid_str);
	}
	/*------ DEBUG LOG END ------*/

	if (sslid_service == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,104, "Could not find such service handle's sslid service.");
		return_value = -1;
		goto service_arg_out;
	}

	/* initialize argument strings */
	memset(sslid_argument, 0, SERVICE_ARG_MAXSIZE);

	/* set sslid args to service argument struct */
	srv_arg_mt->srv_arg.reschedule = sslid_service->reschedule;

	/* create long argument (l7vsadm option -l) */
	strncpy(srv_arg_mt->srv_arg.protomod_key_string, sslid_argument, 256);

	/* create verbose argument (l7vsadm option -V) */
	snprintf(sslid_argument + strlen(sslid_argument), SERVICE_ARG_MAXSIZE - strlen(sslid_argument),
	    "--timeout %d --maxlist %d", sslid_service->timeout, sslid_service->maxlist);
	strncpy(srv_arg_mt->srv_arg.protomod_opt_string, sslid_argument, 512);

	/* set option value */
	s_sarg.timeout = sslid_service->timeout;
	s_sarg.maxlist = sslid_service->maxlist;
	s_sarg.reschedule = sslid_service->reschedule;

	memcpy(srv_arg_mt->protomod_arg, &s_sarg, sizeof(struct l7vs_sslid_service_arg));

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char sslid_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_arg_c_str(sslid_arg_str, &s_sarg);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,121,
		    "pointer assign: srv_arg_mt->protomod_arg=&(%s)", sslid_arg_str);
	}
	/*------ DEBUG LOG END ------*/

service_arg_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,122,
		    "out_function: int service_arg(struct l7vs_service_arg_multi* srv_arg_mt, "
		    "handle_t srv_handle):return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Parse l7vsadm options to sslid argument
 * @param[out] sslid_arg sslid service argument struct
 * @param[in]  argc      number of l7vsadm argument
 * @param[in]  argv      l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void* sslid_arg, int argc, char* argv[])
{
	struct l7vs_sslid_service_arg* sslid_service_arg;
	static struct option opt[] = {
		{"timeout",    required_argument, NULL, 'T'},
		{"maxlist",        required_argument, NULL, 'M'},
		{"reschedule",             no_argument,       NULL, 'R'},
		{"no-reschedule",          no_argument,       NULL, 'N'},
		{NULL,                     0,                 NULL, 0  }
	};
	int c;
	unsigned long buffer;
	int timeout_flag = 0;
	int maxlist_flag = 0;
	int reschedule_flag = 0;
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		int i;
		char argv_str[DEBUG_STR_LEN] = {0};
		char sslid_arg_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_arg_c_str(sslid_arg_str, (struct l7vs_sslid_service_arg*) sslid_arg);
		argv_str[0] = '\0';
		if (argv == NULL)
			snprintf(argv_str, DEBUG_STR_LEN, "NULL");
		else {
			for (i = 0; i < argc; i++) {
				snprintf(argv_str, DEBUG_STR_LEN, "%sargv[%d]=\"%s\", ", argv_str, i, argv[i]);
			}
			i = strnlen(argv_str, DEBUG_STR_LEN);
			if (i > 1)
				argv_str[i - 2] = '\0';
		}
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,123,
		    "in_function: int parse(void* sslid_arg, int argc, char* argv[]):sslid_arg=&(%s), "
		    "argc=%d, %s", sslid_arg_str, argc, argv_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sslid_arg == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,105, "Arg(sslid_arg) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}
	if (argv == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,106, "Arg(argv) is NULL pointer.");
		return_value = -1;
		goto parse_out;
	}

	sslid_service_arg = (struct l7vs_sslid_service_arg*) sslid_arg;
	optind = 0;

	/* check all argument */
	while ((c = getopt_long(argc, argv, "T:M:RN", opt, NULL)) != -1) {
		switch (c) {
		/* --timeout / -T */
		case 'T':
			if (sscanf(optarg, "%lu", &buffer) == 0) {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,107,
				    "-T/--timeout option value '%s' is invalid.", optarg);
				return_value = -1;
				goto parse_out;
			}
			if (buffer > INT_MAX) {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,108,
				    "-T/--timeout option value '%s' is too large.", optarg);
				return_value = -1;
				goto parse_out;
			}
			else
				sslid_service_arg->timeout = buffer;
			timeout_flag++;
			break;

		/* --maxlist / -M */
		case 'M':
			if (sscanf(optarg, "%lu", &buffer) == 0) {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,109,
				    "-M/--maxlist option value '%s' is invalid.", optarg);
				return_value = -1;
				goto parse_out;
			}
			if (buffer > INT_MAX) {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,110,
				    "-M/--maxlist option value '%s' is too large.", optarg);
				return_value = -1;
				goto parse_out;
			}
			else
				sslid_service_arg->maxlist = buffer;
			maxlist_flag++;
			break;

		/* --reschedule / -R */
		case 'R':
			/* reschedule on */
			sslid_service_arg->reschedule = 1;
			reschedule_flag++;
			break;

		/* --no-reschedule / -N */
		case 'N':
			/* reschedule off */
			sslid_service_arg->reschedule = 0;
			reschedule_flag++;
			break;

		/* else error */
		default:
			PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,111, "Option error.");
			return_value = -1;
			goto parse_out;
		}
	}

	/* when set both -R and -N at the same time */
	if (reschedule_flag > 1) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,112,
		    "You should choose either reschdule or no-reschedule.");
		return_value = -1;
		goto parse_out;
	}
	/* same option */
	if (timeout_flag > 1) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,113,
		    "Cannot set multiple option '--timeout/-T'.");
		return_value = -1;
		goto parse_out;
	}
	if (maxlist_flag > 1) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,114,
		    "Cannot set multiple option '--maxlist/-M'.");
		return_value = -1;
		goto parse_out;
	}

	/* set default no reschedule */
	if (reschedule_flag == 0) {
		sslid_service_arg->reschedule = 0;
	}

	/* set default options */
	if (timeout_flag == 0) {
		sslid_service_arg->timeout = 3600;
	}
	if (maxlist_flag == 0 || sslid_service_arg->maxlist == 0) {
		sslid_service_arg->maxlist = 1024;
	}

parse_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,124,
		    "out_function: int parse(void* sslid_arg, int argc, char* argv[]):return_value=%d",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Search sslid service from sslid service list using service handle
 * @param[in] service_handle a unique service ID
 * @return sslid service struct when service was found. NULL when service was not found.
 */
static struct l7vs_sslid_service*
l7vs_protomod_sslid_search_service(handle_t service_handle)
{
	/* sslid service list counter */
	int service_number = 0;
	struct l7vs_sslid_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,125,
		    "in_function: struct l7vs_sslid_service* l7vs_protomod_sslid_search_service(handle_t service_handle):"
		    "service_handle=%d", service_handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check all sslid service list */
	for (service_number = 0; service_number < SSLID_SERVICE_NUMBER; ++service_number) {
		/* found the service has same service handle */
		if (sslid_service_list[service_number] != NULL && 
		    sslid_service_list[service_number]->service_handle == service_handle) {
			return_value = sslid_service_list[service_number];
			break;
		}
	}
	
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,126,
		    "out_function: struct l7vs_sslid_service* l7vs_protomod_sslid_search_service(handle_t service_handle):"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create sslid service.
 * @param  void
 * @return sslid service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_sslid_service*
l7vs_protomod_sslid_create_service()
{
	int service_number = 0;
	struct l7vs_sslid_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,127,
		    "in_function: struct l7vs_sslid_service* l7vs_protomod_sslid_create_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* check all sslid service list */
	for (service_number = 0; service_number < SSLID_SERVICE_NUMBER - 1; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (sslid_service_list[service_number] == NULL) {
			/* create a service at empty pointer */
			sslid_service_list[service_number] = (struct l7vs_sslid_service*) calloc(1, sizeof(struct l7vs_sslid_service));

			/*-------- DEBUG LOG --------*/
			if (sslid_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
				PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,32, "calloc: addr=%p, size=%ld",
				    sslid_service_list[service_number], (unsigned long int) sizeof(struct l7vs_sslid_service));
			}
			/*------ DEBUG LOG END ------*/

			if (sslid_service_list[service_number] == NULL) {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,25, "Could not allocate memory.");
				goto create_service_out;
			}
			return_value = sslid_service_list[service_number];
			goto create_service_out;
		}
	}
	
	/* all service list is full */
	PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,115, "sslid service list is full.");

create_service_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,128,
		    "out_function: struct l7vs_sslid_service* l7vs_protomod_sslid_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Create temporary sslid service.
 * @param  void
 * @return sslid service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_sslid_service*
l7vs_protomod_sslid_create_temp_service()
{
	struct l7vs_sslid_service* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,129,
		    "in_function: struct l7vs_sslid_service* l7vs_protomod_sslid_create_temp_service()");
	}
	/*------ DEBUG LOG END ------*/

	/* if pointer that does not point NULL exists ... */
	if (sslid_service_list[SSLID_SERVICE_NUMBER - 1] != NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,116, "Temporary sslid service is being used by other process.");
		goto create_temp_service_out;
	}

	/* create temp service */
	sslid_service_list[SSLID_SERVICE_NUMBER - 1] = (struct l7vs_sslid_service*) calloc(1, sizeof(struct l7vs_sslid_service));

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,33, "calloc: addr=%p, size=%ld",
		    sslid_service_list[SSLID_SERVICE_NUMBER - 1], (unsigned long int) sizeof(struct l7vs_sslid_service));
	}
	/*------ DEBUG LOG END ------*/

	if (sslid_service_list[SSLID_SERVICE_NUMBER - 1] == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_SYSTEM_MEMORY,26, "Could not allocate memory");
		goto create_temp_service_out;
	}

	return_value = sslid_service_list[SSLID_SERVICE_NUMBER - 1];

create_temp_service_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(ssl_str, return_value);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,130,
		    "out_function: struct l7vs_sslid_service* l7vs_protomod_sslid_create_service():"
		    "return_value=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Read replication area and set session data.
 * @param[in] sslid_service read this sslid service's session.
 * @return void
 */
static void l7vs_protomod_sslid_read_replication_data(struct l7vs_sslid_service* sslid_service, struct l7vs_service* srv)
{
	struct replication_header* head;
	struct replication_header* pick = NULL;
	unsigned int data_size = 0;
	unsigned int used = 0;
	void* data_addr = NULL;
	char exist = -1;
	int srv_num;

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(ssl_str, sslid_service);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,131,
		    "in_function: void l7vs_protomod_sslid_read_replication_data(struct "
		    "l7vs_sslid_service* sslid_service):sslid_service=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sslid_service == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 294, "Arg(sslid_service) is NULL pointer.");
		goto read_replication_data_out;
	}
	if (sslid_service->session == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 295, "Arg(sslid_service->session) is NULL pointer.");
		goto read_replication_data_out;
	}
	if (srv == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 296,
		    "Service that has handle(%d) is not found.", sslid_service->service_handle);
		goto read_replication_data_out;
	}
	if (srv->lsock == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 297,
		    "Conn of service that has handle(%d) is NULL pointer.", sslid_service->service_handle);
		goto read_replication_data_out;
	}

	/* get replication area address */
	data_addr = sslid_protomod.replication_pay_memory(sslid_protomod.modname, &data_size);
	if (data_addr == NULL || data_size <= 0)
		goto read_replication_data_out;

	/* check replication area header */
	for (srv_num = 0; srv_num < SSLID_SERVICE_NUMBER; srv_num++) {
		head = (struct replication_header*) data_addr + srv_num;
		used += head->size;

		if (data_size * DATA_SIZE < sizeof(struct replication_header) * (SSLID_SERVICE_NUMBER - 1) + used) {
			PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 298, "Over replication area.");
			goto read_replication_data_out;
		}

		if (pick == NULL && head->size == 0)
			pick = head;
		if (exist >= 0 && pick != NULL)
			break;

		/* match ip and port, this is pre-used session information */
		if (memcmp(&head->sin_addr, &srv->lsock->addr.sin_addr, sizeof(struct in_addr)) == 0 &&
		    head->sin_port == srv->lsock->addr.sin_port) {
			exist = srv_num;
		}
	}

	if (exist >= 0) {
		head = (struct replication_header*) data_addr + exist;

		/* restore session information */
		if (sslid_service->maxlist * sizeof(struct ssl_session) > head->size) {
			// resize if maxlist set bigger than old
			if (pick != NULL) {
				/* area size check */
				if (sizeof(struct replication_header) * (SSLID_SERVICE_NUMBER - 1) +
				    used + sslid_service->maxlist * sizeof(struct ssl_session) > data_size * DATA_SIZE) {
					PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 299,
					    "Replication area is full.");
					goto read_replication_data_out;
				}
				else {
					memcpy(sslid_service->session, (char*) data_addr + head->offset, head->size);
					memset(sslid_service->replication_addr, 0, head->size);
					sslid_service->replication_addr = (char*) data_addr +
					    sizeof(struct replication_header) * (SSLID_SERVICE_NUMBER - 1) + used;
					pick->sin_addr = srv->lsock->addr.sin_addr;
					pick->sin_port = srv->lsock->addr.sin_port;
					pick->size = sslid_service->maxlist * sizeof(struct ssl_session);
					pick->offset = sizeof(struct replication_header) * (SSLID_SERVICE_NUMBER - 1) + used;

					/*-------- DEBUG LOG --------*/
					if (sslid_protomod.get_log_level != NULL &&
					    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
						char head_str[DEBUG_STR_LEN];
						replication_header_c_str(head_str, pick);
						PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 318,
						    "Write replication area: head=(%s)", head_str);
					}
					/*------ DEBUG LOG END ------*/

					// you should garbage old session area...
				}
			}
			else {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 300,
				    "Replication area is full.");
				goto read_replication_data_out;
			}
		}
		else {
			sslid_service->replication_addr = (char*) data_addr + head->offset;
			memcpy(sslid_service->session, sslid_service->replication_addr,
			    sslid_service->maxlist * sizeof(struct ssl_session));
			head->size = sslid_service->maxlist * sizeof(struct ssl_session);

			/*-------- DEBUG LOG --------*/
			if (sslid_protomod.get_log_level != NULL &&
			    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
				char head_str[DEBUG_STR_LEN];
				replication_header_c_str(head_str, head);
				PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 319,
				    "Write replication area: head=(%s)", head_str);
			}
			/*------ DEBUG LOG END ------*/
		}
	}

	if (exist == -1) {
		if (pick != NULL) {
			/* area size check */
			if (sizeof(struct replication_header) * (SSLID_SERVICE_NUMBER - 1) +
			    used + sslid_service->maxlist * sizeof(struct ssl_session) > data_size * DATA_SIZE) {
				PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 301,
				    "Replication area is full.");
				goto read_replication_data_out;
			}
			else {
				/* initialize replication information */
				sslid_service->replication_addr = (char*) data_addr +
				    sizeof(struct replication_header) * (SSLID_SERVICE_NUMBER - 1) + used;
				pick->sin_addr = srv->lsock->addr.sin_addr;
				pick->sin_port = srv->lsock->addr.sin_port;
				pick->size = sslid_service->maxlist * sizeof(struct ssl_session);
				pick->offset = sizeof(struct replication_header) * (SSLID_SERVICE_NUMBER - 1) + used;

				/*-------- DEBUG LOG --------*/
				if (sslid_protomod.get_log_level != NULL &&
				    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
					char head_str[DEBUG_STR_LEN];
					replication_header_c_str(head_str, pick);
					PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 320,
					    "Write replication area: head=(%s)", head_str);
				}
				/*------ DEBUG LOG END ------*/
			}
		}
		else {
			PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 302,
			    "Replication area is full.");
			goto read_replication_data_out;
		}
	}

read_replication_data_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,132,
		    "out_function: l7vs_protomod_sslid_read_replication_data(struct "
		    "l7vs_sslid_service* sslid_service)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Write session data to replication area.
 * @param[in] sslid_service save this sslid service's session.
 * @return void
 */
static void l7vs_protomod_sslid_write_replication_data(struct l7vs_sslid_service* sslid_service)
{
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char ssl_str[DEBUG_STR_LEN] = {0};
		l7vs_sslid_service_c_str(ssl_str, sslid_service);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,133,
		    "in_function: void l7vs_protomod_sslid_write_replication_data(struct "
		    "l7vs_sslid_service* sslid_service):sslid_service=&(%s)", ssl_str);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (sslid_service == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 303, "Arg(sslid_service) is NULL pointer.");
		goto write_replication_data_out;
	}
	if (sslid_service->session == NULL) {
		PUT_LOG_ERROR(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 304, "Arg(sslid_service->session) is NULL pointer.");
		goto write_replication_data_out;
	}
	/* no replicate setting */
	if (sslid_service->replication_addr == NULL) {
		goto write_replication_data_out;
	}

	/* copy session data */
	memcpy(sslid_service->replication_addr, sslid_service->session,
	    sizeof(struct ssl_session) * sslid_service->maxlist);

	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		char session_str[DEBUG_STR_LEN];
		ssl_session_c_str(session_str, (struct ssl_session*) sslid_service->replication_addr);
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL, 321, "Write replication area: session=(%s)",
		    session_str);
	}
	/*------ DEBUG LOG END ------*/

write_replication_data_out:
	/*-------- DEBUG LOG --------*/
	if (sslid_protomod.get_log_level != NULL &&
	    LOG_LV_DEBUG == sslid_protomod.get_log_level(LOG_CAT_L7VSD_PROTOCOL)) {
		PUT_LOG_DEBUG(sslid_protomod, LOG_CAT_L7VSD_PROTOCOL,134,
		    "out_function: void l7vs_protomod_sslid_write_replication_data(struct "
		    "l7vs_sslid_service* sslid_service)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Serialize struct l7vs_sslid_service for debug log.
 * @param[out] buf   serialized string
 * @param[in]  sslid l7vs_sslid_service struct
 */
static void l7vs_sslid_service_c_str(char* buf, struct l7vs_sslid_service* sslid) {
	if (sslid == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		char session_str[DEBUG_STR_LEN] = {0};
		ssl_session_c_str(session_str, sslid->session);
		snprintf(buf, DEBUG_STR_LEN, "service_handle=%d, timeout=%d, maxlist=%d, "
		    "session=(%s), replication_addr=%p, reschedule=%d", sslid->service_handle,
		    sslid->timeout, sslid->maxlist, session_str, sslid->replication_addr,
		    sslid->reschedule);
	}
}

/*!
 * Serialize struct l7vs_sslid_service_arg for debug log.
 * @param[out] buf       serialized string
 * @param[in]  sslid_arg l7vs_sslid_service_arg struct
 */
void l7vs_sslid_service_arg_c_str(char* buf, struct l7vs_sslid_service_arg* sslid_arg) {
	if (sslid_arg == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "timeout=%d, maxlist=%d, reschedule=%d",
		    sslid_arg->timeout, sslid_arg->maxlist, sslid_arg->reschedule);
	}
}

/*!
 * Serialize struct ssl_session for debug log.
 * @param[out] buf     serialized string
 * @param[in]  session ssl_session struct
 */
static void ssl_session_c_str(char* buf, struct ssl_session* session) {
	if (session == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		char dest_str[DEBUG_STR_LEN] = {0};
		char session_str[SSLID_LENGTH * 2 + 1];
		l7vs_dest_c_str(dest_str, &session->dest);
		id_c_str(session_str, session->id, session->id_len);	// Add length parameter 2009.4.8 T.Motoda@NTTR
		snprintf(buf, DEBUG_STR_LEN, "id=%s, dest=(%s), last_time=%d, valid=%d",
		    session_str, dest_str, (u_int) session->last_time, (int)session->valid);
	}
}

/*!
 * Serialize struct replication_header for debug log.
 * @param[out] buf  serialized string
 * @param[in]  head replication_header struct
 */
static void replication_header_c_str(char* buf, struct replication_header* head) {
	if (head == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		snprintf(buf, DEBUG_STR_LEN, "sin_addr=(s_addr=%ld), sin_port=%d, size=%d, offset=%d",
		    (u_long) head->sin_addr.s_addr, head->sin_port, head->size, head->offset);
	}
}

/*!
 * Convert SSL session ID (binary to hex)
 * @param[out] buf hex string
 * @param[in]  id  SSL session ID
 * @param[in]  id_len SSL session ID length  2009.4.8 by T.Motoda@NTTR
 */
static void id_c_str(char* buf, char* id, int id_len) {
	int i;
	if (id == NULL) {
		snprintf(buf, DEBUG_STR_LEN, "NULL");
	}
	else {
		for (i = 0; i < id_len; i++) {
			snprintf(buf + i * 2, DEBUG_STR_LEN - i * 2, "%02X", (unsigned char)id[i]);

		}
	}
}
