/*
 * @file  conn.c
 * @brief the connection management component
 * @brief transmit data between client and realserver
 * @brief and calls protocol module to decide realserver
 * @brief also modify realserver data
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <assert.h>
#include <glib.h>
#include <limits.h>
#include <errno.h>
#include "logger_wrapper.h"
#include "l7vs_config.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_lsock.h"
#include "l7vs_conn.h"
#include "l7vs_service.h"
#include "l7vs.h"
#include "l7vs_module.h"
#include "l7vs_sched.h"

static int l7vs_conn_cl_callback(struct l7vs_iomux*);
static int l7vs_conn_client_receiving(struct l7vs_iomux*);
static int l7vs_conn_is_rs_connected(struct l7vs_iomux*, struct l7vs_conn*, struct l7vs_service**, struct l7vs_dest**);
static int l7vs_conn_connection_establish(struct l7vs_iomux*, struct l7vs_conn*, struct l7vs_service*, struct l7vs_dest*);
static int l7vs_conn_block(struct l7vs_iomux*, struct l7vs_iomux*);
static int l7vs_conn_busy(struct l7vs_iomux*);
static int l7vs_conn_sending(struct l7vs_iomux*, struct l7vs_iomux*);
static int l7vs_conn_sent_next_state(struct l7vs_iomux*, struct l7vs_iomux*);
static int l7vs_conn_rs_callback(struct l7vs_iomux* );
static int l7vs_conn_realserver_receiving(struct l7vs_iomux*);
static int l7vs_conn_realserver_modify_data(struct l7vs_iomux*);
static int l7vs_conn_recv(struct l7vs_iomux*, int);
static int l7vs_conn_preread(struct l7vs_iomux*, int);
static int l7vs_conn_send(struct l7vs_iomux*, int);
static int l7vs_conn_change_connect_rs(struct l7vs_conn*, struct l7vs_dest*);
static int l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction );
static void l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize, int direction );
static void l7vs_conn_dump_data(char* str_data, const char* data, const size_t len);
static int l7vs_conn_read_bufsize = 20480;              //! conn read buffer size
static int throughput_interval = BPS_DEFAULT_INTERVAL;  //! throughput interval

int
l7vs_conn_init(void)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,353,
            "in_function: int lvs_conn_init() ");
    }
    /*------ DEBUG LOG END ------*/

    int ret = parameter_is_int_exist( PARAM_COMP_CONN, "read_bufsize" );
    if (ret) {
        int bufsize = parameter_get_int_value( PARAM_COMP_CONN, "read_bufsize" );
        if (bufsize >= L7VS_CONN_READ_BUFSIZE_MIN && bufsize <= L7VS_CONN_READ_BUFSIZE_MAX)
            l7vs_conn_read_bufsize = bufsize;
//            LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_NETWORK,5,
//                "l7vsd uses read_bufsize = %d ",bufsize);
        else if (bufsize > L7VS_CONN_READ_BUFSIZE_MAX) {
            l7vs_conn_read_bufsize = L7VS_CONN_READ_BUFSIZE_MAX;
            LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_NETWORK,1,
                "Your read_bufsize is over maximum size. l7vsd uses maximum read_bufsize = %d instead ",L7VS_CONN_READ_BUFSIZE_MAX);
        }
        else {
            l7vs_conn_read_bufsize = L7VS_CONN_READ_BUFSIZE_MIN;
            LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_NETWORK,2,
                "Your read_bufsize is under minimum size. l7vsd uses minimum read_bufsize = %d instead ",L7VS_CONN_READ_BUFSIZE_MIN);
        }
    }
    else {
        LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_NETWORK,3,
            "read_bufsize is not specified. l7vsd uses default read_bufsize = %d ",l7vs_conn_read_bufsize);
    }

    if ( parameter_is_int_exist( PARAM_COMP_L7VSD, "calc_throughput_interval" ) ) {
        throughput_interval = parameter_get_int_value( PARAM_COMP_L7VSD, "calc_throughput_interval" );
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,345,
            "out_function: int lvs_conn_init() ");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * dump data
 *
 * @param[out]  str_data    dump string
 * @param[in]   data        source data
 * @param[in]   len         source data length
 * @return      void
 */
void l7vs_conn_dump_data(char* str_data, const char* data, const size_t len)
{
    if (!str_data) {
        return;
    }

    if (!data) {
        snprintf(str_data, DEBUG_STR_LEN, "(nil)");
        return;
    }

    const char* table[] = {
        "00 ", "01 ", "02 ", "03 ", "04 ", "05 ", "06 ", "07 ", "08 ", "09 ", "0a ", "0b ", "0c ", "0d ", "0e ", "0f ",
        "10 ", "11 ", "12 ", "13 ", "14 ", "15 ", "16 ", "17 ", "18 ", "19 ", "1a ", "1b ", "1c ", "1d ", "1e ", "1f ",
        "20 ", "21 ", "22 ", "23 ", "24 ", "25 ", "26 ", "27 ", "28 ", "29 ", "2a ", "2b ", "2c ", "2d ", "2e ", "2f ",
        "30 ", "31 ", "32 ", "33 ", "34 ", "35 ", "36 ", "37 ", "38 ", "39 ", "3a ", "3b ", "3c ", "3d ", "3e ", "3f ",
        "40 ", "41 ", "42 ", "43 ", "44 ", "45 ", "46 ", "47 ", "48 ", "49 ", "4a ", "4b ", "4c ", "4d ", "4e ", "4f ",
        "50 ", "51 ", "52 ", "53 ", "54 ", "55 ", "56 ", "57 ", "58 ", "59 ", "5a ", "5b ", "5c ", "5d ", "5e ", "5f ",
        "60 ", "61 ", "62 ", "63 ", "64 ", "65 ", "66 ", "67 ", "68 ", "69 ", "6a ", "6b ", "6c ", "6d ", "6e ", "6f ",
        "70 ", "71 ", "72 ", "73 ", "74 ", "75 ", "76 ", "77 ", "78 ", "79 ", "7a ", "7b ", "7c ", "7d ", "7e ", "7f ",
        "80 ", "81 ", "82 ", "83 ", "84 ", "85 ", "86 ", "87 ", "88 ", "89 ", "8a ", "8b ", "8c ", "8d ", "8e ", "8f ",
        "90 ", "91 ", "92 ", "93 ", "94 ", "95 ", "96 ", "97 ", "98 ", "99 ", "9a ", "9b ", "9c ", "9d ", "9e ", "9f ",
        "a0 ", "a1 ", "a2 ", "a3 ", "a4 ", "a5 ", "a6 ", "a7 ", "a8 ", "a9 ", "aa ", "ab ", "ac ", "ad ", "ae ", "af ",
        "b0 ", "b1 ", "b2 ", "b3 ", "b4 ", "b5 ", "b6 ", "b7 ", "b8 ", "b9 ", "ba ", "bb ", "bc ", "bd ", "be ", "bf ",
        "c0 ", "c1 ", "c2 ", "c3 ", "c4 ", "c5 ", "c6 ", "c7 ", "c8 ", "c9 ", "ca ", "cb ", "cc ", "cd ", "ce ", "cf ",
        "d0 ", "d1 ", "d2 ", "d3 ", "d4 ", "d5 ", "d6 ", "d7 ", "d8 ", "d9 ", "da ", "db ", "dc ", "dd ", "de ", "df ",
        "e0 ", "e1 ", "e2 ", "e3 ", "e4 ", "e5 ", "e6 ", "e7 ", "e8 ", "e9 ", "ea ", "eb ", "ec ", "ed ", "ee ", "ef ",
        "f0 ", "f1 ", "f2 ", "f3 ", "f4 ", "f5 ", "f6 ", "f7 ", "f8 ", "f9 ", "fa ", "fb ", "fc ", "fd ", "fe ", "ff "
    };
    const int hx_len = 3;

    size_t pos = 0;
    for (size_t i = 0; i < len; ++i) {
        if (DEBUG_STR_LEN <= pos + hx_len) {
            break;
        }
        strncpy(str_data + pos, table[(unsigned char)*(data + i)], hx_len);
        pos += hx_len;
    }
    *(str_data + pos) = '\0';
}

/*!
 * connection create
 *
 * @param[in]   lfd     listening socket fd
 * @param[in]   lsock   listening socket
 * @return      connection
 */
struct l7vs_conn *
l7vs_conn_create(int lfd, struct l7vs_lsock *lsock)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,50,
            "in_function: struct l7vs_conn* lvs_conn_create(int fd, struct l7vs_lsock *lsock) "
            "lfd=%d: "
            "lsock=%s",
            lfd, lsock_str);
    }
    /*------ DEBUG LOG END ------*/

        socklen_t len = 0;
        struct l7vs_conn *conn = NULL;
        int mss = 0;
        int ret = 0;
    int flags = 0;

    if (-1 == lfd) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,28, "error / lsock fd is not specified");
        goto create_error;
    }

    if (!lsock) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,29, "error / lsock is null");
        goto create_error;
    }
       
        conn = (struct l7vs_conn*)malloc(sizeof(struct l7vs_conn));
        if (!conn) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,8, "error / can not allocate memory");
        goto create_error;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,51,
            "malloc: conn=%p: size=%zu",
            conn, sizeof(struct l7vs_conn));
    }
    /*------ DEBUG LOG END ------*/
    memset(conn, 0, sizeof(struct l7vs_conn));

    conn->ciom = l7vs_iomux_get_from_avail_list();
    if (!conn->ciom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,7, "error / can not get client iom");
        goto create_error;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char ciom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(ciom_str, conn->ciom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,52,
            "iom_get: conn->ciom=%s",
            ciom_str);
    }
    /*------ DEBUG LOG END ------*/

    conn->riom = l7vs_iomux_get_from_avail_list();
    if (!conn->riom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,8, "error / can not get realserver iom");
        goto create_error;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char riom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(riom_str, conn->riom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,53,
            "iom_get: conn->riom=%s",
            riom_str);
    }
    /*------ DEBUG LOG END ------*/

        conn->cldata_len = 0;
        conn->cldata_bufsize = l7vs_conn_read_bufsize;
        conn->cldata = (char *)malloc(conn->cldata_bufsize);
        if (!conn->cldata) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,9, "error / can not allocate memory for buffer");
        goto create_error;
        }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,54,
            "malloc: conn->cldata=%p: size=%zu",
            conn->cldata, conn->cldata_bufsize);
    }
    /*------ DEBUG LOG END ------*/
    memset(conn->cldata, 0, conn->cldata_bufsize);

        len = sizeof(struct sockaddr_in);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char caddr_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_sockaddr_in_c_str(caddr_str, &conn->caddr);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,55,
            "accept param: lfd=%d: conn->caddr=%s: len=%d",
            lfd, caddr_str, len);
    }
    /*------ DEBUG LOG END ------*/
        conn->ciom->fd = accept(lfd, (struct sockaddr *)&conn->caddr, &len);
        if (conn->ciom->fd < 0) {
        char caddr_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_sockaddr_in_c_str(caddr_str, &conn->caddr);
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,8,
            "error / accept error accept param: lfd=%d: conn->caddr=%s: len=%d (%s)",
            lfd, caddr_str, len,strerror(errno));
        goto create_error;
        }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,56,
            "accept result: conn->ciom->fd=%d",
            conn->ciom->fd);
    }
    /*------ DEBUG LOG END ------*/

    // get current flag
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,57,
            "fnctl param: conn->ciom->fd=%d: cmd=F_GETFL: flags=0",
            conn->ciom->fd);
    }
    /*------ DEBUG LOG END ------*/
    flags = fcntl(conn->ciom->fd, F_GETFL, 0);
    if (0 > flags) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,9,
             "error / fcntl((conn->ciom->fd=%d),F_GETFL) error(%s)",
             conn->ciom->fd,strerror(errno));
        goto create_error;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,58,
            "fnctl result: flags=%d",
            flags);
    }
    /*------ DEBUG LOG END ------*/

    // set nonblocking
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,59,
            "fnctl param: conn->ciom->fd=%d: cmd=F_SETFL: flags=%d",
            conn->ciom->fd, (flags | O_NONBLOCK));
    }
    /*------ DEBUG LOG END ------*/
    flags = fcntl(conn->ciom->fd, F_SETFL, flags | O_NONBLOCK);
    if (0 > flags) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,10,
            "error / fcntl((conn->ciom->fd=%d),F_SETFL) error(%s)",
            conn->ciom->fd,strerror(errno));
        goto create_error;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,60,
            "fnctl result: flags=%d",
            flags);
    }
    /*------ DEBUG LOG END ------*/

    // get client mss
        len = sizeof(int);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,61,
            "getsockopt param: conn->ciom->fd=%d: level=IPROTO_TCP: optname=TCP_MAXSEG: "
            "len=%d",
            conn->ciom->fd, len);
    }
    /*------ DEBUG LOG END ------*/
        ret = getsockopt(conn->ciom->fd, IPPROTO_TCP, TCP_MAXSEG, &mss, &len);
        if (0 > ret) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,11, "error / getsockopt TCP_MAXSEG error(%s)", strerror(errno));
        goto create_error;
        }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,62,
            "getsockopt result: mss=%d: len=%d",
            mss, len);
    }
    /*------ DEBUG LOG END ------*/
        conn->cmss = mss;
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,63, "client %s mss %d",
            inet_ntoa(conn->caddr.sin_addr), mss);
    }

        conn->lsock = lsock;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,64,
            "pointer_assign: conn->lsock=%s",
            lsock_str);
    }
    /*------ DEBUG LOG END ------*/
        conn->proto = lsock->proto;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,65,
            "conn->ciom->status change: %d->%d",
            conn->ciom->status, iomux_conn_receiving);
    }
    /*------ DEBUG LOG END ------*/
    conn->ciom->status = iomux_conn_receiving;
    conn->ciom->callback = l7vs_conn_cl_callback;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,66,
            "pointer_assign: conn->ciom->callback=%p",
            l7vs_conn_cl_callback);
    }
    /*------ DEBUG LOG END ------*/
        conn->ciom->data = conn;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,67,
            "pointer_assign: conn->ciom->data=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/
    conn->sorry_conn_flag = 0;
    conn->old_dest = NULL;
    l7vs_iomux_add(conn->ciom, iom_read);

    // connect real server immediately if all lsock services are fast schedule.
    if (lsock->fast_schedule) {
        struct l7vs_service* srv;
        struct l7vs_dest* dest;
        conn->ciom->status = iomux_conn_received;
        if (!l7vs_conn_is_rs_connected(conn->ciom, conn, &srv, &dest)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,38, "error / is realserver connected failed");
        }
        else {
            switch (conn->ciom->status) {
            case iomux_conn_block:
                //already decided real server
                break;
            case iomux_conn_service_selected:
                //service and destination were selected then attempt to connect the real server
                if (!l7vs_conn_connection_establish(conn->ciom, conn, srv, dest)) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,39, "error / client connect rs failed");
                }
                break;
            default:
                //invalid status
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,79, "error / invalid status(%d)", conn->ciom->status);
                break;
            }
        }
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char ret_conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(ret_conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,68,
            "out_function: struct l7vs_conn* lvs_conn_create(int fd, struct l7vs_lsock *lsock) "
            "return_value: conn=%s",
            ret_conn_str);
    }
    /*------ DEBUG LOG END ------*/
        return conn;

create_error:
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,69,
            "entering: create_error_label");
    }
    /*------ DEBUG LOG END ------*/
    if (conn) {
        if (conn->ciom) {
            if (-1 != conn->ciom->fd) {
                /*-------- DEBUG LOG --------*/
                if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                    LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,70,
                        "close: conn->ciom->fd=%d",
                        conn->ciom->fd);
                }
                /*------ DEBUG LOG END ------*/
                close(conn->ciom->fd);
            }
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                char put_ciom_str[DEBUG_STR_LEN] = {0};
                l7vs_iomux_c_str(put_ciom_str, conn->ciom);
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,71,
                    "iom_put: conn->ciom=%s",
                    put_ciom_str);
            }
            /*------ DEBUG LOG END ------*/
            l7vs_iomux_put_to_avail_list(conn->ciom);
        }
        if (conn->riom) {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                char put_riom_str[DEBUG_STR_LEN] = {0};
                l7vs_iomux_c_str(put_riom_str, conn->riom);
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,72,
                    "iom_put: conn->riom=%s",
                    put_riom_str);
            }
            /*------ DEBUG LOG END ------*/
            l7vs_iomux_put_to_avail_list(conn->riom);
        }
        if (conn->cldata) {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,73,
                    "free: conn->cldata=%p",
                    conn->cldata);
            }
            /*------ DEBUG LOG END ------*/
            free(conn->cldata);
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,74,
                "free: conn=%p",
                conn);
        }
        /*------ DEBUG LOG END ------*/
        free(conn);
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,75,
            "out_function: struct l7vs_conn* lvs_conn_create(int fd, struct l7vs_lsock *lsock) "
            "return_value: conn=NULL");
    }
    /*------ DEBUG LOG END ------*/
    return NULL;
}

/*!
 * connection destroy
 *
 * @param[in]   conn    connection
 * @return      void
 */
void
l7vs_conn_destroy(struct l7vs_conn *conn)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,76,
            "in_function: void l7vs_conn_destroy(struct l7vs_conn *conn) "
            "conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/
    if (!conn) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,77,
                "out_function: void l7vs_conn_destroy(struct l7vs_conn *conn) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }

    if (conn->srv) {
        l7vs_service_remove_conn(conn->srv, conn);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,78,
                "pointer_assign: conn->srv=NULL");
        }
        /*------ DEBUG LOG END ------*/
        conn->srv = NULL;
    }

    if (conn->ciom) {
        if (-1 != conn->ciom->fd) {
            l7vs_conn_close_csock(conn);
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            char put_ciom_str[DEBUG_STR_LEN] = {0};
            l7vs_iomux_c_str(put_ciom_str, conn->ciom);
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,79,
                "iom_put: conn->ciom=%s",
                put_ciom_str);
        }
        /*------ DEBUG LOG END ------*/
        l7vs_iomux_put_to_avail_list(conn->ciom);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,80,
                "pointer_assign: conn->ciom=NULL");
        }
        /*------ DEBUG LOG END ------*/
        conn->ciom = NULL;
    }

    if (conn->riom) {
        if (-1 != conn->riom->fd) {
            l7vs_conn_close_rsock(conn);
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            char put_riom_str[DEBUG_STR_LEN] = {0};
            l7vs_iomux_c_str(put_riom_str, conn->riom);
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,81,
                "iom_put: conn->riom=%s",
                put_riom_str);
        }
        /*------ DEBUG LOG END ------*/
        l7vs_iomux_put_to_avail_list(conn->riom);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,82,
                "pointer_assign: conn->riom=NULL");
        }
        /*------ DEBUG LOG END ------*/
        conn->riom = NULL;
    }

    if (conn->cldata) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,83,
                "free: conn->cldata=%p",
                conn->cldata);
        }
        /*------ DEBUG LOG END ------*/
        free(conn->cldata);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,84,
                "pointer_assign: conn->cldata=NULL");
        }
        /*------ DEBUG LOG END ------*/
        conn->cldata = NULL;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,85,
            "free: conn=%p",
            conn);
    }
    /*------ DEBUG LOG END ------*/
    free(conn);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,86,
            "out_function: void l7vs_conn_destroy(struct l7vs_conn *conn) "
            "return_value: void");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * client connection callback
 *
 * @param[in]   iom     connection iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_cl_callback(struct l7vs_iomux *iom )
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,87,
            "in_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
            "iom=%s",
            iom_str);
    }
    /*------ DEBUG LOG END ------*/
    struct l7vs_conn* conn = NULL;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,30, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,88,
                "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    conn = (struct l7vs_conn *)iom->data;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,89,
            "pointer_assign: conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    // invalid status check
    if (iomux_conn_receiving != iom->status &&
        iomux_conn_block != iom->status &&
        iomux_conn_busy != iom->status && 
        iomux_conn_sending != iom->status && 
        iomux_conn_sending_busy != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,31, "error / invalid status(%d)", iom->status);
        l7vs_conn_destroy(conn);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,90,
                "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    // switch by current status
    switch (iom->status) {
    case iomux_conn_receiving:
        if (!l7vs_conn_client_receiving(iom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,9, "error / cl_callback/client receiving failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,91,
                    "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        if (iomux_conn_disconnected == iom->status) {
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,92, "client connection disconnected");
            }
            l7vs_conn_destroy(conn);
        }
        break;
    case iomux_conn_block:
        if (!l7vs_conn_block(iom, conn->riom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,10, "error / cl_callback/block failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,93,
                    "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    case iomux_conn_busy:
        if (!l7vs_conn_busy(iom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,11, "error / cl_callback/busy failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,94,
                    "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    case iomux_conn_sending:
        if (!l7vs_conn_sending(iom, conn->riom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,12, "error / cl_callback/sending failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,95,
                    "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    case iomux_conn_sending_busy:
        if (!l7vs_conn_sending(iom, conn->riom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,13, "error / cl_callback/sending_busy failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,96,
                    "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    default:
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,14, "error / invalid status(%d)", iom->status);
        l7vs_conn_destroy(conn);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,97,
                "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,98,
            "out_function: static int l7vs_conn_cl_callback(struct l7vs_iomux *iom ) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * client receiving
 *
 * @param[in]   iom     connection iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_client_receiving(struct l7vs_iomux *iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,99,
            "in_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
            "iom=%s",
            iom_str);
    }
    /*------ DEBUG LOG END ------*/
    struct l7vs_conn *conn = NULL;
        struct l7vs_service *srv = NULL;
        struct l7vs_dest *dest = NULL;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,32, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,100,
                "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_receiving != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,33, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,101,
                "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    conn = (struct l7vs_conn *)iom->data;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,102,
            "pointer_assign: conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    // when the real server is not set in read-blocked
    if (iomux_conn_block != conn->riom->status && iomux_conn_busy != conn->riom->status) {
        // set the real server in read-blocked
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,103,
                "conn->riom->status change: %d->%d",
                conn->riom->status, iomux_conn_block);
        }
        /*------ DEBUG LOG END ------*/
        conn->riom->status = iomux_conn_block;
    }

    // receive client data
    if (!l7vs_conn_recv(iom, iom->fd)) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,15, "error / recv failed");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,104,
                "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    switch (iom->status) {
    case iomux_conn_receiving:
        // data still remain, epoll re-registration for read
        l7vs_iomux_mod(iom, iom_read);
        break;
    case iomux_conn_received:
        // all data is received, service is selected by using the received data
        if (!l7vs_conn_is_rs_connected(iom, conn, &srv, &dest)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,16, "error / is realserver connected failed");
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,105,
                    "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        switch (iom->status) {
        case iomux_conn_block:
            //already decided real server
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,106,
                    "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                    "return_value: 1");
            }
            /*------ DEBUG LOG END ------*/
            return 1;
        case iomux_conn_service_selected:
            //service and destination were selected then attempt to connect the real server
            if (!l7vs_conn_connection_establish(iom, conn, srv, dest)) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,17, "error / client connect rs failed");
                /*-------- DEBUG LOG --------*/
                if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                    LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,107,
                        "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                        "return_value: 0");
                }
                /*------ DEBUG LOG END ------*/
                return 0;
            }
            l7vs_conn_QoS_recvsize_register( conn, conn->cldata_len, QOS_UP );
            break;
        default:
            //invalid status
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,34, "error / invalid status(%d)", iom->status);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,108,
                    "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    case iomux_conn_disconnected:
        // client connection was disconnected
        break;
    default:
        //invalid status
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,35, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,109,
                "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,110,
            "out_function: static int l7vs_conn_client_receiving(struct l7vs_iomux *iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * check realserver fd, and if real server not connected, call select_service function
 *
 * @param[in]   iom     connection_iomux
 * @param[in]   conn    connection
 * @param[out]  srv     service
 * @param[out]  dest    destination(real server)
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_is_rs_connected(struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        char conn_str[DEBUG_STR_LEN] = {0};
        char srv_str[DEBUG_STR_LEN] = {0};
        char dest_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        l7vs_conn_c_str(conn_str, conn);
        l7vs_service_c_str(srv_str, *srv);
        l7vs_dest_c_str(dest_str, *dest);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,111,
            "in_function: static int l7vs_conn_is_rs_connected("
            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
            "iom=%s: "
            "conn=%s: "
            "srv=%s: "
            "dest=%s",
            iom_str, conn_str, srv_str, dest_str);
    }
    /*------ DEBUG LOG END ------*/

    int ret = 0;
    struct l7vs_dest *d;    // destination for sorry-server
    int sorry_state;        // sorry state

    struct l7vs_lsock *lsock = NULL;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,36, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,112,
                "out_function: static int l7vs_conn_is_rs_connected("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (!conn) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,37, "error / conn is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,113,
                "out_function: static int l7vs_conn_is_rs_connected("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_received != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,38, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,114,
                "out_function: static int l7vs_conn_is_rs_connected("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    lsock = conn->lsock;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,115,
            "pointer_assign: lsock=%s",
            lsock_str);
    }
    /*------ DEBUG LOG END ------*/

    //is real server connected?
    if (-1 != conn->riom->fd) {
        // check sorry status of service
        // argment 0 means no connection count check
        sorry_state = l7vs_sched_sorry_check(conn->srv, 0);
        if (sorry_state) {
            // check connect to sorry-server now
            if (!conn->sorry_conn_flag) {
                // get sorry-server destination (argment 0)
                d = l7vs_sched_sorry_dest(conn->srv, conn, 0);
                /*-------- DEBUG LOG --------*/
                if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                    char sorry_dest_str1[DEBUG_STR_LEN] = {0};
                    l7vs_dest_c_str(sorry_dest_str1, d);
                    LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,116,
                        "pointer_assign: d=%s",
                        sorry_dest_str1);
                }
                /*------ DEBUG LOG END ------*/
                if (!d) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,18, "error / sorry-server dest NG");
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,117,
                            "iom->status change: %d->%d",
                            iom->status, iomux_conn_rs_connect_error);
                    }
                    /*------ DEBUG LOG END ------*/
                    iom->status = iomux_conn_rs_connect_error;
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,118,
                            "out_function: static int l7vs_conn_is_rs_connected("
                            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                            "return_value: 0");
                    }
                    /*------ DEBUG LOG END ------*/
                    return 0;
                }
                conn->old_dest = conn->dest;
                /*-------- DEBUG LOG --------*/
                if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                    char old_dest_str[DEBUG_STR_LEN] = {0};
                    l7vs_dest_c_str(old_dest_str, conn->old_dest);
                    LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,119,
                        "pointer_assign: conn->old_dest=%s",
                        old_dest_str);
                }
                /*------ DEBUG LOG END ------*/
                // change connection to sorry-server
                if (!l7vs_conn_change_connect_rs(conn, d)) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,19, "error / change connection to Sorry NG");
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,120,
                            "iom->status change: %d->%d",
                            iom->status, iomux_conn_rs_connect_error);
                    }
                    /*------ DEBUG LOG END ------*/
                    iom->status = iomux_conn_rs_connect_error;
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,121,
                            "out_function: static int l7vs_conn_is_rs_connected("
                            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                            "return_value: 0");
                    }
                    /*------ DEBUG LOG END ------*/
                    return 0;
                }
                conn->sorry_conn_flag = 1;
            }
        } else {
            // check connect to sorry-server now
            if (conn->sorry_conn_flag) {
                // get old real-server destination (argment 1)
                d = l7vs_sched_sorry_dest(conn->srv, conn, 1);
                /*-------- DEBUG LOG --------*/
                if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                    char sorry_dest_str2[DEBUG_STR_LEN] = {0};
                    l7vs_dest_c_str(sorry_dest_str2, d);
                    LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,122,
                        "pointer_assign: d=%s",
                        sorry_dest_str2);
                }
                /*------ DEBUG LOG END ------*/
                if (!d) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,20, "error / old real-server dest NG");
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,123,
                            "iom->status change: %d->%d",
                            iom->status, iomux_conn_rs_connect_error);
                    }
                    /*------ DEBUG LOG END ------*/
                    iom->status = iomux_conn_rs_connect_error;
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,124,
                            "out_function: static int l7vs_conn_is_rs_connected("
                            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                            "return_value: 0");
                    }
                    /*------ DEBUG LOG END ------*/
                    return 0;
                }
                // change connection to real-server
                if (!l7vs_conn_change_connect_rs(conn, d)) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,21, "error / change connection to RS NG");
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,125,
                            "iom->status change: %d->%d",
                            iom->status, iomux_conn_rs_connect_error);
                    }
                    /*------ DEBUG LOG END ------*/
                    iom->status = iomux_conn_rs_connect_error;
                    /*-------- DEBUG LOG --------*/
                    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,126,
                            "out_function: static int l7vs_conn_is_rs_connected("
                            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                            "return_value: 0");
                    }
                    /*------ DEBUG LOG END ------*/
                    return 0;
                }
                conn->sorry_conn_flag = 0;
            }
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,127,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_block);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_block;
        if (iomux_conn_busy == conn->riom->status) {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,128,
                    "conn->riom->status change: %d->%d",
                    conn->riom->status, iomux_conn_sending_busy);
            }
            /*------ DEBUG LOG END ------*/
            conn->riom->status = iomux_conn_sending_busy;
        }
        else {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,129,
                    "conn->riom->status change: %d->%d",
                    conn->riom->status, iomux_conn_sending);
            }
            /*------ DEBUG LOG END ------*/
            conn->riom->status = iomux_conn_sending;
        }
        //epoll re-registration for write
        l7vs_iomux_mod(conn->riom, iom_write);
    }
    else {
        //select service
        ret = l7vs_lsock_select_service(lsock, conn, conn->cldata,
                        conn->cldata_len, srv, dest);
        if (0 >= ret) {
            *srv = NULL;
            *dest = NULL;
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,22, "error / no matching service found");
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,130,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_receive_error);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_receive_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,131,
                    "out_function: static int l7vs_conn_is_rs_connected("
                    "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,132,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_service_selected);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_service_selected;
    }

    // analyze and modify client data
    if (conn->srv) {
        ret = conn->srv->pm->analyze_cldata(conn->srv, conn, conn->cldata, &conn->cldata_len);
        if (ret != 0) {
            LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_PROTOCOL,19,
                "Failed to modify client data: %d", ret);
        }
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,133,
            "out_function: static int l7vs_conn_is_rs_connected("
            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * connecction establish
 *
 * @param[in]   iom     connection_iomux
 * @param[in]   conn    connection
 * @param[in]   srv     service
 * @param[in]   dest    destination(real server)
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_connection_establish(struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        char conn_str[DEBUG_STR_LEN] = {0};
        char srv_str[DEBUG_STR_LEN] = {0};
        char dest_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        l7vs_conn_c_str(conn_str, conn);
        l7vs_service_c_str(srv_str, srv);
        l7vs_dest_c_str(dest_str, dest);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,134,
            "in_function: static int l7vs_conn_connection_establish("
            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
            "iom=%s: "
            "conn=%s: "
            "srv=%s: "
            "dest=%s",
            iom_str, conn_str, srv_str, dest_str);
    }
    /*------ DEBUG LOG END ------*/
    int ret = 0;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,39, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,135,
                "out_function: static int l7vs_conn_connection_establish("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (!conn) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,40, "error / conn is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,136,
                "out_function: static int l7vs_conn_connection_establish("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (!srv) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,41, "error / srv is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,137,
                "out_function: static int l7vs_conn_connection_establish("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (!dest) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,42, "error / dest is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,138,
                "out_function: static int l7vs_conn_connection_establish("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iom->status != iomux_conn_service_selected) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,43, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,139,
                "out_function: static int l7vs_conn_connection_establish("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    // connect to real server
    if (!l7vs_conn_connect_rs(conn, dest)) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,23, "error / connect rs failed");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,140,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_rs_connect_error);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_rs_connect_error;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,141,
                "out_function: static int l7vs_conn_connection_establish("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,142,
            "iom->status change: %d->%d",
            iom->status, iomux_conn_rs_connected);
    }
    /*------ DEBUG LOG END ------*/
    iom->status = iomux_conn_rs_connected;

    // set pointer for cross reference
    conn->srv = srv;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char service_str[DEBUG_STR_LEN] = {0};
        l7vs_service_c_str(service_str, conn->srv);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,143,
            "pointer_assign: conn->srv=%s",
            service_str);
    }
    /*------ DEBUG LOG END ------*/
    l7vs_service_register_conn(srv, conn);

    // service establish
    ret = l7vs_service_establish(conn->srv, conn);
    if (0 > ret) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,24, "error / service establish failed");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,144,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_rs_connect_error);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_rs_connect_error;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,145,
                "out_function: static int l7vs_conn_connection_establish("
                "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    
    // iom status set 
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,146,
            "iom->status change: %d->%d",
            iom->status, iomux_conn_block);
    }
    /*------ DEBUG LOG END ------*/
    iom->status = iomux_conn_block;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,147,
            "conn->riom->status change: %d->%d",
            conn->riom->status, iomux_conn_sending);
    }
    /*------ DEBUG LOG END ------*/
    conn->riom->status = iomux_conn_sending;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,148,
            "out_function: static int l7vs_conn_connection_establish("
            "struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)"
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * block status:
 * when block status, arrived data turn next epoll reaction
 * and changes state to busy, expect the another iom is already busy
 *
 * @param[in]   iom         connection_iomux
 * @param[in]   another_iom another side's iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_block(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        char a_iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        l7vs_iomux_c_str(a_iom_str, another_iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,149,
            "in_function: static int l7vs_conn_block(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
            "iom=%s: "
            "another_iom=%s",
            iom_str, a_iom_str);
    }
    /*------ DEBUG LOG END ------*/

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,44, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,150,
                "out_function: static int l7vs_conn_block(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (!another_iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,45, "error / another_iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,151,
                "out_function: static int l7vs_conn_block(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_block != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,46, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,152,
                "out_function: static int l7vs_conn_block(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    // epoll re-registraion for read
    l7vs_iomux_mod(iom, iom_read);

    // when another iom status is busy, do not change my status
    if (iomux_conn_busy != another_iom->status &&
        iomux_conn_sending_busy != another_iom->status &&
        iomux_conn_sent_busy != another_iom->status) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,153,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_busy);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_busy;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,154,
            "out_function: static int l7vs_conn_block(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * busy status:
 * when busy status, arrived data turn next epoll reaction and status keep busy
 *
 * @param[in]   iom     connection_iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_busy(struct l7vs_iomux *iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,155,
            "in_function: static int l7vs_conn_busy(struct l7vs_iomux *iom) "
            "iom=%s",
            iom_str);
    }
    /*------ DEBUG LOG END ------*/

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,47, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,156,
                "out_function: static int l7vs_conn_busy(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_busy != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,48, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,157,
                "out_function: static int l7vs_conn_busy(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    //epoll re-registration for read
    // l7vs_iomux_mod(iom, iom_read);

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,158,
            "out_function: static int l7vs_conn_busy(struct l7vs_iomux *iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * sending status:
 * 
 * @param[in]   iom         connection_iomux
 * @param[in]   another_iom another side's iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        char a_iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        l7vs_iomux_c_str(a_iom_str, another_iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,159,
            "in_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
            "iom=%s: "
            "another_iom=%s",
            iom_str, a_iom_str);
    }
    /*------ DEBUG LOG END ------*/

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,49, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,160,
                "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0; 
    }

    if (!another_iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,50, "error / another_iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,161,
                "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0; 
    }

    if (iomux_conn_sending != iom->status && iomux_conn_sending_busy != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,51, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,162,
                "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    // send to handling iom
    if (!l7vs_conn_send(iom, iom->fd)) {
        iom->status = iomux_conn_send_error;
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,25, "error / send failed");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,163,
                "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    if (iomux_conn_sent == iom->status || iomux_conn_sent_busy == iom->status) {
        // when all data was sent, select the next state
        if (!l7vs_conn_sent_next_state(another_iom, iom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,26, "error / send next state failed");
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,164,
                    "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        if (iomux_conn_receiving != iom->status && iomux_conn_block != iom->status) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,52, "error / invalid status(%d)", iom->status);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,165,
                    "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
    }
    else if (iomux_conn_sending == iom->status || iomux_conn_sending_busy == iom->status) {
        // data still remain
        // epoll re-registration for write
        l7vs_iomux_mod(iom, iom_write);
    }
    else  {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,53, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,166,
                "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,167,
            "out_function: static int l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * decide sent next state
 *
 * @param[in,out]   src_iom     source connection_iomux
 * @param[in,out]   dest_iom    destionation connection_iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char s_iom_str[DEBUG_STR_LEN] = {0};
        char d_iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(s_iom_str, src_iom);
        l7vs_iomux_c_str(d_iom_str, dest_iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,168,
            "in_function: static int l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom) "
            "src_iom=%s: "
            "dest_iom=%s",
            s_iom_str, d_iom_str);
    }
    /*------ DEBUG LOG END ------*/

    if (!src_iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,54, "error / src_iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,169,
                "out_function: static int l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return  0;
    }
    if (!dest_iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,55, "error / dest_iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,170,
                "out_function: static int l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return  0;
    }

    // when writing, the dest_status will be "sent" or "sent_busy"
    if (iomux_conn_sent != dest_iom->status && iomux_conn_sent_busy != dest_iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,56, "error / invalid dest status(%d)", dest_iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,171,
                "out_function: static int l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    // and the src_status will be "block" or "busy"
    if (iomux_conn_block != src_iom->status && iomux_conn_busy != src_iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,57, "error / invalid src status(%d)", src_iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,172,
                "out_function: static int l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    // decide the next state
    if (iomux_conn_sent_busy == dest_iom->status) {
        // when receiving from source, request come to the destination, then destination has priority
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,173,
                "src_iom->status change: %d->%d",
                src_iom->status, iomux_conn_block);
        }
        /*------ DEBUG LOG END ------*/
        src_iom->status = iomux_conn_block;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,174,
                "dest_iom->status change: %d->%d",
                dest_iom->status, iomux_conn_receiving);
        }
        /*------ DEBUG LOG END ------*/
        dest_iom->status = iomux_conn_receiving;
    }
    else if (iomux_conn_busy == src_iom->status) {
        // when sending to destination, request come to the source, then source has priority
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,175,
                "src_iom->status change: %d->%d",
                src_iom->status, iomux_conn_receiving);
        }
        /*------ DEBUG LOG END ------*/
        src_iom->status = iomux_conn_receiving;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,176,
                "dest_iom->status change: %d->%d",
                dest_iom->status, iomux_conn_block);
        }
        /*------ DEBUG LOG END ------*/
        dest_iom->status = iomux_conn_block;
    }
    else {
        // request come to neither source nor destnation, ready to read both
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,177,
                "src_iom->status change: %d->%d",
                src_iom->status, iomux_conn_receiving);
        }
        /*------ DEBUG LOG END ------*/
        src_iom->status = iomux_conn_receiving;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,178,
                "dest_iom->status change: %d->%d",
                dest_iom->status, iomux_conn_receiving);
        }
        /*------ DEBUG LOG END ------*/
        dest_iom->status = iomux_conn_receiving;
    }
    l7vs_iomux_mod(src_iom, iom_read);
    l7vs_iomux_mod(dest_iom, iom_read);

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,179,
            "out_function: static int l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * realserver connection callback
 *
 * @param[in]   iom     connection iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_rs_callback(struct l7vs_iomux *iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,180,
            "in_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
            "iom=%s",
            iom_str);
    }
    /*------ DEBUG LOG END ------*/
    struct l7vs_conn* conn = NULL;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,58, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,181,
                "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    conn = (struct l7vs_conn *)iom->data;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,182,
            "pointer_assign: conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    // invalid status check
    if (iomux_conn_receiving != iom->status &&
        iomux_conn_block != iom->status &&
        iomux_conn_busy != iom->status && 
        iomux_conn_sending != iom->status && 
        iomux_conn_sending_busy != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,59, "error / invalid status(%d)", iom->status);
        l7vs_conn_destroy(conn);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,183,
                "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    // switch by current status
    switch (iom->status) {
    case iomux_conn_receiving:
        if (!l7vs_conn_realserver_receiving(iom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,27, "error / rs_callback/realserver receiving failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,184,
                    "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        if (iomux_conn_disconnected == iom->status) {
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,185, "realserver connection disconnected");
            }
            l7vs_conn_destroy(conn);
        }
        break;
    case iomux_conn_block:
        if (!l7vs_conn_block(iom, conn->ciom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,28, "error / rs_callback/block failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,186,
                    "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    case iomux_conn_busy:
        if (!l7vs_conn_busy(iom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,29, "error / rs_callback/busy failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,187,
                    "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    case iomux_conn_sending:
        if (!l7vs_conn_sending(iom, conn->ciom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,30, "error / rs_callback/sending failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,188,
                    "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    case iomux_conn_sending_busy:
        if (!l7vs_conn_sending(iom, conn->ciom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,31, "error / rs_callback/sending_busy failed");
            l7vs_conn_destroy(conn);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,189,
                    "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        break;
    default:
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,60, "error / invalid status(%d)", iom->status);
        l7vs_conn_destroy(conn);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,190,
                "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,191,
            "out_function: static int l7vs_conn_rs_callback(struct l7vs_iomux *iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * realserver receiving 
 *
 * @param[in]   iom     connection iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_realserver_receiving(struct l7vs_iomux *iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,192,
            "in_function: static int l7vs_conn_realserver_receiving(struct l7vs_iomux *iom) "
            "iom=%s",
            iom_str);
    }
    /*------ DEBUG LOG END ------*/

    struct l7vs_conn *conn = NULL;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,61, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,193,
                "out_function: static int l7vs_conn_realserver_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_receiving != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,62, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,194,
                "out_function: static int l7vs_conn_realserver_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    conn = (struct l7vs_conn *)iom->data;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,195,
            "pointer_assign: conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    // when the client is not set in read-blocked
    if (iomux_conn_block != conn->ciom->status && iomux_conn_busy != conn->ciom->status) {
        // set the real server in read-blocked
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,196,
                "conn->ciom->status change: %d->%d",
                conn->ciom->status, iomux_conn_block);
        }
        /*------ DEBUG LOG END ------*/
        conn->ciom->status = iomux_conn_block;
    }

    // receive realserver data
    if (!l7vs_conn_recv(iom, iom->fd)) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,32, "error / recv failed");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,197,
                "out_function: static int l7vs_conn_realserver_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    switch (iom->status) {
    case iomux_conn_receiving:
        // data still remain, epoll re-registration for read
        l7vs_iomux_mod(iom, iom_read);
        break;
    case iomux_conn_received:
        // all data is received, append the session infomation
        if (!l7vs_conn_realserver_modify_data(iom)) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,33, "error / realserver modify data failed");
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,198,
                    "out_function: static int l7vs_conn_realserver_receiving(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,199,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_block);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_block;
        if (iomux_conn_busy == conn->ciom->status) {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,200,
                    "conn->ciom->status change: %d->%d",
                    conn->ciom->status, iomux_conn_sending_busy);
            }
            /*------ DEBUG LOG END ------*/
            conn->ciom->status = iomux_conn_sending_busy;
        } else {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,201,
                    "conn->ciom->status change: %d->%d",
                    conn->ciom->status, iomux_conn_sending);
            }
            /*------ DEBUG LOG END ------*/
            conn->ciom->status = iomux_conn_sending;
        }
        //epoll re-registration for write
        l7vs_iomux_mod(conn->ciom, iom_write);
    case iomux_conn_disconnected:
        // client connection was disconnected
        break;
    default:
        //invalid status
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,63, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,202,
                "out_function: static int l7vs_conn_realserver_receiving(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,203,
            "out_function: static int l7vs_conn_realserver_receiving(struct l7vs_iomux *iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * realserver modify data
 *
 * @param[in]   iom     connection iomux
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,204,
            "in_function: static int l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom) "
            "iom=%s",
            iom_str);
    }
    /*------ DEBUG LOG END ------*/
    struct l7vs_conn *conn = NULL;
        char *newbuf = NULL;
    int (*relayf)(struct l7vs_service *, struct l7vs_conn *, char *, size_t *) = NULL;
    int ret = 0;
    size_t len_ret = 0;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,64, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,205,
                "out_function: static int l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_received != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,65, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,206,
                "out_function: static int l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    conn = (struct l7vs_conn *)iom->data;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,207,
            "pointer_assign: conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    //set the function pointer that modify the real server data
    if (conn->srv) {
        relayf = conn->srv->pm->analyze_rsdata;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,208,
                "pointer_assign: relayf=%p",
                relayf);
        }
        /*------ DEBUG LOG END ------*/
    }

    if (relayf) {
        len_ret = conn->cldata_len;
        newbuf = (char *)realloc(conn->cldata, (conn->cldata_len + L7VS_PROTOMOD_MAX_ADD_BUFSIZE + 1));
        if (!newbuf) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,10, "error / realloc failed");
            iom->status = iomux_conn_buffer_realloc_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,209,
                    "out_function: static int l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,210,
                "realloc: newbuf=%p: size=%zu",
                newbuf, (conn->cldata_len + L7VS_PROTOMOD_MAX_ADD_BUFSIZE + 1));
        }
        /*------ DEBUG LOG END ------*/
        conn->cldata = newbuf;

        ret = (*relayf)(conn->srv, conn, conn->cldata, &len_ret);
        if (0 != ret) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,34, "error / analyze rsdata failed");
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,211,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_rsdata_modify_error);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_rsdata_modify_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,212,
                    "out_function: static int l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        if (L7VS_PROTOMOD_MAX_ADD_BUFSIZE < len_ret - conn->cldata_len) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,35, "error / bufsize too long modified by protomod");
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,213,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_rsdata_modify_error);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_rsdata_modify_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,214,
                    "out_function: static int l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        else {
            conn->cldata_len = len_ret;
        }
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char modify_data[DEBUG_STR_LEN] = {0};
        l7vs_conn_dump_data(modify_data, conn->cldata, conn->cldata_len);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,215,
            "modify data: %s",
            modify_data);
    }
    /*------ DEBUG LOG END ------*/

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,216,
            "iom->status change: %d->%d",
            iom->status, iomux_conn_rsdata_modified);
    }
    /*------ DEBUG LOG END ------*/
    iom->status = iomux_conn_rsdata_modified;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,217,
            "out_function: static int l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * receive data
 *
 * @param[in]   iom         connection_iomux
 * @param[in]   source_fd   the filedescriptor to be read
 * @return      status
 * @retval      0       failed 
 * @retval      1       succeed
 */
static int
l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,218,
            "in_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
            "iom=%s: "
            "source_fd=%d",
            iom_str, source_fd);
    }
    /*------ DEBUG LOG END ------*/
        int ret = 0;
    struct l7vs_conn *conn = NULL;
    char read_buf[l7vs_conn_read_bufsize];
//        char *newbuf = NULL;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,66, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,219,
                "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (0 > source_fd || INT_MAX == source_fd) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,67, "error / source fd is not specified");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,220,
                "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_receiving != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,68, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,221,
                "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    conn = (struct l7vs_conn *)iom->data;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,222,
            "pointer_assign: conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    //QoS Control to server
    if( (source_fd == conn->ciom->fd) && !l7vs_conn_QoS_control( conn, QOS_UP ) ){
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,223,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_receiving);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_receiving;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,224,
                "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    }
    //QoS Control to client
    if( (source_fd == conn->riom->fd) && !l7vs_conn_QoS_control( conn, QOS_DOWN ) ){
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,346,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_receiving);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_receiving;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,347,
                "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    }
    // read the data
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,225,
            "recv param: source_fd=%d: len=%d: flags=%d",
            source_fd, (l7vs_conn_read_bufsize - 1), 0);
    }
    /*------ DEBUG LOG END ------*/
    ret = recv(source_fd, read_buf, (l7vs_conn_read_bufsize - 1), 0);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,226,
            "recv result: ret=%d",
            ret);
    }
    /*------ DEBUG LOG END ------*/
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char recv_data[DEBUG_STR_LEN] = {0};
        l7vs_conn_dump_data(recv_data, read_buf, ret);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,227,
            "recv data: %s",
            recv_data);
    }
    /*------ DEBUG LOG END ------*/
    switch (ret) {
    case -1:
        if (EAGAIN != errno) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,12, "error / recv error(%s)", strerror(errno));
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,228,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_receive_error);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_receive_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,229,
                    "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,230,
                "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    case 0:
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,231, "peer disconnected");
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,232,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_disconnected);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_disconnected;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,233,
                "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    default:
        // reading data is finished
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,239,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_received);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_received;
        break;
    }
    
    // register reveive information for QoS to server
    if( source_fd == conn->ciom->fd ){
        l7vs_conn_QoS_recvsize_register( conn, ret, QOS_UP );
    }
    // register reveive information for QoS to client
    if( source_fd == conn->riom->fd ){
        l7vs_conn_QoS_recvsize_register( conn, ret, QOS_DOWN );
    }
    // copy to conn_buffer
    memcpy(conn->cldata + conn->cldata_len, read_buf, ret);
    conn->cldata_len += ret;
    conn->cldata[conn->cldata_len] = '\0';

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,240,
            "out_function: static int l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * pre-read whether next data exists
 *
 * @param[in]   iom         connection iomux
 * @param[in]   source_fd   source file description
 * @return      status
 * @retval      0       failed
 * @retval      1       succeed
 */
static int
l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,241,
            "in_function: static int l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd) "
            "iom=%s: "
            "source_fd=%d",
            iom_str, source_fd);
    }
    /*------ DEBUG LOG END ------*/
    int preread_ret = 0;
    char preread_buf[1] = {0};

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,69, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,242,
                "out_function: static int l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (-1 == source_fd) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,70, "error / source fd is not specified");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,243,
                "out_function: static int l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_receiving != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,71, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,244,
                "out_function: static int l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,245,
            "recv param: source_fd=%d: len=1: flags=MSG_PEEK",
            source_fd);
    }
    /*------ DEBUG LOG END ------*/
    preread_ret = recv(source_fd, preread_buf, 1, MSG_PEEK);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,246,
            "recv result: preread_ret=%d",
            preread_ret);
    }
    /*------ DEBUG LOG END ------*/
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char recv_data[DEBUG_STR_LEN] = {0};
        l7vs_conn_dump_data(recv_data, preread_buf, preread_ret);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,247,
            "recv data: %s",
            recv_data);
    }
    /*------ DEBUG LOG END ------*/
    switch (preread_ret) {
    case -1:
        if (EAGAIN != errno) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,13, "error / recv error(%s)", strerror(errno));
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,248,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_receive_error);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_receive_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,249,
                    "out_function: static int l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        // finished reading all data
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,250,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_received);
        }
        /*------ DEBUG LOG END ------*/
        iom->status = iomux_conn_received;
        break;
    case 0:
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,251, "peer disconnected");
        }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,252,
                "iom->status change: %d->%d",
                iom->status, iomux_conn_disconnected);
        }
        /*------ DEBUG LOG END ------*/
        iom->status =  iomux_conn_disconnected;
        break;
    case 1:
        // the data that should be read remains
        // status does not change
        break;
    default:
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,37, "error / preread error");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,253,
                "out_function: static int l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,254,
            "out_function: static int l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * send data
 *
 * @param[in]   iom         connection_iomux
 * @param[in]   dest_fd     the filedescriptor to be write
 * @return      status
 * @retval      0       failed 
 * @retval      1       succeed
 */
static int
l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char iom_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iom_str, iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,255,
            "in_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
            "iom=%s: "
            "dest_fd=%d",
            iom_str, dest_fd);
    }
    /*------ DEBUG LOG END ------*/

    int ret = 0;
    struct l7vs_conn *conn = NULL;
    char *newbuf = NULL;
    size_t sent_len = 0;

    if (!iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,72, "error / iom is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,256,
                "out_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (-1 == dest_fd) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,73, "error / dest fd is not specified");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,257,
                "out_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    if (iomux_conn_sending != iom->status &&
        iomux_conn_sending_busy != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,74, "error / invalid status(%d)", iom->status);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,258,
                "out_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

    conn = (struct l7vs_conn *)iom->data;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,259,
            "pointer_assign: conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char send_data[DEBUG_STR_LEN] = {0};
        l7vs_conn_dump_data(send_data, conn->cldata, conn->cldata_len);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,260,
            "send data: %s",
            send_data);
    }
    /*------ DEBUG LOG END ------*/
    // write the data
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,261,
            "send param: dest_fd=%d: len=%zu: flags=MSG_NOSIGNAL",
            dest_fd, conn->cldata_len);
    }
    /*------ DEBUG LOG END ------*/
    ret = send(dest_fd, conn->cldata, conn->cldata_len, MSG_NOSIGNAL);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,262,
            "send result: ret=%d",
            ret);
    }
    /*------ DEBUG LOG END ------*/
    if (0 > ret) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,14, "error / send error(%s)", strerror(errno));
        iom->status = iomux_conn_send_error;
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,263,
                "out_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    sent_len = ret;
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,264, "sent %zu bytes", sent_len);
    }

    // decide the status
    if (conn->cldata_len == sent_len) {
        // finished sending all data
        if (iomux_conn_sending == iom->status) {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,265,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_sent);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_sent;
        } else {
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,266,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_sent_busy);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_sent_busy;
        }
        //cldata memory realloc
        conn->cldata_len = 0;
        conn->cldata_bufsize = l7vs_conn_read_bufsize;
        newbuf = (char *)malloc(conn->cldata_bufsize);
        if (!newbuf) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,12, "error / malloc failed");
            iom->status = iomux_conn_send_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,267,
                    "out_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        free(conn->cldata);
        conn->cldata = newbuf;
        memset(conn->cldata, 0, l7vs_conn_read_bufsize);
    }
    else {
        // shift data for next sending
        newbuf = (char *)malloc((conn->cldata_len - sent_len) + 1);
        if (!newbuf) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,13, "error / malloc failed");
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,268,
                    "iom->status change: %d->%d",
                    iom->status, iomux_conn_send_error);
            }
            /*------ DEBUG LOG END ------*/
            iom->status = iomux_conn_send_error;
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,269,
                    "out_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        memcpy(newbuf, (conn->cldata + sent_len), conn->cldata_len - sent_len);
        free(conn->cldata);
        conn->cldata = newbuf;
        conn->cldata_len -= sent_len;
        conn->cldata[conn->cldata_len] = '\0';

        // the data that should be sent remains, status does not change
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,270,
            "out_function: static int l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;

}

/*!
 * connect to real server
 *
 * @param[in]   conn    connection
 * @param[in]   dest    destination
 * @return      status
 * @retval      0       failed 
 * @retval      1       succeed
 */
int
l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        char dest_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        l7vs_dest_c_str(dest_str, dest);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,271,
            "in_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
            "conn=%s: "
            "dest=%s",
            conn_str, dest_str);
    }
    /*------ DEBUG LOG END ------*/

        int s;
        int stype;
        int flags;
        int ret;
    enum iomaction action;

    if (!conn) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,75, "error / conn is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,272,
                "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    if (!dest) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,76, "error / dest is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,273,
                "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }

        switch (conn->proto) {
        case IPPROTO_TCP:
                stype = SOCK_STREAM;
                break;
        case IPPROTO_UDP:
                stype = SOCK_DGRAM;
                break;
        default:
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,15, "error / unknown socket type");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,274,
                "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
                return 0;
        }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,275,
            "socket param: family=PF_INET: type=%d: protocol=%d",
            stype, conn->proto);
    }
    /*------ DEBUG LOG END ------*/
        s = socket(PF_INET, stype, conn->proto);  
        if (0 > s) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,16, "error / socket error(%s)", strerror(errno));
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,276,
                "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
                return 0;
        }
        
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,277,
            "fnctl param: s=%d: cmd=F_GETFL: flags=0",
            s);
    }
    /*------ DEBUG LOG END ------*/
        flags = fcntl(s, F_GETFL, 0);
        if (0 > flags) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,17, "error / fcntl(s=%d,F_GETFL) error(%s)", s,strerror(errno));
                close(s);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,278,
                "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
                return 0;
        }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,279,
            "fnctl result: flags=%d",
            flags);
    }
    /*------ DEBUG LOG END ------*/

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,280,
            "fnctl param: s=%d: cmd=F_SETFL: flags=%d",
            s, (flags | O_NONBLOCK));
    }
    /*------ DEBUG LOG END ------*/
        flags = fcntl(s, F_SETFL, flags | O_NONBLOCK);
        if (0 > flags) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,18, "error / fcntl(s=%d,F_SETFL) error(%s)", s,strerror(errno));
                close(s);
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,281,
                "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
                return 0;
        }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,282,
            "fnctl result: flags=%d",
            flags);
    }
    /*------ DEBUG LOG END ------*/

        if (conn->cmss > 0) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,283,
                "setsockopt param: s=%d: level=IPROTO_TCP: optname=TCP_MAXSEG: optval=%d: "
                "len=%zu",
                s, conn->cmss, sizeof(int));
        }
        /*------ DEBUG LOG END ------*/
                ret = setsockopt(s, IPPROTO_TCP, TCP_MAXSEG, &conn->cmss,
                                 sizeof(int));
                if (ret < 0) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,19, "error / setsockopt(TCP_MAXSEG) error(%s)", strerror(errno));
                }
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,284,
                "setsockopt result: ret=%d: len=%zu",
                ret, sizeof(int));
        }
        /*------ DEBUG LOG END ------*/
        }

#ifdef IP_TRANSPARENT
        if (dest->forward == L7VS_DEST_FORWARD_TPROXY) {
                int value = 1;
                ret = setsockopt(s, SOL_IP, IP_TRANSPARENT, &value, sizeof(int));
                if (ret != 0) {
                        LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_SYSTEM_SOCKET,/*XXX*/999, "setsockopt(IP_TRANSPARENT) not supported on this platform.");
                } else {
                        ret = bind(s, (struct sockaddr*) &conn->caddr, sizeof(struct sockaddr));
                        if (ret != 0) {
                                LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_SYSTEM_SOCKET,/*XXX*/999, "Cannot bind client address.");
                        }
                }
        }
#endif
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char addr_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_sockaddr_in_c_str(addr_str, &dest->addr);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,285,
            "connect param: s=%d: dest->addr=%s: len=%zu",
            s, addr_str, sizeof(struct sockaddr_in));
    }
    /*------ DEBUG LOG END ------*/
        ret = connect(s, (struct sockaddr *)&dest->addr,
                      sizeof(struct sockaddr_in)); 
        if (0 > ret) {
                if (errno != EINPROGRESS) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,20, "error / connect error(%s)", strerror(errno));
                        close(s);
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,286,
                    "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
                        return 0;
                }
    }
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,287,
            "connect result: ret=%d",
            ret);
    }
    /*------ DEBUG LOG END ------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,288, "dest %p nactive %d->%d",
            dest, dest->nactive,
            dest->nactive + 1);
    }

    action = iom_write;
    dest->nactive++;   
    conn->dest = dest;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char dest_str[DEBUG_STR_LEN] = {0};
        l7vs_dest_c_str(dest_str, conn->dest);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,289,
            "pointer_assign: conn->dest=%s",
            dest_str);
    }
    /*------ DEBUG LOG END ------*/
    conn->riom->fd = s;
    conn->riom->callback = l7vs_conn_rs_callback;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,290,
            "pointer_assign: conn->riom->callback=%p",
            conn->riom->callback);
    }
    /*------ DEBUG LOG END ------*/
    conn->riom->data = conn;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,291,
            "pointer_assign: conn->riom->data=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/
    l7vs_iomux_add(conn->riom, action);

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,292,
            "out_function: int l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
            "return_value: 1");
    }

    if ( dest ){
                /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,341,
                "RealServer Connecting Count : nactive is %d",
                dest->nactive);
        }
        /*-------- DEBUG LOG --------*/

        /*-------- INFO LOG (Access Log) --------*/
        if (LOG_LV_INFO >= logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            char addr_str_tmp[ADDR_STR_LEN] = {0};
            memcpy(addr_str_tmp, inet_ntoa(conn->caddr.sin_addr), ADDR_STR_LEN);
            LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_NETWORK,342,
                "[[AccessLog] (SRC)%s:%d -> (DST)%s:%d ]",
                addr_str_tmp,
                ntohs(conn->caddr.sin_port),
                inet_ntoa(conn->dest->addr.sin_addr),
                ntohs(conn->dest->addr.sin_port));
        }
        /*------ INFO LOG END (Access Log) ------*/
    }

    /*------ DEBUG LOG END ------*/
        return 1;
}

/*!
 * close client socket 
 *
 * @param[in]   conn    connection
 * @return      void 
 */
void
l7vs_conn_close_csock(struct l7vs_conn *conn)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,293,
            "in_function: void l7vs_conn_close_csock(struct l7vs_conn *conn) "
            "conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    if (!conn) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,294,
                "out_function: void l7vs_conn_close_csock(struct l7vs_conn *conn) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    if (!conn->ciom) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,295,
                "out_function: void l7vs_conn_close_csock(struct l7vs_conn *conn) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    if (-1 == conn->ciom->fd) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,296,
                "out_function: void l7vs_conn_close_csock(struct l7vs_conn *conn) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }

    l7vs_iomux_remove(conn->ciom);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,297,
            "close: conn->ciom->fd=%d",
            conn->ciom->fd);
    }
    /*------ DEBUG LOG END ------*/
    close(conn->ciom->fd);
    conn->ciom->fd = -1;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,298,
            "out_function: void l7vs_conn_close_csock(struct l7vs_conn *conn) "
            "return_value: void");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * close real server socket
 *
 * @param[in]   conn    connection
 * @return      void 
 */
void
l7vs_conn_close_rsock(struct l7vs_conn *conn)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,299,
            "in_function: void l7vs_conn_close_rsock(struct l7vs_conn *conn) "
            "conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    if (!conn) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,300,
                "out_function: void l7vs_conn_close_rsock(struct l7vs_conn *conn) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    if (!conn->riom) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,301,
                "out_function: void l7vs_conn_close_rsock(struct l7vs_conn *conn) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    if (-1 == conn->riom->fd) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,302,
                "out_function: void l7vs_conn_close_rsock(struct l7vs_conn *conn) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }

    if (conn->dest) {
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,303, "dest %p nactive %d->%d",
                conn->dest, conn->dest->nactive,
                conn->dest->nactive - 1);
        }
        conn->dest->nactive--;
        conn->dest->ninact++;
        if(conn->dest->ninact == INT_MAX) {
            conn->dest->ninact = 0;
        }

        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,343,
                "RealServer Connected Count : ninact is %d",
                conn->dest->ninact);
        }
        /*------ DEBUG LOG END ------*/
    }

    l7vs_iomux_remove(conn->riom);
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,304,
            "close: conn->riom->fd=%d",
            conn->riom->fd);
    }
    /*------ DEBUG LOG END ------*/
    close(conn->riom->fd);
    conn->riom->fd = -1;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,305,
            "out_function: void l7vs_conn_close_rsock(struct l7vs_conn *conn) "
            "return_value: void");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * confirmation of whether conn is closed
 *
 * @param[in]   conn    connection
 * @return      closed or not
 * @retval      0       not closed 
 * @retval      1       closed
 */
int
l7vs_conn_closed(struct l7vs_conn *conn)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,306,
            "in_function: int l7vs_conn_closed(struct l7vs_conn *conn) "
            "conn=%s",
            conn_str);
    }
    /*------ DEBUG LOG END ------*/

    if (!conn) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,307,
                "out_function: int l7vs_conn_closed(struct l7vs_conn *conn) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    }
    if ((conn->ciom && -1 != conn->ciom->fd) || (conn->riom && -1 != conn->riom->fd)) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,308,
                "out_function: int l7vs_conn_closed(struct l7vs_conn *conn) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    } else {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,309,
                "out_function: int l7vs_conn_closed(struct l7vs_conn *conn) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    }
}

/*!
 * Change connection to specified destination (to real-server or sorry-server).
 *
 * @param[in]   *conn   connection pointer
 * @param[in]   *dest   destination pointer
 * @return      int     change connection result 0=NG, 1=OK
 */
static int l7vs_conn_change_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        char dest_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        l7vs_dest_c_str(dest_str, dest);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,310,
            "in_function: static int l7vs_conn_change_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
            "conn=%s: "
            "dest=%s",
            conn_str, dest_str);
    }
    /*------ DEBUG LOG END ------*/

    int ret;

    if (!conn) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,77, "error / conn is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,311,
                "out_function: static int l7vs_conn_change_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    if (!dest) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_PROGRAM,78, "error / dest is null");
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,312,
                "out_function: static int l7vs_conn_change_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
                "return_value: 0");
        }
        /*------ DEBUG LOG END ------*/
        return 0;
    }
    // save old destination
    conn->old_dest = conn->dest;
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char old_dest_str[DEBUG_STR_LEN] = {0};
        l7vs_dest_c_str(old_dest_str, conn->old_dest);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,313,
            "pointer_assign: conn->old_dest=%s",
            old_dest_str);
    }
    /*------ DEBUG LOG END ------*/
    // close real-server connection
    l7vs_conn_close_rsock(conn);
    // connect to new destination
    ret = l7vs_conn_connect_rs(conn, dest);

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,314,
            "out_function: static int l7vs_conn_change_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest) "
            "return_value: %d",
            ret);
    }
    /*------ DEBUG LOG END ------*/
    return ret;
}

/*!
 * Judges from the data size received last time and recieve-interval
 *
 * @param[in]   *conn       connection pointer
 * @param[in]   direction   traffic direction
 * @return      int         permitte for recv function call 0=NG(not permitted), 1=OK(permitted)
 */
static int
l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction )
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,315,
            "in_function: static int l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction ) "
            "conn=%s, direction=%d",
            conn_str, direction);
    }
    /*------ DEBUG LOG END ------*/

    if (!conn) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,316,
                "out_function: static int l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction ) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    }
    if (!conn->srv) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,348,
                "out_function: static int l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction ) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    }

    struct timeval CurrTime;
    gettimeofday( &CurrTime, NULL );
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char time_str[DEBUG_STR_LEN] = {0};
        snprintf(time_str, DEBUG_STR_LEN,
            "timeval="
            "{tv_sec=%ld: "
            "tv_usec=%ld}"
            , CurrTime.tv_sec
            , CurrTime.tv_usec);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,339,
            "gettimeofday result: CurrTime=%s",
            time_str);
    }
    /*------ DEBUG LOG END ------*/

    unsigned long long cur_recvtime = (CurrTime.tv_sec * 1000000ULL + CurrTime.tv_usec) / throughput_interval;
    l7vs_service_update_throughput(conn->srv, cur_recvtime);

    switch (direction) {
    case QOS_UP:
        //QoS Control check : to server
        conn->srv->throughput_to_server = conn->srv->cur_recvsize_from_client * 1000000ULL / throughput_interval;
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,323, "QoS threshold to server:%lld ", conn->srv->qos_threshold_up);
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,324, "(current throughput:%lld)", conn->srv->throughput_to_server);
        }
        if( (conn->srv->qos_threshold_up) && (conn->srv->throughput_to_server + l7vs_conn_read_bufsize > conn->srv->qos_threshold_up) ){
            if (conn->srv->qos_up_flag == 0) {
                LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_NETWORK_QOS,1, "Start QoS traffic control to server.");
                conn->srv->qos_up_flag = 1;
            }
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,326,
                    "out_function: static int l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction ) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        if (conn->srv->qos_up_flag == 1) {
            LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_NETWORK_QOS,2, "Stop QoS traffic control to server.");
            conn->srv->qos_up_flag = 0;
        }
        break;
    case QOS_DOWN:
        //QoS Control check : to client
        conn->srv->throughput_to_client = conn->srv->cur_recvsize_from_server * 1000000ULL / throughput_interval;
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,349, "QoS threshold to client:%lld ", conn->srv->qos_threshold_down);
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,350, "(current throughput:%lld)", conn->srv->throughput_to_client);
        }
        if( (conn->srv->qos_threshold_down) && (conn->srv->throughput_to_client + l7vs_conn_read_bufsize > conn->srv->qos_threshold_down) ){
            if (conn->srv->qos_down_flag == 0) {
                LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_NETWORK_QOS,3, "Start QoS traffic control to client.");
                conn->srv->qos_down_flag = 1;
            }
            /*-------- DEBUG LOG --------*/
            if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,351,
                    "out_function: static int l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction ) "
                    "return_value: 0");
            }
            /*------ DEBUG LOG END ------*/
            return 0;
        }
        if (conn->srv->qos_down_flag == 1) {
            LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_NETWORK_QOS,4, "Stop QoS traffic control to client.");
            conn->srv->qos_down_flag = 0;
        }
        break;
    default:
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,317,
                "out_function: static int l7vs_conn_QoS_control( struct l7vs_conn *conn, int direction ) "
                "return_value: 1");
        }
        /*------ DEBUG LOG END ------*/
        return 1;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,335,
            "out_function: static int l7vs_conn_QoS_control( struct l7vs_conn *conn ) "
            "return_value: 1");
    }
    /*------ DEBUG LOG END ------*/
    return 1;
}

/*!
 * Regsiter recive time and recieved data size
 *
 * @param[in]   *conn       connection pointer
 * @param[in]   in_recvsize recieved size
 * @param[in]   direction   traffic direction
 * @return      void 
 */
static void
l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize, int direction )
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char conn_str[DEBUG_STR_LEN] = {0};
        l7vs_conn_c_str(conn_str, conn);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,336,
            "in_function: static void l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize ) "
            "conn=%s: "
            "in_recvsize=%zu",
            conn_str, in_recvsize);
    }
    /*------ DEBUG LOG END ------*/

    if (!conn) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,337,
                "out_function: static void l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize ) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    if (!conn->srv) {
        /*-------- DEBUG LOG --------*/
        if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,338,
                "out_function: static void l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize ) "
                "return_value: void");
        }
        /*------ DEBUG LOG END ------*/
        return;
    }
    struct timeval CurrTime;
    gettimeofday( &CurrTime, NULL );
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        char time_str[DEBUG_STR_LEN] = {0};
        snprintf(time_str, DEBUG_STR_LEN,
            "timeval="
            "{tv_sec=%ld: "
            "tv_usec=%ld}"
            , CurrTime.tv_sec
            , CurrTime.tv_usec);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,352,
            "gettimeofday result: CurrTime=%s",
            time_str);
    }
    /*------ DEBUG LOG END ------*/

    int throughput_interval = BPS_DEFAULT_INTERVAL;
    if ( parameter_is_int_exist( PARAM_COMP_L7VSD, "calc_throughput_interval" ) ) {
        throughput_interval = parameter_get_int_value( PARAM_COMP_L7VSD, "calc_throughput_interval" );
    }
    unsigned long long cur_recvtime = (CurrTime.tv_sec * 1000000ULL + CurrTime.tv_usec) / throughput_interval;
    l7vs_service_update_throughput(conn->srv, cur_recvtime);

    switch (direction) {
    case QOS_UP:
        conn->srv->cur_recvsize_from_client += in_recvsize;
        break;
    case QOS_DOWN:
        conn->srv->cur_recvsize_from_server += in_recvsize;
        break;
    default:
        break;
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_NETWORK)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,340,
            "out_function: static void l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize ) "
            "return_value: void");
    }
    /*------ DEBUG LOG END ------*/
}

