/**
 * @file uzume_callback.c
 * @brief オーディオ信号処理を行うユーザー・コールバック
 */
#include "t_services.h"
#include "s_services.h"
#include "kernel_id.h"
#include "uzume.h"
#include "i2s_subsystem.h"

/**
 * @brief 信号処理初期化関数
 * @details
 * この関数はUZUMEプラットフォームの初期化を行う。
 * フィルタの構築や変数の初期化などを実行すると良い。
 *
 * @ingroup AUDIO
 */
void init_audio(void)
{
    /* ブロックサイズは UZUME_BLOCKSIZE マクロを使用すればわかる */
}

/**
 * @brief 信号処理関数
 * @param input_left 左チャンネル0入力配列
 * @param input_right 右チャンネル0入力配列
 * @param output_left 左チャンネル0出力配列
 * @param output_right 右チャンネル0出力配列
 * @ingroup AUDIO
 *
 * ユーザーが信号処理を行う関数。オーディオ信号に同期して呼び出される。
 * それぞれの引数は引数countがあらわす個数だけデータを持っている。
 * データは24bit符号付整数。32bit変数に左詰で渡される。
 *
 * この関数はステレオ信号を２系統あつかえる。すなわち、系統０と系統1がそれぞれ左右チャンネルを持つ。
 * この関数が使う資源で、あらかじめ初期化が必要なものは、@ref init_audio()の中で初期化しておく。
 */
void process_audio(
        AUDIOSAMPLE input_left[],
        AUDIOSAMPLE input_right[],
        AUDIOSAMPLE output_left[],
        AUDIOSAMPLE output_right[]
        )
{
    // ループ変数
    fract32 sample;
    fract32 volume;

    // ch 0 のボリューム値(32bit)
    volume = get_volume(0) >> 15;

    // count個のサンプルすべてを処理する
    for (sample = 0; sample < UZUME_BLOCKSIZE; sample++) {
        output_left[sample]  = (input_left[sample]  >> 16) * volume;
        output_right[sample] = (input_right[sample] >> 16) * volume;
    }
}

