/*
  compatible functions

  Copyright (C) 2005 Video maid project. (http://vmaid.sourceforge.jp)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/* ja: 忠実な互換性はありません。 */

#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include "compatible.h"


#ifdef __cplusplus
extern "C" {
#endif


#define NATIVE_CODEC_DIR ".vmaid_codec"  /* XXX: */
#define FILE_INIT_POSTFIX "-2.1.1"  /* XXX: */


/* 
 * huffyuv.cpp
 */

static gchar *
_resolve_filename(const gchar * name)
{
  gchar * ret = NULL;

  if ( '/' == name[0] ) {
    ret = g_strdup(name);
  }
  else {
    ret = g_build_filename(g_get_home_dir(),
                           g_strconcat(".", name, FILE_INIT_POSTFIX, NULL),
                           NULL);  /* XXX: */
#if 0  /* XXX: */
    if ( TRUE == g_file_test(ret, G_FILE_TEST_EXISTS) ) {
    }
#if defined(__FreeBSD__)
    ret = g_build_filename("/usr/local/etc", name, NULL);
#endif
#if defined(__FreeBSD__) || defined(__linux__)
    ret = g_build_filename("/etc", name, NULL);
#endif
#endif /* 0 */
  }

  return ret;
}


UINT
compat_GetPrivateProfileInt(LPCTSTR lpAppName, LPCTSTR lpKeyName,
                            INT nDefault, LPCTSTR lpFileName)
{
  void _root_key_destroy_func(gpointer data){ g_free(data); }
  void _root_value_destroy_func(gpointer data){ g_hash_table_destroy(data); }
  void _key_key_destroy_func(gpointer data){ g_free(data); }

  gchar *name;
  GIOChannel* ch = NULL;
  GError *err = NULL;
  gchar *buf;
  GHashTable *root;
  UINT ret = nDefault;

  if ( NULL == lpAppName || NULL == lpKeyName || NULL == lpFileName ) {
    return ret;
  }

  name = _resolve_filename((gchar*)lpFileName);

  ch = g_io_channel_new_file(name, "r", &err);

  if ( NULL == ch ) {
    g_free(name);
    return ret;
  }

  root = g_hash_table_new_full(g_str_hash, g_str_equal,
                               _root_key_destroy_func, _root_value_destroy_func);

  while ( G_IO_STATUS_NORMAL ==
          g_io_channel_read_line(ch, &buf, NULL, NULL, &err) ) {

    static gchar *sec = NULL;
    gsize len;

    g_strstrip(buf);
    len = strlen(buf);

    if (0 >= len) {
    }
    else if ( '[' == buf[0]  &&  ']' == buf[len - 1] ) {

      sec = g_strndup(buf + 1, len -2);
      g_hash_table_insert(root, sec,
                          g_hash_table_new_full(g_str_hash, g_str_equal,
                                                _key_key_destroy_func, NULL));
    }
    else if ( NULL != sec ) {

      GHashTable *t;
      gchar *key, *value;

      key = g_strdup(buf);
      value = strchr(key, '=');
      *value = '\0';
      value++;

      if ( NULL != (t = g_hash_table_lookup(root, sec)) ) {
        g_hash_table_insert(t, key, value);
      }
    }
    g_free(buf);
  }

  {
    GHashTable *t;
    gpointer p;

    if ( NULL != (t = g_hash_table_lookup(root, lpAppName)) ) {
      if ( NULL != (p = g_hash_table_lookup(t , lpKeyName)) ) {
        ret = atoi(p);  /* XXX: */
      }
    }
  }

  g_hash_table_destroy(root);
  g_io_channel_unref(ch);
  g_free(name);

  return ret;
}


BOOL
compat_WritePrivateProfileString(LPCTSTR lpAppName, LPCTSTR lpKeyName,
                                 LPCTSTR lpString, LPCTSTR lpFileName)  /* XXX: */
{
  struct _pack {
    GList *list;
    GHashTable *table;
  };

  void _root_key_destroy_func(gpointer data){ g_free(data); }
  void _root_value_destroy_func(gpointer data) {
    g_hash_table_destroy(((struct _pack *)data)->table);
    g_free(data);
  }
  void _key_key_destroy_func(gpointer data){ g_free(data); }

  gchar *name;
  GIOChannel* ch;
  GError *err = NULL;
  GHashTable *root = NULL;
  GList *linebuf = NULL;

  if ( NULL == lpAppName || NULL == lpFileName ) {
    return FALSE;
  }

  if ( NULL == lpKeyName ) {
    /* XXX: delete section */
  }
  if ( NULL == lpString ) {
    /* XXX: delete key */
  }

  name = _resolve_filename((gchar*)lpFileName);

  root = g_hash_table_new_full(g_str_hash, g_int_equal,
                               _root_key_destroy_func,
                               _root_value_destroy_func);

  ch = g_io_channel_new_file(name, "r", &err);

  if ( NULL != ch ) {

    gchar *buf;

    while ( G_IO_STATUS_NORMAL ==
            g_io_channel_read_line(ch, &buf, NULL, NULL, &err) ) {

      static gchar *sec = NULL;
      gsize len;

      linebuf = g_list_append(linebuf, g_strdup(buf));

      g_strstrip(buf);
      len = strlen(buf);

      if ( 0 >= len ) {
      }
      else if ( '[' == buf[0]  &&  ']' == buf[len - 1] ) {

        struct _pack *pack = g_new(struct _pack, 1);

        sec = g_strndup(buf + 1, len -2);

        pack->list = g_list_last(linebuf);
        pack->table = g_hash_table_new_full(g_str_hash, g_int_equal,
                                            _key_key_destroy_func, NULL);

        g_hash_table_insert(root, sec, pack);
      }
      else if ( NULL != sec ) {

        struct _pack *pack;
        gchar *key;

        key = g_strdup(buf);
        *strchr(key, '=') = '\0';
        g_strstrip(buf);

        if ( NULL != (pack = g_hash_table_lookup(root, sec)) ) {
          g_hash_table_insert(pack->table, key, g_list_last(linebuf));
        }
      }
      g_free(buf);
    }
  }
  g_io_channel_unref(ch);

#if 1
  {
    struct _pack *pack;
    GList *list;

    if ( NULL == (pack = g_hash_table_lookup(root, lpAppName)) ) {
      linebuf = g_list_append(linebuf,
                              g_strconcat("[", lpAppName, "]\n", NULL));
      linebuf = g_list_append(linebuf,
                              g_strconcat(lpKeyName, "=", lpString, "\n", NULL));
    }
    else if ( NULL == (list = g_hash_table_lookup(pack->table, lpKeyName)) ) {
      linebuf =
        g_list_insert_before(linebuf, g_list_next(pack->list),
                             g_strconcat(lpKeyName, "=", lpString, "\n", NULL));
    }
    else {  /* XXX: */

      gchar *s;

      *(strchr(list->data, '=') + 1) = '\0';
      s = g_strconcat(list->data, lpString, "\n", NULL);
      g_free(list->data);
      list->data = s;
    }
  }
#endif

  /* XXX: flock */

#if 1
  ch = NULL;
  err = NULL;
  ch = g_io_channel_new_file(name, "w+", &err);

  if ( NULL != ch ) {
    void __write(gpointer data, gpointer user_data) {
      g_io_channel_write_chars(ch, data, -1, NULL, &err);
    }
    g_list_foreach(linebuf, __write, NULL);   
  }
  else {
    g_warning("HFYU CODEC: Can't open file '%s': msg(%s)", name, err->message);
    return FALSE;
  }

  g_io_channel_unref(ch);
#endif

  {
    void _list_remove_func(gpointer data, gpointer user_data) { g_free(data); }
    g_list_foreach(linebuf, _list_remove_func, NULL);
  }

  g_hash_table_destroy(root);
  g_free(name);

  return TRUE;
}


/* */

void
compat_About(void)
{
  gchar *name;
  GladeXML *xml;
  GtkWidget *about;

  name = g_build_filename(g_get_home_dir(),
                          NATIVE_CODEC_DIR, "huffyuv.glade", NULL);  /* XXX: */

  if ( NULL == (xml = glade_xml_new(name, "aboutDialog", NULL)) ) {
    g_warning("HFYU CODEC: NULL == glade_xml_new(aboutDialog)");
  }

  glade_xml_signal_autoconnect(xml);

  if ( NULL == (about = glade_xml_get_widget(xml, "aboutDialog")) ) {
    g_warning("HFYU CODEC: NULL == glade_xml_get_widget(aboutDialog)");
  }

  if( GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(about)) ) {
    g_message("HFYU CODEC: GTK_RESPONSE_OK == gtk_dialog_run(about)");
  }
}


void
compat_Configure(struct MethodName *yuv_method_names,
                 struct MethodName *rgb_method_names)
{
  gchar *name;
  GladeXML *xml;
  GtkWidget *config, *button;
  GtkWidget *widgetHBox, *widgetYUY2Method, *widgetRGBMethod;

  name = g_build_filename(g_get_home_dir(),
                          NATIVE_CODEC_DIR, "huffyuv.glade", NULL);  /* XXX: */

  if ( NULL == (xml = glade_xml_new(name, "configureDialog", NULL)) ) {
    g_warning("HFYU CODEC: NULL == glade_xml_new(configureDialog)");
  }

  glade_xml_signal_autoconnect(xml);

  if ( NULL == (config = glade_xml_get_widget(xml, "configureDialog")) ) {
    g_warning("HFYU CODEC: NULL == glade_xml_get_widget(configureDialog)");
  }

  if ( NULL == (widgetHBox = glade_xml_get_widget(xml, "widgetHBox")) ) {
    g_warning("HFYU CODEC: NULL == glade_xml_get_widget(widgetHBox)");
  }


  widgetYUY2Method = gtk_combo_box_new_text();
  gtk_widget_show(widgetYUY2Method);
  {
    int i;
    UINT yuy2method = 
      GetPrivateProfileInt("general", "yuy2method", methodMedian, "huffyuv.ini");

    for (i=0; NULL != yuv_method_names[i].name; i++) {
      gtk_combo_box_append_text(GTK_COMBO_BOX(widgetYUY2Method),
                                (gchar*)yuv_method_names[i].name);
      if ( yuv_method_names[i].method == yuy2method ) {
        gtk_combo_box_set_active(GTK_COMBO_BOX(widgetYUY2Method), i);
      }
    }
  }
  gtk_box_pack_start(GTK_BOX(widgetHBox), widgetYUY2Method, TRUE, TRUE, 0);


  widgetRGBMethod = gtk_combo_box_new_text();
  gtk_widget_show(widgetRGBMethod);
  {
    int i;
    UINT rgbmethod =
      GetPrivateProfileInt("general", "rgbmethod",
                           methodGrad+flagDecorrelate, "huffyuv.ini");

    for (i=0; NULL != rgb_method_names[i].name; i++) {
      gtk_combo_box_append_text(GTK_COMBO_BOX(widgetRGBMethod),
                                (gchar*)rgb_method_names[i].name);
      if ( rgb_method_names[i].method == rgbmethod ) {
        gtk_combo_box_set_active(GTK_COMBO_BOX(widgetRGBMethod), i);
      }
    }
  }
  gtk_box_pack_start(GTK_BOX(widgetHBox), widgetRGBMethod, TRUE, TRUE, 0);


  button = glade_xml_get_widget(xml, "widgetRGBoutput");

  if ( NULL != button ) {
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON(button),
       GetPrivateProfileInt("debug", "rgboutput", FALSE, "huffyuv.ini"));
  }
  else {
    g_warning("HFYU CODEC: NULL == glade_xml_get_widget(widgetRGBoutput)");
  }

  button = glade_xml_get_widget(xml, "widgetEnableRGBA");

  if ( NULL != button ) {
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON(button),
       GetPrivateProfileInt("general", "enable_rgba", FALSE, "huffyuv.ini"));
  }
  else {
    g_warning("HFYU CODEC: NULL == glade_xml_get_widget(widgetEnableRGBA)");
  }

  button = glade_xml_get_widget(xml, "widgetDecompSwapFields");

  if ( NULL != button ) {
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON(button),
       GetPrivateProfileInt("debug", "decomp_swap_fields", FALSE, "huffyuv.ini"));
  }
  else {
    g_warning("HFYU CODEC: NULL == glade_xml_get_widget(widgetDecompSwapFields)");
  }

  button = glade_xml_get_widget(xml, "widgetLog");

  if ( NULL != button ) {
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON(button),
       GetPrivateProfileInt("debug", "log", FALSE, "huffyuv.ini"));
  }
  else {
    g_warning("HFYU CODEC: NULL == glade_xml_get_widget(widgetLog)");
  }


  if( GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(config)) ) {
    gchar methodstring[4];
    gint index;

    index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgetYUY2Method));
    g_sprintf(methodstring, "%d", yuv_method_names[index].method);
    WritePrivateProfileString("general", "yuy2method",
                              methodstring, "huffyuv.ini");
    index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgetRGBMethod));
    g_sprintf(methodstring, "%d", rgb_method_names[index].method);
    WritePrivateProfileString("general", "rgbmethod",
                              methodstring,"huffyuv.ini");

#define _CHECKED(n) gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(n)) ? "1" : "0"

    button = glade_xml_get_widget(xml, "widgetRGBoutput");

    if ( NULL != button ) {
      WritePrivateProfileString("debug", "rgboutput",
                                _CHECKED(button), "huffyuv.ini");
    }

    button = glade_xml_get_widget(xml, "widgetEnableRGBA");

    if ( NULL != button ) {
      WritePrivateProfileString("general", "enable_rgba",
                                _CHECKED(button), "huffyuv.ini");
    }

    button = glade_xml_get_widget(xml, "widgetDecompSwapFields");

    if ( NULL != button ) {
      WritePrivateProfileString("debug", "decomp_swap_fields",
                                _CHECKED(button), "huffyuv.ini");
    }

    button = glade_xml_get_widget(xml, "widgetLog");

    if ( NULL != button ) {
      WritePrivateProfileString("debug", "log", _CHECKED(button), "huffyuv.ini");
    }

    gtk_widget_destroy(config);

    g_message("HFYU CODEC: GTK_RESPONSE_OK == gtk_dialog_run(config)");
  }
}


/* */

void
compat_wvsprintf(LPTSTR lpOutput, LPCTSTR lpFmt, va_list arglist)
{
  g_vsprintf((gchar*)lpOutput, (gchar*)lpFmt, arglist);
}


void
compat_WriteConsole(const VOID* lpBuffer)
{
  g_print((gchar*)lpBuffer);
}


void
compat_MultiByteToWideChar(LPWSTR dst, LPCSTR src)
{
  g_strlcpy((gchar*)dst,
            g_locale_to_utf8((gchar*)src, -1, NULL, NULL, NULL),
            128);
}


int
compat_wsprintfA(LPTSTR lpOut, LPCTSTR lpFmt, ...)
{
  gint ret;

  va_list args;
  va_start (args, lpFmt);

  ret = g_vsprintf((gchar*)lpOut, (gchar*)lpFmt, args);

  va_end (args);

  return ret;
}


int
compat_lstrlen(LPCTSTR lpString)
{
  return strlen(lpString);
}


LPTSTR
compat_lstrcpy(LPTSTR lpString1, LPCTSTR lpString2)
{
  return (LPTSTR)strcpy((char*)lpString1, (char*)lpString2);
}


/*
 * drvproc.cpp
 */

LRESULT
compat_DefDriverProc(DWORD dwDriverId, HDRVR hdrvr,
                     UINT msg, LPARAM lParam1, LPARAM lParam2)
{
	switch(msg) {
  case DRV_LOAD:
  case DRV_FREE:
  case DRV_ENABLE:
  case DRV_DISABLE:
    return 1;
  case DRV_OPEN:
  case DRV_CLOSE:
  case DRV_QUERYCONFIGURE:
    return 0;
  case DRV_CONFIGURE:
    g_warning("HFYU CODEC: Driver isn't configurable !");
    return 0;
  case DRV_INSTALL:
  case DRV_REMOVE:
    return 1;
	}
	return 0;
}


#ifdef __cplusplus
}
#endif
