/*
    dllloader
    (C)Copyright 2000, 2001 by Hiroshi Takekawa
    copyright (c) 2002 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "pe_image.h"

/* for LDT */
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#include <sys/mman.h>
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifndef __i386__
# error i386 or later required
#endif

#ifdef __linux__
# include <asm/ldt.h>
# include <asm/unistd.h>
# ifdef HAVE_STRUCT_USER_DESC
#  define modify_ldt_ldt_s user_desc
# endif
#elif defined(HAVE_I386_SET_LDT)
# include <machine/segments.h>
# include <machine/sysarch.h>
#else
# error Sorry, LDT is not supported. Please submit a patch.
#endif

/* import dlls */
#include "advapi32.h"
#include "borlndmm.h"
#include "gdi32.h"
#include "kernel32.h"
#include "msvcrt.h"
#include "ole32.h"
#include "shell32.h"
#include "user32.h"
#include "version.h"
#include "winmm.h"


/* Argument 1 passed to the DllEntryProc. */
#define	DLL_PROCESS_DETACH	0	/* detach process (unload library) */
#define	DLL_PROCESS_ATTACH	1	/* attach process (load library) */
#define	DLL_THREAD_ATTACH	2	/* attach new thread */
#define	DLL_THREAD_DETACH	3	/* detach thread */


SystemDll system_dll[]={
	{"advapi32.dll",advapi32_get_export_symbols},
	{"borlndmm.dll",borlndmm_get_export_symbols},
	{"gdi32.dll",	gdi32_get_export_symbols},
	{"kernel32.dll",kernel32_get_export_symbols},
	{"msvcrt.dll",	msvcrt_get_export_symbols},
	{"ole32.dll",	ole32_get_export_symbols},
	{"shell32.dll",	shell32_get_export_symbols},
	{"user32.dll",	user32_get_export_symbols},
	{"version.dll",	version_get_export_symbols},
	{"winmm.dll",	winmm_get_export_symbols},
	{NULL,NULL}
};


static SymbolInfo *get_dll_symbols(const gchar *name)
{
	gint i;

	for (i=0;system_dll[i].name!=NULL;i++)
		if (g_strcasecmp(system_dll[i].name,name)==0)
			return system_dll[i].get_symbols();
	return NULL;
}


/* for internal use */
static VOID WINAPI UnknownSymbol(VOID)
{
	g_message("unknown symbol called");
}


/* LDT related */
#define LDT_INDEX 1
#define LDT_TABLE_IDENT 1
#define LDT_RPL 3
#define LDT_SELECTOR(i, t, rpl) ((i << 3) | (t << 2) | rpl)

#if defined(__linux__)
#ifdef PIC
static int modify_ldt(int func, struct modify_ldt_ldt_s *p, unsigned long c)
{
  int res;
	__asm__ __volatile__("pushl %%ebx\n\t"
						 "movl %2,%%ebx\n\t"
						 "int $0x80\n\t"
						 "popl %%ebx"
						 : "=a" (res)
						 : "0" (__NR_modify_ldt),
						   "r" (func),
						   "c" (p),
						   "d" (c));
	return res;
}
#else
static _syscall3(int,modify_ldt,int,func,
								struct modify_ldt_ldt_s *,p,unsigned long,c);
#endif
#endif /* defined(linux) */

static int fs_installed=0;
static char *fs_seg=NULL;

#if defined(__linux__)
static int install_fs(void)
{
	struct modify_ldt_ldt_s ldt;
	unsigned int fs;
	int fd, ret;
	void *prev;

	if (fs_installed)
		return 0;

	fd=open("/dev/zero",O_RDWR);
	if ((fs_seg=mmap(NULL,getpagesize(),
							PROT_READ | PROT_WRITE, MAP_PRIVATE,fd,0))==NULL) {
		g_message("No enough memory for fs. Probably raising SIGSEGV...");
		return -1;
	}
	ldt.base_addr=(int)fs_seg;
	ldt.entry_number=LDT_INDEX;
	ldt.limit=ldt.base_addr+getpagesize()-1;
	ldt.seg_32bit=1;
	ldt.read_exec_only=0;
	ldt.seg_not_present=0;
	ldt.contents=MODIFY_LDT_CONTENTS_DATA;
	ldt.limit_in_pages=0;
	if ((ret=modify_ldt(1,&ldt,sizeof(ldt)))<0) {
		g_message("install_fs");
		g_message("modify_ldt() failed. Probably raising SIGSEGV...");
	}
	fs=LDT_SELECTOR(LDT_INDEX,LDT_TABLE_IDENT,LDT_RPL);
	__asm__("movl %0, %%eax; movw %%ax, %%fs\n\t" : : "r" (fs));
	prev=g_malloc(8);
	*(void **)ldt.base_addr=prev;
	close(fd);
	fs_installed = 1;
	return 0;
}
#elif defined(HAVE_I386_SET_LDT)
static int
install_fs(void)
{
	unsigned int fs;
	union descriptor d;
	int ret;
	caddr_t addr;
	size_t psize;

	psize=getpagesize();
	addr=fs_seg=(caddr_t)mmap(0,psize*2,
				PROT_READ | PROT_WRITE | PROT_EXEC,
				MAP_ANON | MAP_PRIVATE,
				-1,0);
	if (addr==(void *)-1)
		return -1;

	g_memset(addr,0x0,psize*2);
	*(uint32_t *)(addr+0)=(uint32_t)addr+psize;
	*(uint32_t *)(addr+4)=(uint32_t)addr+psize*2;
	d.sd.sd_lolimit=2;
	d.sd.sd_hilimit=0;
	d.sd.sd_lobase=((uint32_t)addr&0xffffff);
	d.sd.sd_hibase=((uint32_t)addr>>24)&0xff;
	d.sd.sd_type = SDT_MEMRW;
	d.sd.sd_dpl=SEL_UPL;
	d.sd.sd_p=1;
	d.sd.sd_def32=1;
	d.sd.sd_gran=1;
#if defined(__FreeBSD__) && defined(LDT_AUTO_ALLOC)
#define LDT_SEL_START LDT_AUTO_ALLOC
#else
#define LDT_SEL_START 0xb
#endif
	ret=i386_set_ldt(LDT_SEL_START,&d,1);
	if (ret<0) {
		g_message("i386_set_ldt");
		return ret;
	}
	fs=LSEL(ret,SEL_UPL);
	__asm__("movl %0, %%eax; movw %%ax, %%fs\n\t" : : "r" (fs));
	return 0;
}
#else
#error No install_fs()
#endif

#if 0
static int uninstall_fs(void)
{
	if (fs_seg==NULL)
		return -1;
	munmap(fs_seg, getpagesize());
	fs_seg=0;
	fs_installed=0;

	return 0;
}
#endif


/*
0x0001 = Cursor
0x0002 = Bitmap
0x0003 = Icon
0x0004 = Menu
0x0005 = Dialog
0x0006 = String Table
0x0007 = Font Directory
0x0008 = Font
0x0009 = Accelerators Table
0x000A = RC Data (custom binary data)
0x000B = Message table
0x000C = Group Cursor
0x000E = Group Icon
0x0010 = Version Information
0x0011 = Dialog Include
0x0013 = Plug'n'Play
0x0014 = VXD
0x0015 = Animated Cursor
0x2002 = Bitmap (new version)
0x2004 = Menu (new version)
0x2005 = Dialog (new version)
*/
static void traverse_directory(PE_image *pe,
									IMAGE_RESOURCE_DIRECTORY *ird,gchar *name)
{
	gchar *new_name,*buf;
	guint8 *data;
	guint i,j;
	IMAGE_RESOURCE_DIRECTORY_ENTRY *irde;
	IMAGE_RESOURCE_DIRECTORY *new_ird;
	IMAGE_RESOURCE_DATA_ENTRY *de;
	IMAGE_RESOURCE_DIR_STRING_U *irdsu;

	for (i=0;i<ird->NumberOfNamedEntries+ird->NumberOfIdEntries;i++) {
		irde=(IMAGE_RESOURCE_DIRECTORY_ENTRY *)
								((LPBYTE)ird+sizeof(IMAGE_RESOURCE_DIRECTORY)
									+sizeof(IMAGE_RESOURCE_DIRECTORY_ENTRY)*i);
		if (irde->Id&0x80000000) {
			irdsu=(IMAGE_RESOURCE_DIR_STRING_U *)(pe->image
							+pe->opt_header.DataDirectory
								[IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress
							+(irde->Id&0x7fffffff));
			buf=g_malloc((irdsu->Length+1)*sizeof(gchar));
			for (j=0;j<irdsu->Length;j++)/* UNICODE */
				buf[j]=irdsu->NameString[j];
			buf[j]='\0';
			g_strdown(buf);
			new_name=g_strdup_printf("%s/%s",name,buf);
			g_free(buf);
		} else {
			new_name=g_strdup_printf("%s/0x%x",name,irde->Id);
		}
		if (irde->OffsetToData&0x80000000) {
			new_ird=(IMAGE_RESOURCE_DIRECTORY *)(pe->image
						+pe->opt_header.DataDirectory
								[IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress
						+(irde->OffsetToData&0x7fffffff));
			traverse_directory(pe,new_ird,new_name);
			g_free(new_name);
		} else {
			de=(IMAGE_RESOURCE_DATA_ENTRY *)(pe->image
						+pe->opt_header.DataDirectory
								[IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress
						+irde->OffsetToData);
			if ((data=g_malloc(de->Size+sizeof(DWORD)))==NULL)
				g_error("No enough memory for resource.\n");
			*(LPDWORD)data=de->Size;
			g_memmove(data+sizeof(DWORD),pe->image+de->OffsetToData,de->Size);
			g_hash_table_insert(pe->resource,new_name,data);
			g_debug("Resource:%s,%X,%X",new_name,de->OffsetToData,de->Size);
		}
	}
}


gboolean isPE(const gchar *file)
{
	guint8 dos_header[DOS_HEADER_SIZE];
	guint32 pe_signature;
	guint32 pe_header_start;
	IMAGE_FILE_HEADER pe_header;
	FILE *fpr;
	gboolean retval;

	if ((fpr = fopen(file, "rb")) == NULL)
		return FALSE;

	if (fread(dos_header, 1, DOS_HEADER_SIZE, fpr) != DOS_HEADER_SIZE) {
		retval = FALSE;
		goto err_out;
	}
		
	pe_header_start = PE_HEADER_START(dos_header);

	if (fseek(fpr ,pe_header_start, SEEK_SET) ||
	    (fread(&pe_signature, 1, 4, fpr) != 4)) {
		retval = FALSE;
		goto err_out;
	}

	if (pe_signature != PE_SIGNATURE) {
		retval = FALSE;
		goto err_out;
	}

	if (fread(&pe_header, 1, PE_HEADER_SIZE, fpr) != PE_HEADER_SIZE) {
		retval = FALSE;
		goto err_out;
	}

	switch (pe_header.Machine) {
	case 0x14c:
	case 0x14d:
	case 0x14e:
		retval = TRUE;
		break;
	default:
    		g_message("PE_image: %s: unsupported architecture %03X",
		   __FUNCTION__, pe_header.Machine);
		retval = FALSE;
	}
err_out:
	fclose(fpr);
	return retval;
}


/* methods */
static gboolean load(PE_image *pe,const gchar *file,const gboolean flags)
{
	guint8 dos_header[DOS_HEADER_SIZE];
	guint32 pe_signature;
	guint32 pe_header_start;
	gint i;
	FILE *fp;

	if ((fp=fopen(file,"rb"))==NULL)
		return FALSE;

	fread(dos_header,1,DOS_HEADER_SIZE,fp);
	pe_header_start=PE_HEADER_START(dos_header);

	fseek(fp,pe_header_start,SEEK_SET);
	fread(&pe_signature,1,4,fp);
	if (pe_signature!=PE_SIGNATURE) {
		g_message("PE_image: %s: Not PE file.",__FUNCTION__);
		fclose(fp);
		return FALSE;
	}

	fread(&pe->pe_header,1,PE_HEADER_SIZE,fp);
	switch (pe->pe_header.Machine) {
		case 0x14c:
		case 0x14d:
		case 0x14e:
			break;
		default:
    		g_message("PE_image: %s: unsupported architecture %03X",
										__FUNCTION__,pe->pe_header.Machine);
			fclose(fp);
			return FALSE;
	}

	if (pe->pe_header.SizeOfOptionalHeader!=OPTIONAL_HEADER_SIZE) {
		g_message("PE_image: %s: Optional Header Size %d != %d",__FUNCTION__,
					pe->pe_header.SizeOfOptionalHeader,OPTIONAL_HEADER_SIZE);
		fclose(fp);
		return FALSE;
	}

	fread(&pe->opt_header,1,OPTIONAL_HEADER_SIZE,fp);
	if (pe->opt_header.Magic!=OPTIONAL_SIGNATURE) {
		g_message("PE_image: %s: PE file but corrupted optional header.",
																__FUNCTION__);
		fclose(fp);
		return FALSE;
	}

	if ((pe->image=g_malloc0(pe->opt_header.SizeOfHeaders
										+pe->opt_header.SizeOfImage))==NULL) {
    	g_message("No enough memory for image (%d bytes)",
					pe->opt_header.SizeOfImage+pe->opt_header.SizeOfHeaders);
		fclose(fp);
		return FALSE;
	}
	fseek(fp,0,SEEK_SET);
	fread(pe->image,1,pe->opt_header.SizeOfHeaders,fp);

	pe->sect_headers=(IMAGE_SECTION_HEADER *)
			(pe->image+pe_header_start+4+PE_HEADER_SIZE+OPTIONAL_HEADER_SIZE);
	for (i=0;i<pe->pe_header.NumberOfSections;i++) {
		if (pe->sect_headers[i].Characteristics
										&IMAGE_SCN_CNT_UNINITIALIZED_DATA) {
			g_memset(pe->image+pe->sect_headers[i].VirtualAddress,0,
									pe->opt_header.SizeOfUninitializedData);
		} else if (pe->sect_headers[i].SizeOfRawData>0) {
			fseek(fp,pe->sect_headers[i].PointerToRawData,SEEK_SET);
			fread(pe->image+pe->sect_headers[i].VirtualAddress,
									1,pe->sect_headers[i].SizeOfRawData,fp);
		}
	}

	fclose(fp);
	pe->file=g_strdup(file);

	/* export symbols */
	if (pe->opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_EXPORT].Size) {
		IMAGE_EXPORT_DIRECTORY *ied=(IMAGE_EXPORT_DIRECTORY *)(pe->image
							+pe->opt_header.DataDirectory
								[IMAGE_DIRECTORY_ENTRY_EXPORT].VirtualAddress);
		DWORD *functions=(DWORD *)(pe->image+ied->AddressOfFunctions);
		DWORD *names=(DWORD *)(pe->image+ied->AddressOfNames);
		WORD *name_ordinals=(WORD *)(pe->image+ied->AddressOfNameOrdinals);

		pe->export=g_hash_table_new(g_str_hash,g_str_equal);
		for (i=0;i<ied->NumberOfNames;i++)
			g_hash_table_insert(pe->export,pe->image+names[i],
										pe->image+functions[name_ordinals[i]]);
	}

	/* import symbols */
	if (pe->opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].Size) {
		IMAGE_IMPORT_DESCRIPTOR *iid=(IMAGE_IMPORT_DESCRIPTOR *)(pe->image
							+pe->opt_header.DataDirectory
								[IMAGE_DIRECTORY_ENTRY_IMPORT].VirtualAddress);
		IMAGE_THUNK_DATA *thunk,*othunk;
		IMAGE_IMPORT_BY_NAME *iibn;
		gint j;
		SymbolInfo *syminfo;

		/* per DLL */
		for (i=0;iid[i].Name;i++) {
			if (iid[i].u.OriginalFirstThunk)
				othunk=(IMAGE_THUNK_DATA *)
								(pe->image+(gint)iid[i].u.OriginalFirstThunk);
			else
				othunk=(IMAGE_THUNK_DATA *)(pe->image+(gint)iid[i].FirstThunk);
			thunk=(IMAGE_THUNK_DATA *)(pe->image+(gint)iid[i].FirstThunk);

			/* symbols in each DLL */
			for (j=0;othunk[j].u1.AddressOfData;j++)
				if ((othunk[j].u1.Ordinal&IMAGE_ORDINAL_FLAG)==0) {
					iibn=(IMAGE_IMPORT_BY_NAME *)
								(pe->image+(int)othunk[j].u1.AddressOfData);
					/* import */
					if ((syminfo=get_dll_symbols(
									(gchar *)pe->image+iid[i].Name))!=NULL) {
						while (syminfo->name!=NULL) {
							if (g_strcmp((gchar *)iibn->Name,syminfo->name)==0)
								break;
							syminfo++;
						}
						thunk[j].u1.Function=(FARPROC)syminfo->value;
						if (syminfo->name==NULL)
							g_message("Unknown Symbol:%s,%s",
											pe->image+iid[i].Name,iibn->Name);
					} else {
						thunk[j].u1.Function=(FARPROC)UnknownSymbol;
						g_message("Unknown DLL:%s,%s",
											pe->image+iid[i].Name,iibn->Name);
					}
				}
		}
	}

	/* resource */
	pe->resource=g_hash_table_new(g_str_hash,g_str_equal);
	if (pe->opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_RESOURCE].Size) {
		gchar *name;
		IMAGE_RESOURCE_DIRECTORY *ird=(IMAGE_RESOURCE_DIRECTORY *)(pe->image
						+pe->opt_header.DataDirectory
							[IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress);

		name=g_strdup("");
		traverse_directory(pe,ird,name);
		g_free(name);
	}

	/* relocation */
	if (pe->opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].Size) {
		IMAGE_BASE_RELOCATION *ibr=(IMAGE_BASE_RELOCATION *)(pe->image
						+pe->opt_header.DataDirectory
							[IMAGE_DIRECTORY_ENTRY_BASERELOC].VirtualAddress);
		gint adjust=(gint)(pe->image-pe->opt_header.ImageBase);

		while (ibr->VirtualAddress) {
			LPWORD reloc=ibr->TypeOffset;
			guint nrelocs=(ibr->SizeOfBlock-sizeof(DWORD)*2)>>1;

			for (i=0;i<nrelocs;i++) {
				gint type=(reloc[i]>>12)&0xf;
				gint offset=reloc[i]&0xfff;

				switch (type) {
					case 0:
						break;
					case 3:
						{
							LPDWORD patched=(LPDWORD)(pe->image
												+ibr->VirtualAddress+offset);
							*patched+=adjust;
						}
						break;
					default:
						g_message("Unsupported relocation type %d...", type);
						break;
				}
			}
			ibr=(IMAGE_BASE_RELOCATION *)(((guint8 *)ibr)+ibr->SizeOfBlock);
		}
 	}

	if (!fs_installed)
		install_fs();

	if (flags && pe->pe_header.Characteristics&0x2000) {
		if ((pe->DllMain=(DllEntryProc)peimage_resolve(pe,"DllMain"))==NULL)
			pe->DllMain=(DllEntryProc)(pe->image
										+pe->opt_header.AddressOfEntryPoint);
		if (pe->DllMain!=NULL) {
			if (pe->opt_header.AddressOfEntryPoint
					>pe->opt_header.SizeOfHeaders+pe->opt_header.SizeOfImage) {
				g_message("%s: InitDll %p is bad address",file,pe->DllMain);
				pe->DllMain=NULL;
			}
			if (pe->opt_header.AddressOfEntryPoint
											<pe->opt_header.SizeOfHeaders) {
				g_message("EntryPoint %d < %d SizeOfHeaders",
											pe->opt_header.AddressOfEntryPoint,
											pe->opt_header.SizeOfHeaders);
				pe->DllMain=NULL;
			}
		}
		if (pe->DllMain!=NULL) {
			BOOL fResult;

			/*
			 * some dll (e.g. ir32_32.dll) break %ebx
			 * but do as below will cause segmentation fault...
			 */
//			__asm__ __volatile__("pushl %ebx\n\t");
			fResult=pe->DllMain((HMODULE)pe,DLL_PROCESS_ATTACH,NULL);
//			__asm__ __volatile__("popl %ebx\n\t");
			if (!fResult)
				g_message("DllMain returns %d",fResult);
		}
	}

	return TRUE;
}


static gpointer resolve(PE_image *pe,const gchar *name)
{
	return g_hash_table_lookup(pe->export,name);
}


static void g_hash_table_callback(gpointer key,gpointer value,gpointer hash)
{
	gpointer orig_key,orig_value;

	if (g_hash_table_lookup_extended((GHashTable*)hash,key,
													&orig_key,&orig_value)) {
		g_free(orig_key);
		g_free(orig_value);
	}
}


static void destroy(PE_image *pe)
{
	if (pe->DllMain!=NULL) {
		BOOL fResult;

		/*
		 * some dll (e.g. ir32_32.dll) break %ebx
		 * but do as below will cause segmentation fault...
		 */
//		__asm__ __volatile__("pushl %ebx\n\t");
		fResult=pe->DllMain((HMODULE)pe,DLL_PROCESS_DETACH,NULL);
//		__asm__ __volatile__("popl %ebx\n\t");
		if (!fResult)
			g_message("DllMain returns %d",fResult);
	}
	if (pe->resource!=NULL) {
		g_hash_table_foreach(pe->resource,g_hash_table_callback,pe->resource);
		g_hash_table_destroy(pe->resource);
	}
	if (pe->export!=NULL)
		g_hash_table_destroy(pe->export);
	g_free(pe->file);
	g_free(pe->image);
	g_free(pe);
}


PE_image *peimage_create(void)
{
	PE_image *pe;

	pe=g_malloc0(sizeof(PE_image));
	pe->load=load;
	pe->resolve=resolve;
	pe->destroy=destroy;
	return pe;
}
