/*
    Video maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "file.h"
#include "sigfile.h"
#include "sigmain.h"
#include "size.h"
#include "thread.h"
#include "misc/fileio.h"
#include "misc/misc.h"


/******************************************************************************
*                                                                             *
* ja:ファイル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの履歴に加える
    file,ファイル名                                                         */
void
file_set_history (const gchar *file)
{
  gchar *label;
  gint i, count;
  GList *glist;
  GtkWidget *menu_shell, *menu_item;

  menu_shell = gtk_item_factory_get_widget (ifactory_menu, "<main>/File");
  glist = gtk_container_children (GTK_CONTAINER (menu_shell));
  count = g_list_length (glist);
  for (i = 0; i < count - MENUFILE - 1; i++)
    {
      menu_item = g_list_nth_data (glist, i + MENUFILE - 1);
      label = g_object_get_data (G_OBJECT (menu_item), "user_data");
      if (!g_strfilecmp (label, file))
        {
          gtk_container_remove (GTK_CONTAINER (menu_shell), menu_item);
          break;
        }
    }
  g_list_free (glist);
  if (i >= history)
    {
      glist = gtk_container_children (GTK_CONTAINER (menu_shell));
      gtk_container_remove (GTK_CONTAINER (menu_shell),
                        g_list_nth_data (glist, MENUFILE + history + 1 - 3));
      g_list_free (glist);
    }
  if (count <= MENUFILE)
    {
      menu_item = gtk_menu_item_new ();
      gtk_menu_insert (GTK_MENU (menu_shell), menu_item, MENUFILE - 2);
        gtk_widget_show (menu_item);
    }
  label = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  menu_item = gtk_menu_item_new_with_label (label);
  g_free (label);
  label = g_strdup (file);
  g_signal_connect (G_OBJECT (menu_item), "activate",
                            G_CALLBACK (signal_activate_menu_history), label);
  g_signal_connect (G_OBJECT (menu_item), "destroy",
                            G_CALLBACK (signal_destroy_menu_history), label);
  g_object_set_data (G_OBJECT (menu_item), "user_data", label);
  gtk_menu_insert (GTK_MENU (menu_shell), menu_item, MENUFILE - 1);
  gtk_widget_show (menu_item);
}


/*  ja:ファイルを加える
    file,加えようとするファイルのフルパス
    same,フルパスに付属する数値
     RET,加えようとするファイルのラベル                                     */
gchar *
file_add_edit (const gchar *file, gint *same)
{
  gchar *label, *utf8str;
  const gchar *name0, *name1;
  gint i;
  GList *glist;
  GtkWidget *child;
  VmaidWindow *vmaid;

  *same=-1;
  /* ja:ファイル名を調べる */
  glist = gtk_container_children (GTK_CONTAINER (notebook));
  /* ja:同名同パスを調べる */
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      child = g_list_nth_data (glist, i);
      vmaid = g_object_get_data (G_OBJECT (child), "user_data");
      if (!g_strfilecmp (file, vmaid->file))
        {
          if (vmaid->same < 0)
            {
              vmaid->same = 0;
              utf8str = g_filename_to_utf8 (vmaid->file, -1, NULL, NULL, NULL);
              label = g_strdup_printf ("%s:%d", utf8str, vmaid->same);
              g_free (utf8str);
              gtk_label_set_text (GTK_LABEL (vmaid->label), label);
              g_free (label);
            }
            if (*same <= vmaid->same)
              *same = vmaid->same + 1;
        }
    }
  if (*same >= 0)
    {
      /* ja:同名同パスがあるとき */
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      label = g_strdup_printf ("%s:%d", utf8str, *same);
      g_free (utf8str);
    }
  else
    {
      /* ja:同名同パスはないとき */
      /* ja:同名異パスを調べる */
      name0 = g_basename (file);
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          child = g_list_nth_data (glist, i);
          vmaid = g_object_get_data (G_OBJECT (child), "user_data");
          name1 = g_basename (vmaid->file);
          if (!g_strfilecmp (name0, name1))
            {
              if (vmaid->same < 0)
                {
                  utf8str = g_filename_to_utf8 (vmaid->file, -1,
                                                            NULL, NULL, NULL);
                  gtk_label_set_text (GTK_LABEL (vmaid->label), utf8str);
                  g_free (utf8str);
                }
              break;
            }
        }
      label = g_filename_to_utf8 (i >= 0 ? file : name0, -1, NULL, NULL, NULL);
    }
  g_list_free (glist);
  return label;
}


/*  ja:ファイルを削除する
    file,削除しようとするファイルのフルパス                                 */
void
file_delete_edit (const gchar *file)
{
  gchar *utf8str;
  const gchar *name0,*name1;
  gint i,count=0;
  GList *glist;
  VmaidWindow *vmaid;
  GtkWidget *child;

  /* ja:ファイル名を調べる */
  glist = gtk_container_children (GTK_CONTAINER (notebook));
  /* ja:同名異パスを調べる */
  name0 = g_basename (file);
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      vmaid = g_object_get_data (g_list_nth_data (glist, i), "user_data");
      name1 = g_basename (vmaid->file);
      if (!g_strfilecmp (name0, name1))
        count++;
    }
  if (count <= 2)
    {
      /* ja:同名異パスのファイルを通常表示に変更する */
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          child = g_list_nth_data (glist, i);
          vmaid = g_object_get_data (G_OBJECT (child), "user_data");
          name1 = g_basename (vmaid->file);
          if (!g_strfilecmp (name0, name1))
            {
              utf8str = g_filename_to_utf8 (name1, -1, NULL, NULL, NULL);
              gtk_label_set_text (GTK_LABEL (vmaid->label), utf8str);
              g_free (utf8str);
            }
        }
    }
  else
    {
      /* ja:同名同パスを調べる */
      count = 0;
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          vmaid = g_object_get_data (g_list_nth_data (glist, i), "user_data");
          if (!g_strfilecmp (file, vmaid->file))
            count++;
        }
      if (count <= 2)/* ja:同名同パスのファイルをフルパス表示に変更する */
        for (i = g_list_length (glist) - 1; i >= 0; i--)
          {
            child = g_list_nth_data (glist, i);
            vmaid = g_object_get_data (G_OBJECT (child), "user_data");
            if (!g_strfilecmp (file,  vmaid->file))
              {
                vmaid->same = -1;
                utf8str = g_filename_to_utf8 (vmaid->file, -1,
                                                            NULL, NULL, NULL);
                gtk_label_set_text (GTK_LABEL (vmaid->label), utf8str);
                g_free (utf8str);
              }
          }
    }
}


/******************************************************************************
*                                                                             *
* ja:ファイル入力関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:AVIファイルを開く
     avi,AVI編集構造体
    file,ファイル名                                                         */
static void
file_open_avi (AviEdit     *avi_edit[],
               const gchar *file)
{
  gint i;
  guint32 streams;
  AviEdit *avi_edit_tmp;

  avi_edit[0] = avi_edit[1] = NULL;
  streams = MAX (avi_main_streams (file), 1);
  for (i = 0; i < streams && (!avi_edit[0] || !avi_edit[1]); i++)
    if ((avi_edit_tmp = avi_open (file, i)))
      switch (avi_type (avi_edit_tmp))
        {
          case AVI_STREAM_VIDEO:
            if (!avi_edit[0])
              avi_edit[0] = avi_edit_tmp;
            else
              avi_release (avi_edit_tmp);
              break;
          case AVI_STREAM_AUDIO:
            if (!avi_edit[1])
              avi_edit[1] = avi_edit_tmp;
            else
              avi_release (avi_edit_tmp);
            break;
          default:
            avi_release (avi_edit_tmp);
        }
}


/*  ja:ファイルを開く(独立スレッド)
    file_open,Open情報
     RET,ウインドウ情報                                                     */
VmaidWindow *
file_open_edit (FileOpen *file_open)
{
  FileIO *fio;
  gboolean result;
  gchar *text = NULL, *format = NULL, *file, *dir, *head, *scenario;
  gchar *utf8str;
  const gchar *name, *foot;
  gint i, j, value,max = 0;
  goffset offset;
#ifdef USE_THREAD
  GThread *id;
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */
  AviEdit *avi_edit_old[2], *avi_edit_tmp[2], **avi_edit;
  GdkBitmap *mask;
  GdkPixmap *pixmap;
  GtkWidget *xbox, *hbox, *vbox, *menu_shell, *button, *frame;
  VmaidWindow *vmaid;
  static gint fcount = 0;   /* ja:新規ファイルの数 */
  static gchar *xpm[] = {
"8 7 2 1",
"   c None",
".  c #000000",
"..    ..",
" ..  .. ",
"  ....  ",
"   ..   ",
"  ....  ",
" ..  .. ",
"..    .."};

#ifdef USE_THREAD
  id = g_thread_self();
#endif /* USE_THREAD */
  thread_insert (id, THREAD_MODE_OPEN, file_open ? file_open->file : NULL);

  vmaid = g_malloc (sizeof (VmaidWindow));
  vmaid->edit = FALSE;
  vmaid->same = -1;
  vmaid->undo = vmaid->redo = NULL;
  vmaid->rate = default_rate;
  vmaid->scale = 1;
  vmaid->cursor.stream = 0;
  vmaid->cursor.frame = 0;
  vmaid->select.stream = -1;
  vmaid->select.frame = -1;
  vmaid->top = 0;
  vmaid->avi_edit[0] = NULL;
  vmaid->avi_edit[1] = NULL;
  vmaid->timer_id = 0;
  if (!file_open)
    {
      /* ja:新規 */
      vmaid->file = g_strdup_printf ("new%04d", fcount++ % 1000);
    }
  else
    {
      vmaid->file = fileio_get_full_path (file_open->file);
      g_free (file_open->file);
      file_open_avi (vmaid->avi_edit, vmaid->file);
      if ((vmaid->avi_edit[0] || vmaid->avi_edit[1]) && file_open->merge)
        {
          /* ja:連番 */
          name = g_basename (vmaid->file);
          for (i = g_strlen (name) - 1; i >= 0; i--)
            if ('0' <= name[i] && name[i] <= '9')
              break;
          if (i >= 0)
            {
              dir = g_dirname (vmaid->file);
              head = g_strdup (name);
              foot = name + i + 1;
              for (j = i - 1; j >= 0; j--)
                if (head[j] < '0' || '9' < head[j])
                  break;
              misc_str_to_val (&value, head + j + 1, 10, FALSE);
              head[j + 1] = '\0';
              format = g_strdup_printf ("%s/%s%%0%dd%s",
                                                    dir, head, i - j, foot);
              g_free (dir);
              g_free (head);
              max = 1;
              while (j < i)
                {
                  max *= 10;
                  j++;
                }
            }
        }
      if (format)
        {
          /* ja:連番 */
          for (i = value - 1, result = TRUE; i >= 0 && result; i--)
            {
              /* ja:連番前 */
              file = g_strdup_printf (format, i);
              file_open_avi (avi_edit_tmp, file);
              g_free (file);
              if (!avi_edit_tmp[0] && !avi_edit_tmp[1])
                break;
              for (j = 0; j < 2; j++)
                if (vmaid->avi_edit[j])
                  {
                    avi_edit_old[j] = avi_clone (vmaid->avi_edit[j]);
                    if (!avi_paste (vmaid->avi_edit[j], 0, avi_edit_tmp[j]))
                      result = FALSE;
                    }
                  else
                    {
                      avi_edit_old[j] = NULL;
                      vmaid->avi_edit[j] = avi_edit_tmp[j];
                    }
              if (result)
                {
                  for (j = 0; j < 2; j++)
                    if (avi_edit_old[j])
                      avi_release (avi_edit_old[j]);
                }
              else
                {
                  for (j = 0; j < 2; j++)
                    {
                      if (vmaid->avi_edit[j])
                        avi_release (vmaid->avi_edit[j]);
                      vmaid->avi_edit[j] = avi_edit_old[j];
                    }
                }
            }
          for (i = value + 1, result = TRUE; i < max && result; i++)
            {
              /* ja:連番後 */
              file = g_strdup_printf (format, i);
              file_open_avi (avi_edit_tmp, file);
              g_free (file);
              if (!avi_edit_tmp[0] && !avi_edit_tmp[1])
                break;
              for (j = 0; j < 2; j++)
                if (vmaid->avi_edit[j])
                  {
                    avi_edit_old[j] = avi_clone (vmaid->avi_edit[j]);
                    if (!avi_paste (vmaid->avi_edit[j],
                            avi_length (vmaid->avi_edit[j]), avi_edit_tmp[j]))
                      result=FALSE;
                  }
                else
                  {
                    avi_edit_old[j] = NULL;
                    vmaid->avi_edit[j] = avi_edit_tmp[j];
                  }
              if (result)
                {
                  for (j = 0; j < 2; j++)
                    if (avi_edit_old[j])
                      avi_release (avi_edit_old[j]);
                }
              else
                {
                  for (j = 0; j < 2; j++)
                    {
                      if (vmaid->avi_edit[j])
                        avi_release(vmaid->avi_edit[j]);
                      vmaid->avi_edit[j]=avi_edit_old[j];
                    }
                }
            }
        }
      if (!vmaid->avi_edit[0] && !vmaid->avi_edit[1]
                        && avi_file_type (vmaid->file) == AVI_TYPE_SCENARIO)
        {
          /* ja:ファイルを開く */
          fio = fileio_open (vmaid->file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING);
          if (!fio)
            {
              g_free (file_open);
              g_free (vmaid->file);
              g_free (vmaid);
              return NULL;
            }
          /* ja:読み込み */
          offset = fileio_seek (fio, 0, SEEK_END);
          if (offset == -1 || fileio_seek (fio, 0, FILEIO_SEEK_SET) == -1
                                || !(scenario = g_malloc ((gssize)offset + 1)))
            {
              fileio_close (fio);
              g_free (file_open);
              g_free (vmaid->file);
              g_free (vmaid);
              return 0;
            }
          scenario[offset] = '\0';
          if (fileio_read (fio, scenario, (gssize)offset) != offset)
            {
              fileio_close (fio);
              g_free (scenario);
              g_free (file_open);
              g_free (vmaid->file);
              g_free (vmaid);
              return 0;
            }
          if (!fileio_close (fio))
            {
              g_free (scenario);
              g_free (file_open);
              g_free (vmaid->file);
              g_free (vmaid);
              return 0;
            }
          if ((avi_edit = avi_from_scenario (scenario)))
            for (i = 0; avi_edit[i]; i++)
              switch (avi_type (avi_edit[i]))
                {
                  case AVI_STREAM_VIDEO:
                    if (!vmaid->avi_edit[0])
                      vmaid->avi_edit[0]=avi_edit[i];
                    else
                      avi_release (avi_edit[i]);
                    break;
                  case AVI_STREAM_AUDIO:
                    if (!vmaid->avi_edit[1])
                      vmaid->avi_edit[1]=avi_edit[i];
                    else
                      avi_release (avi_edit[i]);
                    break;
                  default:
                    avi_release(avi_edit[i]);
                }
        }
    }

  if (!thread_idling (id, 100))
    {
      g_free (file_open);
      g_free (vmaid->file);
      for (i = 0; i < 2; i++)
        if (vmaid->avi_edit[i])
          avi_release(vmaid->avi_edit[i]);
      g_free (vmaid);
      return NULL;
    }

  if (vmaid->avi_edit[0])
    {
      size_set_scale (default_view, &vmaid->width, &vmaid->height,
                                        avi_get_width (vmaid->avi_edit[0]),
                                        avi_get_height (vmaid->avi_edit[0]));
      vmaid->rate = avi_get_rate (vmaid->avi_edit[0]);
      vmaid->scale = avi_get_scale (vmaid->avi_edit[0]);
    }
  else
    {
      size_set_scale (default_view, &vmaid->width, &vmaid->height, -1, -1);
    }
  if (!vmaid->avi_edit[0] && vmaid->avi_edit[1])
    vmaid->cursor.stream = 1;

#ifdef USE_THREAD
  gdk_threads_enter();
#endif /* USE_THREAD */

  /* ja:ファイル名を調べる */
  text = file_add_edit (vmaid->file, &vmaid->same);

  /* ja:表示エリア */
  vmaid->drawing = gtk_drawing_area_new ();
  g_signal_connect (G_OBJECT (vmaid->drawing), "focus-in-event",
                                G_CALLBACK (signal_focus_in), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "focus-out-event",
                                G_CALLBACK (signal_focus_out), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "realize",
                                G_CALLBACK (signal_realize), NULL);
  g_signal_connect (G_OBJECT (vmaid->drawing), "configure-event",
                                G_CALLBACK (signal_config), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "expose-event",
                                G_CALLBACK (signal_expose), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "button-press-event",
                                G_CALLBACK (signal_button_press_draw), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "motion-notify-event",
                                G_CALLBACK (signal_motion_notify), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "button-release-event",
                                G_CALLBACK (signal_button_release), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "scroll-event",
                                G_CALLBACK (signal_scroll), vmaid);
  g_signal_connect_after (G_OBJECT (vmaid->drawing), "key-press-event",
                                G_CALLBACK (signal_key_press), vmaid);
  g_signal_connect (GTK_OBJECT (vmaid->drawing), "destroy",
                                G_CALLBACK (signal_destroy_draw), vmaid);
  gtk_widget_add_events (vmaid->drawing, GDK_POINTER_MOTION_MASK
                            | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
                            | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
                            | GDK_FOCUS_CHANGE_MASK);
  gtk_widget_set_usize (vmaid->drawing,
                vmaid->width * 3 + 4, system_font_height * 2
                + ((vmaid->avi_edit[0] ? 1 : 0) + (vmaid->avi_edit[1] ? 1 : 0))
                                                            * vmaid->height);
  GTK_WIDGET_SET_FLAGS (vmaid->drawing, GTK_CAN_FOCUS);
  /* ja:スクロールバー */
  vmaid->hscroll = gtk_hscrollbar_new
                    (GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 0, 1, 1, 1)));
  /* ja:ボックス */
  xbox = gtk_hbox_new (FALSE, 0);
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), vmaid->drawing, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), vmaid->hscroll, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (xbox), vbox, FALSE, FALSE,0);
  g_object_set_data (G_OBJECT (xbox), "user_data", vmaid);
  /* ja:メニュー */
  menu_shell = gtk_item_factory_get_widget (ifactory_menu, "<main>/Window");
  utf8str = g_filename_to_utf8 (vmaid->file, -1, NULL, NULL, NULL);
  vmaid->menu_item = gtk_menu_item_new_with_label (vmaid->file);
  g_free (utf8str);
  g_signal_connect (G_OBJECT (vmaid->menu_item), "activate",
                            G_CALLBACK (signal_activate_menu_window), xbox);
  gtk_menu_append (GTK_MENU (menu_shell), vmaid->menu_item);
  gtk_widget_show (vmaid->menu_item);
  /* ja:ボタン */
  button = gtk_button_new ();
  frame = gtk_frame_new (NULL);
  gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
  pixmap = gdk_pixmap_create_from_xpm_d (window->window, &mask, NULL, xpm);
  gtk_container_add (GTK_CONTAINER (frame),
                                    gtk_image_new_from_pixmap (pixmap, mask));
  gtk_container_add (GTK_CONTAINER (button), frame);
  g_signal_connect (G_OBJECT (button), "clicked",
                                            G_CALLBACK (signal_clicked), xbox);
  /* ja:ラベル */
  vmaid->label = gtk_label_new (text);
  g_free (text);
  /* ja:マーク */
  vmaid->mark = gtk_label_new ("*");
  /* ja:表示 */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vmaid->label, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vmaid->mark, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), xbox, hbox);
  gtk_widget_show (hbox);
  gtk_widget_show (vmaid->label);
  gtk_widget_show_all (button);
  gtk_widget_show_all (xbox);
  gtk_notebook_set_page (GTK_NOTEBOOK (notebook),
                        gtk_notebook_page_num (GTK_NOTEBOOK (notebook), xbox));
  gtk_widget_grab_focus (vmaid->drawing);

  /* ja:新規作成ではなく、ファイルが存在する時には履歴に加える */
  if (file_open && history > 0 && fileio_isfile (vmaid->file))
    file_set_history (vmaid->file);

#ifdef USE_THREAD
  gdk_threads_leave();
#endif /* USE_THREAD */

  g_free (file_open);
  thread_delete (id);
  return vmaid;
}


/******************************************************************************
*                                                                             *
* ja:ファイル出力関数群                                                       *
*                                                                             *
******************************************************************************/
static gboolean
file_save_edit_callback (gint percent, gpointer user_data)
{
#ifdef USE_THREAD
  return thread_idling (g_thread_self (), percent);
#else /* not USE_THREAD */
  return thread_idling (THREAD_ID_STANDARD, percent);
#endif /* not USE_THREAD */
}


/*  ja:AVIファイルを保存する(独立スレッド)
    file_save,Save情報
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
file_save_edit (FileSave *file_save)
{
  gboolean result;
  gint i;
#ifdef USE_THREAD
  GThread *id;
  id = g_thread_self ();
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */

  thread_insert (id, THREAD_MODE_SAVE, file_save->file);

  result = avi_save_with_options (file_save->avi_edit, file_save->file,
                        file_save->avi_save, file_save_edit_callback, NULL);
  g_free (file_save->file);
  for (i = 0; i < 2; i++)
    if (file_save->avi_edit[i])
      avi_release (file_save->avi_edit[i]);
  avi_save_free (file_save->avi_save);
  g_free (file_save);

  if (!result)
    while (thread_idling (id, -1));

  thread_delete (id);
  return result;
}
