/*
    fileio
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "fileio.h"
#include <time.h>
#ifdef HAVE_UTIME_H
# include <utime.h>
#endif /* HAVE_UTIME_H */
#if defined TM_IN_SYS_TIME && defined HAVE_SYS_TIME
# include <sys/time.h>
#endif /* defined TM_IN_SYS_TIME && defined HAVE_SYS_TIME */


/******************************************************************************
*                                                                             *
* ja:ファイル入出力関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの存在確認
    file,ファイル名
    test,条件
     RET,TRUE:合致する,FALSE:合致しない                                     */
gboolean
fileio_test (const gchar *file,
             const guint  test)
{
#ifdef G_OS_WIN32
# ifdef UNICODE
  gchar *utf8file;
  LPTSTR lpszFile;
# endif /* UNICODE */
  HANDLE hFind; /* 検索ハンドル */
  SHFILEINFO rcInfo;
  WIN32_FIND_DATA rcFind;

  if (!file)
    return FALSE;
# ifdef UNICODE
  utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8file)
    return FALSE;
  lpszFile = g_utf8_to_utf16 (utf8file, -1, NULL, NULL, NULL);
  if (!lpszFile)
    return FALSE;
# else /* not UNICODE */
#   define lpszFile file
# endif /* not UNICODE */
  if ((test & FILEIO_TEST_IS_EXECUTABLE) && SHGetFileInfo (lpszFile, 0,
                                &rcInfo, sizeof (SHFILEINFO), SHGFI_EXETYPE))
    {
# ifdef UNICODE
      g_free (lpszFile);
# endif /* UNICODE */
      return TRUE;
    }
  hFind = FindFirstFile (lpszFile, &rcFind);
# ifdef UNICODE
  g_free (lpszFile);
# else /* not UNICODE */
#   undef lpszFile
# endif /* not UNICODE */
  if (hFind != INVALID_HANDLE_VALUE)
    {
      FindClose (hFind);
      if ((test & FILEIO_TEST_EXISTS)
            || ((test & FILEIO_TEST_IS_REGULAR)
                    && !(rcFind.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
            || ((test & FILEIO_TEST_IS_DIR)
                    && (rcFind.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)))
        return TRUE;
    }
  return FALSE;
#else /* not G_OS_WIN32 */
  struct stat buf;

  if (((test & FILEIO_TEST_EXISTS) && !access (file, F_OK))
        || ((test & FILEIO_TEST_IS_EXECUTABLE)
                                        && !access (file, X_OK) && getuid ()))
    return TRUE;
  if (lstat (file, &buf))
    return FALSE;
  return ((test & FILEIO_TEST_IS_SYMLINK) && S_ISLNK (buf.st_mode))
                || ((test & FILEIO_TEST_IS_REGULAR) && S_ISREG (buf.st_mode))
                || ((test & FILEIO_TEST_IS_DIR) && S_ISDIR (buf.st_mode));
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルのフルパスを取得する
    file,ファイル名
     RET,フルパス                                                           */
gchar *
fileio_get_full_path (const gchar *file)
{
#ifdef G_OS_WIN32
  int i = 0, j, s;
  DWORD dwSize;             /* フルパスの長さ */
  HANDLE hFind;             /* 検索ハンドル */
  LPTSTR p;
  TCHAR szFile[MAX_PATH];   /* フルパス */
  TCHAR lpszLongName[MAX_PATH];
  WIN32_FIND_DATA rcf;
# ifdef UNICODE
  gchar *utf8file, *utf16file, filename;
# endif /* UNICODE */

  if (!file)
    return NULL;
# ifdef UNICODE
  utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8file)
    return g_strdup (file);
  utf16file = g_utf8_to_utf16 (utf8file, -1, NULL, NULL, NULL);
    g_free (utf8file);
  if (!utf16file)
    return g_strdup (file);
  dwSize = GetFullPathName (utf16file, MAX_PATH, szFile, &p);
  g_free (utf16file);
# else /* not UNICODE */
  dwSize = GetFullPathName (file, MAX_PATH, szFile, &p);
# endif /* not UNICODE */
  if (dwSize == 0 || dwSize >= MAX_PATH)
    return g_strdup (file);
  if (dwSize >= 2)
    {
      if (szFile[1] == ':')
        {
          lpszLongName[0] = (TCHAR)CharUpper ((LPTSTR)szFile[0]);
          lpszLongName[1] = ':';
          if (szFile[2] == G_DIR_SEPARATOR)
            {
              lpszLongName[2] = G_DIR_SEPARATOR;
              i = 3;
            }
          else
            {
              i = 2;
            }
        }
      else if (szFile[0] == G_DIR_SEPARATOR && szFile[1] == G_DIR_SEPARATOR)
        {
          i = 2;
          while (i < dwSize)
            {
              if (szFile[i] == G_DIR_SEPARATOR)
                break;
# ifdef UNICODE
              i++;
# else /* not UNICODE */
              i += IsDBCSLeadByteEx (CP_ACP, szFile[i]) ? 2 : 1;
# endif /* not UNICODE */
            }
          i++;
          g_memmove (lpszLongName, szFile, i * sizeof (TCHAR));
        }
    }
  lpszLongName[i] = '\0';
  j = i;
  while (i <= dwSize)
    {
      if (szFile[i] == '\0' || szFile[i] == G_DIR_SEPARATOR)
        {
          s = lstrlen (lpszLongName);
          g_memmove (lpszLongName + s, szFile + j, (i - j) * sizeof (TCHAR));
          lpszLongName[s + i - j] = '\0';
          hFind = FindFirstFile (lpszLongName, &rcf);
          if (hFind != INVALID_HANDLE_VALUE)
            FindClose (hFind);
          else
            lstrcpy (rcf.cFileName, _T("."));
          if (lstrcmp (rcf.cFileName, _T(".")))
            {
              g_memmove (lpszLongName + s, rcf.cFileName,
                                    lstrlen (rcf.cFileName) * sizeof (TCHAR));
              s += lstrlen (rcf.cFileName);
            }
          else
            {
              s += i - j;
            }
          if (szFile[i] == G_DIR_SEPARATOR)
            lpszLongName[s++] = G_DIR_SEPARATOR;
          lpszLongName[s] = '\0';
          j = i + 1;
        }
# ifdef UNICODE
      i++;
# else /* not UNICODE */
      i += IsDBCSLeadByteEx (CP_ACP, szFile[i]) ? 2 : 1;
# endif /* not UNICODE */
    }
# ifdef UNICODE
  utf8file = g_utf16_to_utf8 (lpszLongName, -1, NULL, NULL, NULL);
  if (!utf8file)
    return g_strdup (file);
  filename = g_filename_from_utf8 (utf8file, -1, NULL, NULL, NULL);
  g_free (utf8file);
  return filename ? filename : g_strdup (file);
# else /* not UNICODE */
  return g_strdup (lpszLongName);
# endif /* not UNICODE */
#else /* not G_OS_WIN32 */
  gchar *dir, *path;
  gint i, j, leng;

  if (!file)
    return NULL;
  if (file[0] != G_DIR_SEPARATOR)
    {
      dir = g_get_current_dir ();
      path = g_strjoin (G_DIR_SEPARATOR_S, dir, file, NULL);
      g_free (dir);
    }
  else
    {
      path = g_strdup (file);
    }
  leng = g_strlen (path);
  i = 0;
  while (i < leng)
    if (path[i] == G_DIR_SEPARATOR && path[i + 1] == G_DIR_SEPARATOR)
      {
        leng--;
        g_memmove (path + i + 1, path + i + 2, leng - i);
      }
    else
      {
        i++;
      }
  i = 0;
  while (i < leng)
    if (path[i] == G_DIR_SEPARATOR && path[i + 1] == '.'
                                            && path[i + 2] == G_DIR_SEPARATOR)
      {
        leng -= 2;
        g_memmove (path + i + 1, path + i + 3, leng - i);
      }
    else
      {
        i++;
      }
  i = 0;
  while (i < leng)
    if (path[i] == G_DIR_SEPARATOR && path[i + 1] == '.'
                    && path[i + 2] == '.' && path[i + 3] == G_DIR_SEPARATOR)
      {
        leng -= 3;
        g_memmove (path + i + 1, path + i + 4, leng - i);
        for (j = i - 1; j >= 0; j--)
          if (path[j] == G_DIR_SEPARATOR)
            break;
        if (j >= 0)
          {
            g_memmove (path + j + 1, path + i + 1, leng - i);
            leng -= i - j;
            i = j;
          }
      }
    else
      {
        i++;
      }
  return path;
#endif /* not G_OS_WIN32 */
}


#ifndef G_OS_WIN32
static GList *glist_fileio = NULL;
# ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
# endif /* USE_THREAD */
#endif /* not G_OS_WIN32 */


/*  ja:ファイルの作成
      file,ファイル名
    access,アクセス
     share,共有
      mode,モード
       RET,ファイルポインタ,NULL:エラー                                     */
FileIO *
fileio_open (const gchar *file,
             const guint access,
             const guint share,
             const guint mode)
{
  FileIO *fio;
#ifdef G_OS_WIN32
  HANDLE hFile;
# ifdef UNICODE
  gchar *utf8file, *utf16file;
  LPTSTR lpszFile;
# endif /* UNICODE */

  if (!file)
    return NULL;
# ifdef UNICODE
  utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8file)
    return NULL;
  utf16file = g_utf8_to_utf16 (utf8file, -1, NULL, NULL, NULL);
  g_free (utf8file);
  if (!utf16file)
    return NULL;
  hFile = CreateFile (utf16file, access, share, NULL, mode,
                                                FILE_ATTRIBUTE_NORMAL, NULL);
  g_free (utf16file);
# else /* not UNICODE */
  hFile = CreateFile (file, access, share, NULL, mode,
                                                FILE_ATTRIBUTE_NORMAL, NULL);
# endif /* not UNICODE */
  if (hFile == INVALID_HANDLE_VALUE)
    return NULL;
  fio = g_malloc (sizeof (FileIO));
  fio->hFile = hFile;
#else /* not G_OS_WIN32 */
  int fd, flags;
  gchar *path;
  gint i;
  guint length;

  /* ja:フルパスを求める */
  path = fileio_get_full_path (file);
  if (!path)
    return NULL;
  /* ja:フラグ設定 */
  switch (access)
    {
      case FILEIO_ACCESS_READ:  flags = O_RDONLY;   break;
      case FILEIO_ACCESS_WRITE: flags = O_WRONLY;   break;
      case FILEIO_ACCESS_ALL:   flags = O_RDWR;     break;
      default:                  g_free (path);      return NULL;
    }
  switch (mode)
    {
      case FILEIO_MODE_CREATE_NEW:      /* ja:無ければ新規作成,あれば失敗 */
        flags |= O_CREAT | O_EXCL;
        break;
      case FILEIO_MODE_CREATE_ALWAYS:   /* ja:常に新規作成 */
        flags |= O_CREAT;
        break;
      case FILEIO_MODE_OPEN_EXISTING:   /* ja:あれば開く,無ければ失敗 */
        break;
      case FILEIO_MODE_OPEN_ALWAYS:     /* ja:あれば開く,無ければ新規作成 */
        flags |= O_CREAT;
        break;
      default:
        g_free (path);
        return NULL;
    }
  /* ja:共有をチェック */
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  length = g_list_length (glist_fileio);
  for (i = 0; i < length; i++)
    {
      fio = g_list_nth_data (glist_fileio, i);
      if (!g_strcmp (fio->file, path) && (fio->share & access) != access)
        break;
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  if (i < length)
    {
      g_free (path);
      return NULL;
    }

  fd = open (path, flags,
                    S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
  if (fd == -1)
    {
      g_free (path);
      return NULL;
    }
  if (
# ifdef HAVE_FLOCK
      flock (fd,
        (access == FILEIO_ACCESS_READ ? LOCK_SH : LOCK_EX) | LOCK_NB) != 0 ||
# endif /* HAVE_FLOCK */
        (mode == FILEIO_MODE_CREATE_ALWAYS && (
# ifdef HAVE_FTRUNCATE
                    ftruncate (fd, 0) || 
# endif /* HAVE_FTRUNCATE */
                    lseek (fd, 0, SEEK_SET) == -1)))
    {
      close (fd);
      g_free (path);
      return NULL;
    }
  fio = g_malloc (sizeof (FileIO));
  fio->fd = fd;
  fio->file = path;
  fio->share = share;
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  glist_fileio = g_list_append (glist_fileio, fio);
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
#endif /* not G_OS_WIN32 */
  return fio;
}


/*  ja:ファイルポインタを閉じる
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_close (FileIO *fio)
{
  gboolean result;

#ifdef G_OS_WIN32
  result = fio && CloseHandle (fio->hFile);
  g_free (fio);
#else /* not G_OS_WIN32 */
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (fio && g_list_find (glist_fileio, fio))
    {
      glist_fileio = g_list_remove (glist_fileio, fio);
      result = !close (fio->fd);
      g_free (fio->file);
      g_free (fio);
    }
  else
    {
      result = FALSE;
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
#endif /* not G_OS_WIN32 */
  return result;
}


/*  ja:ファイルから読み込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_read (FileIO       *fio,
             gpointer      data,
             const gssize  length)
{
#ifdef G_OS_WIN32
  DWORD dwRead;

  return fio && ReadFile (fio->hFile, data, length, &dwRead, NULL)
                                                                ? dwRead : -1;
#else /* not G_OS_WIN32 */
  gssize result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  result = fio && g_list_find (glist_fileio, fio) != NULL
                                        ? read (fio->fd, data, length) : -1;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルに書き込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_write (FileIO       *fio,
              gpointer      data,
              const gssize  length)
{
#ifdef G_OS_WIN32
  DWORD dwWrite;

  return fio && WriteFile (fio->hFile, data, length, &dwWrite, NULL)
                                                                ? dwWrite : -1;
#else /* not G_OS_WIN32 */
  gssize result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  result = fio && g_list_find (glist_fileio, fio)
                                        ? write (fio->fd, data, length) : -1;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルをシークする
       fio,ファイルポインタ
    offset,位置
    whence,モード
       RET,位置,-1:エラー                                                   */
goffset
fileio_seek (FileIO        *fio,
             const goffset  offset,
             const guint    whence)
{
#ifdef G_OS_WIN32
# ifdef G_HAVE_GINT64
  LONG lHigh, lLow;

  if (!fio)
    return -1;
  lHigh = offset >> 32;
  lLow = SetFilePointer (fio->hFile, (LONG)offset, &lHigh, whence);
  return lLow != 0xFFFFFFFF || GetLastError () == NO_ERROR
                                ? ((gint64)lHigh << 32) | (guint32)lLow : -1;
# else /* not G_HAVE_GINT64 */
  return SetFilePointer (fio->hFile, offset, NULL, whence);
# endif /* not G_HAVE_GINT64 */
#else /* not G_OS_WIN32 */
  gssize result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  result = fio && g_list_find (glist_fileio, fio)
                                        ? lseek (fio->fd, offset, whence) : -1;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
#endif /* not G_OS_WIN32 */
}


/*  ja:1文字読み込む
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_getc (FileIO *fio)
{
  gchar c;

  return fileio_read (fio, &c, sizeof (gchar)) == sizeof (gchar) ? c : EOF;
}


/*  ja:1文字書き込む
      c,文字
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_putc (gint c, FileIO *fio)
{
  return fileio_write (fio, &c, sizeof (gchar)) == sizeof (gchar) ? c : EOF;
}


/*  ja:1行読み込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,data:正常終了,NULL:エラー                                        */
gchar *
fileio_gets (gchar       *data,
             const gsize  length,
             FileIO      *fio)
{
  gchar c;
  gsize i = 0;
  gssize result;

  if (!data || length <= 0)
    return NULL;
  while (i <= length - 2)
    {
      result = fileio_read (fio, &c, sizeof (gchar));
      if (result < 0)
        return NULL;
      if (result < sizeof (gchar))
        break;
      data[i++] = c;
      if (c == '\n')
        break;
    }
  if (i == 0)
    return NULL;
  data[i] = '\0';
  return data;
}


/*  ja:1行書き込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,正数:正常終了,EOF:エラー                                         */
gint
fileio_puts (gchar  *data,
             FileIO *fio)
{
  gsize i;
  gssize result;

  if (!data)
    return EOF;
  for (i = 0; data[i] != '\0'; i++);
  result = fileio_write (fio, data, i * sizeof (gchar));
  return result >= 0 ? result : EOF;
}


/*  ja:ファイルの時刻を取得する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_gettime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime)
{
#ifdef G_OS_WIN32
# ifdef UNICODE
  gchar *utf8file;
  LPTSTR lpszFile;
# endif /* UNICODE */
  HANDLE hFind;
  WIN32_FIND_DATA rcFind;
#else /* not G_OS_WIN32 */
  struct stat buf;
#endif /* not G_OS_WIN32 */

  if (!file)
    return FALSE;
#ifdef G_OS_WIN32
# ifdef UNICODE
  utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8file)
    return FALSE;
  lpszFile = g_utf8_to_utf16 (utf8file, -1, NULL, NULL, NULL);
  if (!lpszFile)
    return FALSE;
# else /* not UNICODE */
#   define lpszFile file
# endif /* not UNICODE */

  hFind = FindFirstFile (lpszFile, &rcFind);
# ifdef UNICODE
  g_free (lpszFile);
# else /* not UNICODE */
#   undef lpszFile
# endif /* not UNICODE */
  if (hFind == INVALID_HANDLE_VALUE || !FindClose (hFind))
    return FALSE;
  if (atime)
    *atime = rcFind.ftLastAccessTime;
  if (mtime)
    *mtime = rcFind.ftLastWriteTime;
#else /* not G_OS_WIN32 */
  if (stat (file, &buf))
    return FALSE;
  if (atime)
    *atime = buf.st_atime;
  if (mtime)
    *mtime = buf.st_mtime;
#endif /* not G_OS_WIN32 */
  return TRUE;
}


/*  ja:ファイルの時刻を設定する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_settime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime)
{
#ifdef G_OS_WIN32
# ifdef UNICODE
  gchar *utf8file;
  LPTSTR lpszFile;
# endif /* UNICODE */
  HANDLE hFile;
#else /* not G_OS_WIN32 */
  struct stat sbuf;
  struct utimbuf ubuf;
#endif /* not G_OS_WIN32 */

  if (!file)
    return FALSE;
#ifdef G_OS_WIN32
# ifdef UNICODE
  utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8file)
    return FALSE;
  lpszFile = g_utf8_to_utf16 (utf8file, -1, NULL, NULL, NULL);
  if (!lpszFile)
    return FALSE;
# else /* not UNICODE */
#   define lpszFile file
# endif /* not UNICODE */
  hFile = CreateFile (lpszFile, GENERIC_WRITE,
                                FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
# ifdef UNICODE
  g_free (lpszFile);
# else /* not UNICODE */
#   undef lpszFile
# endif /* not UNICODE */
  if (hFile == INVALID_HANDLE_VALUE)
    return FALSE;
  if (!SetFileTime (hFile, NULL, atime, mtime))
    {
      CloseHandle (hFile);
      return FALSE;
    }
  return CloseHandle (hFile);
#else /* not G_OS_WIN32 */
# ifdef HAVE_UTIME
  if (stat (file, &sbuf))
    return FALSE;
  ubuf.actime  = atime ? *atime : sbuf.st_atime;
  ubuf.modtime = mtime ? *mtime : sbuf.st_mtime;
  return !utime (file, &ubuf);
# else /* not HAVE_UTIME */
  return FALSE;
# endif /* not HAVE_UTIME */
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルの時刻を変換する
    ftime,ファイルの時刻
    stime,システム時間                                                      */
void
fileio_getsystime (FileIOTime    *ftime,
                   FileIOSysTime *stime)
{
#ifdef G_OS_WIN32
  SYSTEMTIME st;

  FileTimeToSystemTime (ftime, &st);
  stime->year = st.wYear;
  stime->mon  = st.wMonth;
  stime->mday = st.wDay;
  stime->hour = st.wHour;
  stime->min  = st.wMinute;
  stime->sec  = st.wSecond;
  stime->wday = st.wDayOfWeek;
#else /* not G_OS_WIN32 */
  struct tm *gmt;

  gmt = gmtime (ftime);
  stime->year = gmt->tm_year + 1900;
  stime->mon  = gmt->tm_mon + 1;
  stime->mday = gmt->tm_mday;
  stime->hour = gmt->tm_hour;
  stime->min  = gmt->tm_min;
  stime->sec  = gmt->tm_sec;
  stime->wday = gmt->tm_wday;
#endif /* not G_OS_WIN32 */
}
