/*
    avicore
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef __ICM_H__
#define __ICM_H__


#include "common.h"
#include "bitmap.h"
#include "chunk.h"
#ifdef W32CODECDIR
# include "icm32.h"
#endif /* W32CODECDIR */
#ifdef G_OS_WIN32
# include <tchar.h>
# include <windows.h>
# include <vfw.h>
#endif /* G_OS_WIN32 */


#ifdef __cplusplus
extern "C" {
#endif


#define ICM_ATTR_NATIVE             0
#define ICM_ATTR_WIN32              1

#define ICM_FLAG_QUALITY            0x0001
#define ICM_FLAG_CRUNCH             0x0002
#define ICM_FLAG_TEMPORAL           0x0004
#define ICM_FLAG_COMPRESSFRAMES     0x0008
#define ICM_FLAG_DRAW               0x0010
#define ICM_FLAG_FASTTEMPORALC      0x0020
#define ICM_FLAG_FASTTEMPORALD      0x0080
#define ICM_FLAG_QUALITYTIME        0x0040

#define ICM_MODE_COMPRESS           1
#define ICM_MODE_DECOMPRESS         2
#define ICM_MODE_FASTDECOMPRESS     3
#define ICM_MODE_QUERY              4
#define ICM_MODE_FASTCOMPRESS       5
#define ICM_MODE_DRAW               8

#define ICM_DECOMPRESS_HURRYUP      0x80000000
#define ICM_DECOMPRESS_UPDATE       0x40000000
#define ICM_DECOMPRESS_PREROLL      0x20000000
#define ICM_DECOMPRESS_NULLFRAME    0x10000000
#define ICM_DECOMPRESS_NOTKEYFRAME  0x08000000


typedef struct IcmDriver_Tag
{
#if defined (G_OS_WIN32)
  LPWSTR lpszName;
#elif defined (W32CODECDIR)
  PE_image *hLib;
  LONG lOpenID;
  LONG WINAPI (*DriverProc)(DWORD  dwDriverId,
                            HANDLE hdrvr,
                            UINT   msg,
                            LONG   lParam1,
                            LONG   lParam2);
#else /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
  guint8 dummy;
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
} IcmDriver;


typedef struct IcmObject_Tag
{
  gint counter;
  guint32 handler;
  gint mode;
  BitmapInfoHeader *bmih_in;
  BitmapInfoHeader *bmih_out;
  gsize size;
  gint frame;
  gint key_point;
  gint key_frame;
  gint quality;
  gpointer prev;
  gsize prev_size;
  IcmDriver *icm_driver;
#ifdef G_OS_WIN32
  HDRVR hDrvr;
#endif /* G_OS_WIN32 */
#ifdef W32CODECDIR
  DWORD dwDriverId;
#endif /* W32CODECDIR */
} IcmObject;


typedef struct IcmInfo_Tag
{
  gchar *name;
  guint32 handler;
  guint32 flags;
  guint32 attribute;
} IcmInfo;


/******************************************************************************
*                                                                             *
* ja:ICMマクロ                                                                *
*                                                                             *
******************************************************************************/
/*  ja:Codecの参照数を増やす
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
#define icm_ref(icm_object)                                                 \
        ((icm_object)!=NULL?((icm_object)->icm_object->counter++,TRUE):FALSE)


/*  ja:Codecの参照数を減らす
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
#define icm_unref(icm_object) icm_close(icm_object)


/*  ja:Codecをコピーする
    icm_object,IcmObject
           RET,IcmObject,NULL:エラー                                        */
#define icm_dup(icm_object) ((icm_object)!=NULL                             \
                    ?icm_open((icm_object)->handler,(icm_object)->mode):NULL)


/*  ja:Codecの4文字コードを取得する
    icm_object,IcmObject
           RET,4文字コード,0:エラー                                         */
#define icm_get_handler(icm_object) ((icm_object)?(icm_object)->handler:0)


/******************************************************************************
*                                                                             *
* ja:ICM初期化                                                                *
*                                                                             *
******************************************************************************/
/*  ja:ICMを初期化する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_init (void);


/*  ja:ICMを終了する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_exit (void);


/******************************************************************************
*                                                                             *
* ja:ICM4文字コード処理                                                       *
*                                                                             *
******************************************************************************/
/*  ja:4文字コード→文字列
    handler,4文字コード
        RET,文字列                                                          */
gchar *
icm_handler_to_string (const guint32 handler);


/*  ja:文字列→4文字コード
    text,文字列
     RET,4文字コード                                                        */
guint32
icm_string_to_handler (const gchar *text);


/******************************************************************************
*                                                                             *
* ja:ICM基本                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:登録されている4文字コードを取得する
    RET,4文字コードのリスト,NULL:登録なし                                   */
GList *
icm_get_handler_list (void);


/*  ja:Codecを開く
    handler,4文字コード
       mode,モード
        RET,IcmObject,NULL:エラー                                           */
IcmObject *
icm_open (const guint32 handler,
          const gint    mode);


/*  ja:Codecを閉じる
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_close (IcmObject *icm_object);


/*  ja:Codecの情報を取得する
    icm_object,IcmObject
      icm_info,情報
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_get_info (IcmObject *icm_object,
              IcmInfo   *icm_info);


/*  ja:パラメータのサイズを取得する
    icm_object,IcmObject
           RET,サイズ                                                       */
gsize
icm_get_state_size (IcmObject *icm_object);


/*  ja:パラメータを取得する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_get_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size);


/*  ja:パラメータを設定する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_set_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size);


/******************************************************************************
*                                                                             *
* ja:ICMダイアログ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:Aboutダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_about (IcmObject *icm_object);


/*  ja:Aboutダイアログを表示する
    icm_object,IcmObject                                                    */
void
icm_dialog_about (IcmObject *icm_object);


/*  ja:設定ダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_configure (IcmObject *icm_object);


/*  ja:設定ダイアログを表示する
    icm_object,IcmObject                                                    */
void
icm_dialog_configure (IcmObject *icm_object);


/******************************************************************************
*                                                                             *
* en:ICM Compress                                                             *
*                                                                             *
******************************************************************************/
/*  ja:圧縮可能か確認する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
icm_compress_query (IcmObject              *icm_object,
                    const BitmapInfoHeader *bmih);


/*  ja:圧縮のデフォルトの周期を取得する
    icm_object,IcmObject
           RET,デフォルト(0:未対応)                                         */
gint
icm_compress_get_default_key_frame (IcmObject *icm_object);


/*  ja:圧縮のデフォルトの品質を取得する
    icm_object,IcmObject
           RET,デフォルト(-1:未対応)                                        */
gint
icm_compress_get_default_quality (IcmObject *icm_object);


/*  ja:圧縮後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_format_size (IcmObject              *icm_object,
                              const BitmapInfoHeader *bmih);


/*  ja:圧縮後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_get_format (IcmObject              *icm_object,
                         const BitmapInfoHeader *bmih_in,
                         BitmapInfoHeader       *bmih_out);


/*  ja:圧縮後のデータの最大サイズを取得する
    icm_object,IcmObject
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_size (IcmObject *icm_object);


/*  ja:圧縮を設定する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
     bmih_size,圧縮されたビットマップのヘッタのサイズ
     key_frame,周期(0:すべてキーフレーム)
       quality,品質0～10000,-1:デフォルト
     data_rate,1秒間のサイズ(0:無効)
          rate,レート
         scale,スケール
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_frames_info (IcmObject              *icm_object,
                          const BitmapInfoHeader *bmih_in,
                          const BitmapInfoHeader *bmih_out,
                          const gsize             bmih_size,
                          const gint              key_frame,
                          const gint              quality,
                          const gsize             data_rate,
                          const guint32           rate,
                          const guint32           scale);


/*  ja:圧縮を開始する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_begin (IcmObject *icm_object);


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_end (IcmObject *icm_object);


/*  ja:圧縮する
    icm_object,IcmObject
     key_frame,TRUE:キーフレーム,FALSE:非キーフレーム
         input,圧縮するデータ
        output,圧縮されたデータ
           RET,フレームのサイズ,0:エラー                                    */
gsize
icm_compress (IcmObject *icm_object,
              gboolean  *key_frame,
              gpointer   input,
              gpointer   output);


/******************************************************************************
*                                                                             *
* en:ICM Decompress                                                           *
*                                                                             *
******************************************************************************/
/*  ja:展開可能か確認する
    icm_object,IcmObject
         bmih,圧縮されたビットマップのヘッタ
          RET,TRUE:可能,FALSE:不可能                                        */
gboolean
icm_decompress_query (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih);


/*  ja:展開後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_decompress_get_format_size (IcmObject              *icm_object,
                                const BitmapInfoHeader *bmih);


/*  ja:展開後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮されたビットマップのヘッタ
      bmih_out,展開されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_get_format (IcmObject              *icm_object,
                           const BitmapInfoHeader *bmih_in,
                           BitmapInfoHeader       *bmih_out);


/*  ja:展開を開始する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_begin (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih);


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_end (IcmObject *icm_object);


/*  ja:展開する
    icm_object,IcmObject
         flags,フラグ
         input,圧縮されたデータ
        output,展開されたデータ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress (IcmObject   *icm_object,
                const guint  flags,
                gpointer     input,
                gpointer     output);


#ifdef __cplusplus
}
#endif


#endif
