/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "avibase.h"
#include "avifmt.h"


#define AVI_HEADERSIZE 2048
#define AVI_ALIGNSIZE (AVI_HEADERSIZE * 8)
#define AVI_RIFFSIZE (1024 * 1024 * 1024)
#define AVI_SUPERINDEX_ENTRY                                                \
                ((AVI_ALIGNSIZE - sizeof (guint32) * 2 - ASI_SIZE) / ASE_SIZE)


/******************************************************************************
*                                                                             *
* ja:保存関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:保存用のオプションを解放する
    avi_save,オプション
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_save_free (AviSave *avi_save)
{
  if (!avi_save)
    return FALSE;
  if (avi_save->avi_opt)
    {
      gint i;

      for (i = 0; avi_save->avi_opt[i].type != AVI_TYPE_UNKNOWN; i++)
        {
          g_free (avi_save->avi_opt[i].param);
          g_free (avi_save->avi_opt[i].bmih);
          g_free (avi_save->avi_opt[i].wfx);
        }
      g_free (avi_save->avi_opt);
    }
  g_free (avi_save);
  return TRUE;
}


/*  ja:すべて保存されるかチェックする
    avi_edit,AVI編集ハンドル
    avi_save,オプション
         RET,TRUE:すべて保存される,FALSE:すべて保存されない                 */
gboolean
avi_save_is_all (AviEdit **avi_edit,
                 AviSave  *avi_save)
{
  return avi_edit && avi_edit[0] && avi_save
    && (avi_save->type == AVI_FILE_AVI || avi_save->type == AVI_FILE_SCENARIO
                        || (!avi_edit[0] && avi_save->type == AVI_FILE_WAVE));
}


typedef struct _AviSaveAviStream
{
  gboolean active;          /* ja:TRUE:圧縮on,FALSE:圧縮off */
  gboolean recompress;      /* ja:TRUE:次のキーフレームまで再圧縮中 */
  gsize bmih_size;          /* ja:圧縮されたときの最大サイズ */
  gsize max;                /* ja:書き込んだチャンクの最大サイズ */
  gpointer fmt;             /* ja:ストリームのヘッタ */
  gsize fmt_size;           /* ja:ストリームのヘッタのサイズ */
  gint pos;                 /* ja:ストリームの現在位置 */
  guint32 fourcc;           /* ja:4文字コード */
  AviFrame *avi_frame;
  AviPcm *avi_pcm;
  IcmObject *icm_object;
  AviSuperIndexEntry super[AVI_SUPERINDEX_ENTRY];
  AviBaseIndexEntry *base;
  gint entries;             /* ja:AviBaseIndexEntryの数 */
  gint samples;             /* ja:RIFF内のサンプル数 */
  gint written;             /* ja:書き込んだサンプル数 */
} AviSaveAviStream;


/*  ja:ファイルに書き込む
          fio,ファイルポインタ
         data,データ
       length,dataのバイト数
      samples,サンプル数
    key_frame,0:キーフレーム,1:非キーフレーム,2:オーディオ
        index,AviIndexEntry構造体
      entries,AviIndexEntry構造体の数
         riff,現在のRIFFのオフセット
         movi,現在のmoviのオフセット
       stream,ストリームのデータ
      streams,ストリームの数
      current,データの属するストリーム
          RET,TRUE:正常終了,FALSE:エラー                                    */
static gboolean
avi_save_avi_write (FileIO            *fio,
                    gpointer           data,
                    gsize              length,
                    gint               samples,
                    gint               key_frame,
                    AviIndexEntry    **index,
                    gint              *entries,
                    goffset           *riff,
                    goffset           *movi,
                    AviSaveAviStream  *stream, 
                    gint               streams,
                    gint               current)
{
  gint i, s, size = -1;
  guint8 *p, *dummy;
  guint32 fcc, cb, type;
  goffset offset;

  if (!fio || !index || !riff || !movi || !stream || streams <= 0)
    return FALSE;
  offset = fileio_seek (fio, 0, FILEIO_SEEK_CUR);
  if (offset == -1)
    return FALSE;
  if (length > 0 && data)
    {
      if (current < 0 || streams < current || samples <= 0)
        return FALSE;
      /* ja:書き込み後の現在のRIFFのデータサイズを計算する */
      size = offset - *riff - sizeof (guint32) * 2          /* ja:既存データ */
                + sizeof (guint32)                          /* ja:FOURCC */
                + sizeof (guint32)                          /* ja:サイズ */
                + ((length + 1) & ~1);                      /* ja:データ */
      /* ja:各ストリームのAviBaseIndexを加える */
      for (i = 0; i < streams; i++)
        if (stream[i].entries > 0 || i == current)
          {
            s = sizeof (guint32) * 2 + ABI_SIZE + stream[i].entries * ABE_SIZE;
            if (i == current)
              s += ABE_SIZE;
            if (s % AVI_ALIGNSIZE != 0)
              s = (s / AVI_ALIGNSIZE + 1) * AVI_ALIGNSIZE;
            size += s;
          }
      /* ja:最初のRIFFならばidx1チャンクも加える */
      if (*riff <= 0)
        size += sizeof (guint32) * 2 + (*entries + 1) * AIE_SIZE;
    }
  if (size < 0 || AVI_RIFFSIZE <= size)
    { /* ja:RIFFを更新する処理 */
      if (*riff <= 0)
        { /* ja:最初のRIFF */
          /* ja:すべてのストリームは出力されている */
          for (i = 0; i < streams; i++)
            if (stream[i].entries <= 0)
              return FALSE;
        }
      else
        {
          /* ja:最低でも1つのストリームは出力されている */
          for (i = 0; i < streams; i++)
            if (stream[i].entries > 0)
              break;
          if (i >= streams)
            return FALSE;
        }
      /* ja:ix??の出力 */
      for (i = 0; i < streams; i++)
        if (stream[i].entries > 0)
          {
            gint j;
            AviBaseIndex abi;

            s = stream[i].entries * ABE_SIZE;
            size = sizeof (guint32) * 2 + ABI_SIZE + s;
            if (size % AVI_ALIGNSIZE != 0)
              size = (size / AVI_ALIGNSIZE + 1) * AVI_ALIGNSIZE;
            /* en:AviSuperIndexEntry */
            for (j = 0, p = (guint8 *)stream[i].super;
                                j < AVI_SUPERINDEX_ENTRY; j++, p += ASE_SIZE)
              if (ase_get_size (p) == 0)
                break;
            if (j >= AVI_SUPERINDEX_ENTRY)
              return FALSE;
            ase_set_offset (p, offset);
            ase_set_size (p, size);
            ase_set_duration (p, stream[i].samples);
            stream[i].samples = 0;
            /* en:AviBaseIndex */
            offset += size;
            size -= sizeof (guint32) * 2;
            fcc = GUINT32_TO_LE (make4cc ('i', 'x',
                                                i / 10 + '0', i % 10 + '0'));
            cb = GUINT32_TO_LE (size);
            abi_set_longs (&abi, 2);
            abi_set_sub_type (&abi, AVI_IDX_SUB_DEFAULT);
            abi_set_type (&abi, AVI_IDX_OF_CHUNKS);
            abi_set_entries (&abi, stream[i].entries);
            abi_set_ckid (&abi, stream[i].fourcc);
            abi_set_offset (&abi, *movi);
            abi_set_reserve (&abi, 0);
            if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
                    || fileio_write (fio, &cb, sizeof (guint32))
                                                            != sizeof (guint32)
                    || fileio_write (fio, &abi, ABI_SIZE) != ABI_SIZE
                    || fileio_write (fio, stream[i].base, s) != s)
              return FALSE;
            /* ja:ダミー */
            size -= ABI_SIZE + s;
            if (size > 0)
              {
                dummy = g_malloc0 (size);
                if (fileio_write (fio, dummy, size) != size)
                  {
                    g_free (dummy);
                    return FALSE;
                  }
                g_free (dummy);
              }
            stream[i].entries = 0;
          }
      /* moviのサイズ */
      cb = GUINT32_TO_LE (offset - *movi - sizeof (guint32) * 2);
      if (fileio_seek (fio, *movi + sizeof (guint32), SEEK_SET) == -1
            || fileio_write (fio, &cb, sizeof (guint32)) != sizeof (guint32))
        return FALSE;
      if (*riff <= 0)
        { /* 最初のRIFF */
          /* インデックス */
          s = *entries * AIE_SIZE;
          fcc = GUINT32_TO_LE (CK_ID_AVINEWINDEX);
          cb = GUINT32_TO_LE (s);
          if (!*index || *entries < 0
                                || fileio_seek (fio, 0, FILEIO_SEEK_END) == -1
                                || fileio_write (fio, &fcc, sizeof (guint32))
                                                            != sizeof (guint32)
                                || fileio_write (fio, &cb, sizeof (guint32))
                                                            != sizeof (guint32)
                                || fileio_write (fio, *index, s) != s)
            return FALSE;
          offset += sizeof (guint32) * 2 + s;
          g_free (*index);
          *index = NULL;
          *entries = 0;
        }
      /* ja:RIFFのサイズ */
      cb = GUINT32_TO_LE (offset - *riff - sizeof (guint32) * 2);
      if (fileio_seek (fio, *riff + sizeof (guint32), SEEK_SET) == -1
            || fileio_write (fio, &cb, sizeof (guint32)) != sizeof (guint32)
                                || fileio_seek (fio, 0, FILEIO_SEEK_END) == -1)
        return FALSE;
      /* ja:新しいRIFF */
      if (length > 0 && data)
        {
          *riff = offset;
          fcc = GUINT32_TO_LE (CK_DEF_RIFF);
          type = GUINT32_TO_LE (CK_FORM_AVIX);
          if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
                || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
                || fileio_write (fio, &type, sizeof (guint32))
                                                        != sizeof (guint32))
            return FALSE;
          offset += sizeof (guint32) * 3;
          if ((offset + sizeof (guint32) * 3) % AVI_HEADERSIZE != 0)
            { /* en:JUNK */
              size = ((offset + sizeof (guint32) * 5) / AVI_HEADERSIZE + 1)
                            * AVI_HEADERSIZE - (offset + sizeof (guint32) * 5);
              fcc = GUINT32_TO_LE (CK_ID_AVIPADDING);
              cb = GUINT32_TO_LE (size);
              dummy = g_malloc0 (size);
              if (fileio_write (fio, &fcc, sizeof (guint32))
                                                            != sizeof (guint32)
                                || fileio_write (fio, &cb, sizeof (guint32))
                                                            != sizeof (guint32)
                                    || fileio_write (fio, dummy, size) != size)
                {
                  g_free (dummy);
                  return FALSE;
                }
              g_free (dummy);
              offset += sizeof (guint32) * 2 + size;
            }
          *movi = offset;
          fcc = GUINT32_TO_LE (CK_DEF_LIST);
          type = GUINT32_TO_LE (CK_LIST_AVIMOVIE);
          if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
                || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
                || fileio_write (fio, &type, sizeof (guint32))
                                                        != sizeof (guint32))
            return FALSE;
          offset += sizeof (guint32) * 3;
        }
    }
  /* ja:データを書き加える */
  if (length >= 0 && data)
    {
      const static guint8 dmy = 0;

      /* en:AviBaseIndexEntry */
      stream[current].base = g_realloc (stream[current].base,
                                    (stream[current].entries + 1) * ABE_SIZE);
      p = (guint8 *)stream[current].base
                                        + stream[current].entries++ * ABE_SIZE;
      abe_set_offset (p, offset + sizeof (guint32) * 2 - *movi);
      abe_set_size (p, length | (key_frame == 1 ? 1 << 31 : 0));
      if (*riff <= 0)
        { /* ja:最初のRIFF */
          /* ja:インデックス */
          *index = g_realloc (*index, (*entries + 1) * AIE_SIZE);
          p = (guint8 *)*index + (*entries)++ * AIE_SIZE;
          aie_set_type (p, stream[current].fourcc);
          aie_set_flags (p, key_frame == 0 ? AVI_IF_KEYFRAME : 0);
          aie_set_offset (p, offset - *movi - sizeof (guint32) * 2);
          aie_set_length (p, length);
        }
      /* ja:データ */
      fcc = GUINT32_TO_LE (stream[current].fourcc);
      cb = GUINT32_TO_LE ((length + 1) & ~1);
      if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
            || fileio_write (fio, &cb, sizeof (guint32)) != sizeof (guint32)
            || fileio_write (fio, data, length) != length
            || (length % 2 != 0
            && fileio_write (fio, &dmy, sizeof (guint8)) != sizeof (guint8)))
        return FALSE;
      stream[current].samples += samples;
      if (stream[current].max < length)
        stream[current].max = length;
    }
  stream[current].written += samples;
  return TRUE;
}


/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
      avi_opt,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_avi (AviEdit      *avi_edit[],
              const gchar  *file,
              AviOptions   *avi_opt,
              gboolean    (*func)(gint, gpointer),
              gpointer      user_data)
{
  FileIO *fio;
  gint current;         /* ja:最も1秒間のフレーム数が少ないストリーム */
  gint longer = 0;      /* ja:最も時間が長いストリーム */
  gint times;           /* ja:最も1秒間のフレーム数が少ないストリームの時間 */
  gint timer = 0;       /* ja:current終了後の書き込みの基準となる時間 */
  gint streams;         /* ja:ストリームの数 */
  gint head;            /* ja:アライメントされたヘッタのサイズ */
  gint header;          /* ja:ヘッタのサイズ */
  gint i, j, length, samples;
  guint8 *p, *data = NULL;
  guint32 size;
  AviIndexEntry *index = NULL;
  gint entries = 0;     /* ja:AviIndexEntry構造体の数 */
  goffset riff = 0;     /* ja:RIFFのオフセット */
  goffset movi;         /* ja:moviのオフセット */
  AviSaveAviStream *stream;
  static guint32 fcc0 = GUINT32_TO_LE (CK_DEF_RIFF);
  static guint32 type0 = GUINT32_TO_LE (CK_FORM_AVI);
  static guint32 fcc1 = GUINT32_TO_LE (CK_DEF_LIST);
  static guint32 type1 = GUINT32_TO_LE (CK_LIST_AVIMOVIE);

  if (!avi_edit || !file)
    return FALSE;
  for (streams = 0; avi_edit[streams]; streams++);
  if (streams <= 0)
    return FALSE;
  if (streams == 1 && avi_type (avi_edit[0]) == AVI_TYPE_AUDIO)
    return avi_save_wave (avi_edit[0], file, func, user_data);
  stream = g_malloc0 (streams * sizeof (AviSaveAviStream));

  /* ja:ヘッタサイズを計算する */
  header = sizeof (guint32)                                     /* en:'RIFF' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + sizeof (guint32)                                  /* en:'AVI ' */
                + sizeof (guint32)                              /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32)                              /* en:'hdrl' */
                    + sizeof (guint32)                          /* en:'avih' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((AMH_SIZE + 1) & ~1)                     /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'LIST' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + sizeof (guint32)                          /* en:'odml' */
                        + sizeof (guint32)                      /* en:'odmh' */
                        + sizeof (guint32)                      /* ja:サイズ */
                        + DML_SIZE                              /* ja:構造体 */
                + sizeof (guint32)                              /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32);                             /* en:'movi' */
  for (i = 0; i < streams; i++)
    {
      switch (avi_type (avi_edit[i]))
        {
          case AVI_TYPE_VIDEO:/* ja:ビデオ */
            if (avi_opt && avi_opt[i].handler != 0
                    && (stream[i].icm_object = icm_open (avi_opt[i].handler,
                                                            ICM_MODE_COMPRESS))
                    && (avi_opt[i].param_size <= 0
                            || icm_set_state (stream[i].icm_object,
                                    avi_opt[i].param, avi_opt[i].param_size))
                    && icm_compress_frames_info (stream[i].icm_object,
                                                 avi_edit[i]->bmih,
                                                 avi_opt[i].bmih,
                                                 avi_opt[i].bmih_size,
                                                 avi_opt[i].key_frame,
                                                 avi_opt[i].quality,
                                                 avi_opt[i].data_rate,
                                                 avi_get_rate (avi_edit[i]),
                                                 avi_get_scale (avi_edit[i]))
                    && (stream[i].fmt_size = icm_compress_get_format_size
                                            (stream[i].icm_object,
                                            avi_edit[i]->bmih)) >= BMIH_SIZE)
              {
                /* ja:圧縮 */
                stream[i].fmt = g_malloc (stream[i].fmt_size);
                if (!icm_compress_get_format (stream[i].icm_object,
                                            avi_edit[i]->bmih, stream[i].fmt))
                  {
                    g_free (stream[i].fmt);
                    stream[i].fmt = NULL;
                  }
                stream[i].fourcc = make4cc (i / 10 + '0',
                                                    i % 10 + '0', 'd', 'c');
              }
            if (!stream[i].fmt)
              {
                /* ja:未圧縮 */
                if (stream[i].icm_object)
                  {
                    icm_close (stream[i].icm_object);
                    stream[i].icm_object = NULL;
                  }
                stream[i].fmt_size = bx_header_bytes
                                (bmih_get_width (avi_edit[i]->bmih),
                                bmih_get_height (avi_edit[i]->bmih),
                                bmih_get_bit_count (avi_edit[i]->bmih),
                                bmih_get_compression (avi_edit[i]->bmih), 0);
                stream[i].fmt = g_malloc (stream[i].fmt_size);
                g_memmove (stream[i].fmt, avi_edit[i]->bmih, BMIH_SIZE);
                bmih_set_color_used (stream[i].fmt, 0);
                bmih_set_color_important (stream[i].fmt, 0);
                switch (bmih_get_bit_count (stream[i].fmt))
                  {
                    case 1:g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb2, RGBQUAD_SIZE * 2); break;
                    case 4:g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb16, RGBQUAD_SIZE * 16); break;
                    case 8:g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb256, RGBQUAD_SIZE * 256);
                  }
                stream[i].fourcc = make4cc (i / 10 + '0',
                                                    i % 10 + '0', 'd', 'b');
              }
            break;
          case AVI_TYPE_AUDIO:/* ja:オーディオ */
            stream[i].fmt_size = wf_header_bytes (avi_edit[i]->wfx);
            stream[i].fmt = g_memdup (avi_edit[i]->wfx, stream[i].fmt_size);
            stream[i].fourcc = make4cc (i / 10 + '0', i % 10 + '0', 'w', 'b');
        }
      header += sizeof (guint32)                                /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32)                              /* en:'strl' */
                    + sizeof (guint32)                          /* en:'strh' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((ASH_SIZE + 1) & ~1)                     /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'strf' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((stream[i].fmt_size + 1) & ~1)           /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'indx' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + AVI_ALIGNSIZE - sizeof (guint32) * 2;     /* ja:配列 */
      if (avi_get_name (avi_edit[i]))
        header +=     sizeof (guint32)                          /* en:'strn' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((g_strlen (avi_get_name (avi_edit[i])) + 2) & ~1);
                                                                /* ja:文字列 */
    }
  head = header % AVI_HEADERSIZE != 0
                        ? ((header + sizeof(guint32) * 2) / AVI_HEADERSIZE + 1)
                                                    * AVI_HEADERSIZE : header;
  movi = head - sizeof (guint32) * 3;

  /* ja:ファイルを開く */
  fio = fileio_open (file, FILEIO_ACCESS_ALL,
                                    FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW);
  if (!fio
      || fileio_write (fio, &fcc0, sizeof (guint32)) != sizeof (guint32)
      || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
      || fileio_write (fio, &type0, sizeof (guint32)) != sizeof (guint32)
      || fileio_seek (fio, head - sizeof (guint32) * 3, FILEIO_SEEK_SET) == -1
      || fileio_write (fio, &fcc1, sizeof (guint32)) != sizeof (guint32)
      || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
      || fileio_write (fio, &type1, sizeof (guint32)) != sizeof (guint32))
    {
      fileio_close (fio);
      for (i = 0; i < streams; i++)
        {
          g_free (stream[i].fmt);
          if (stream[i].icm_object)
            icm_close (stream[i].icm_object);
        }
      g_free (stream);
      return FALSE;
    }

  /* ja:最も1秒間のフレーム数が少ないストリームを探す */
  for (current = 0; current < streams; current++)
    if (avi_type (avi_edit[current]) == AVI_TYPE_VIDEO)
      break;
  if (current >= streams)
    {
      /* ja:ビデオストリームがないとき */
      current = -1;
      times = 1000;
    }
  else
    {
      for (i = 0; i < streams; i++)
        if (avi_type (avi_edit[i]) == AVI_TYPE_VIDEO
                    && (glonglong)avi_get_rate (avi_edit[i])
                                            * avi_get_scale (avi_edit[current])
                            < (glonglong)avi_get_rate (avi_edit[current])
                                                * avi_get_scale (avi_edit[i]))
          current = i;
      times = avi_length_time (avi_edit[current])
                                            / avi_length (avi_edit[current]);
    }
  /* ja:最も時間が長いストリームを探す */
  for (i = 1; i < streams; i++)
    if (avi_length_time (avi_edit[longer]) < avi_length_time (avi_edit[i]))
      longer = i;
  /* ja:フレーム */
  for (i = 0; i < streams; i++)
    if ((avi_type (avi_edit[i]) == AVI_TYPE_VIDEO
                && !(stream[i].avi_frame = avi_get_frame_open (avi_edit[i])))
        || (avi_type (avi_edit[i]) == AVI_TYPE_AUDIO
                && !(stream[i].avi_pcm = avi_get_pcm_open (avi_edit[i]))))
      {
        while (i >= 0)
          {
            if (stream[i].avi_frame)
              avi_get_frame_close (stream[i].avi_frame);
            if (stream[i].avi_pcm)
              avi_get_pcm_close (stream[i].avi_pcm);
            i--;
          }
        fileio_close (fio);
        for (i = 0; i < streams; i++)
          {
            g_free (stream[i].fmt);
            if (stream[i].icm_object)
              icm_close (stream[i].icm_object);
          }
        g_free (stream);
        return FALSE;
      }
  while (TRUE)
    {
      /* ja:中断チェック */
      if (func && !func (stream[longer].pos * 100
                                / avi_length (avi_edit[longer]), user_data))
        break;
      /* ja:すべて出力したら終了 */
      for (i = 0; i < streams; i++)
        if (stream[i].pos < avi_length (avi_edit[i]))
          break;
      if (i >= streams)
        break;
      for (i = 0; i < streams; i++)
        {
          if (avi_length (avi_edit[i]) <= stream[i].pos)
            continue;
          /* ja:出力サンプル数を計算 */
          samples = MIN (current >= 0
                    ? i == current ? 1 : avi_time_to_sample (avi_edit[i],
                                    avi_sample_to_time (avi_edit[current],
                                    stream[current].pos + 1)) - stream[i].pos
                    : avi_time_to_sample (avi_edit[i], timer) - stream[i].pos,
                                    avi_length (avi_edit[i]) - stream[i].pos);
          if (samples <= 0)
            continue;
          if (avi_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            { /* ja:ビデオ */
              for (j = 0; j < samples; j++)
                {
                  BitmapInfoHeader *bmih;

                  if (!stream[i].icm_object)
                    {
                      /* ja:未圧縮 */
                      bmih = avi_get_frame (stream[i].avi_frame,
                                        stream[i].pos + j,
                                        bmih_get_width (stream[i].fmt),
                                        bmih_get_height (stream[i].fmt),
                                        bmih_get_bit_count (stream[i].fmt));
                      if (!bmih)
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].base);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_get_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_pcm)
                                avi_get_pcm_close (stream[i].avi_pcm);
                            }
                          g_free (stream);
                          g_free (data);
                          g_free (index);
                          return FALSE;
                        }
                      if (!avi_save_avi_write (fio,
                                    (guint8 *)bmih + bm_header_bytes (bmih),
                                    bm_image_bytes (bmih), 1, 0,
                                    &index, &entries, &riff, &movi,
                                                        stream, streams, i))
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].base);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_get_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_pcm)
                                avi_get_pcm_close (stream[i].avi_pcm);
                            }
                          g_free (stream);
                          g_free (data);
                          g_free (index);
                          return FALSE;
                        }
                    }
                  else
                    {
                      gboolean key_frame;
                                /* ja:TRUE:キーフレーム,FALSE:非キーフレーム */
                      gboolean recompress;
                                /* ja:TRUE:再圧縮,FALSE:再圧縮なし */

                      /* ja:圧縮 */
                      recompress = avi_opt[i].recompress;
                      key_frame = avi_is_keyframe (avi_edit[i],
                                                            stream[i].pos + j);
                      if (key_frame)
                        stream[i].recompress = FALSE;
                      if (!recompress)
                        {
                          /* ja:AVIファイルを求める */
                          avi_get_file (avi_edit[i],stream[i].pos + j);
                          recompress = stream[i].recompress
                                || avi_edit[i]->file->handler
                                                        != avi_opt[i].handler
                                || bmih_get_size (avi_edit[i]->file->bmih)
                                        != bmih_get_size (stream[i].fmt)
                                || bmih_get_width (avi_edit[i]->file->bmih)
                                        != bmih_get_width (stream[i].fmt)
                                || bmih_get_height (avi_edit[i]->file->bmih)
                                        != bmih_get_height (stream[i].fmt)
                                || bmih_get_planes (avi_edit[i]->file->bmih)
                                        != bmih_get_planes (stream[i].fmt)
                                || bmih_get_bit_count (avi_edit[i]->file->bmih)
                                        != bmih_get_bit_count (stream[i].fmt)
                                || bmih_get_compression
                                                    (avi_edit[i]->file->bmih)
                                        != bmih_get_compression (stream[i].fmt)
                                || (avi_edit[i]->offset == stream[i].pos + j
                                                                && !key_frame);
                        }
                      if (recompress)
                        {
                          stream[i].recompress = TRUE;
                          /* ja:再圧縮 */
                          if (!stream[i].active)
                            {
                              icm_compress_begin (stream[i].icm_object);
                              stream[i].active = TRUE;
                              stream[i].bmih_size = icm_compress_get_size
                                                        (stream[i].icm_object);
                            }
                          data = g_realloc (data, stream[i].bmih_size);
                          if (!(bmih = avi_get_frame
                                    (stream[i].avi_frame,
                                    stream[i].pos + j,
                                    bmih_get_width (avi_edit[i]->bmih),
                                    bmih_get_height (avi_edit[i]->bmih),
                                    bmih_get_bit_count (avi_edit[i]->bmih)))
                                || (length = icm_compress
                                      (stream[i].icm_object,
                                      &key_frame,
                                      (guint8 *)bmih + bm_header_bytes (bmih),
                                      data)) <= 0)
                            {
                              fileio_close (fio);
                              for (i = 0; i < streams; i++)
                                {
                                  g_free (stream[i].fmt);
                                  g_free (stream[i].base);
                                  if (stream[i].active)
                                    icm_compress_end (stream[i].icm_object);
                                  if (stream[i].icm_object)
                                    icm_close (stream[i].icm_object);
                                  if (stream[i].avi_frame)
                                    avi_get_frame_close (stream[i].avi_frame);
                                  if (stream[i].avi_pcm)
                                    avi_get_pcm_close (stream[i].avi_pcm);
                                }
                              g_free (stream);
                              g_free (data);
                              g_free (index);
                              return FALSE;
                            }
                        }
                      else
                        {
                          /* ja:再圧縮なし */
                          if (stream[i].active)
                            {
                              icm_compress_end (stream[i].icm_object);
                              stream[i].active = FALSE;
                            }
                          length = avi_read_video_size (avi_edit[i],
                                                            stream[i].pos + j);
                          data = g_realloc (data, length);
                          avi_read_video (avi_edit[i],
                                                    stream[i].pos + j, data);
                        }
                      if (!avi_save_avi_write (fio, data, length, 1,
                                                key_frame ? 0 : 1,
                                                &index, &entries, &riff, &movi,
                                                        stream, streams, i))
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].base);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_get_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_pcm)
                                avi_get_pcm_close (stream[i].avi_pcm);
                            }
                          g_free (stream);
                          g_free (data);
                          g_free (index);
                          return FALSE;
                        }
                    }
                }
            }
          else
            { /* ja:オーディオ */
              gpointer out_buf = NULL;
              gint out_samples;

              if (!avi_get_pcm (stream[i].avi_pcm,
                                stream[i].pos, samples, &out_buf, &out_samples)
                    || !avi_save_avi_write (fio, out_buf,
                        out_samples * wfx_get_block_align (avi_edit[i]->wfx),
                                                out_samples, 2,
                                                &index, &entries, &riff, &movi,
                                                        stream, streams, i))
                {
                  g_free (out_buf);
                  fileio_close (fio);
                  for (i = 0; i < streams; i++)
                    {
                      g_free (stream[i].fmt);
                      g_free (stream[i].base);
                      if (stream[i].active)
                        icm_compress_end (stream[i].icm_object);
                      if (stream[i].icm_object)
                        icm_close (stream[i].icm_object);
                      if (stream[i].avi_frame)
                        avi_get_frame_close (stream[i].avi_frame);
                      if (stream[i].avi_pcm)
                        avi_get_pcm_close (stream[i].avi_pcm);
                    }
                  g_free (stream);
                  g_free (data);
                  g_free (index);
                  return FALSE;
                }
              g_free (out_buf);
            }
          if (i != current)
            stream[i].pos += samples;
        }
      if (current >= 0)
        {
          stream[current].pos++;
          if (avi_length (avi_edit[current]) <= stream[current].pos)
            {
              timer = avi_length_time (avi_edit[current]);
              current = -1;
            }
        }
      else
        {
          timer += times;
        }
    }
  g_free (data);
  /* ja:フレーム/圧縮 */
  for (i = 0; i < streams; i++)
    if ((stream[i].active && !icm_compress_end (stream[i].icm_object))
        | (stream[i].icm_object && !icm_close (stream[i].icm_object))
        | (stream[i].avi_frame && !avi_get_frame_close (stream[i].avi_frame))
        | (stream[i].avi_pcm && !avi_get_pcm_close (stream[i].avi_pcm)))
      {
        fileio_close (fio);
        while (++i < streams)
          {
            if (stream[i].active)
              icm_compress_end (stream[i].icm_object);
            if (stream[i].icm_object)
              icm_close (stream[i].icm_object);
            if (stream[i].avi_frame)
              avi_get_frame_close (stream[i].avi_frame);
          }
        for (i = 0; i < streams; i++)
          {
            g_free (stream[i].fmt);
            g_free (stream[i].base);
          }
        g_free (stream);
        g_free (index);
        return FALSE;
      }
  /* ja:ファイナライズ */
  if (!avi_save_avi_write (fio, NULL, 0, 0, -1,
                        &index, &entries, &riff, &movi, stream, streams, -1))
    {
      fileio_close (fio);
      for (i = 0; i < streams; i++)
        {
          g_free (stream[i].fmt);
          g_free (stream[i].base);
        }
      g_free (stream);
      g_free (index);
      return FALSE;
    }
  /* ja:最も長いビデオストリームを求める */
  current = -1;
  for (i = 0;i < streams; i++)
    if (avi_type (avi_edit[i]) == AVI_TYPE_VIDEO
                    && (current == -1 || stream[i].pos > stream[current].pos))
      current = i;
  if (current < 0)
    {
      fileio_close (fio);
      for (i = 0; i < streams; i++)
        {
          g_free (stream[i].fmt);
          g_free (stream[i].base);
        }
      g_free (stream);
    }

  /* ja:ヘッタを設定する */
  p = data = g_malloc0 (head - sizeof (guint32) * 6);

  *(guint32 *)p = GUINT32_TO_LE (CK_DEF_LIST);
  ((guint32 *)p)++;                                             /* en:'LIST' */
  size = sizeof (guint32)                                       /* en:'hdrl' */
            + sizeof (guint32)                                  /* en:'avih' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + AMH_SIZE                                          /* ja:構造体 */
            + sizeof (guint32)                                  /* en:'LIST' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + sizeof (guint32)                                  /* en:'odml' */
                + sizeof (guint32)                              /* en:'odmh' */
                + sizeof (guint32)                              /* ja:サイズ */
                + DML_SIZE;                                     /* ja:構造体 */
  for (i = 0; i < streams; i++)
    {
      size += sizeof (guint32)                                  /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32)                              /* en:'strl' */
                    + sizeof (guint32)                          /* en:'strh' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ASH_SIZE                                  /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'strf' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((stream[i].fmt_size + 1) & ~1)           /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'indx' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + AVI_ALIGNSIZE - sizeof (guint32) * 2;     /* ja:配列 */
      if (avi_get_name (avi_edit[i]))
        size +=       sizeof (guint32)                          /* en:'strn' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((g_strlen (avi_get_name (avi_edit[i])) + 2) & ~1);
                                                                /* ja:文字列 */
    }
  *(guint32 *)p = GUINT32_TO_LE (size);
  ((guint32 *)p)++;                                             /* ja:サイズ */
  *(guint32 *)p = GUINT32_TO_LE (CK_LIST_AVIHEADER);
  ((guint32 *)p)++;                                             /* en:'hdrl' */
  *(guint32 *)p = GUINT32_TO_LE (CK_ID_AVIMAINHDR);
  ((guint32 *)p)++;                                             /* en:'avih' */
  *(guint32 *)p = GUINT32_TO_LE (AMH_SIZE);
  ((guint32 *)p)++;                                             /* ja:サイズ */
  amh_set_micro_sec_per_frame (p,0);
  amh_set_padding_granularity (p, AVI_HEADERSIZE);
  amh_set_flags (p, AVI_FILE_HASINDEX | AVI_FILE_MUSTUSEINDEX);
  amh_set_total_frames (p, stream[current].pos);
  for (i = 0; i < streams; i++)
    if (stream[i].pos > 0)
      amh_set_streams (p, amh_get_streams (p) + 1);
  amh_set_width (p, bmih_get_width (stream[current].fmt));
  amh_set_height (p, bmih_get_height (stream[current].fmt));
  p += AMH_SIZE;                                                /* ja:構造体 */

  for (i = 0; i < streams; i++)
    {
      if (stream[i].pos > 0)
        {
          guint8 *q;

          *(guint32 *)p = GUINT32_TO_LE (CK_DEF_LIST);
          ((guint32 *)p)++;                                     /* en:'LIST' */
          size = sizeof (guint32)                               /* en:'strl' */
                    + sizeof (guint32)                          /* en:'strh' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ASH_SIZE                                  /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'strf' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((stream[i].fmt_size + 1) & ~1)           /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'indx' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + AVI_ALIGNSIZE - sizeof (guint32) * 2;     /* ja:配列 */
          if (avi_get_name (avi_edit[i]))
            size +=   sizeof(guint32)                           /* en:'strn' */
                    + sizeof(guint32)                           /* ja:サイズ */
                    + ((g_strlen (avi_get_name (avi_edit[i])) + 2) & ~1);
                                                                /* ja:文字列 */
          *(guint32 *)p = GUINT32_TO_LE (size);
          ((guint32 *)p)++;                                     /* ja:サイズ */
          *(guint32 *)p = GUINT32_TO_LE (CK_LIST_STREAMHEADER);
          ((guint32 *)p)++;                                     /* en:'strl' */

          *(guint32 *)p = GUINT32_TO_LE (CK_ID_STREAMHEADER);
          ((guint32 *)p)++;                                     /* en:'strh' */
          *(guint32 *)p = GUINT32_TO_LE (ASH_SIZE);
          ((guint32 *)p)++;                                     /* ja:サイズ */
          ash_set_type (p, avi_type (avi_edit[i]) == AVI_TYPE_VIDEO
                                        ? AVI_STREAM_VIDEO : AVI_STREAM_AUDIO);
          if (avi_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            ash_set_handler (p, !avi_opt || avi_opt[i].handler == 0
                                                    || !stream[i].icm_object
                                ? AVI_STREAM_COMP_DIB : avi_opt[i].handler);
          ash_set_priority (p, avi_get_priority (avi_edit[i]));
          ash_set_language (p, avi_get_language (avi_edit[i]));
          ash_set_scale (p, avi_get_scale (avi_edit[i]));
          ash_set_rate (p, avi_get_rate (avi_edit[i]));
          ash_set_length (p, stream[i].written);
          ash_set_suggested_buffer_size (p, stream[i].max);
          ash_set_quality (p, -1);
          if (avi_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            {
              ash_set_frame_right (p, bmih_get_width (avi_edit[i]->bmih));
              ash_set_frame_bottom (p, bmih_get_height (avi_edit[i]->bmih));
            }
          p += ASH_SIZE;                                        /* ja:構造体 */

          *(guint32 *)p = GUINT32_TO_LE (CK_ID_STREAMFORMAT);
          ((guint32 *)p)++;                                     /* en:'strf' */
          *(guint32 *)p = GUINT32_TO_LE ((stream[i].fmt_size + 1) & ~1);
          ((guint32 *)p)++;                                     /* ja:サイズ */
          g_memmove (p, stream[i].fmt,stream[i].fmt_size);
          p += (stream[i].fmt_size + 1) & ~1;                   /* ja:構造体 */

          if (avi_get_name (avi_edit[i]))
            {
              *(guint32 *)p = GUINT32_TO_LE (CK_ID_STREAMNAME);
              ((guint32 *)p)++;                                 /* en:'strn' */
              *(guint32 *)p = GUINT32_TO_LE ((g_strlen
                                    (avi_get_name (avi_edit[i])) + 2) & ~1);
              ((guint32 *)p)++;                                 /* ja:サイズ */
              g_memmove (p, avi_get_name (avi_edit[i]),
                                        g_strlen (avi_get_name (avi_edit[i])));
              p += (g_strlen (avi_get_name (avi_edit[i])) + 2) & ~1;
                                                                /* ja:文字列 */
            }

          *(guint32 *)p = GUINT32_TO_LE (CK_ID_AVISUPERINDEX);
          ((guint32 *)p)++;                                     /* en:'indx' */
          *(guint32 *)p = AVI_ALIGNSIZE - sizeof (guint32) * 2;
          ((guint32 *)p)++;                                     /* ja:サイズ */
          for (j = 0, q = (guint8 *)stream[i].super;
                                j < AVI_SUPERINDEX_ENTRY; j++, q += ASE_SIZE)
            if (ase_get_size (q) == 0)
              break;
          asi_set_longs (p, 4);
          asi_set_sub_type (p, 0);
          asi_set_type (p, AVI_IDX_OF_INDEXES);
          asi_set_entries (p, j);
          asi_set_ckid (p, stream[i].fourcc);
          p += ASI_SIZE;
          g_memmove (p, stream[i].super, AVI_SUPERINDEX_ENTRY * ASE_SIZE);
          p += AVI_ALIGNSIZE - sizeof (guint32) * 2 - ASI_SIZE; /* ja:配列 */
        }
      g_free (stream[i].fmt);
    }
  *(guint32 *)p = GUINT32_TO_LE (CK_DEF_LIST);
  ((guint32 *)p)++;                                             /* en:'LIST' */
  *(guint32 *)p = GUINT32_TO_LE (sizeof (guint32)               /* en:'odml' */
                                    + sizeof (guint32)          /* en:'odmh' */
                                    + sizeof (guint32)          /* ja:サイズ */
                                    + DML_SIZE);                /* ja:構造体 */
  ((guint32 *)p)++;                                             /* ja:サイズ */
  *(guint32 *)p = GUINT32_TO_LE (CK_LIST_OPENDML);
  ((guint32 *)p)++;                                             /* en:'odml' */

  *(guint32 *)p = GUINT32_TO_LE (CK_ID_OPENDML);
  ((guint32 *)p)++;                                             /* en:'dmlh' */
  *(guint32 *)p = GUINT32_TO_LE (DML_SIZE);
  ((guint32 *)p)++;                                             /* ja:サイズ */
  dml_set_total_frames (p, stream[current].pos);
  p += DML_SIZE;                                                /* ja:構造体 */
  g_free (stream);

  if (head - header > 0)
    {
      *(guint32 *)p = GUINT32_TO_LE (CK_ID_AVIPADDING);
      ((guint32 *)p)++;                                         /* en:'JUNK' */
      *(guint32 *)p = GUINT32_TO_LE (head - header - sizeof (guint32) * 2);
    }                                                           /* ja:サイズ */

  if (fileio_seek (fio, sizeof (guint32) * 3, FILEIO_SEEK_SET) == -1
                    || fileio_write (fio, data, head - sizeof (guint32) * 6)
                                                != head - sizeof (guint32) * 6)
    {
      fileio_close (fio);
      g_free (data);
    }
  g_free (data);
  return fileio_close (fio);
}


/*  ja:ビットマップファイルに保存する
     avi_edit,AVI編集ハンドル
         file,ファイル名
        start,最初のフレーム
          end,最後のフレーム
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_bitmap (AviEdit      *avi_edit,
                 const gchar  *file,
                 const gint    start,
                 const gint    end,
                 gboolean    (*func)(gint, gpointer),
                 gpointer      user_data)
{
  gboolean result = TRUE;
  gchar *format, *dir, *name, *ext;
  gint i, s;
  AviFrame *avi_frame;
  BitmapFileHeader bmfh;

  if (!avi_edit || !file)
    return FALSE;
  dir = g_dirname (file);
  name = g_strdup (g_basename (file));
  for (i = g_strlen (name) - 1; i >= 0; i--)
    if (name[i] == '.')
      break;
  if (i >= 0)
    {
      ext = g_strdup (name + i);
      name[i] = '\0';
    }
  else
    {
      ext = g_strdup ("");
    }
  for (i = 0, s = end; s != 0; s /= 10)
    i++;
  format = g_strdup_printf ("%s/%s%%0%dd%s", dir, name, i, ext);
  g_free (dir);
  g_free (name);
  g_free (ext);

  g_memset (&bmfh, 0, BMFH_SIZE);
  bmfh_set_type (&bmfh ,0x4d42);
  bmfh_set_size (&bmfh, bm_all_bytes (avi_edit->bmih) + BMFH_SIZE);
  bmfh_set_off_bits (&bmfh, bm_header_bytes (avi_edit->bmih) + BMFH_SIZE);

  avi_frame = avi_get_frame_open (avi_edit);
  if (!avi_frame)
    {
      g_free (format);
      return FALSE;
    }
  for (i = start; i <= end; i++)
    {
      FileIO *fio;
      gint length;
      BitmapInfoHeader *bmih;

      /* ja:中断チェック */
      if (func && !func ((i - start) * 100/(end - start + 1), user_data))
        break;

      bmih = avi_get_frame (avi_frame, i, bmih_get_width (avi_edit->bmih),
                                          bmih_get_height (avi_edit->bmih),
                                          bmih_get_bit_count (avi_edit->bmih));
      if (!bmih)
        {
          result = FALSE;
          break;
        }
      bmih_set_x_pixels_per_meter (bmih,
                                bmih_get_x_pixels_per_meter (avi_edit->bmih));
      bmih_set_y_pixels_per_meter (bmih,
                                bmih_get_y_pixels_per_meter (avi_edit->bmih));
      length = bm_all_bytes (bmih);

      name = start == end ? g_strdup (file) : g_strdup_printf (format, i);
      /* bmihファイルを開く */
      fio = fileio_open (name, FILEIO_ACCESS_ALL,
                                    FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW);
      g_free (name);
      if (!fio)
        {
          result = FALSE;
          break;
        }
      if (fileio_write (fio, &bmfh, BMFH_SIZE) != BMFH_SIZE
                                || fileio_write (fio, bmih, length) != length)
        {
          fileio_close (fio);
          result = FALSE;
          break;
        }
      if (!fileio_close (fio))
        {
          result = FALSE;
          break;
        }
    }
  g_free (format);
  return avi_get_frame_close (avi_frame) && result;
}


/*  ja:WAVEファイルに保存する
     avi_edit,AVI編集ハンドル
         file,ファイル名
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_wave (AviEdit      *avi_edit,
               const gchar  *file,
               gboolean    (*func)(gint, gpointer),
               gpointer      user_data)
{
  FileIO *fio;
  gpointer head;
  guint8 *p;
  gint pos = 0, samples = AVI_PCM_SAMPLES_PER_SEC, written = 0, header;
  AviPcm *avi_pcm;

  if (!avi_edit || !file)
    return FALSE;
  /* ja:ヘッタサイズを計算する */
  header = sizeof (guint32)                                     /* en:'RIFF' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + sizeof (guint32)                                  /* en:'WAVE' */
            + sizeof (guint32)                                  /* en:'fmt ' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + ((wf_header_bytes (avi_edit->wfx) + 1) & ~1)      /* ja:構造体 */
            + sizeof (guint32)                                  /* en:'data' */
            + sizeof (guint32);                                 /* ja:サイズ */
  /* ja:ファイルを開く */
  fio = fileio_open (file, FILEIO_ACCESS_ALL,
                                    FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW);
  if (!fio)
    return FALSE;
  if (fileio_seek (fio, header, FILEIO_SEEK_SET) == -1)
    {
      fileio_close (fio);
      return FALSE;
    }
  /* ja:データ */
  avi_pcm = avi_get_pcm_open (avi_edit);
  if (!avi_pcm)
    {
      fileio_close (fio);
      return FALSE;
    }
  while (pos < avi_length (avi_edit))
    {
      gpointer out_buf;
      gint out_samples;
      gint length;

      /* ja:中断チェック */
      if (func && !func (pos * 100 / avi_length (avi_edit), user_data))
        break;
      if (avi_length (avi_edit) < pos + samples)
        samples = avi_length (avi_edit) - pos;
      if (!avi_get_pcm (avi_pcm, pos, samples, &out_buf, &out_samples))
        {
          avi_get_pcm_close (avi_pcm);
          fileio_close (fio);
          return FALSE;
        }
      pos += samples;
      length = out_samples * wfx_get_block_align (avi_edit->wfx);
      if (fileio_write (fio, out_buf, length) != length)
        {
          avi_get_pcm_close (avi_pcm);
          g_free (out_buf);
          fileio_close (fio);
          return FALSE;
        }
      g_free (out_buf);
      written += length;
    }
  if (!avi_get_pcm_close (avi_pcm))
    {
      fileio_close (fio);
      return FALSE;
    }
  if (written & 1)
    {
      const static guint8 dmy = 0;

      if (fileio_write (fio, &dmy, 1) != 1)
        {
          fileio_close (fio);
          return FALSE;
        }
    }
  /* ja:ヘッタを設定する */
  p = head = g_malloc (header);
  *(guint32 *)p = GUINT32_TO_LE (CK_DEF_RIFF);
  ((guint32 *)p)++;                                             /* en:'RIFF' */
  *(guint32 *)p = GUINT32_TO_LE (header - sizeof (guint32) * 2
                                                    + ((written + 1) & ~1));
  ((guint32 *)p)++;                                             /* ja:サイズ */
  *(guint32 *)p = GUINT32_TO_LE (CK_FORM_WAVE);
  ((guint32 *)p)++;                                             /* en:'WAVE' */
  *(guint32 *)p = GUINT32_TO_LE (CK_ID_WAVEFORMAT);
  ((guint32 *)p)++;                                             /* en:'fmt ' */
  *(guint32 *)p = GUINT32_TO_LE (wf_header_bytes (avi_edit->wfx));
  ((guint32 *)p)++;                                             /* ja:サイズ */
  g_memmove (p, avi_edit->wfx, wf_header_bytes (avi_edit->wfx));
  if (wf_header_bytes (avi_edit->wfx) & 1)
    p[wf_header_bytes (avi_edit->wfx)] = 0;
  p += (wf_header_bytes (avi_edit->wfx) + 1) & ~1;
  *(guint32 *)p = GUINT32_TO_LE (CK_ID_WAVEDATA);
  ((guint32 *)p)++;                                             /* en:'data' */
  *(guint32 *)p = GUINT32_TO_LE (written);                      /* ja:サイズ */
  if (fileio_seek (fio, 0 , FILEIO_SEEK_SET) == -1
                                || fileio_write (fio, head, header) != header)
    {
      fileio_close (fio);
      g_free (head);
      return FALSE;
    }
  g_free (head);
  return fileio_close (fio);
}


/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
     avi_save,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_with_options (AviEdit      *avi_edit[],
                       const gchar  *file,
                       AviSave      *avi_save,
                       gboolean    (*func)(gint, gpointer),
                       gpointer      user_data)
{
  FileIO *fio;
  gboolean result = FALSE;
  gchar *scenario;

  switch (avi_save->type)
    {
      case AVI_FILE_AVI:
        result = avi_save_avi (avi_edit, file,
                                        avi_save->avi_opt, func, user_data);
        break;
      case AVI_FILE_BITMAP:
        result = avi_save_bitmap (avi_edit[avi_save->stream], file,
                            avi_save->start, avi_save->end, func, user_data);
        break;
      case AVI_FILE_WAVE:
        result = avi_save_wave (avi_edit[avi_save->stream], file,
                                                            func, user_data);
        break;
      default:
        if ((scenario = avi_to_scenario (avi_edit, FALSE))
                && (fio = fileio_open (file, FILEIO_ACCESS_ALL,
                                FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW)))
          {
            gssize length;

            length = g_strlen (scenario);
            result = fileio_write (fio, scenario, length) == length;
            if (!fileio_close (fio))
              result = FALSE;
          }
        g_free (scenario);
    }
  return result;
}
