/*
    Video maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "general.h"
#include "thread.h"
#include "toolbar.h"
#include "orz/orzmdi.h"
#ifndef USE_THREAD
#include "misc/misc.h"
#endif /* not USE_THREAD */
#include <time.h>


/******************************************************************************
*                                                                             *
* ja:スレッド関数群                                                           *
*                                                                             *
******************************************************************************/
#ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gint critical = FALSE;
#else /* not USE_THREAD */
/*  ja:メニューを設定する
    task,TRUE:通常,FALSE:終了のみ                                           */
static void
thread_set_menu_bar (gboolean task)
{
  gint i;
  GList *glist;
  GtkWidget *menu_shell;

  menu_shell = misc_find_menu (menu_entries, "/file/");
  glist = gtk_container_children (GTK_CONTAINER (menu_shell));
  for (i = g_list_length (glist) - 2; i >= 0; i--)
    gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)), task);
  g_list_free (glist);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/edit"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/view"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/option"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/task"), !task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/window"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/help"), task);
  for (i = 0; toolbar_entries[i].path; i++)
    if (toolbar_entries[i].tool_item)
      gtk_widget_set_sensitive (GTK_WIDGET (toolbar_entries[i].tool_item),
                                                                        task);
  if (task)
    set_menu_bar ((VmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1));
}
#endif /* not USE_THREAD */


/*  ja:リストに加える
      id,スレッドID
    mode,モード(0:終了,1:開く,2:保存,3:フレーム数,4:反転)
    file,ファイル名
     RET,TRUE:正常終了,FALSE:エラー                                         */
#ifdef USE_THREAD
gboolean
thread_insert (GThread     *id,
               const gint   mode,
               const gchar *file)
#else /* not USE_THREAD */
gboolean
thread_insert (gint         id,
               const gint   mode,
               const gchar *file)
#endif /* not USE_THREAD */
{
  gchar *text[4];
  gchar *name[] = {N_("Exit"), N_("Open"), N_("Save"), N_("Change Frames"),
                                                                N_("Reverse")};
  gint i;
  GtkTreeIter iter;
  time_t tm;
  ThreadList *tl;

  tl = g_malloc (sizeof (ThreadList));
  tl->mode = mode;
  tl->percent = 0;
  tl->userbreak = TRUE;
  tl->id = id;
  /* ja:モード */
  text[0] = _(name[mode]);
  /* ja:ファイル名 */
  text[1] = g_filename_to_utf8 (file != NULL ? file : "", -1,
                                                            NULL, NULL, NULL);
  /* ja:時刻 */
  tm = time (NULL);
  text[2] = ctime (&tm);
  /* ja:進行状況 */
  text[3] = "0";
  /* ja:設定 */
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */
  gtk_list_store_append (list_store, &iter);
  for (i = 0; i < 4; i++)
    gtk_list_store_set (list_store, &iter, i, text[i], -1);
  gtk_list_store_set (list_store, &iter, 4, tl, -1);
  gtk_tree_view_columns_autosize (GTK_TREE_VIEW (tview));
#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
  g_free (text[1]);
#ifndef USE_THREAD
  thread_set_menu_bar (FALSE);
  gtk_widget_set_sensitive (mdi, FALSE);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:リストから削除する
     id,スレッドID
    RET,TRUE:正常終了,FALSE:エラー                                          */
#ifdef USE_THREAD
gboolean
thread_delete (GThread *id)
#else /* not USE_THREAD */
gboolean
thread_delete(gint id)
#endif /* not USE_THREAD */
{
  gint i, count;
  GtkTreeIter iter;
  ThreadList *tl;
#ifndef USE_THREAD
  gint id_standard = 0, id_close = 0;
#endif /* not USE_THREAD */

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */
  /* ja:IDに一致するアイテムを探す */
  count = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (list_store), NULL);
  for (i = count - 1; i >= 0; i--)
    {
      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list_store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter, 4, &tl, -1);
      if (tl->id == id)
        {
          /* ja:リストにアイテムがあるとき */
          g_free (tl);
          gtk_list_store_remove (list_store, &iter);
        }
    }
#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  /* ja:IDに一致するアイテムを探す */
  count = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (list_store), NULL);
  for (i = count - 1; i >= 0; i--)
    {
      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list_store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter, 4, &tl, -1);
      switch (tl->id)
        {
          case THREAD_ID_STANDARD: id_standard++; break;
          case THREAD_ID_CLOSE:    id_close++;    break;
        }
    }
  if (id_standard <= 0)
    {
      thread_set_menu_bar (TRUE);
      gtk_widget_set_sensitive (mdi, TRUE);
    }
  if (id_close > 0)
    gtk_widget_destroy (window);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:リストを更新する
         id,スレッドID
    percent,進行状況
        RET,TRUE:継続,FALSE:中断                                            */
#ifdef USE_THREAD
gboolean
thread_idling (GThread    *id,
               const gint  percent)
#else /* not USE_THREAD */
gboolean
thread_idling (gint       id,
               const gint percent)
#endif /* not USE_THREAD */
{
  gchar *text;
  gint i;
  GtkTreeIter iter;
  ThreadList *tl = NULL;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */
  /* ja:IDに一致するアイテムを探す */
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list_store),
                                                        &iter, NULL, i); i++)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter, 4, &tl, -1);
      if (tl->id == id)
        break;
    }
  if (!tl)
    {
#ifdef USE_THREAD
      gdk_threads_leave ();
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
      return FALSE;
    }
  if (tl->userbreak && tl->percent != percent)
    {
      /* ja:中断ではなくかつ進行状況が異なるとき */
      tl->percent = percent;
      text = percent >= 0
                    ? g_strdup_printf ("%d", percent) : g_strdup (_("Error"));
      gtk_list_store_set (list_store, &iter, 3, text, -1);
      g_free (text);
    }
#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  while (gtk_events_pending ())
    gtk_main_iteration ();
#endif /* not USE_THREAD */
  return tl->userbreak;
}


/*  ja:スレッドを停止させる
    all,TRUE:すべてのスレッド,FALSE:選択されたスレッドのみ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
thread_break (const gboolean all)
{
  gint i;
  GtkTreeIter iter;
  GtkTreeSelection *select;
  ThreadList *tl;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tview));
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list_store),
                                                        &iter, NULL, i); i++)
    if (all || gtk_tree_selection_iter_is_selected (select, &iter))
      {
        gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter, 4, &tl, -1);
        if (tl->userbreak)
          {
            tl->userbreak = FALSE;
            gtk_list_store_set (list_store, &iter, 3, "*", -1);
          }
      }
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
  return TRUE;
}


/*  ja:プログラムの終了(独立スレッド)                                       */
void *
close_program (void)
{
  gboolean result;
  gint i, count, length;
  GtkTreeIter iter;
  ThreadList *tl;
#ifdef USE_THREAD
  GThread *id;

  id = g_thread_self ();
#else /* not USE_THREAD */
  gint id = 1;
#endif /* not USE_THREAD */

#ifdef USE_THREAD
  thread_insert (id, THREAD_MODE_EXIT, NULL);

  /* ja:既にエラーになっているスレッドを終了させる */
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */

  /* ja:エラーになっているアイテムを探す */
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list_store),
                                                        &iter, NULL, i); i++)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter, 4, &tl, -1);
      if (tl->percent < 0 && tl->userbreak) /* ja:リストにアイテムがあるとき */
        {
          /* ja:表示を変更する */
          tl->userbreak = FALSE;
          gtk_list_store_set (list_store, &iter, 3, "*", -1);
        }
    }

#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);

  do
    {
      g_thread_yield ();
      G_LOCK (critical);
      critical = TRUE;
      gdk_threads_enter ();
#endif /* USE_THREAD */

      /* ja:終了スレッドに一致するアイテムを探す */
      result = FALSE;
      count = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (list_store),
                                                                        NULL);
      for (i = count - 1; i >= 0; i--)
        {
          gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list_store), &iter,
                                                                    NULL, i);
          gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter, 4, &tl, -1);
          if (tl->mode == THREAD_MODE_EXIT)
            {
              /* ja:リストにアイテムがあるとき */
              result = tl->id != id;
              break;
            }
        }
      /* ja:MDIの子ウインドウ数 */
      length = orz_mdi_get_n_pages (ORZ_MDI (mdi));

#ifdef USE_THREAD
      gdk_threads_leave ();
      critical = FALSE;
      G_UNLOCK (critical);

      /* ja:リストに終了があり、このスレッド以外ならばこのスレッドは破棄 */
      if (result || !thread_idling (id, 0))
        {
          thread_delete (id);
          return NULL;
        }
    }
  while (count > 1 || length > 0);
  thread_delete (id);
  gdk_threads_enter ();
  gtk_widget_destroy (window);
  gdk_threads_leave ();
#else /* not USE_THREAD */
  /* ja:終了する */
  if (length <= 0)
    {
      if (count <= 0)
        gtk_widget_destroy (window);
      else if (i < 0)
        thread_insert (THREAD_ID_CLOSE, THREAD_MODE_EXIT, NULL);
    }
#endif /* USE_THREAD */
  return NULL;
}
