/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "wave.h"


/******************************************************************************
*                                                                             *
* ja:PCM生成関数                                                              *
*                                                                             *
******************************************************************************/
/*  ja:無音PCMを作る
        wfx,WaveFormatEx構造体へのポインタ
    samples,出力データのサンプル数
        RET,無音のPCM                                                       */
gpointer
wave_fill_silent_pcm (WaveFormatEx *wfx,
                      const gsize   samples)
{
  gpointer buffer;
  if (wfx_get_bits_per_sample (wfx) == 8)
    {
      gsize bytes;

      bytes = samples * wfx_get_channels (wfx) * sizeof (guint8);
      buffer = g_malloc (bytes);
      g_memset (buffer, 128, bytes);
    }
  else
    {
      buffer = g_malloc0 (samples * wfx_get_channels (wfx) * sizeof (gint16));
    }
  return buffer;
}


/******************************************************************************
*                                                                             *
* ja:PCM加工関数                                                              *
*                                                                             *
******************************************************************************/
typedef struct _WaveConvChannels
{
  guint16 channels;
  guint16 bits_per_sample;
} WaveConvChannels;


/*  ja:PCMのチャンネル変換を開始する
    wfx0,元のWaveFormatEx構造体へのポインタ
    wfx1,新しいWaveFormatEx構造体へのポインタ
     RET,変換ハンドル,NULL:変換不可能                                       */
WaveConvChannels *
wave_convert_channels_begin (WaveFormatEx *wfx0,
                             WaveFormatEx *wfx1)
{
  WaveConvChannels *wcc;

  wcc = g_malloc (sizeof (WaveConvChannels));
  wcc->channels = wfx_get_channels (wfx0);
  wcc->bits_per_sample = wfx_get_bits_per_sample (wfx0);
  return wcc;
}


/*  ja:PCMのチャンネル変換をする
            wcc,変換ハンドル
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_channels (WaveConvChannels *wcc,
                       gconstpointer     in_buf,
                       const gsize       in_samples,
                       gpointer         *out_buf,
                       gsize            *out_samples)
{
  gpointer result;
  gint i;

  if (wcc->channels == 1)
    {
      if (wcc->bits_per_sample == 8)
        {
          result = g_malloc (in_samples * 2 * sizeof (guint8));
          for (i = 0; i < in_samples; i++)
            ((guint8 *)result)[i * 2] =
            ((guint8 *)result)[i * 2 + 1] = ((guint8 *)in_buf)[i];
        }
      else
        {
          result = g_malloc (in_samples * 2 * sizeof (gint16));
          for (i = 0; i < in_samples; i++)
            ((gint16 *)result)[i * 2] =
            ((gint16 *)result)[i * 2 + 1] = ((gint16 *)in_buf)[i];
        }
    }
  else
    {
      if (wcc->bits_per_sample == 8)
        {
          result = g_malloc (in_samples * sizeof (guint8));
          for (i = 0; i < in_samples; i++)
            ((guint8 *)result)[i] = (((guint8 *)in_buf)[i * 2]
                                        + ((guint8 *)in_buf)[i * 2 + 1]) / 2;
        }
      else
        {
          result = g_malloc (in_samples * sizeof (gint16));
          for (i = 0; i < in_samples; i++)
            ((gint16 *)result)[i] =
                    GINT16_TO_LE ((GINT16_FROM_LE (((gint16 *)in_buf)[i * 2])
                        + GINT16_FROM_LE (((gint16 *)in_buf)[i * 2 + 1])) / 2);
        }
    }
  if (out_buf)
    *out_buf = result;
  else
    g_free (result);
  if (out_samples)
    *out_samples = in_samples;
  return TRUE;
}


/*  ja:PCMのチャンネル変換を終了する
            wcc,変換ハンドル
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_channels_end (WaveConvChannels *wcc,
                           gpointer         *out_buf,
                           gsize            *out_samples)
{
  if (out_buf)
    *out_buf = NULL;
  if (out_samples)
    *out_samples = 0;
  g_free (wcc);
  return TRUE;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _WaveConvSamplesPerSec
{
  gint in_samples;
  gint out_samples;
  gint last_sample[2];
  guint32 samples_per_sec0;
  guint32 samples_per_sec1;
  guint16 channels;
  guint16 bits_per_sample;
} WaveConvSamplesPerSec;


/*  ja:PCMのサンプリング周波数変換を開始する
    wfx0,元のWaveFormatEx構造体へのポインタ
    wfx1,新しいWaveFormatEx構造体へのポインタ
     RET,変換ハンドル,NULL:変換不可能                                       */
WaveConvSamplesPerSec *
wave_convert_samples_per_sec_begin (WaveFormatEx *wfx0,
                                    WaveFormatEx *wfx1)
{
  WaveConvSamplesPerSec *wcs;

  wcs = g_malloc0 (sizeof (WaveConvSamplesPerSec));
  wcs->samples_per_sec0 = wfx_get_samples_per_sec (wfx0);
  wcs->samples_per_sec1 = wfx_get_samples_per_sec (wfx1);
  wcs->channels = wfx_get_channels (wfx0);
  wcs->bits_per_sample = wfx_get_bits_per_sample (wfx0);
  if (wcs->bits_per_sample == 8)
    wcs->last_sample[0] = wcs->last_sample[1] = 128;
  return wcs;
}


#include <windows.h>
/*  ja:PCMのサンプリング周波数変換をする
            wcs,変換ハンドル
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_samples_per_sec (WaveConvSamplesPerSec *wcs,
                              gconstpointer          in_buf,
                              const gsize            in_samples,
                              gpointer              *out_buf,
                              gsize                 *out_samples)
{
  gpointer result;
  gsize samples;
  gint i, j, d, block_align, pos;

  block_align = wcs->channels * wcs->bits_per_sample / 8;
  samples = (glonglong)((wcs->in_samples + in_samples) * wcs->samples_per_sec1
    + wcs->samples_per_sec0 - 1) / wcs->samples_per_sec0 - wcs->out_samples;
  result = g_malloc (samples * block_align);
  if (wcs->bits_per_sample == 8)
    {
      const guint8 *i_buf;
      guint8 *o_buf;

      i_buf = in_buf;
      o_buf = result;
      for (i = 0; i < samples; i++)
        {
          pos = (glonglong)(wcs->out_samples + i) * wcs->samples_per_sec0
                                    / wcs->samples_per_sec1 - wcs->in_samples;
          d = (glonglong)(wcs->out_samples + i) * wcs->samples_per_sec0
                                                    % wcs->samples_per_sec1;
          if (d > 0 && pos + 1 >= in_samples)
            break;
          for (j = 0; j < wcs->channels; j++)
            o_buf[i * wcs->channels + j] = d == 0
                ? i_buf[pos * wcs->channels + j]
                : ((pos >= 0 ? i_buf[pos * wcs->channels + j]
                             : (guint8)wcs->last_sample[j])
                        * (wcs->samples_per_sec1 - d)
                    + i_buf[(pos + 1) * wcs->channels + j] * d)
                                                    / wcs->samples_per_sec1;
        }
      if (in_samples > 0)
        for (j = 0; j < wcs->channels; j++)
          wcs->last_sample[j] = i_buf[(in_samples - 1) * wcs->channels + j];
    }
  else
    {
      const gint16 *i_buf;
      gint16 *o_buf;

      i_buf = in_buf;
      o_buf = result;
      for (i = 0; i < samples; i++)
        {
          pos = (glonglong)(wcs->out_samples + i) * wcs->samples_per_sec0
                                    / wcs->samples_per_sec1 - wcs->in_samples;
          d = (glonglong)(wcs->out_samples + i) * wcs->samples_per_sec0
                                                    % wcs->samples_per_sec1;
          if (d > 0 && pos + 1 >= in_samples)
            break;
          for (j = 0; j < wcs->channels; j++)
            if (d == 0)
              {
                o_buf[i * wcs->channels + j] = i_buf[pos * wcs->channels + j];
              }
            else
              {
                gint16 n, n0, n1;

                n0 = pos >= 0 ? GINT16_FROM_LE (i_buf[pos * wcs->channels + j])
                                                : (gint16)wcs->last_sample[j];
                n1 = GINT16_FROM_LE (i_buf[(pos + 1) * wcs->channels + j]);
                n = ((glonglong)n0 * (wcs->samples_per_sec1 - d)
                                + (glonglong)n1 * d) / wcs->samples_per_sec1;
                o_buf[i * wcs->channels + j] = GINT16_TO_LE (n);
              }
        }
      if (in_samples > 0)
        for (j = 0; j < wcs->channels; j++)
          wcs->last_sample[j]
                = GINT16_FROM_LE (i_buf[(in_samples - 1) * wcs->channels + j]);
    }
  wcs->in_samples += in_samples;
  wcs->out_samples += i;
  result = g_realloc (result, i * block_align);
  if (out_buf)
    *out_buf = result;
  else
    g_free (result);
  if (out_samples)
    *out_samples = i;
  return TRUE;
}


/*  ja:PCMのサンプリング周波数変換を終了する
            wcs,変換ハンドル
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_samples_per_sec_end (WaveConvSamplesPerSec *wcs,
                                  gpointer              *out_buf,
                                  gsize                 *out_samples)
{
  gpointer result;
  gsize samples;
  gint i, j, d;

  samples = (glonglong)(wcs->in_samples * wcs->samples_per_sec1
    + wcs->samples_per_sec0 - 1) / wcs->samples_per_sec0 - wcs->out_samples;
  result = g_malloc (samples * (wcs->channels * wcs->bits_per_sample / 8));
  if (wcs->bits_per_sample == 8)
    {
      guint8 *o_buf;

      o_buf = result;
      for (i = 0; i < samples; i++)
        {
          d = (glonglong)(wcs->out_samples + i) * wcs->samples_per_sec0
                                                    % wcs->samples_per_sec1;
          for (j = 0; j < wcs->channels; j++)
            o_buf[i * wcs->channels + j]
                = (wcs->last_sample[j] * (wcs->samples_per_sec1 - d) + 128 * d)
                                                    / wcs->samples_per_sec1;
        }
    }
  else
    {
      gint16 *o_buf;

      o_buf = result;
      for (i = 0; i < samples; i++)
        {
          d = (glonglong)(wcs->out_samples + i) * wcs->samples_per_sec0
                                                    % wcs->samples_per_sec1;
          for (j = 0; j < wcs->channels; j++)
            {
              gint16 n;

              n = (glonglong)wcs->last_sample[j]
                        * (wcs->samples_per_sec1 - d) / wcs->samples_per_sec1;
              o_buf[i * wcs->channels + j] = GINT16_TO_LE (n);
            }
        }
    }
  if (out_buf)
    *out_buf = result;
  else
    g_free (result);
  if (out_samples)
    *out_samples = samples;
  g_free (wcs);
  return TRUE;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _WaveConvBitPerSample
{
  guint16 channels;
  guint16 bits_per_sample;
} WaveConvBitPerSample;


/*  ja:PCMのビット数変換を開始する
    wfx0,元のWaveFormatEx構造体へのポインタ
    wfx1,新しいWaveFormatEx構造体へのポインタ
     RET,変換ハンドル,NULL:変換不可能                                       */
WaveConvBitPerSample *
wave_convert_bits_per_sample_begin (WaveFormatEx *wfx0,
                                    WaveFormatEx *wfx1)
{
  WaveConvBitPerSample *wcb;

  wcb = g_malloc (sizeof (WaveConvBitPerSample));
  wcb->channels = wfx_get_channels (wfx0);
  wcb->bits_per_sample = wfx_get_bits_per_sample (wfx0);
  return wcb;
}


/*  ja:PCMのビット数変換をする
            wcb,変換ハンドル
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_bits_per_sample (WaveConvBitPerSample *wcb,
                              gconstpointer         in_buf,
                              const gsize           in_samples,
                              gpointer             *out_buf,
                              gsize                *out_samples)
{
  gpointer result;
  gint i;

  if (wcb->bits_per_sample == 8)
    {
      result = g_malloc (in_samples * wcb->channels * sizeof (gint16));
      for (i = in_samples * wcb->channels - 1; i >= 0; i--)
        {
          gint c;

          c = ((guint8 *)in_buf)[i] - 128;
          c = c * 256 + (c % 2 == 0 ? 0 : 255);
          ((gint16 *)result)[i] = GINT16_TO_LE (c);
        }
    }
  else
    {
      result = g_malloc (in_samples * wcb->channels * sizeof (guint8));
      for (i = in_samples * wcb->channels - 1; i >= 0; i--)
        ((guint8 *)result)[i] = (GINT16_FROM_LE (((gint16 *)in_buf)[i])
                                                                + 32768) / 256;
    }
  if (out_buf)
    *out_buf = result;
  else
    g_free (result);
  if (out_samples)
    *out_samples = in_samples;
  return TRUE;
}


/*  ja:PCMのビット数変換を終了する
            wcb,変換ハンドル
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_bits_per_sample_end (WaveConvBitPerSample *wcb,
                                  gpointer             *out_buf,
                                  gsize                *out_samples)
{
  if (out_buf)
    *out_buf = NULL;
  if (out_samples)
    *out_samples = 0;
  g_free (wcb);
  return TRUE;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _WaveConvGeneric *WaveConvGeneric;
typedef WaveConvGeneric (*WaveConvBeginFunc)(WaveFormatEx *wfx0,
                                             WaveFormatEx *wfx1);
typedef gboolean (*WaveConvFunc)(WaveConvGeneric  wch,
                                 gconstpointer    in_buf,
                                 const gsize      in_samples,
                                 gpointer        *out_buf,
                                 gsize           *out_samples);
typedef gboolean (*WaveConvEndFunc)(WaveConvGeneric  wch,
                                    gpointer        *out_buf,
                                    gsize           *out_samples);
typedef struct _WaveConvProccess
{
  WaveConvBeginFunc func_begin;
  WaveConvFunc func;
  WaveConvEndFunc func_end;
  WaveConvGeneric handle;
} WaveConvProccess;
struct _WaveConv
{
  guint16 block_align0;
  guint16 block_align1;
  gint procs;
  WaveConvProccess *proc;
};


/*  ja:PCMの変換を開始する
    wfx0,元のWaveFormatEx構造体へのポインタ
    wfx1,新しいWaveFormatEx構造体へのポインタ
     RET,変換ハンドル,NULL:変換不可能                                       */
WaveConv *
wave_convert_begin (WaveFormatEx *wfx0,
                    WaveFormatEx *wfx1)
{
  gint i;
  WaveConv *wcv;
  WaveFormatEx *wfx;

  wcv = g_malloc0 (sizeof (WaveConv));
  wcv->block_align0 = wfx_get_block_align (wfx0);
  wcv->block_align1 = wfx_get_block_align (wfx1);
  wfx = g_memdup (wfx0, wf_header_bytes (wfx0));
  if (wfx_get_bits_per_sample (wfx0) == 8
                                    && wfx_get_bits_per_sample (wfx1) == 16)
    {
      wcv->proc = g_realloc (wcv->proc,
                                (wcv->procs + 1) * sizeof (WaveConvProccess));
      wcv->proc[wcv->procs].func_begin
                    = (WaveConvBeginFunc)wave_convert_bits_per_sample_begin;
      wcv->proc[wcv->procs].func = (WaveConvFunc)wave_convert_bits_per_sample;
      wcv->proc[wcv->procs].func_end
                        = (WaveConvEndFunc)wave_convert_bits_per_sample_end;
      wcv->proc[wcv->procs].handle
                                = wcv->proc[wcv->procs].func_begin (wfx, wfx1);
      if (!wcv->proc[wcv->procs].handle)
        {
          for (i = wcv->procs - 1; i >= 0; i--)
            wcv->proc[i].func_end (wcv->proc[i].handle, NULL, NULL);
          g_free (wcv);
          g_free (wfx);
          return NULL;
        }
      wfx_set_bits_per_sample (wfx, 16);
      wcv->procs++;
    }
  if (wfx_get_samples_per_sec (wfx0) != wfx_get_samples_per_sec (wfx1))
    {
      wcv->proc = g_realloc (wcv->proc,
                                (wcv->procs + 1) * sizeof (WaveConvProccess));
      wcv->proc[wcv->procs].func_begin
                    = (WaveConvBeginFunc)wave_convert_samples_per_sec_begin;
      wcv->proc[wcv->procs].func = (WaveConvFunc)wave_convert_samples_per_sec;
      wcv->proc[wcv->procs].func_end
                        = (WaveConvEndFunc)wave_convert_samples_per_sec_end;
      wcv->proc[wcv->procs].handle
                                = wcv->proc[wcv->procs].func_begin (wfx, wfx1);
      if (!wcv->proc[wcv->procs].handle)
        {
          for (i = wcv->procs - 1; i >= 0; i--)
            wcv->proc[i].func_end (wcv->proc[i].handle, NULL, NULL);
          g_free (wcv);
          g_free (wfx);
          return NULL;
        }
      wfx_set_samples_per_sec (wfx, wfx_get_samples_per_sec (wfx1));
      wcv->procs++;
    }
  if (wfx_get_channels (wfx0) != wfx_get_channels (wfx1))
    {
      wcv->proc = g_realloc (wcv->proc,
                                (wcv->procs + 1) * sizeof (WaveConvProccess));
      wcv->proc[wcv->procs].func_begin
                            = (WaveConvBeginFunc)wave_convert_channels_begin;
      wcv->proc[wcv->procs].func = (WaveConvFunc)wave_convert_channels;
      wcv->proc[wcv->procs].func_end
                                = (WaveConvEndFunc)wave_convert_channels_end;
      wcv->proc[wcv->procs].handle
                                = wcv->proc[wcv->procs].func_begin (wfx, wfx1);
      if (!wcv->proc[wcv->procs].handle)
        {
          for (i = wcv->procs - 1; i >= 0; i--)
            wcv->proc[i].func_end (wcv->proc[i].handle, NULL, NULL);
          g_free (wcv);
          g_free (wfx);
          return NULL;
        }
      wfx_set_channels (wfx, wfx_get_channels (wfx1));
      wcv->procs++;
    }
  if (wfx_get_bits_per_sample (wfx0) == 16
                                        && wfx_get_bits_per_sample (wfx1) == 8)
    {
      wcv->proc = g_realloc (wcv->proc,
                                (wcv->procs + 1) * sizeof (WaveConvProccess));
      wcv->proc[wcv->procs].func_begin
                    = (WaveConvBeginFunc)wave_convert_bits_per_sample_begin;
      wcv->proc[wcv->procs].func = (WaveConvFunc)wave_convert_bits_per_sample;
      wcv->proc[wcv->procs].func_end
                        = (WaveConvEndFunc)wave_convert_bits_per_sample_end;
      wcv->proc[wcv->procs].handle
                                = wcv->proc[wcv->procs].func_begin (wfx, wfx1);
      if (!wcv->proc[wcv->procs].handle)
        {
          for (i = wcv->procs - 1; i >= 0; i--)
            wcv->proc[i].func_end (wcv->proc[i].handle, NULL, NULL);
          g_free (wcv);
          g_free (wfx);
          return NULL;
        }
      wfx_set_bits_per_sample (wfx, 8);
      wcv->procs++;
    }
  g_free (wfx);
  return wcv;
}


/*  ja:PCMの変換をする
            wcv,変換ハンドル
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert (WaveConv      *wcv,
              gconstpointer  in_buf,
              const gsize    in_samples,
              gpointer      *out_buf,
              gsize         *out_samples)
{
  gpointer buf;
  gsize samples;
  gint i;

  buf = g_memdup (in_buf, in_samples * wcv->block_align0);
  samples = in_samples;
  for (i = 0; i < wcv->procs; i++)
    {
      gpointer x_buf;

      if (!wcv->proc[i].func (wcv->proc[i].handle,
                                            buf, samples, &x_buf, &samples))
        {
          g_free (buf);
          return FALSE;
        }
      g_free (buf);
      buf = x_buf;
    }
  if (out_buf)
    *out_buf = buf;
  else
    g_free (buf);
  if (out_samples)
    *out_samples = samples;
  return TRUE;
}


/*  ja:PCMの変換を終了する
      wave_conv,変換ハンドル
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_end (WaveConv *wcv,
                  gpointer *out_buf,
                  gsize    *out_samples)
{
  gboolean result = TRUE;
  gpointer buf = NULL;
  gsize samples = 0;
  gint i;

  for (i = 0; i < wcv->procs; i++)
    {
      gpointer e_buf;
      gint e_samples;
      gint j;

      if (!wcv->proc[i].func_end (wcv->proc[i].handle, &e_buf, &e_samples))
        result = FALSE;
      if (!result)
        continue;
      for (j = i + 1; j < wcv->procs; j++)
        {
          gpointer x_buf;

          wcv->proc[j].func (wcv->proc[j].handle,
                                        e_buf, e_samples, &x_buf, &e_samples);
          g_free (e_buf);
          e_buf = x_buf;
        }
      buf = g_realloc (buf, (samples + e_samples) * wcv->block_align1);
      g_memmove ((guint8 *)buf + samples * wcv->block_align1, e_buf,
                                                e_samples * wcv->block_align1);
      g_free (e_buf);
      samples += e_samples;
    }
  if (result && out_buf)
    *out_buf = buf;
  else
    g_free (buf);
  if (result && out_samples)
    *out_samples = samples;
  g_free (wcv->proc);
  g_free (wcv);
  return result;
}


/*  ja:PCMの変換をする
           wfx0,元のWaveFormatEx構造体へのポインタ
           wfx1,新しいWaveFormatEx構造体へのポインタ
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_direct (WaveFormatEx  *wfx0,
                     WaveFormatEx  *wfx1,
                     gconstpointer  in_buf,
                     const gsize    in_samples,
                     gpointer      *out_buf,
                     gsize         *out_samples)
{
  gpointer buf, e_buf;
  gsize samples, e_samples;
  WaveConv *wcv;

  wcv = wave_convert_begin (wfx0, wfx1);
  wave_convert (wcv, in_buf, in_samples, &buf, &samples);
  wave_convert_end (wcv, &e_buf, &e_samples);
  buf = g_realloc (buf, (samples + e_samples) * wfx_get_block_align (wfx1));
  g_memmove ((guint8 *)buf + samples * wfx_get_block_align (wfx1), e_buf,
                                    e_samples * wfx_get_block_align (wfx1));
  g_free (e_buf);
  if (out_buf)
    *out_buf = buf;
  if (out_samples)
    *out_samples = samples;
  return TRUE;
}
