/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"


/******************************************************************************
*                                                                             *
* ja:AVIメモリ関数                                                            *
*                                                                             *
******************************************************************************/
struct _AviMemory
{
  gint counter;
  gpointer data;
  gsize length;
};


static GList *glist_memory = NULL;
#ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
#endif /* USE_THREAD */


/*  ja:AVIメモリを登録する
      data,登録するメモリ
    length,登録するバイト数
       RET,AVIメモリ,NULL:エラー                                            */
AviMemory *
avi_memory_register (gpointer    data,
                     const gsize length)
{
  gint i;
  AviMemory *avi_memory = NULL;

  if (!data || length <= 0)
    return NULL;
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  for (i = g_list_length (glist_memory) - 1; i >= 0; i--)
    {
      AviMemory *avi_mem;

      avi_mem = g_list_nth_data (glist_memory, i);
      if (avi_mem->length == length
                                && g_memcmp (avi_mem->data, data, length) == 0)
        {
          g_free (data);
          avi_memory = avi_mem;
          avi_memory->counter++;
          break;
        }
    }
  if (!avi_memory)
    {
      avi_memory = g_malloc (length);
      avi_memory->counter = 1;
      avi_memory->length = length;
      avi_memory->data = data;
      glist_memory = g_list_append (glist_memory, avi_memory);
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return avi_memory;
}


/*  ja:AVIメモリを複製する
      data,複製するメモリ
    length,複製するバイト数
       RET,AVIメモリ,NULL:エラー                                            */
AviMemory *
avi_memory_alloc (gconstpointer data,
                  const gsize   length)
{
  gint i;
  AviMemory *avi_memory = NULL;

  if (!data || length <= 0)
    return NULL;
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  for (i = g_list_length (glist_memory) - 1; i >= 0; i--)
    {
      AviMemory *avi_mem;

      avi_mem = g_list_nth_data (glist_memory, i);
      if (avi_mem->length == length
                                && g_memcmp (avi_mem->data, data, length) == 0)
        {
          avi_memory = avi_mem;
          avi_memory->counter++;
          break;
        }
    }
  if (!avi_memory)
    {
      avi_memory = g_malloc (length);
      avi_memory->counter = 1;
      avi_memory->length = length;
      avi_memory->data = g_memdup (data, length);
      glist_memory = g_list_append (glist_memory, avi_memory);
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return avi_memory;
}


/*  ja:AVIメモリのデータを取得する
    avi_memory,AVIメモリ
           RET,データ,NULL:エラー                                           */
gconstpointer
avi_memory_get_data (AviMemory *avi_memory)
{
  gconstpointer data;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  data = avi_memory && g_list_find (glist_memory, avi_memory)
                                                    ? avi_memory->data : NULL;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return data;
}


/*  ja:AVIメモリのバイト数を取得する
    avi_memory,AVIメモリ
           RET,バイト数                                                     */
gsize
avi_memory_get_length (AviMemory *avi_memory)
{
  gsize length;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  length = avi_memory && g_list_find (glist_memory, avi_memory)
                                                    ? avi_memory->length : 0;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return length;
}


/*  ja:AVIメモリの参照数を増やす
    avi_memory,AVIメモリ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
avi_memory_ref (AviMemory *avi_memory)
{
  gboolean result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  result = avi_memory && g_list_find (glist_memory, avi_memory)
                                        ? avi_memory->counter++, TRUE : FALSE;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
}


/*  ja:AVIメモリの参照数を減らす
    avi_memory,AVIメモリ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
avi_memory_unref (AviMemory *avi_memory)
{
  gboolean result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (avi_memory && g_list_find (glist_memory, avi_memory))
    {
      avi_memory->counter--;
      if (avi_memory->counter <= 0)
        {
          glist_memory = g_list_remove (glist_memory, avi_memory);
          g_free (avi_memory->data);
          g_free (avi_memory);
        }
      result = TRUE;
    }
  else
    {
      result = FALSE;
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
}
