/*
    peimage
    copyright (c) 1998-2010 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "peimage.h"
#include "misc/fileio.h"


/******************************************************************************
*                                                                             *
* ja:PEイメージ関数                                                           *
*                                                                             *
******************************************************************************/
/*  ja:PEイメージを検証する
     image,PEイメージ(ヘッダ)
    length,バイト数
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
peimage_file_is_valid (const guint8 *image,
                       const gssize  length)
{
  gint i, count;
  gssize leng = 0;
  const ImageSectionHeader *ish;

  if (!image || length < sizeof (ImageDosHeader)
                || pe_idh_get_magic (image) != PEIMAGE_DOS_SIGNATURE
                || length < pe_idh_get_lfanew (image) + sizeof (guint32)
                    + sizeof (ImageFileHeader) + sizeof (ImageOptionalHeader)
                || pe_get_signature (image) != PEIMAGE_NT_SIGNATURE
                || pe_ifh_get_machine (image) != PEIMAGE_FILE_MACHINE_I386
                || pe_ifh_get_size_of_optional_header (image)
                                                != sizeof (ImageOptionalHeader)
                || pe_ioh_get_magic (image) != PEIMAGE_NT_OPTIONAL_HDR_MAGIC
                || length < peimage_file_header_bytes (image))
    return FALSE;
  ish = pe_image_section_header (image);
  count = pe_ifh_get_number_of_sections (image);
  for (i = 0; i < count; i++)
    {
      gssize size;

      size = ish_get_virtual_address (ish)
            + MAX (ish_get_virtual_size (ish), ish_get_size_of_raw_data (ish));
      if (leng < size)
        leng = size;
      ish++;
    }
  return leng <= pe_ioh_get_size_of_image (image);
}


/*  ja:PEイメージを読み込む
    file,ファイル名
     RET,PEイメージ,NULL:エラー                                             */
guint8 *
peimage_file_load (const gchar *file)
{
  guint8 *image;
  gint i, number_of_sections;
  gssize length, header_bytes, size_of_image;
  FileIO *fio;
  ImageSectionHeader *ish;

  if (!file)
    return NULL;
  fio = fileio_open (file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (!fio)
    return NULL;
  image = g_malloc (sizeof (ImageDosHeader));
  if (!image || fileio_read (fio, image, sizeof (ImageDosHeader))
                                                    != sizeof (ImageDosHeader)
                        || pe_idh_get_magic (image) != PEIMAGE_DOS_SIGNATURE)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  length = idh_get_lfanew (image) + sizeof (guint32)
                    + sizeof (ImageFileHeader) + sizeof (ImageOptionalHeader);
  image = g_realloc (image, length);
  length -= sizeof (ImageDosHeader);
  if (!image || fileio_read (fio, image + sizeof (ImageDosHeader), length)
                                                                    != length
                || pe_get_signature (image) != PEIMAGE_NT_SIGNATURE
                || pe_ifh_get_machine (image) != PEIMAGE_FILE_MACHINE_I386
                || pe_ifh_get_size_of_optional_header (image)
                                                != sizeof (ImageOptionalHeader)
                || pe_ioh_get_magic (image) != PEIMAGE_NT_OPTIONAL_HDR_MAGIC)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  number_of_sections = pe_ifh_get_number_of_sections (image);
  header_bytes = peimage_file_header_bytes (image);
  size_of_image = pe_ioh_get_size_of_image (image);
  image = g_realloc (image, header_bytes);
  ish = pe_image_section_header (image);
  length = sizeof (ImageSectionHeader) * number_of_sections;
  if (!image || fileio_read (fio, ish, length) != length)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  length = 0;
  for (i = 0; i < number_of_sections; i++)
    {
      gssize size;

      size = ish_get_virtual_address (ish) + MAX (ish_get_virtual_size (ish),
                                            ish_get_size_of_raw_data (ish));
      if (length < size)
        length = size;
      ish++;
    }
  if (length > size_of_image)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  image = g_realloc (image, size_of_image);
  if (!image)
    {
      fileio_close (fio);
      return NULL;
    }
  ish = pe_image_section_header (image);
  length = pe_ioh_get_size_of_headers (image);
  g_memset (image + length, 0, size_of_image - length);
  length -= header_bytes;
  if (fileio_read (fio, image + header_bytes, length) != length)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  for (i = 0; i < number_of_sections; i++)
    {
      length = ish_get_size_of_raw_data (ish);
      if (fileio_seek (fio, ish_get_pointer_to_raw_data (ish),
                                                        FILEIO_SEEK_SET) < 0
            || fileio_read (fio, image + ish_get_virtual_address (ish), length)
                                                                    != length)
        {
          g_free (image);
          fileio_close (fio);
          return NULL;
        }
      if (ish_get_virtual_size (ish) > length)
        length = ish_get_virtual_size (ish);
      ish_set_virtual_size (ish, length);
      ish_set_size_of_raw_data (ish, length);
      ish++;
    }
  if (!fileio_close (fio))
    {
      g_free (image);
      return NULL;
    }
  return image;
}


/*  ja:PEイメージを再配置する
    image,PEイメージ
    delta,差                                                                */
void
peimage_file_relocate (guint8     *image,
                       const gint  delta)
{
  if (image && delta != 0 && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC))
    {
      const ImageBaseRelocation *ibr;

      for (ibr = pe_image_base_relocation (image);
                                                ibr_get_virtual_address (ibr);
                        ibr = (const ImageBaseRelocation *)((const guint8 *)ibr
                                                + ibr_get_size_of_block (ibr)))
        {
          gint i, count;
          guint32 address;

          address = ibr_get_virtual_address (ibr);
          count = (ibr_get_size_of_block (ibr) - sizeof (guint32) * 2)
                                                            / sizeof (guint16);
          for (i = 0; i < count; i++)
            {
              guint16 typeoffset;

              typeoffset = ibr_get_type_offset (ibr, i);
              if (((typeoffset >> 12) & 0xf) == PEIMAGE_REL_BASED_HIGHLOW)
                {
                  guint32 *p;

                  p = (guint32 *)(image + address + (typeoffset & 0xfff));
                  *p = GUINT32_TO_LE (GUINT32_FROM_LE (*p) + delta);
                }
            }
        }
    }
}


/*  ja:PEイメージからAPIを取得する
    image,PEイメージ
     name,API名
      RET,アドレス,NULL:エラー                                              */
gconstpointer
peimage_file_get_func (const guint8 *image,
                       const gchar  *name)
{
  gconstpointer func = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      gint i, count;
      const guint16 *ordinals;
      const guint32 *functions, *names;
      const ImageExportDirectory *ied;

      ied = pe_image_export_directory (image);
      functions = (const guint32 *)(image
                                    + ied_get_address_of_functions (ied));
      names = (const guint32 *)(image
                                    + ied_get_address_of_names (ied));
      ordinals = (const guint16 *)(image
                                    + ied_get_address_of_name_ordinals (ied));
      count = ied_get_number_of_names (ied);
      for (i = 0; i < count; i++)
        if (g_strcmp ((const gchar *)(image + GUINT32_FROM_LE (names[i])),
                                                                    name) == 0)
          {
            func = image
                + GUINT32_FROM_LE (functions[GUINT16_FROM_LE (ordinals[i])]);
            break;
          }
    }
  return func;
}


/*  ja:PEイメージからAPIを取得する
      image,PEイメージ
    ordinal,オーディナル値
        RET,アドレス,NULL:エラー                                            */
gconstpointer
peimage_file_get_proc (const guint8  *image,
                       const guint16  ordinal)
{
  gconstpointer proc = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      gint index;
      const guint32 *functions;
      const ImageExportDirectory *ied;

      ied = pe_image_export_directory (image);
      functions = (const guint32 *)(image + ied_get_address_of_functions (ied));
      index = ordinal - ied_get_base (ied);
      if (index < ied_get_number_of_functions (ied))
        proc = image + GUINT32_FROM_LE (functions[index]);
    }
  return proc;
}


/*  ja:PEイメージからリソースを取得する
    image,PEイメージ
      key,リソース
      RET,データ,NULL:エラー                                                */
gconstpointer
peimage_file_get_resource (const guint8 *image,
                           const gchar  *key)
{
  gconstpointer resource = NULL;

  if (image && key && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE))
    {
      gchar **element;

      element = g_strsplit (key[0] == '/' ? key + 1 : key, "/", G_MAXINT);
      if (element)
        {
          gint i;
          guint32 address, offset = 0;

          address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE);
          for (i = 0; element[i]; i++)
            {
              gint j, count;
              const ImageResourceDirectory *ird;
              const ImageResourceDirectoryEntry *irde;

              ird = (const ImageResourceDirectory *)(image + address + offset);
              irde = (const ImageResourceDirectoryEntry *)(ird + 1);
              count = ird_get_number_of_named_entries (ird)
                                        + ird_get_number_of_id_entries (ird);
              for (j = 0; j < count; j++)
                {
                  gboolean result;
                  gchar *name;

                  if (irde_get_id (irde) & 0x80000000)
                    {
                      gchar *utf8str;
                      const ImageResourceDirStringU *irdsu;

                      irdsu = (const ImageResourceDirStringU *)(image + address
                                        + (irde_get_id (irde) & 0x7fffffff));
                      utf8str = g_utf16_to_utf8 (irdsu_get_name_string (irdsu),
                                irdsu_get_length (irdsu), NULL, NULL, NULL);
                      name = g_utf8_strup (utf8str, -1);
                      g_free (utf8str);
                    }
                  else
                    {
                      name = g_strdup_printf ("0x%x", irde_get_id (irde));
                    }
                  result = g_strcmp (element[i], name) == 0;
                  g_free (name);
                  if (result)
                    break;
                  irde++;
                }
              if (j < count && irde_get_offset_to_data (irde) & 0x80000000)
                offset = irde_get_offset_to_data (irde) & 0x7fffffff;
              else if (j < count && !element[i + 1])
                resource = image + address + irde_get_offset_to_data (irde);
              else
                break;
            }
          g_strfreev (element);
        }
    }
  return resource;
}


/******************************************************************************
*                                                                             *
* ja:モジュール関数群                                                         *
*                                                                             *
******************************************************************************/
typedef struct _PeModule
{
  guint8 *image;
  gchar *file, *name;
  gint counter;
  GList *glist;
} PeModule;


# ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
# endif /* USE_THREAD */


static gchar *
peimage_get_basename (const gchar *file)
{
  gchar *name = NULL;

  if (file)
    {
      name = g_path_get_basename (file);
      if (!g_strrchr (name, '.'))
        {
          gchar *tmp;

          tmp = g_strconcat (name, ".dll", NULL);
          g_free (name);
          name = tmp;
        }
    }
  return name;
}


/*  ja:既にロードされているPEイメージを取得する
    process,プロセス
       file,イメージ名
        RET,PEイメージ,NULL:ロードされていない                              */
guint8 *
peimage_module_get_library (PeProcess   *process,
                            const gchar *file)
{
  guint8 *image = NULL;

  if (file)
    {
      gchar *base, *name;
      GList *glist;

      base = g_path_get_basename (file);
      name = peimage_get_basename (file);
#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
        if (g_ascii_strcasecmp (((PeModule *)glist->data)->name, base) == 0
        || g_ascii_strcasecmp (((PeModule *)glist->data)->name, name) == 0)
          {
            image = ((PeModule *)glist->data)->image;
            break;
          }
#ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
      g_free (base);
      g_free (name);
    }
  return image;
}


/*  ja:モジュール構造体を取得する
    process,プロセス
      image,PEイメージ
        RET,モジュール構造体,NULL:エラー                                    */
static PeModule *
peimage_module_get_info (PeProcess    *process,
                         const guint8 *image)
{
  PeModule *module = NULL;

  if (image)
    {
      GList *glist;

#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
        if (((PeModule *)glist->data)->image == image)
          {
            module = glist->data;
            break;
          }
#ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
    }
  return module;
}


/*  ja:イメージをロードする
    process,プロセス
       file,イメージ名
        RET,イメージ,NULL:エラー                                            */
guint8 *
peimage_module_load_library (PeProcess        *process,
                             const gchar      *file,
                             PeModuleSystem    pemodule_system,
                             PeModuleRelocate  pemodule_relocate,
                             PeModuleImport    pemodule_import,
                             PeModuleEntry     pemodule_entry,
                             gpointer          user_data)
{
  gboolean result = TRUE;
  guint8 *image;
  gint i;
  PeModule *module;

  if (!process || !file)
    return NULL;
  image = peimage_module_get_library (process, file);
  if (image)
    {
      /* ja:既読 */
      module = peimage_module_get_info (process, image);
      if (module)
        {
          module->counter++;
          return module->image;
        }
    }
  module = g_malloc0 (sizeof (PeModule));
  module->counter = 1;
  module->name = peimage_get_basename (file);
  if (pemodule_system)
    module->image = pemodule_system (process, module->name, user_data);
  if (module->image)
    {
      module->file = g_build_filename
            (peimage_dir_get_path (PEIMAGE_DIR_SYSTEM), module->name, NULL);
#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      process->module = g_list_append (process->module, module);
#ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
      return module->image;
    }
  g_free (module->name);

  /* ja:ファイル読み込み */
  module->name = g_path_get_basename (file);
  module->file = fileio_get_full_path (file);
  module->image = peimage_file_load (module->file);
  for (i = 0; i < 8; i++)
    if (!module->image)
      {
        gchar *path;

        g_free (module->name);
        g_free (module->file);
        module->name = i % 2 == 0 ? g_path_get_basename (file)
                                  : peimage_get_basename (file);
        path = i < 4 ? g_build_filename (peimage_dir_get_path
                (i / 2 == 0 ? PEIMAGE_DIR_SYSTEM : PEIMAGE_DIR_WINDOWS),
                                                        module->name, NULL)
                : g_build_filename (peimage_dir_get_path (PEIMAGE_DIR_HOME),
                                    i < 6 ? "bin" : "lib", module->name, NULL);
        module->file = peimage_dir_get_case_insensitive (path);
        g_free (path);
        module->image = peimage_file_load (module->file);
      }
  if (!module->image)
    {
      g_free (module->name);
      g_free (module->file);
      g_free (module);
      return NULL;
    }
  /* ja:再配置 */
  if (pemodule_relocate)
    peimage_file_relocate (module->image,
                        pemodule_relocate (process, module->image, user_data));

  /* ja:プロセス登録 */
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  process->module = g_list_append (process->module, module);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */

  /* ja:インポート */
  if (pe_ioh_get_data_directory_size (module->image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT))
    {
      ImageImportDescriptor *iid;

      for (iid = pe_image_import_descriptor (module->image);
                                                    iid_get_name (iid); iid++)
        {
          gchar *name;

          name = (gchar *)(module->image + iid_get_name (iid));
          image = peimage_module_load_library (process,
                                               name,
                                               pemodule_system,
                                               pemodule_relocate,
                                               pemodule_import,
                                               pemodule_entry,
                                               user_data);
          if (image)
            {
              ImageThunkData *iat, *ilt;

              iat = (ImageThunkData *)(module->image
                                        + (iid_get_original_first_thunk (iid)
                                        ? iid_get_original_first_thunk (iid)
                                        : iid_get_first_thunk (iid)));
              ilt = (ImageThunkData *)(module->image
                                                + iid_get_first_thunk (iid));
              while (itd_get_address_of_data (iat))
                {
                  gconstpointer func;

                  if (itd_get_ordinal (iat) & PEIMAGE_ORDINAL_FLAG)
                    {
                      guint16 ordinal;

                      ordinal = itd_get_ordinal (iat) & ~PEIMAGE_ORDINAL_FLAG;
                      func = peimage_file_get_proc (image, ordinal);
                    }
                  else
                    {
                      const ImageImportByName *iibn;

                      iibn = (const ImageImportByName *)(module->image
                                            + itd_get_address_of_data (iat));
                      func = peimage_file_get_func (image,
                                                        iibn_get_name (iibn));
                    }
                  itd_set_function (ilt, pemodule_import
                                ? pemodule_import (process, func, user_data)
                                : GPOINTER_TO_UINT (func));
                  iat++;
                  ilt++;
                }
              module->glist = g_list_append (module->glist, image);
            }
          else
            {
              result = FALSE;
            }
        }
    }
  if (!result)
    {
      peimage_module_free_library (process, module->image, NULL, user_data);
      return NULL;
    }

  if (pemodule_entry
                && !pemodule_entry (process, module->image, TRUE, user_data))
    {
      peimage_module_free_library (process,
                                    module->image, pemodule_entry, user_data);
      return NULL;
    }
  return module->image;
}


/*  ja:イメージを解放する
    process,プロセス
      image,イメージ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
peimage_module_free_library (PeProcess     *process,
                             const guint8  *image,
                             PeModuleEntry  pemodule_entry,
                             gpointer       user_data)
{
  gboolean result = FALSE;
  PeModule *module;

  module = peimage_module_get_info (process, image);
  if (module)
    {
      result = TRUE;
      module->counter--;
      if (module->counter <= 0)
        {
          if (pemodule_entry)
            pemodule_entry (process, module->image, FALSE, user_data);
#ifdef USE_THREAD
          G_LOCK (critical);
          critical = TRUE;
#endif /* USE_THREAD */
          process->module = g_list_remove (process->module, module);
#ifdef USE_THREAD
          critical = FALSE;
          G_UNLOCK (critical);
#endif /* USE_THREAD */
          for (module->glist = g_list_first (module->glist); module->glist;
            module->glist = g_list_delete_link (module->glist, module->glist))
            if (!peimage_module_free_library (process,
                                    ((PeModule *)module->glist->data)->image,
                                                    pemodule_entry, user_data))
              result = FALSE;
          g_free (module->image);
          g_free (module->name);
          g_free (module->file);
          g_free (module);
        }
    }
  return result;
}


/*  ja:イメージの名前を取得する
    process,プロセス
      image,イメージ
        RET,名前,NULL:エラー                                                */
const gchar *
peimage_module_get_filename (PeProcess    *process,
                             const guint8 *image)
{
  PeModule *module;

  module = peimage_module_get_info (process, image);
  return module ? module->file : NULL;
}


/******************************************************************************
*                                                                             *
* ja:ディレクトリ関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ディレクトリを取得する
    dir,ディレクトリの種類
    RET,ディレクトリ                                                        */
const gchar *
peimage_dir_get_path (const guint dir)
{
  gchar *file;
  const gchar *key, *path = NULL;
  const static gchar *dir_windows = NULL, *dir_system = NULL, *dir_temp = NULL;
  const static gchar *dir_data = NULL, *dir_home = NULL;
  gint i;
  GKeyFile *key_file;

  switch (dir)
    {
      case PEIMAGE_DIR_WINDOWS: if (dir_windows) return dir_windows; break;
      case PEIMAGE_DIR_SYSTEM:  if (dir_system)  return dir_system;  break;
      case PEIMAGE_DIR_TEMP:    if (dir_temp)    return dir_temp;    break;
      case PEIMAGE_DIR_DATA:    if (dir_data)    return dir_data;    break;
      case PEIMAGE_DIR_HOME:    if (dir_home)    return dir_home;
    }
  for (i = 0; i < 2 && !path; i++)
    {
      switch (dir)
        {
          case PEIMAGE_DIR_WINDOWS:
            file = peimage_dir_get_filename ("system.ini",
                                i == 0 ? PEIMAGE_DIR_HOME : PEIMAGE_DIR_DATA);
            key = "windows";
            break;
          case PEIMAGE_DIR_SYSTEM:
            file = peimage_dir_get_filename ("system.ini",
                                i == 0 ? PEIMAGE_DIR_HOME : PEIMAGE_DIR_DATA);
            key = "system";
            break;
          case PEIMAGE_DIR_TEMP:
            file = peimage_dir_get_filename ("system.ini",
                                i == 0 ? PEIMAGE_DIR_HOME : PEIMAGE_DIR_DATA);
            key = "temp";
            break;
          case PEIMAGE_DIR_DATA:
            if (i == 0)
              {
                file = g_build_filename (g_get_home_dir (),
                                    ".maid.org", "w32ldr", "system.ini", NULL);
                key = "data";
                break;
              }
          default:
            file = NULL;
            key = NULL;
        }
      if (file && key)
        {
          gchar *tmp, *utf8str;

          key_file = g_key_file_new ();
          tmp = peimage_dir_get_case_insensitive (file);
          g_key_file_load_from_file (key_file, tmp, G_KEY_FILE_NONE, NULL);
          g_free (tmp);
          utf8str = g_key_file_get_string (key_file, "directory", key, NULL);
          g_key_file_free (key_file);
          if (utf8str)
            {
              path = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
            }
        }
      g_free (file);
    }
  switch (dir)
    {
      case PEIMAGE_DIR_WINDOWS:
        return dir_windows = path ? path
#ifdef BINDIR
                                  : g_build_filename (BINDIR, "w32ldr", NULL);
#else /* not BINDIR */
                                  : "/usr/local/bin/w32ldr";
#endif /* not BINDIR */
      case PEIMAGE_DIR_SYSTEM:
        return dir_system = path ? path
#ifdef LIBDIR
                                 : g_build_filename (LIBDIR, "w32ldr", NULL);
#else /* not LIBDIR */
                                 : "/usr/local/lib/w32ldr";
#endif /* not LIBDIR */
      case PEIMAGE_DIR_TEMP:
        return dir_temp = path ? path : g_get_tmp_dir ();
      case PEIMAGE_DIR_DATA:
        return dir_data = path ? path
#ifdef SYSCONFDIR
                               : g_build_filename (SYSCONFDIR, "w32ldr", NULL);
#else /* not SYSCONFDIR */
                               : "/usr/local/etc/w32ldr";
#endif /* not SYSCONFDIR */
      case PEIMAGE_DIR_HOME:
        if (!dir_home)
          dir_home = g_build_filename (g_get_home_dir (),
                                                ".maid.org", "w32ldr", NULL);
        return dir_home;
    }
  return NULL;
}


/*  ja:大文字小文字を区別しないパス名を求める
    path,ファイル名
     RET,大文字小文字を区別しないパス名                                     */
gchar *
peimage_dir_get_case_insensitive (const gchar *path)
{
  gchar **element, *file = NULL;
  gint i;

  if (!path)
    return NULL;
  if (path[0] == '\0')
    return g_malloc0 (sizeof (gchar));
  element = g_strsplit (path, G_DIR_SEPARATOR_S, 0);
  for (i = 0; element[i]; i++)
    if ((element[i])[0] == '\0')
      {
        if (!file)
          file = g_strdup (G_DIR_SEPARATOR_S);
      }
    else
      {
        gchar *tmp;

        tmp = file ? g_build_filename (file, element[i], NULL)
                   : g_strdup (element[i]);
        if (g_file_test (tmp, G_FILE_TEST_EXISTS))
          {
            g_free (file);
            file = tmp;
          }
        else
          {
            const gchar *name;
            GDir *dir;

            g_free (tmp);
            tmp = NULL;
            dir = g_dir_open (file ? file : ".", 0, NULL);
            while ((name = g_dir_read_name (dir)))
              if (g_ascii_strcasecmp (element[i], name) == 0)
                {
                  tmp = file ? g_build_filename (file, name, NULL)
                             : g_strdup (name);
                  break;
                }
            g_dir_close (dir);
            if (!tmp)
              {
                while (element[i])
                  {
                    if ((element[i])[0] != '\0')
                      {
                        tmp = file ? g_build_filename (file, element[i], NULL)
                                   : g_strdup (element[i]);
                        g_free (file);
                        file = tmp;
                      }
                    i++;
                  }
                break;
              }
            g_free (file);
            file = tmp;
          }
      }
  g_strfreev (element);
  return file;
}


/*  ja:特定のディレクトリのファイルを取得する
    file,元のファイル名
     dir,ディレクトリの種類
     RET,ファイル                                                           */
gchar *
peimage_dir_get_filename (const gchar *file,
                          const guint  dir)
{
  gchar *name, *tmp, *result;

  name = g_path_get_basename (file);
  tmp = g_build_filename (peimage_dir_get_path (dir), name, NULL);
  g_free (name);
  result = peimage_dir_get_case_insensitive (tmp);
  g_free (tmp);
  return result;
}
