/*
    peimage
    copyright (c) 1998-2010 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __PEIMAGE_H__
#define __PEIMAGE_H__


#include "gcommon.h"


G_BEGIN_DECLS


#define PEIMAGE_DOS_SIGNATURE 0x5a4d
#define PEIMAGE_NT_SIGNATURE  0x00004550
/* ja:ファイルヘッダ */
#define PEIMAGE_FILE_MACHINE_I386 0x014c
#define PEIMAGE_FILE_EXECUTABLE_IMAGE    0x0002
#define PEIMAGE_FILE_LINE_NUMS_STRIPPED  0x0004
#define PEIMAGE_FILE_LOCAL_SYMS_STRIPPED 0x0008
#define PEIMAGE_FILE_32BIT_MACHINE       0x0100
#define PEIMAGE_FILE_DLL                 0x2000
/* ja:オプションヘッダ */
#define PEIMAGE_NT_OPTIONAL_HDR_MAGIC 0x10b
#define PEIMAGE_SUBSYSTEM_UNKNOWN        0
#define PEIMAGE_SUBSYSTEM_NATIVE         1
#define PEIMAGE_SUBSYSTEM_WINDOWS_GUI    2
#define PEIMAGE_SUBSYSTEM_WINDOWS_CUI    3
#define PEIMAGE_SUBSYSTEM_POSIX_CUI      7
#define PEIMAGE_SUBSYSTEM_WINDOWS_CE_GUI 9
#define PEIMAGE_NUMBEROF_DIRECTORY_ENTRIES 16
#define PEIMAGE_DIRECTORY_ENTRY_EXPORT 0
#define PEIMAGE_DIRECTORY_ENTRY_IMPORT 1
#define PEIMAGE_DIRECTORY_ENTRY_RESOURCE 2
#define PEIMAGE_DIRECTORY_ENTRY_BASERELOC 5
/* ja:セクションヘッダ */
#define PEIMAGE_SIZEOF_SHORT_NAME 8
#define PEIMAGE_SCN_CNT_CODE               0x00000020
#define PEIMAGE_SCN_CNT_INITIALIZED_DATA   0x00000040
#define PEIMAGE_SCN_CNT_UNINITIALIZED_DATA 0x00000080
#define PEIMAGE_SCN_LNK_OVFL               0x01000000
#define PEIMAGE_SCN_MEM_DISCARDABLE        0x02000000
#define PEIMAGE_SCN_MEM_NOT_CACHED         0x04000000
#define PEIMAGE_SCN_MEM_NOT_PAGED          0x08000000
#define PEIMAGE_SCN_MEM_SHARED             0x10000000
#define PEIMAGE_SCN_MEM_EXECUTE            0x20000000
#define PEIMAGE_SCN_MEM_READ               0x40000000
#define PEIMAGE_SCN_MEM_WRITE              0x80000000
/* ja:エクスポートテーブル */
#define PEIMAGE_ORDINAL_FLAG (1<<31)
/* ja:再配置テーブル */
#define PEIMAGE_REL_BASED_HIGHLOW 3


#define IDH_SIZE 64
#define IFH_SIZE 20
#define IDD_SIZE 8
#define IOH_SIZE (96+IDD_SIZE*PEIMAGE_NUMBEROF_DIRECTORY_ENTRIES)
#define ISH_SIZE (PEIMAGE_SIZEOF_SHORT_NAME+32)
#define IED_SIZE 40
#define IIBN_SIZE 2
#define ITD_SIZE 4
#define IID_SIZE 20
#define IBR_SIZE 8
#define IRD_SIZE 16
#define IRDE_SIZE 8
#define IRDSU_SIZE 2
#define IRDA_SIZE 16


#ifdef USE_GCC


/* ja:MS-DOSヘッダ */
typedef struct _ImageDosHeader
{
  guint16 e_magic __attribute__((packed));
  guint16 e_cblp __attribute__((packed));
  guint16 e_cp __attribute__((packed));
  guint16 e_crlc __attribute__((packed));
  guint16 e_cparhdr __attribute__((packed));
  guint16 e_minalloc __attribute__((packed));
  guint16 e_maxalloc __attribute__((packed));
  guint16 e_ss __attribute__((packed));
  guint16 e_sp __attribute__((packed));
  guint16 e_csum __attribute__((packed));
  guint16 e_ip __attribute__((packed));
  guint16 e_cs __attribute__((packed));
  guint16 e_lfarlc __attribute__((packed));
  guint16 e_ovno __attribute__((packed));
  guint16 e_res[4] __attribute__((packed));
  guint16 e_oemid __attribute__((packed));
  guint16 e_oeminfo __attribute__((packed));
  guint16 e_res2[10] __attribute__((packed));
  gint32 e_lfanew __attribute__((packed));
} ImageDosHeader;
/* ja:ファイルヘッダ */
typedef struct _ImageFileHeader
{
  guint16 machine __attribute__((packed));
  guint16 number_of_sections __attribute__((packed));
  guint32 time_date_stamp __attribute__((packed));
  guint32 pointer_to_symbol_table __attribute__((packed));
  guint32 number_of_symbols __attribute__((packed));
  guint16 size_of_optional_header __attribute__((packed));
  guint16 characteristics __attribute__((packed));
} ImageFileHeader;
/* ja:オプションヘッダ */
typedef struct _ImageDataDirectory
{
  guint32 vrtual_address __attribute__((packed));
  guint32 size __attribute__((packed));
} ImageDataDirectory;
typedef struct _ImageOptionalHeader
{
  guint16 magic __attribute__((packed));
  guint8 major_linker_version __attribute__((packed));
  guint8 minor_linker_version __attribute__((packed));
  guint32 size_of_code __attribute__((packed));
  guint32 size_of_initialized_data __attribute__((packed));
  guint32 size_of_uninitialized_data __attribute__((packed));
  guint32 address_of_entry_point __attribute__((packed));
  guint32 base_of_code __attribute__((packed));
  guint32 base_of_data __attribute__((packed));
  guint32 image_base __attribute__((packed));
  guint32 section_alignment __attribute__((packed));
  guint32 file_alignment __attribute__((packed));
  guint16 major_operating_system_version __attribute__((packed));
  guint16 minor_operating_system_version __attribute__((packed));
  guint16 major_image_version __attribute__((packed));
  guint16 minor_image_version __attribute__((packed));
  guint16 major_subsystem_version __attribute__((packed));
  guint16 minor_subsystem_version __attribute__((packed));
  guint32 win32_version_value __attribute__((packed));
  guint32 size_of_image __attribute__((packed));
  guint32 size_of_headers __attribute__((packed));
  guint32 check_sum __attribute__((packed));
  guint16 subsystem __attribute__((packed));
  guint16 dll_characteristics __attribute__((packed));
  guint32 size_of_stack_reserve __attribute__((packed));
  guint32 size_of_stack_commit __attribute__((packed));
  guint32 size_of_heap_reserve __attribute__((packed));
  guint32 size_of_heap_commit __attribute__((packed));
  guint32 loader_flags __attribute__((packed));
  guint32 number_of_rva_and_sizes __attribute__((packed));
  ImageDataDirectory data_directory[PEIMAGE_NUMBEROF_DIRECTORY_ENTRIES] __attribute__((packed));
} ImageOptionalHeader;
/* ja:セクションヘッダ */
typedef struct _ImageSectionHeader
{
  guint8 name[PEIMAGE_SIZEOF_SHORT_NAME] __attribute__((packed));
  union {
    guint32 physical_address __attribute__((packed));
    guint32 virtual_size __attribute__((packed));
  } misc __attribute__((packed));
  guint32 virtual_address __attribute__((packed));
  guint32 size_of_raw_data __attribute__((packed));
  guint32 pointer_to_raw_data __attribute__((packed));
  guint32 pointer_to_relocations __attribute__((packed));
  guint32 pointer_to_linenumbers __attribute__((packed));
  guint16 number_of_relocations __attribute__((packed));
  guint16 number_of_linenumbers __attribute__((packed));
  guint32 characteristics __attribute__((packed));
} ImageSectionHeader;
/* ja:エクスポートテーブル */
typedef struct _ImageExportDirectory
{
  guint32 characteristics __attribute__((packed));
  guint32 time_date_stamp __attribute__((packed));
  guint16 major_version __attribute__((packed));
  guint16 minor_version __attribute__((packed));
  guint32 name __attribute__((packed));
  guint32 base __attribute__((packed));
  guint32 number_of_functions __attribute__((packed));
  guint32 number_of_Names __attribute__((packed));
  guint32 address_of_functions __attribute__((packed));
  guint32 address_of_names __attribute__((packed));
  guint32 address_of_name_ordinals __attribute__((packed));
} ImageExportDirectory;
/* ja:インポートテーブル */
typedef struct _ImageImportByName
{
  guint16 hint __attribute__((packed));
  guint8 name[1] __attribute__((packed));
} ImageImportByName;
typedef struct _ImageThunkData
{
  union {
    guint32 forwarder_string __attribute__((packed));
    guint32 function __attribute__((packed));
    guint32 ordinal __attribute__((packed));
    guint32 address_of_data __attribute__((packed));
  } u1 __attribute__((packed));
} ImageThunkData;
typedef struct _ImageImportDescriptor
{
  union {
    guint32 characteristics __attribute__((packed));
    guint32 original_first_thunk __attribute__((packed));
  } DummyUnionName __attribute__((packed));
  guint32 time_date_stamp __attribute__((packed));
  guint32 forwarder_chain __attribute__((packed));
  guint32 name __attribute__((packed));
  guint32 first_thunk __attribute__((packed));
} ImageImportDescriptor;
/* ja:再配置テーブル */
typedef struct _ImageBaseRelocation
{
  guint32 virtual_address __attribute__((packed));
  guint32 size_of_block __attribute__((packed));
  guint16 type_offset[1] __attribute__((packed));
} ImageBaseRelocation;
/* ja:リソース */
typedef struct _ImageResourceDirectory
{
  guint32 characteristics __attribute__((packed));
  guint32 time_date_stamp __attribute__((packed));
  guint16 major_version __attribute__((packed));
  guint16 minor_version __attribute__((packed));
  guint16 number_of_named_entries __attribute__((packed));
  guint16 number_of_id_entries __attribute__((packed));
} ImageResourceDirectory;
typedef struct _ImageResourceDirectoryEntry
{
  guint32 id __attribute__((packed));
  guint32 offset_to_data __attribute__((packed));
} ImageResourceDirectoryEntry;
typedef struct _ImageResourceDirStringU
{
  guint16 length __attribute__((packed));
  gunichar2 name_string[1] __attribute__((packed));
} ImageResourceDirStringU;
typedef struct _ImageResourceDataEntry
{
  guint32 offset_to_data __attribute__((packed));
  guint32 size __attribute__((packed));
  guint32 code_page __attribute__((packed));
  guint32 reserved __attribute__((packed));
} ImageResourceDataEntry;


#else /* not USE_GCC */

typedef struct _ImageDosHeader { guint8 dummy[IDH_SIZE]; } ImageDosHeader;
typedef struct _ImageFileHeader { guint8 dummy[IFH_SIZE]; } ImageFileHeader;
typedef struct _ImageDataDirectory { guint8 dummy[IDD_SIZE]; } ImageDataDirectory;
typedef struct _ImageOptionalHeader { guint8 dummy[IOH_SIZE]; } ImageOptionalHeader;
typedef struct _ImageSectionHeader { guint8 dummy[ISH_SIZE]; } ImageSectionHeader;
typedef struct _ImageExportDirectory { guint8 dummy[IED_SIZE]; } ImageExportDirectory;
typedef struct _ImageImportByName { guint8 dummy[IIBN_SIZE]; } ImageImportByName;
typedef struct _ImageThunkData { guint8 dummy[ITD_SIZE]; } ImageThunkData;
typedef struct _ImageImportDescriptor { guint8 dummy[IID_SIZE]; } ImageImportDescriptor;
typedef struct _ImageBaseRelocation { guint8 dummy[IBR_SIZE]; } ImageBaseRelocation;
typedef struct _ImageResourceDirectory { guint8 dummy[IRD_SIZE]; } ImageResourceDirectory;
typedef struct _ImageResourceDirectoryEntry { guint8 dummy[IRDE_SIZE]; } ImageResourceDirectoryEntry;
typedef struct _ImageResourceDirStringU { guint8 dummy[IRDSU_SIZE]; } ImageResourceDirStringU;
typedef struct _ImageResourceDataEntry { guint8 dummy[IRDA_SIZE]; } ImageResourceDataEntry;


#endif /* not USE_GCC */


/******************************************************************************
*                                                                             *
* ja:PEイメージ構造体マクロ                                                   *
*                                                                             *
******************************************************************************/
#define idh_get_magic(idh) GUINT16_FROM_LE(*(guint16 *)(idh))
#define idh_get_lfanew(idh) GINT32_FROM_LE(*(gint32 *)((guint8 *)(idh)+60))
#define idh_set_magic(idh,magic) (*(guint16 *)(idh)=GUINT16_TO_LE(magic))
#define idh_set_lfanew(idh,lfanew) (*(guint32 *)((guint8 *)(idh)+60)=GINT32_TO_LE(lfanew))

#define ifh_get_machine(ifh) GUINT16_FROM_LE(*(guint16 *)(ifh))
#define ifh_get_number_of_sections(ifh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ifh)+2))
#define ifh_get_time_date_stamp(ifh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ifh)+4))
#define ifh_get_pointer_to_symbol_table(ifh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ifh)+8))
#define ifh_get_number_of_symbols(ifh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ifh)+12))
#define ifh_get_size_of_optional_header(ifh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ifh)+16))
#define ifh_get_characteristics(ifh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ifh)+18))
#define ifh_set_machine(ifh,machine) (*(guint16 *)(ifh)=GUINT16_TO_LE(machine))
#define ifh_set_number_of_sections(ifh,number_of_sections) (*(guint16 *)((guint8 *)(ifh)+2)=GUINT16_TO_LE(number_of_sections))
#define ifh_set_time_date_stamp(ifh,time_date_stamp) (*(guint32 *)((guint8 *)(ifh)+4)=GUINT32_TO_LE(time_date_stamp))
#define ifh_set_pointer_to_symbol_table(ifh,pointer_to_symbol_table) (*(guint32 *)((guint8 *)(ifh)+8)=GUINT32_TO_LE(pointer_to_symbol_table))
#define ifh_set_number_of_symbols(ifh,number_of_symbols) (*(guint32 *)((guint8 *)(ifh)+12)=GUINT32_TO_LE(number_of_symbols))
#define ifh_set_size_of_optional_header(ifh,size_of_optional_header) (*(guint16 *)((guint8 *)(ifh)+16)=GUINT16_TO_LE(size_of_optional_header))
#define ifh_set_characteristics(ifh,characteristics) (*(guint16 *)((guint8 *)(ifh)+18)=GUINT16_TO_LE(characteristics))


#define idd_get_virtual_address(idd) GUINT32_FROM_LE(*(guint32 *)(idd))
#define idd_get_size(idd) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(idd)+4))
#define idd_set_virtual_address(idd,virtual_address) (*(guint32 *)(idd)=GUINT32_TO_LE(virtual_address))
#define idd_set_size(idd,size) (*(guint32 *)((guint8 *)(idd)+4)=GUINT32_TO_LE(size))


#define ioh_get_magic(ioh) GUINT16_FROM_LE(*(guint16 *)(ioh))
#define ioh_get_major_linker_version(ioh) (*((guint8 *)(ioh)+2))
#define ioh_get_minor_linker_version(ioh) (*((guint8 *)(ioh)+3))
#define ioh_get_size_of_code(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+4))
#define ioh_get_size_of_initialized_data(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+8))
#define ioh_get_size_of_uninitialized_data(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+12))
#define ioh_get_address_of_entry_point(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+16))
#define ioh_get_base_of_code(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+20))
#define ioh_get_base_of_data(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+24))
#define ioh_get_image_base(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+28))
#define ioh_get_section_alignment(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+32))
#define ioh_get_file_alignment(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+36))
#define ioh_get_major_operating_system_version(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+40))
#define ioh_get_minor_operating_system_version(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+42))
#define ioh_get_major_image_version(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+44))
#define ioh_get_minor_image_version(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+46))
#define ioh_get_major_subsystem_version(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+48))
#define ioh_get_minor_subsystem_version(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+50))
#define ioh_get_win32_version_value(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+52))
#define ioh_get_size_of_image(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+56))
#define ioh_get_size_of_headers(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+60))
#define ioh_get_check_sum(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+64))
#define ioh_get_subsystem(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+68))
#define ioh_get_dll_characteristics(ioh) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ioh)+70))
#define ioh_get_size_of_stack_reserve(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+72))
#define ioh_get_size_of_stack_commit(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+76))
#define ioh_get_size_of_heap_reserve(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+80))
#define ioh_get_size_of_heap_commit(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+84))
#define ioh_get_loader_flags(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+88))
#define ioh_get_number_of_rva_and_sizes(ioh) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ioh)+92))
#define ioh_get_data_directory_virtual_address(ioh,entry) idd_get_virtual_address((guint8 *)(ioh)+96+(entry)*IDD_SIZE)
#define ioh_get_data_directory_size(ioh,entry) idd_get_size((guint8 *)(ioh)+96+(entry)*IDD_SIZE)
#define ioh_set_magic(ioh,magic) (*(guint16 *)(ioh)=GUINT16_TO_LE(magic))
#define ioh_set_major_linker_version(ioh,major_linker_version) (*((guint8 *)(ioh)+2)=(major_linker_version))
#define ioh_set_minor_linker_version(ioh,minor_linker_version) (*((guint8 *)(ioh)+3)=(minor_linker_version))
#define ioh_set_size_of_code(ioh,size_of_code) (*(guint32 *)((guint8 *)(ioh)+4)=GUINT32_TO_LE(size_of_code))
#define ioh_set_size_of_initialized_data(ioh,size_of_initialized_data) (*(guint32 *)((guint8 *)(ioh)+8)=GUINT32_TO_LE(size_of_initialized_data))
#define ioh_set_size_of_uninitialized_data(ioh,size_of_uninitialized_data) (*(guint32 *)((guint8 *)(ioh)+12)=GUINT32_TO_LE(size_of_uninitialized_data))
#define ioh_set_address_of_entry_point(ioh,address_of_entry_point) (*(guint32 *)((guint8 *)(ioh)+16)=GUINT32_TO_LE(address_of_entry_point))
#define ioh_set_base_of_code(ioh,base_of_code) (*(guint32 *)((guint8 *)(ioh)+20)=GUINT32_TO_LE(base_of_code))
#define ioh_set_base_of_data(ioh,base_of_data) (*(guint32 *)((guint8 *)(ioh)+24)=GUINT32_TO_LE(base_of_data))
#define ioh_set_image_base(ioh,image_base) (*(guint32 *)((guint8 *)(ioh)+28)=GUINT32_TO_LE(image_base))
#define ioh_set_section_alignment(ioh,section_alignment) (*(guint32 *)((guint8 *)(ioh)+32)=GUINT32_TO_LE(section_alignment))
#define ioh_set_file_alignment(ioh,file_alignment) (*(guint32 *)((guint8 *)(ioh)+36)=GUINT32_TO_LE(file_alignment))
#define ioh_set_major_operating_system_version(ioh,major_operating_system_version) (*(guint16 *)((guint8 *)(ioh)+40)=GUINT16_TO_LE(major_operating_system_version))
#define ioh_set_minor_operating_system_version(ioh,minor_operating_system_version) (*(guint16 *)((guint8 *)(ioh)+42)=GUINT16_TO_LE(minor_operating_system_version))
#define ioh_set_major_image_version(ioh,major_image_version) (*(guint16 *)((guint8 *)(ioh)+44)=GUINT16_TO_LE(major_image_version))
#define ioh_set_minor_image_version(ioh,minor_image_version) (*(guint16 *)((guint8 *)(ioh)+46)=GUINT16_TO_LE(minor_image_version))
#define ioh_set_major_subsystem_version(ioh,major_subsystem_version) (*(guint16 *)((guint8 *)(ioh)+48)=GUINT16_TO_LE(major_subsystem_version))
#define ioh_set_minor_subsystem_version(ioh,minor_subsystem_version) (*(guint16 *)((guint8 *)(ioh)+50)=GUINT16_TO_LE(minor_subsystem_version))
#define ioh_set_win32_version_value(ioh,win32_version_value) (*(guint32 *)((guint8 *)(ioh)+52)=GUINT32_TO_LE(win32_version_value))
#define ioh_set_size_of_image(ioh,size_of_image) (*(guint32 *)((guint8 *)(ioh)+56)=GUINT32_TO_LE(size_of_image))
#define ioh_set_size_of_headers(ioh,size_of_headers) (*(guint32 *)((guint8 *)(ioh)+60)=GUINT32_TO_LE(size_of_headers))
#define ioh_set_check_sum(ioh,check_sum) (*(guint32 *)((guint8 *)(ioh)+64)=GUINT32_TO_LE(check_sum))
#define ioh_set_subsystem(ioh,subsystem) (*(guint16 *)((guint8 *)(ioh)+68)=GUINT16_TO_LE(subsystem))
#define ioh_set_dll_characteristics(ioh,dll_characteristics) (*(guint16 *)((guint8 *)(ioh)+70)=GUINT16_TO_LE(dll_characteristics))
#define ioh_set_size_of_stack_reserve(ioh,size_of_stack_reserve) (*(guint32 *)((guint8 *)(ioh)+72)=GUINT32_TO_LE(size_of_stack_reserve))
#define ioh_set_size_of_stack_commit(ioh,size_of_stack_commit) (*(guint32 *)((guint8 *)(ioh)+76)=GUINT32_TO_LE(size_of_stack_commit))
#define ioh_set_size_of_heap_reserve(ioh,size_of_heap_reserve) (*(guint32 *)((guint8 *)(ioh)+80)=GUINT32_TO_LE(size_of_heap_reserve))
#define ioh_set_size_of_heap_commit(ioh,size_of_heap_commit) (*(guint32 *)((guint8 *)(ioh)+84)=GUINT32_TO_LE(size_of_heap_commit))
#define ioh_set_loader_flags(ioh,loader_flags) (*(guint32 *)((guint8 *)(ioh)+88)=GUINT32_TO_LE(loader_flags))
#define ioh_set_number_of_rva_and_sizes(ioh,number_of_rva_and_sizes) (*(guint32 *)((guint8 *)(ioh)+92)=GUINT32_TO_LE(number_of_rva_and_sizes))
#define ioh_set_data_directory_virtual_address(ioh,entry,virtual_address) idd_set_virtual_address((guint8 *)(ioh)+96+(entry)*IDD_SIZE,virtual_address)
#define ioh_set_data_directory_size(ioh,entry,size) idd_set_size((guint8 *)(ioh)+96+(entry)*IDD_SIZE,size)


#define ish_get_name(ish) (gchar *)(ish)
#define ish_get_physical_address(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+8))
#define ish_get_virtual_size(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+8))
#define ish_get_virtual_address(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+12))
#define ish_get_size_of_raw_data(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+16))
#define ish_get_pointer_to_raw_data(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+20))
#define ish_get_pointer_to_relocations(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+24))
#define ish_get_pointer_to_linenumbers(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+28))
#define ish_get_number_of_relocations(ish) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ish)+32))
#define ish_get_number_of_linenumbers(ish) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ish)+34))
#define ish_get_characteristics(ish) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ish)+36))
#define ish_set_name(ish,name) g_strncpy((gchar *)(ish),(gchar *)(name),PEIMAGE_SIZEOF_SHORT_NAME)
#define ish_set_physical_address(ish,physical_address) (*(guint32 *)((guint8 *)(ish)+8)=GUINT32_TO_LE(physical_address))
#define ish_set_virtual_size(ish,virtual_size) (*(guint32 *)((guint8 *)(ish)+8)=GUINT32_TO_LE(virtual_size))
#define ish_set_virtual_address(ish,virtual_address) (*(guint32 *)((guint8 *)(ish)+12)=GUINT32_TO_LE(virtual_address))
#define ish_set_size_of_raw_data(ish,size_of_raw_data) (*(guint32 *)((guint8 *)(ish)+16)=GUINT32_TO_LE(size_of_raw_data))
#define ish_set_pointer_to_raw_data(ish,pointer_to_raw_data) (*(guint32 *)((guint8 *)(ish)+20)=GUINT32_TO_LE(pointer_to_raw_data))
#define ish_set_pointer_to_relocations(ish,pointer_to_relocations) (*(guint32 *)((guint8 *)(ish)+24)=GUINT32_TO_LE(pointer_to_relocations))
#define ish_set_pointer_to_linenumbers(ish,pointer_to_linenumbers) (*(guint32 *)((guint8 *)(ish)+28)=GUINT32_TO_LE(pointer_to_linenumbers))
#define ish_set_number_of_relocations(ish,number_of_relocations) (*(guint16 *)((guint8 *)(ish)+32)=GUINT16_TO_LE(number_of_relocations))
#define ish_set_number_of_linenumbers(ish,number_of_linenumbers) (*(guint16 *)((guint8 *)(ish)+34)=GUINT16_TO_LE(number_of_linenumbers))
#define ish_set_characteristics(ish,characteristics) (*(guint32 *)((guint8 *)(ish)+36)=GUINT32_TO_LE(characteristics))


#define ied_get_characteristics(ied) GUINT32_FROM_LE(*(guint32 *)(ied))
#define ied_get_time_date_stamp(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+4))
#define ied_get_major_version(ied) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ied)+8))
#define ied_get_minor_version(ied) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ied)+10))
#define ied_get_name(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+12))
#define ied_get_base(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+16))
#define ied_get_number_of_functions(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+20))
#define ied_get_number_of_names(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+24))
#define ied_get_address_of_functions(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+28))
#define ied_get_address_of_names(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+32))
#define ied_get_address_of_name_ordinals(ied) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ied)+36))
#define ied_set_characteristics(ied,characteristics) (*(guint32 *)(ied)=GUINT32_TO_LE(characteristics))
#define ied_set_time_date_stamp(ied,time_date_stamp) (*(guint32 *)((guint8 *)(ied)+4)=GUINT32_TO_LE(time_date_stamp))
#define ied_set_major_version(ied,major_version) (*(guint16 *)((guint8 *)(ied)+8)=GUINT16_TO_LE(major_version))
#define ied_set_minor_version(ied,minor_version) (*(guint16 *)((guint8 *)(ied)+10)=GUINT16_TO_LE(minor_version))
#define ied_set_name(ied,name) (*(guint32 *)((guint8 *)(ied)+12)=GUINT32_TO_LE(name))
#define ied_set_base(ied,base) (*(guint32 *)((guint8 *)(ied)+16)=GUINT32_TO_LE(base))
#define ied_set_number_of_functions(ied,number_of_functions) (*(guint32 *)((guint8 *)(ied)+20)=GUINT32_TO_LE(number_of_functions))
#define ied_set_number_of_names(ied,number_of_names) (*(guint32 *)((guint8 *)(ied)+24)=GUINT32_TO_LE(number_of_names))
#define ied_set_address_of_functions(ied,address_of_functions) (*(guint32 *)((guint8 *)(ied)+28)=GUINT32_TO_LE(address_of_functions))
#define ied_set_address_of_names(ied,address_of_names) (*(guint32 *)((guint8 *)(ied)+32)=GUINT32_TO_LE(address_of_names))
#define ied_set_address_of_name_ordinals(ied,address_of_name_ordinals) (*(guint32 *)((guint8 *)(ied)+36)=GUINT32_TO_LE(address_of_name_ordinals))


#define iibn_get_hint(iibn) GUINT16_FROM_LE(*(guint16 *)(iibn))
#define iibn_get_name(iibn) ((gchar *)(iibn)+2)
#define iibn_set_hint(iibn,hint) (*(guint16 *)(iibn)=GUINT16_TO_LE(hint))
#define iibn_set_name(iibn,name) g_strcpy((gchar *)(iibn)+2,(gchar *)(name))


#define itd_get_forwarder_string(itd) GUINT32_FROM_LE(*(guint32 *)(itd))
#define itd_get_function(itd) GUINT32_FROM_LE(*(guint32 *)(itd))
#define itd_get_ordinal(itd) GUINT32_FROM_LE(*(guint32 *)(itd))
#define itd_get_address_of_data(itd) GUINT32_FROM_LE(*(guint32 *)(itd))
#define itd_set_forwarder_string(itd,forwarder_string) (*(guint32 *)(itd)=GUINT32_TO_LE(forwarder_string))
#define itd_set_function(itd,function) (*(guint32 *)(itd)=GUINT32_TO_LE(function))
#define itd_set_ordinal(itd,ordinal) (*(guint32 *)(itd)=GUINT32_TO_LE(ordinal))
#define itd_set_address_of_data(itd,address_of_data) (*(guint32 *)(itd)=GUINT32_TO_LE(address_of_data))


#define iid_get_characteristics(iid) GUINT32_FROM_LE(*(guint32 *)(iid))
#define iid_get_original_first_thunk(iid) GUINT32_FROM_LE(*(guint32 *)(iid))
#define iid_get_time_date_stamp(iid) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(iid)+4))
#define iid_get_forwarder_chain(iid) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(iid)+8))
#define iid_get_name(iid) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(iid)+12))
#define iid_get_first_thunk(iid) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(iid)+16))
#define iid_set_characteristics(iid,characteristics) (*(guint32 *)(iid)=GUINT32_TO_LE(characteristics))
#define iid_set_original_first_thunk(iid,original_first_thunk) (*(guint32 *)(iid)=GUINT32_TO_LE(original_first_thunk))
#define iid_set_time_date_stamp(iid,time_date_stamp) (*(guint32 *)((guint8 *)(iid)+4)=GUINT32_TO_LE(time_date_stamp))
#define iid_set_forwarder_chain(iid,forwarder_chain) (*(guint32 *)((guint8 *)(iid)+8)=GUINT32_TO_LE(forwarder_chain))
#define iid_set_name(iid,name) (*(guint32 *)((guint8 *)(iid)+12)=GUINT32_TO_LE(name))
#define iid_set_first_thunk(iid,first_thunk) (*(guint32 *)((guint8 *)(iid)+16)=GUINT32_TO_LE(first_thunk))


#define ibr_get_virtual_address(ibr) GUINT32_FROM_LE(*(guint32 *)(ibr))
#define ibr_get_size_of_block(ibr) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ibr)+4))
#define ibr_get_type_offset(ibr,n) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ibr)+8+(n)*2))
#define ibr_set_virtual_address(ibr,virtual_address) (*(guint32 *)(ibr)=GUINT32_TO_LE(virtual_address))
#define ibr_set_size_of_block(ibr,size_of_block) (*(guint32 *)((guint8 *)(ibr)+4)=GUINT32_TO_LE(size_of_block))
#define ibr_set_type_offset(ibr,n,type_offset) (*(guint16 *)((guint8 *)(ibr)+8+(n)*2)=GUINT16_TO_LE(type_offset))


#define ird_get_characteristics(ird) GUINT32_FROM_LE(*(guint32 *)(ird))
#define ird_get_time_date_stamp(ird) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(ird)+4))
#define ird_get_major_version(ird) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ird)+8))
#define ird_get_minor_version(ird) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ird)+10))
#define ird_get_number_of_named_entries(ird) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ird)+12))
#define ird_get_number_of_id_entries(ird) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(ird)+14))
#define ird_set_characteristics(ird,characteristics) (*(guint32 *)(ird)=GUINT32_TO_LE(characteristics))
#define ird_set_time_date_stamp(ird,time_date_stamp) (*(guint32 *)((guint8 *)(ird)+4)=GUINT32_TO_LE(time_date_stamp))
#define ird_set_major_version(ird,major_version) (*(guint16 *)((guint8 *)(ird)+8)=GUINT16_TO_LE(major_version))
#define ird_set_minor_version(ird,minor_version) (*(guint16 *)((guint8 *)(ird)+10)=GUINT16_TO_LE(minor_version))
#define ird_set_number_of_named_entries(ird,number_of_named_entries) (*(guint16 *)((guint8 *)(ird)+12)=GUINT16_TO_LE(number_of_named_entries))
#define ird_set_number_of_id_entries(ird,number_of_id_entries) (*(guint16 *)((guint8 *)(ird)+14)=GUINT16_TO_LE(number_of_id_entries))


#define irde_get_id(irde) GUINT32_FROM_LE(*(guint32 *)(irde))
#define irde_get_offset_to_data(irde) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(irde)+4))
#define irde_set_id(irde,id) (*(guint32 *)(irde)=GUINT32_TO_LE(id))
#define irde_set_offset_to_data(irde,offset_to_data) (*(guint32 *)((guint8 *)(irde)+4)=GUINT32_TO_LE(offset_to_data))


#define irdsu_get_length(irdsu) GUINT16_FROM_LE(*(guint16 *)(irdsu))
#define irdsu_get_name_string(irdsu) (gunichar2 *)((guint8 *)(irdsu)+2)
#define irdsu_set_length(irdsu,length) (*(guint16 *)(irdsu)=GUINT16_TO_LE(length))


#define irda_get_offset_to_data(irda) GUINT32_FROM_LE(*(guint32 *)(irda))
#define irda_get_size(irda) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(irda)+4))
#define irda_get_code_page(irda) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(irda)+8))
#define irda_get_reserved(irda) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(irda)+12))
#define irda_set_offset_to_data(irda,offset_to_data) (*(guint32 *)(irda)=GUINT32_TO_LE(offset_to_data))
#define irda_set_size(irda,size) (*(guint32 *)((guint8 *)(irda)+4)=GUINT32_TO_LE(size))
#define irda_set_code_page(irda,code_page) (*(guint32 *)((guint8 *)(irda)+8)=GUINT32_TO_LE(code_page))
#define irda_set_reserved(irda,reserved) (*(guint32 *)((guint8 *)(irda)+12)=GUINT32_TO_LE(reserved))


/******************************************************************************
*                                                                             *
* ja:PEイメージマクロ                                                         *
*                                                                             *
******************************************************************************/
#define pe_signature(image) (guint32 *)((guint8 *)(image)+idh_get_lfanew(image))
#define pe_get_signature(image) GUINT32_FROM_LE(*(guint32 *)((guint8 *)(image)+idh_get_lfanew(image)))
#define pe_set_signature(image,signature) (*(guint32 *)((guint8 *)(image)+idh_get_lfanew(image))=GUINT32_TO_LE(signature))


#define pe_image_dos_header(image) (ImageDosHeader *)(image)
#define pe_idh_get_magic(image) idh_get_magic(pe_image_dos_header(image))
#define pe_idh_get_lfanew(image) idh_get_lfanew(pe_image_dos_header(image))
#define pe_idh_set_magic(image,magic) idh_set_magic(pe_image_dos_header(image),magic)
#define pe_idh_set_lfanew(image,lfanew) idh_set_lfanew(pe_image_dos_header(image),lfanew)


#define pe_image_file_header(image) (ImageFileHeader *)((guint8 *)(image)+idh_get_lfanew(image)+4)
#define pe_ifh_get_machine(image) ifh_get_machine(pe_image_file_header(image))
#define pe_ifh_get_number_of_sections(image) ifh_get_number_of_sections(pe_image_file_header(image))
#define pe_ifh_get_time_date_stamp(image) ifh_get_time_date_stamp(pe_image_file_header(image))
#define pe_ifh_get_pointer_to_symbol_table(image) ifh_get_pointer_to_symbol_table(pe_image_file_header(image))
#define pe_ifh_get_number_of_symbols(image) ifh_get_number_of_symbols(pe_image_file_header(image))
#define pe_ifh_get_size_of_optional_header(image) ifh_get_size_of_optional_header(pe_image_file_header(image))
#define pe_ifh_get_characteristics(image) ifh_get_characteristics(pe_image_file_header(image))
#define pe_ifh_setmachine(imageh,machine) ifh_set_machine(pe_image_file_header(image),machine)
#define pe_ifh_set_number_of_sections(image,number_of_sections) ifh_set_number_of_sections(pe_image_file_header(image),number_of_sections)
#define pe_ifh_set_time_date_stamp(image,time_date_stamp) ifh_set_time_date_stamp(pe_image_file_header(image),time_date_stamp)
#define pe_ifh_set_pointer_to_symbol_table(image,pointer_to_symbol_table) ifh_set_pointer_to_symbol_table(pe_image_file_header(image),pointer_to_symbol_table)
#define pe_ifh_set_number_of_symbols(image,number_of_symbols) ifh_set_number_of_symbols(pe_image_file_header(image),number_of_symbols)
#define pe_ifh_set_size_of_optional_header(image,size_of_optional_header) ifh_set_size_of_optional_header(pe_image_file_header(image),size_of_optional_header)
#define pe_ifh_set_characteristics(image,characteristics) ifh_set_characteristics(pe_image_file_header(image),characteristics)


#define pe_image_optional_header(image) (ImageOptionalHeader *)((guint8 *)(image)+idh_get_lfanew(image)+4+IFH_SIZE)
#define pe_ioh_get_magic(image) ioh_get_magic(pe_image_optional_header(image))
#define pe_ioh_get_major_linker_version(image) ioh_get_major_linker_version(pe_image_optional_header(image))
#define pe_ioh_get_minor_linker_version(image) ioh_get_minor_linker_version(pe_image_optional_header(image))
#define pe_ioh_get_size_of_code(image) ioh_get_size_of_code(pe_image_optional_header(image))
#define pe_ioh_get_size_of_initialized_data(image) ioh_get_size_of_initialized_data(pe_image_optional_header(image))
#define pe_ioh_get_size_of_uninitialized_data(image) ioh_get_size_of_uninitialized_data(pe_image_optional_header(image))
#define pe_ioh_get_address_of_entry_point(image) ioh_get_address_of_entry_point(pe_image_optional_header(image))
#define pe_ioh_get_base_of_code(image) ioh_get_base_of_code(pe_image_optional_header(image))
#define pe_ioh_get_base_of_data(image) ioh_get_base_of_data(pe_image_optional_header(image))
#define pe_ioh_get_image_base(image) ioh_get_image_base(pe_image_optional_header(image))
#define pe_ioh_get_section_alignment(image) ioh_get_section_alignment(pe_image_optional_header(image))
#define pe_ioh_get_file_alignment(image) ioh_get_file_alignment(pe_image_optional_header(image))
#define pe_ioh_get_major_operating_system_version(image) ioh_get_major_operating_system_version(pe_image_optional_header(image))
#define pe_ioh_get_minor_operating_system_version(image) ioh_get_minor_operating_system_version(pe_image_optional_header(image))
#define pe_ioh_get_major_image_version(image) ioh_get_major_image_version(pe_image_optional_header(image))
#define pe_ioh_get_minor_image_version(image) ioh_get_minor_image_version(pe_image_optional_header(image))
#define pe_ioh_get_major_subsystem_version(image) ioh_get_major_subsystem_version(pe_image_optional_header(image))
#define pe_ioh_get_minor_subsystem_version(image) ioh_get_minor_subsystem_version(pe_image_optional_header(image))
#define pe_ioh_get_win32_version_value(image) ioh_get_win32_version_value(pe_image_optional_header(image))
#define pe_ioh_get_size_of_image(image) ioh_get_size_of_image(pe_image_optional_header(image))
#define pe_ioh_get_size_of_headers(image) ioh_get_size_of_headers(pe_image_optional_header(image))
#define pe_ioh_get_check_sum(image) ioh_get_check_sum(pe_image_optional_header(image))
#define pe_ioh_get_subsystem(image) ioh_get_subsystem(pe_image_optional_header(image))
#define pe_ioh_get_dll_characteristics(image) ioh_get_dll_characteristics(pe_image_optional_header(image))
#define pe_ioh_get_size_of_stack_reserve(image) ioh_get_size_of_stack_reserve(pe_image_optional_header(image))
#define pe_ioh_get_size_of_stack_commit(image) ioh_get_size_of_stack_commit(pe_image_optional_header(image))
#define pe_ioh_get_size_of_heap_reserve(image) ioh_get_size_of_heap_reserve(pe_image_optional_header(image))
#define pe_ioh_get_size_of_heap_commit(image) ioh_get_size_of_heap_commit(pe_image_optional_header(image))
#define pe_ioh_get_loader_flags(image) ioh_get_loader_flags(pe_image_optional_header(image))
#define pe_ioh_get_number_of_rva_and_sizes(image) ioh_get_number_of_rva_and_sizes(pe_image_optional_header(image))
#define pe_ioh_get_data_directory_virtual_address(image,entry) ioh_get_data_directory_virtual_address(pe_image_optional_header(image),entry)
#define pe_ioh_get_data_directory_size(image,entry) ioh_get_data_directory_size(pe_image_optional_header(image),entry)
#define pe_ioh_set_magic(image,magic) ioh_set_magic(pe_image_optional_header(image),magic)
#define pe_ioh_set_major_linker_version(image,major_linker_version) ioh_set_major_linker_version(pe_image_optional_header(image),major_linker_version)
#define pe_ioh_set_minor_linker_version(image,minor_linker_version) ioh_set_minor_linker_version(pe_image_optional_header(image),minor_linker_version)
#define pe_ioh_set_size_of_code(image,size_of_code) ioh_set_size_of_code(pe_image_optional_header(image),size_of_code)
#define pe_ioh_set_size_of_initialized_data(image,size_of_initialized_data) ioh_set_size_of_initialized_data(pe_image_optional_header(image),size_of_initialized_data)
#define pe_ioh_set_size_of_uninitialized_data(image,size_of_uninitialized_data) ioh_set_size_of_uninitialized_data(pe_image_optional_header(image),size_of_uninitialized_data)
#define pe_ioh_set_address_of_entry_point(image,address_of_entry_point) ioh_set_address_of_entry_point(pe_image_optional_header(image),address_of_entry_point)
#define pe_ioh_set_base_of_code(image,base_of_code) ioh_set_base_of_code(pe_image_optional_header(image),base_of_code)
#define pe_ioh_set_base_of_data(image,base_of_data) ioh_set_base_of_data(pe_image_optional_header(image),base_of_data)
#define pe_ioh_set_image_base(image,image_base) ioh_set_image_base(pe_image_optional_header(image),image_base)
#define pe_ioh_set_section_alignment(image,section_alignment) ioh_set_section_alignment(pe_image_optional_header(image),section_alignment)
#define pe_ioh_set_file_alignment(image,file_alignment) ioh_set_file_alignment(pe_image_optional_header(image),file_alignment)
#define pe_ioh_set_major_operating_system_version(image,major_operating_system_version) ioh_set_major_operating_system_version(pe_image_optional_header(image),major_operating_system_version)
#define pe_ioh_set_minor_operating_system_version(image,minor_operating_system_version) ioh_set_minor_operating_system_version(pe_image_optional_header(image),minor_operating_system_version)
#define pe_ioh_set_major_image_version(image,major_image_version) ioh_set_major_image_version(pe_image_optional_header(image),major_image_version)
#define pe_ioh_set_minor_image_version(image,minor_image_version) ioh_set_minor_image_version(pe_image_optional_header(image),minor_image_version)
#define pe_ioh_set_major_subsystem_version(image,major_subsystem_version) ioh_set_major_subsystem_version(pe_image_optional_header(image),major_subsystem_version)
#define pe_ioh_set_minor_subsystem_version(image,minor_subsystem_version) ioh_set_minor_subsystem_version(pe_image_optional_header(image),minor_subsystem_version)
#define pe_ioh_set_win32_version_value(image,win32_version_value) ioh_set_win32_version_value(pe_image_optional_header(image),win32_version_value)
#define pe_ioh_set_size_of_image(image,size_of_image) ioh_set_size_of_image(pe_image_optional_header(image),size_of_image)
#define pe_ioh_set_size_of_headers(image,size_of_headers) ioh_set_size_of_headers(pe_image_optional_header(image),size_of_headers)
#define pe_ioh_set_check_sum(image,check_sum) ioh_set_check_sum(pe_image_optional_header(image),check_sum)
#define pe_ioh_set_subsystem(image,subsystem) ioh_set_subsystem(pe_image_optional_header(image),subsystem)
#define pe_ioh_set_dll_characteristics(image,dll_characteristics) ioh_set_dll_characteristics(pe_image_optional_header(image),dll_characteristics)
#define pe_ioh_set_size_of_stack_reserve(image,size_of_stack_reserve) ioh_set_size_of_stack_reserve(pe_image_optional_header(image),size_of_stack_reserve)
#define pe_ioh_set_size_of_stack_commit(image,size_of_stack_commit) ioh_set_size_of_stack_commit(pe_image_optional_header(image),size_of_stack_commit)
#define pe_ioh_set_size_of_heap_reserve(image,size_of_heap_reserve) ioh_set_size_of_heap_reserve(pe_image_optional_header(image),size_of_heap_reserve)
#define pe_ioh_set_size_of_heap_commit(image,size_of_heap_commit) ioh_set_size_of_heap_commit(pe_image_optional_header(image),size_of_heap_commit)
#define pe_ioh_set_loader_flags(image,loader_flags) ioh_set_loader_flags(pe_image_optional_header(image),loader_flags)
#define pe_ioh_set_number_of_rva_and_sizes(image,number_of_rva_and_sizes) ioh_set_number_of_rva_and_sizes(pe_image_optional_header(image),number_of_rva_and_sizes)
#define pe_ioh_set_data_directory_virtual_address(image,entry,virtual_address) ioh_set_data_directory_virtual_address(pe_image_optional_header(image),entry,virtual_address)
#define pe_ioh_set_data_directory_size(image,entry,virtual_address) ioh_set_data_directory_size(pe_image_optional_header(image),entry,size)


#define pe_image_section_header_nth(image,n) (ImageSectionHeader *)((guint8 *)(image)+idh_get_lfanew(image)+4+IFH_SIZE+IOH_SIZE+(n)*ISH_SIZE)
#define pe_image_section_header(image) pe_image_section_header_nth(image,0)
#define pe_ish_get_name(image,n) ish_get_name(pe_image_section_header_nth(image,n))
#define pe_ish_get_physical_address(image,n) ish_get_physical_address(pe_image_section_header_nth(image,n))
#define pe_ish_get_virtual_size(image,n) ish_get_virtual_size(pe_image_section_header_nth(image,n))
#define pe_ish_get_virtual_address(image,n) ish_get_virtual_address(pe_image_section_header_nth(image,n))
#define pe_ish_get_size_of_raw_data(image,n) ish_get_size_of_raw_data(pe_image_section_header_nth(image,n))
#define pe_ish_get_pointer_to_raw_data(image,n) ish_get_pointer_to_raw_data(pe_image_section_header_nth(image,n))
#define pe_ish_get_pointer_to_relocations(image,n) ish_get_pointer_to_relocations(pe_image_section_header_nth(image,n))
#define pe_ish_get_pointer_to_linenumbers(image,n) ish_get_pointer_to_linenumbers(pe_image_section_header_nth(image,n))
#define pe_ish_get_number_of_relocations(image,n) ish_get_number_of_relocations(pe_image_section_header_nth(image,n))
#define pe_ish_get_number_of_linenumbers(image,n) ish_get_number_of_linenumbers(pe_image_section_header_nth(image,n))
#define pe_ish_get_characteristics(image,n) ish_get_characteristics(pe_image_section_header_nth(image,n))
#define pe_ish_set_name(image,n,name) ish_get_name(pe_image_section_header_nth(image,n),name)
#define pe_ish_set_physical_address(image,n,physical_address) ish_set_physical_address(pe_image_section_header_nth(image,n),physical_address)
#define pe_ish_set_virtual_size(image,n,virtual_size) ish_set_virtual_size(pe_image_section_header_nth(image,n),virtual_size)
#define pe_ish_set_virtual_address(image,n,virtual_address) ish_set_virtual_address(pe_image_section_header_nth(image,n),virtual_address)
#define pe_ish_set_size_of_raw_data(image,n,size_of_raw_data) ish_set_size_of_raw_data(pe_image_section_header_nth(image,n),size_of_raw_data)
#define pe_ish_set_pointer_to_raw_data(image,n,pointer_to_raw_data) ish_set_pointer_to_raw_data(pe_image_section_header_nth(image,n),pointer_to_raw_data)
#define pe_ish_set_pointer_to_relocations(image,n,pointer_to_relocations) ish_set_pointer_to_relocations(pe_image_section_header_nth(image,n),pointer_to_relocations)
#define pe_ish_set_pointer_to_linenumbers(image,n,pointer_to_linenumbers) ish_set_pointer_to_linenumbers(pe_image_section_header_nth(image,n),pointer_to_linenumbers)
#define pe_ish_set_number_of_relocations(image,n,number_of_relocations) ish_set_number_of_relocations(pe_image_section_header_nth(image,n),number_of_relocations)
#define pe_ish_set_number_of_linenumbers(image,n,number_of_linenumbers) ish_set_number_of_linenumbers(pe_image_section_header_nth(image,n),number_of_linenumbers)
#define pe_ish_set_characteristics(image,n,characteristics) ish_set_characteristics(pe_image_section_header_nth(image,n),characteristics)


#define pe_image_export_directory(image) (ImageExportDirectory *)((guint8 *)(image)+pe_ioh_get_data_directory_virtual_address(image,PEIMAGE_DIRECTORY_ENTRY_EXPORT))
#define pe_ied_get_characteristics(image) ied_get_characteristics(pe_image_export_directory(image))
#define pe_ied_get_time_date_stamp(image) ied_get_time_date_stamp(pe_image_export_directory(image))
#define pe_ied_get_major_version(image) ied_get_major_version(pe_image_export_directory(image))
#define pe_ied_get_minor_version(image) ied_get_minor_version(pe_image_export_directory(image))
#define pe_ied_get_name(image) ied_get_name(pe_image_export_directory(image))
#define pe_ied_get_base(image) ied_get_base(pe_image_export_directory(image))
#define pe_ied_get_number_of_functions(image) ied_get_number_of_functions(pe_image_export_directory(image))
#define pe_ied_get_number_of_names(image) ied_get_number_of_names(pe_image_export_directory(image))
#define pe_ied_get_address_of_functions(image) ied_get_address_of_functions(pe_image_export_directory(image))
#define pe_ied_get_address_of_names(image) ied_get_address_of_names(pe_image_export_directory(image))
#define pe_ied_get_address_of_name_ordinals(image) ied_get_address_of_name_ordinals(pe_image_export_directory(image))
#define pe_ied_set_characteristics(image,characteristics) ied_set_characteristics(pe_image_export_directory(image),characteristics)
#define pe_ied_set_time_date_stamp(image,time_date_stamp) ied_set_time_date_stamp(pe_image_export_directory(image),time_date_stamp)
#define pe_ied_set_major_version(image,major_version) ied_set_major_version(pe_image_export_directory(image),major_version)
#define pe_ied_set_minor_version(image,minor_version) ied_set_minor_version(pe_image_export_directory(image),minor_version)
#define pe_ied_set_name(image,name) ied_set_name(pe_image_export_directory(image),name)
#define pe_ied_set_base(image,base) ied_set_base(pe_image_export_directory(image),base)
#define pe_ied_set_number_of_functions(image,number_of_functions) ied_set_number_of_functions(pe_image_export_directory(image),number_of_functions)
#define pe_ied_set_number_of_names(image,number_of_names) ied_set_number_of_names(pe_image_export_directory(image),number_of_names)
#define pe_ied_set_address_of_functions(image,address_of_functions) ied_set_address_of_functions(pe_image_export_directory(image),address_of_functions)
#define pe_ied_set_address_of_names(image,address_of_names) ied_set_address_of_names(pe_image_export_directory(image),address_of_names)
#define pe_ied_set_address_of_name_ordinals(image,address_of_name_ordinals) ied_set_address_of_name_ordinals(pe_image_export_directory(image),address_of_name_ordinals)


#define pe_image_import_descriptor_nth(image,n) (ImageImportDescriptor *)((guint8 *)(image)+pe_ioh_get_data_directory_virtual_address(image,PEIMAGE_DIRECTORY_ENTRY_IMPORT)+(n)*IID_SIZE)
#define pe_image_import_descriptor(image) pe_image_import_descriptor_nth(image,0)
#define pe_iid_get_characteristics(image,n) iid_get_characteristics(pe_image_import_descriptor_nth(image,n))
#define pe_iid_get_original_first_thunk(image,n) iid_get_original_first_thunk(pe_image_import_descriptor_nth(image,n))
#define pe_iid_get_time_date_stamp(image,n) iid_get_time_date_stamp(pe_image_import_descriptor_nth(image,n))
#define pe_iid_get_forwarder_chain(image,n) iid_get_forwarder_chain(pe_image_import_descriptor_nth(image,n))
#define pe_iid_get_name(image,n) iid_get_name(pe_image_import_descriptor_nth(image,n))
#define pe_iid_get_first_thunk(image,n) iid_get_first_thunk(pe_image_import_descriptor_nth(image,n))
#define pe_iid_set_characteristics(image,n,characteristics) iid_set_characteristics(pe_image_import_descriptor_nth(image,n),characteristics)
#define pe_iid_set_original_first_thunk(image,n,original_first_thunk) iid_set_original_first_thunk(pe_image_import_descriptor_nth(image,n),original_first_thunk)
#define pe_iid_set_time_date_stamp(image,n,time_date_stamp) iid_set_time_date_stamp(pe_image_import_descriptor_nth(image,n),time_date_stamp)
#define pe_iid_set_forwarder_chain(image,n,forwarder_chain) iid_set_forwarder_chain(pe_image_import_descriptor_nth(image,n),forwarder_chain)
#define pe_iid_set_name(image,n,name) iid_set_name(pe_image_import_descriptor_nth(image,n),name)
#define pe_iid_set_first_thunk(image,n,first_thunk) iid_set_first_thunk(pe_image_import_descriptor_nth(image,n),first_thunk)


#define pe_image_base_relocation(image) (ImageBaseRelocation *)((guint8 *)(image)+pe_ioh_get_data_directory_virtual_address(image,PEIMAGE_DIRECTORY_ENTRY_BASERELOC))


#define pe_image_resource_directory(image) (ImageResourceDirectory *)((guint8 *)(image)+pe_ioh_get_data_directory_virtual_address(image,PEIMAGE_DIRECTORY_ENTRY_RESOURCE))


/******************************************************************************
*                                                                             *
* ja:PEイメージ関数                                                           *
*                                                                             *
******************************************************************************/
/*  ja:PEイメージのヘッダサイズを取得する
    image,PEイメージ(ヘッダ)
      RET,バイト数                                                          */
#define peimage_file_header_bytes(image)                                    \
        (pe_idh_get_lfanew(image)+sizeof(guint32)                           \
        +sizeof(ImageFileHeader)+sizeof(ImageOptionalHeader)                \
        +sizeof(ImageSectionHeader)*pe_ifh_get_number_of_sections(image))


/*  ja:PEイメージを検証する
     image,PEイメージ(ヘッダ)
    length,バイト数
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
peimage_file_is_valid (const guint8 *image,
                       const gssize  length);


/*  ja:PEイメージを読み込む
    file,ファイル名
     RET,PEイメージ,NULL:エラー                                             */
guint8 *
peimage_file_load (const gchar *file);


/*  ja:PEイメージを再配置する
    image,PEイメージ
    delta,差                                                                */
void
peimage_file_relocate (guint8     *image,
                       const gint  delta);


/*  ja:PEイメージからAPIを取得する
    image,PEイメージ
     name,API名
      RET,アドレス,NULL:エラー                                              */
gconstpointer
peimage_file_get_func (const guint8 *image,
                       const gchar  *name);


/*  ja:PEイメージからAPIを取得する
      image,PEイメージ
    ordinal,オーディナル値
        RET,アドレス,NULL:エラー                                            */
gconstpointer
peimage_file_get_proc (const guint8  *image,
                       const guint16  ordinal);


/*  ja:PEイメージからリソースを取得する
    image,PEイメージ
      key,リソース
      RET,データ,NULL:エラー                                                */
gconstpointer
peimage_file_get_resource (const guint8 *image,
                           const gchar  *key);


/******************************************************************************
*                                                                             *
* ja:モジュール関数群                                                         *
*                                                                             *
******************************************************************************/
typedef struct _PeProcess
{
  GList *module;
} PeProcess;
typedef guint8 *(*PeModuleSystem)(PeProcess *, const gchar *, gpointer);
typedef gint (*PeModuleRelocate)(PeProcess *, const guint8 *, gpointer);
typedef guint32 (*PeModuleImport)(PeProcess *, gconstpointer, gpointer);
typedef gboolean (*PeModuleEntry)(PeProcess *, const guint8 *, const gboolean, gpointer);


/*  ja:既にロードされているPEイメージを取得する
    process,プロセス
       file,イメージ名
        RET,PEイメージ,NULL:ロードされていない                              */
guint8 *
peimage_module_get_library (PeProcess   *process,
                            const gchar *file);


/*  ja:PEイメージをロードする
              process,プロセス
                 file,イメージ名
      pemodule_system,システムイメージ用コールバック
    pemodule_relocale,再配置用コールバック
      pemodule_import,インポート用コールバック
       pemodule_entry,エントリーポイント用コールバック
            user_data,ユーザデータ
                  RET,PEイメージ,NULL:エラー                                */
guint8 *
peimage_module_load_library (PeProcess        *process,
                             const gchar      *file,
                             PeModuleSystem    pemodule_system,
                             PeModuleRelocate  pemodule_relocale,
                             PeModuleImport    pemodule_import,
                             PeModuleEntry     pemodule_entry,
                             gpointer          user_data);


/*  ja:PEイメージを解放する
           process,プロセス
             image,PEイメージ
    pemodule_entry,エントリーポイント用コールバック
         user_data,ユーザデータ
               RET,TRUE:正常終了,FALSE:エラー                               */
gboolean
peimage_module_free_library (PeProcess     *process,
                             const guint8  *image,
                             PeModuleEntry  pemodule_entry,
                             gpointer       user_data);


/*  ja:イメージの名前を取得する
    process,プロセス
      image,イメージ
        RET,名前,NULL:エラー                                                */
const gchar *
peimage_module_get_filename (PeProcess    *process,
                             const guint8 *image);


/******************************************************************************
*                                                                             *
* ja:ディレクトリ関数群                                                       *
*                                                                             *
******************************************************************************/
#define PEIMAGE_DIR_WINDOWS 0
#define PEIMAGE_DIR_SYSTEM  1
#define PEIMAGE_DIR_TEMP    2
#define PEIMAGE_DIR_DATA    3
#define PEIMAGE_DIR_HOME    4


/*  ja:ディレクトリを取得する
    dir,ディレクトリの種類
    RET,ディレクトリ                                                        */
const gchar *
peimage_dir_get_path (const guint dir);


/*  ja:大文字小文字を区別しないパス名を求める
    path,ファイル名
     RET,大文字小文字を区別しないパス名                                     */
gchar *
peimage_dir_get_case_insensitive (const gchar *path);


/*  ja:特定のディレクトリのファイルを取得する
    file,元のファイル名
     dir,ディレクトリの種類
     RET,ファイル                                                           */
gchar *
peimage_dir_get_filename (const gchar *file,
                          const guint  dir);



G_END_DECLS


#endif /* __PEIMAGE_H__ */
