/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "commctrl.h"
#include "misc/misc.h"


typedef struct _W32LdrUpdownData
{
  W32LdrWindowData wd;
  gint base;  /* ja:基数(デフォルト10) */
  gint min;   /* ja:最小値(デフォルト0) */
  gint max;   /* ja:最大値(デフォルト100) */
  gint value; /* ja:現在値(デフォルト0) */
  gint x;
  gint y;
  gint width;
  gint height;
  gulong handler;
  gint nAccels;
  LPUDACCEL aAccels;
  GtkWidget *button_up;
  GtkWidget *button_down;
  GtkWidget *entry;
} W32LdrUpdownData;


static void
w32updwn_destroy (GtkWidget        *entry,
                  W32LdrUpdownData *ud)
{
  if (ud->handler)
    g_signal_handler_disconnect (G_OBJECT (entry), ud->handler);
  ud->handler = 0;
  ud->entry = NULL;
}


static LRESULT WINAPI
w32updwn_DefWindowProc (HWND   hWnd,
                        UINT   Msg,
                        WPARAM wParam,
                        LPARAM lParam)
{
  W32LdrWindowData *wd;
  W32LdrUpdownData *ud;

  ud = g_object_get_data (G_OBJECT (hWnd), "user_data");
  if (!ud)
    return 0;
  wd = &ud->wd;
  switch (Msg)
    {
      case UDM_SETRANGE:
        return w32updwn_DefWindowProc (hWnd, UDM_SETRANGE32,
                                            LOWORD (lParam), HIWORD (lParam));
      case UDM_GETRANGE:
        return MAKELRESULT (ud->min, ud->max);
      case UDM_SETPOS:
        return w32updwn_DefWindowProc (hWnd, UDM_SETPOS32, 0, LOWORD (lParam));
      case UDM_GETPOS:
        return w32updwn_DefWindowProc (hWnd, UDM_GETPOS32,
                                                    0, GPOINTER_TO_INT (NULL));
      case UDM_SETBUDDY:
        if (GTK_IS_ENTRY (wParam))
          {
            GtkWidget *entry;

            entry = ud->entry;
            ud->entry = GINT_TO_POINTER (wParam);
            if (ud->entry != entry)
              {
                if (entry)
                  {
                    if (ud->handler != 0)
                      g_signal_handler_disconnect (G_OBJECT (entry),
                                                                ud->handler);
                    if (wd->dwStyle & (UDS_ALIGNLEFT | UDS_ALIGNRIGHT))
                      MoveWindow (entry, ud->x, ud->y, ud->width, ud->height,
                                                                        TRUE);
                  }
                if (ud->entry)
                  {
                    RECT rcEdit, rcUpdown;

                    if (wd->hWndParent
                            && (wd->dwStyle & (UDS_ALIGNLEFT | UDS_ALIGNRIGHT))
                            && GetWindowRect (ud->entry, &rcEdit)
                            && GetWindowRect (hWnd, &rcUpdown))
                      {
                        gint width;
                        POINT pt;

                        /* ja:以前のエントリーの位置とサイズを保存 */
                        pt.x = rcEdit.left;
                        pt.y = rcEdit.top;
                        ScreenToClient (wd->hWndParent, &pt);
                        ud->x = pt.x;
                        ud->y = pt.y;
                        ud->width = rcEdit.right - rcEdit.left + 1;
                        ud->height = rcEdit.bottom - rcEdit.top + 1;
                        /* ja:アップダウンを移動 */
                        width = rcUpdown.right - rcUpdown.left + 1;
                        MoveWindow (hWnd, ud->x + (wd->dwStyle & UDS_ALIGNRIGHT
                                            ? ud->width - width : 0), ud->y,
                                                    width, ud->height, TRUE);
                        /* ja:エントリーを移動 */
                        MoveWindow (ud->entry,
                            ud->x + (wd->dwStyle & UDS_ALIGNLEFT ? width : 0),
                                ud->y, ud->width - width, ud->height, TRUE);
                      }
                    ud->handler = g_signal_connect (G_OBJECT (ud->entry),
                                "destroy", G_CALLBACK (w32updwn_destroy), ud);
                  }
                else
                  {
                    ud->handler = 0;
                  }
              }
            return GPOINTER_TO_INT (entry);
          }
        return GPOINTER_TO_INT (ud->entry);
      case UDM_GETBUDDY:
        return GPOINTER_TO_INT (ud->entry);
      case UDM_SETACCEL:
        g_free (ud->aAccels);
        if (wParam > 0 && lParam)
          {
            ud->nAccels = wParam;
            ud->aAccels = g_memdup (GINT_TO_POINTER (lParam),
                                            ud->nAccels * sizeof (UDACCEL));
          }
        else
          {
            ud->nAccels = 0;
            ud->aAccels = NULL;
          }
        return TRUE;
      case UDM_GETACCEL:
        if (wParam > 0 && lParam)
          {
            gint accels;

            accels = MIN (wParam, ud->nAccels);
            g_memmove (GINT_TO_POINTER (lParam), ud->aAccels,
                                                    accels * sizeof (UDACCEL));
            return accels;
          }
        return ud->nAccels;
      case UDM_SETBASE:
        if (wParam == 10 || wParam == 16)
          {
            gint base;

            base = ud->base;
            ud->base = wParam;
            if (ud->base != base && ud->entry)
              {
                gchar *str;

                str = misc_str_from_val (ud->value, ud->base, 0,
                                                            ud->base == 10);
                gtk_entry_set_text (GTK_ENTRY (ud->entry), str);
                g_free (str);
              }
            return base;
          }
        return 0;
      case UDM_GETBASE:
        return ud->base;
      case UDM_SETRANGE32:
        ud->min = MIN (wParam, lParam);
        ud->max = MAX (wParam, lParam);
        w32updwn_DefWindowProc (hWnd, UDM_SETPOS32,
                                    0, CLAMP (ud->value, ud->min, ud->max));
        return 0;
      case UDM_GETRANGE32:
        {
          LPINT pLow, pHigh;

          pLow = GINT_TO_POINTER (wParam);
          pHigh = GINT_TO_POINTER (lParam);
          if (pLow)
            *pLow = ud->min;
          if (pHigh)
            *pHigh = ud->max;
        }
        return 0;
      case UDM_SETUNICODEFORMAT:
        {
          gboolean widechar;

          widechar = wd->widechar;
          wd->widechar = wParam;
          return widechar;
        }
      case UDM_GETUNICODEFORMAT:
        return wd->widechar;
      case UDM_SETPOS32:
        if (ud->value != lParam)
          {
            gint value;

            value = ud->value;
            ud->value = lParam;
            if (!(wd->dwStyle & UDS_WRAP))
              {
                gtk_widget_set_sensitive (ud->button_up,
                                                        ud->value <= ud->max);
                gtk_widget_set_sensitive (ud->button_down,
                                                        ud->value >= ud->min);
              }
            if (ud->entry)
              {
                gchar *str;

                str = misc_str_from_val (ud->value, ud->base, 0,
                                                            ud->base == 10);
                gtk_entry_set_text (GTK_ENTRY (ud->entry), str);
                g_free (str);
              }
            return value;
          }
        return ud->value;
      case UDM_GETPOS32:
        {
          LPBOOL pfError;

          pfError = GINT_TO_POINTER (lParam);
          if (ud->entry && (wd->dwStyle & UDS_SETBUDDYINT))
            {
              const gchar *str;
              gint value;

              str = gtk_entry_get_text (GTK_ENTRY (ud->entry));
              if (misc_str_to_val (&value, str, ud->base, ud->base == 10))
                {
                  ud->value = CLAMP (value, ud->min, ud->max);
                  if (pfError)
                    *pfError = FALSE;
                }
              else if (pfError)
                {
                  *pfError = TRUE;
                }
            }
          else if (pfError)
            {
              *pfError = FALSE;
            }
          return ud->value;
        }
      case WM_CREATE:
        if ((wd->dwStyle & UDS_AUTOBUDDY) && wd->hWndParent)
          {
            RECT rcUpdown;

            if (GetWindowRect (hWnd, &rcUpdown))
              {
                gint i;
                RECT rcUnion, rcEdit;
                W32LdrWindowData *wd_p;

                wd_p = g_object_get_data (G_OBJECT (wd->hWndParent),
                                                                "user_data");
                if (wd_p->child)
                  for (i = 0; wd_p->child[i]; i++)
                    if (GTK_IS_ENTRY (wd_p->child[i])
                            && GetWindowRect (wd_p->child[i], &rcEdit)
                            && IntersectRect (&rcUnion, &rcUpdown, &rcEdit))
                      {
                        w32updwn_DefWindowProc (hWnd, UDM_SETBUDDY,
                                        GPOINTER_TO_INT (wd_p->child[i]), 0);
                        break;
                      }
              }
          }
        break;
      case WM_DESTROY:
        if (ud->entry && ud->handler)
          g_signal_handler_disconnect (G_OBJECT (ud->entry), ud->handler);
        g_free (ud->aAccels);
        ud->handler = 0;
        ud->nAccels = 0;
        ud->aAccels = NULL;
        ud->entry = NULL;
    }
  return wd->widechar ? DefWindowProcW (hWnd, Msg, wParam, lParam)
                       : DefWindowProcA (hWnd, Msg, wParam, lParam);
}


/* ja:ボタンが押されたとき */
static void
w32updwn_clicked (GtkWidget *button,
                  gpointer   user_data)
{
  gint step, value;
  GtkWidget *widget;
  W32LdrWindowData *wd;
  W32LdrUpdownData *ud;

  widget = gtk_widget_get_parent (button);
  ud = g_object_get_data (G_OBJECT (widget), "user_data");
  wd = &ud->wd;
  step = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "user_data"));
  value = ud->value + step;
  if (wd->dwStyle & UDS_WRAP)
    {
      if (value < ud->min)
        value = ud->max;
      else if (ud->max < value)
        value = ud->min;
    }
  else
    {
      value = CLAMP (value, ud->min, ud->max);
    }
  if (ud->value != value)
    {
      NM_UPDOWN nmu;

      nmu.hdr.hwndFrom = widget;
      nmu.hdr.idFrom = wd->wID;
      nmu.hdr.code = UDN_DELTAPOS;
      nmu.iPos = ud->value;
      nmu.iDelta = step;
      if (!(wd->widechar ? SendMessageW (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nmu))
                             : SendMessageA (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nmu))))
        w32updwn_DefWindowProc (widget, UDM_SETPOS32, 0, value);
    }
}


GtkWidget *
w32ldr_control_create_updwn (const gchar    *windowname,
                             const guint32   style,
                             const guint32   exstyle,
                             const gint      width,
                             const gint      height,
                             const guint16   id,
                             HWND            hWndParent,
                             HINSTANCE       hInstance,
                             const gboolean  widechar)
{
  GtkWidget *vbox, *button_up, *button_down;
  W32LdrWindowData *wd;
  W32LdrUpdownData *ud;
#if GTK_CHECK_VERSION(2,6,0)
  GdkPixbuf *pixbuf;
  GtkWidget *image;
  const static gchar *xpm_up[] = {
"5 3 2 1",
"   c None",
".  c #000000",
"  .  ",
" ... ",
"....."};
  const static gchar *xpm_down[] = {
"5 3 2 1",
"   c None",
".  c #000000",
".....",
" ... ",
"  .  "};
#endif /* not GTK_CHECK_VERSION(2,6,0) */

  button_up = gtk_button_new ();
  button_down = gtk_button_new ();
#if GTK_CHECK_VERSION(2,6,0)
  pixbuf = gdk_pixbuf_new_from_xpm_data (xpm_up);
  image = gtk_image_new_from_pixbuf (pixbuf);
  g_object_unref (G_OBJECT (pixbuf));
  gtk_widget_show (image);
  gtk_button_set_image (GTK_BUTTON (button_up), image);
  pixbuf = gdk_pixbuf_new_from_xpm_data (xpm_down);
  image = gtk_image_new_from_pixbuf (pixbuf);
  g_object_unref (G_OBJECT (pixbuf));
  gtk_widget_show (image);
  gtk_button_set_image (GTK_BUTTON (button_down), image);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
  g_signal_connect (G_OBJECT (button_up), "clicked",
                                        G_CALLBACK (w32updwn_clicked), NULL);
  g_signal_connect (G_OBJECT (button_down), "clicked",
                                        G_CALLBACK (w32updwn_clicked), NULL);
  g_object_set_data (G_OBJECT (button_up), "user_data", GINT_TO_POINTER (1));
  g_object_set_data (G_OBJECT (button_down), "user_data",
                                                        GINT_TO_POINTER (-1));
  gtk_widget_show (button_up);
  gtk_widget_show (button_down);
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), button_up, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), button_down, TRUE, TRUE, 0);
  ud = g_malloc (sizeof (W32LdrUpdownData));
  wd = &ud->wd;
  wd->dwStyle = style;
  wd->dwExStyle = exstyle;
  wd->hInstance = hInstance;
  wd->lpfnWndProc = w32updwn_DefWindowProc;
  wd->hWndParent = hWndParent;
  wd->wID = id;
  wd->dwUserData = 0;
  wd->lpDialogFunc = NULL;
  wd->lResult = -1;
  wd->dwUser = 0;
  wd->csa = NULL;
  wd->csw = NULL;
  wd->dwInitParam = 0;
  wd->widechar = widechar;
  wd->resizable = TRUE;
  wd->classname = g_strdup ("msctls_updown32");
  wd->focus = NULL;
  wd->child = NULL;
  ud->base = 10;
  ud->min = G_MININT;
  ud->max = G_MAXINT;
  ud->value = 0;
  ud->x = 0;
  ud->y = 0;
  ud->width = 0;
  ud->height = 0;
  ud->handler = 0;
  ud->nAccels = 0;
  ud->aAccels = NULL;
  ud->button_up = button_up;
  ud->button_down = button_down;
  ud->entry = NULL;
  g_object_set_data (G_OBJECT (vbox), "user_data", wd);
  return vbox;
}
