 /*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "kernel32.h"
#include <gdk/gdkkeysyms.h>


GHashTable *ghash_class = NULL;


/******************************************************************************
*                                                                             *
* ja:クラス関数群                                                             *
*                                                                             *
******************************************************************************/
ATOM
w32ldr_RegisterClassEx (UINT            style,
                        WNDPROC         lpfnWndProc,
                        INT             cbClsExtra,
                        INT             cbWndExtra,
                        HINSTANCE       hInstance,
                        HICON           hIcon,
                        HCURSOR         hCursor,
                        HBRUSH          hbrBackground,
                        const gchar    *menuname,
                        const gchar    *classname,
                        HICON           hIconSm,
                        const gboolean  widechar)
{
  guint16 atom;
  W32LdrClassData *w32class;

  atom = w32ldr_atom_add_name (classname);
  if (!ghash_class)
    ghash_class = g_hash_table_new (g_direct_hash, g_direct_equal);
  else if (g_hash_table_lookup (ghash_class, GUINT_TO_POINTER ((guint)atom)))
    return 0;
  w32class = g_malloc (sizeof (W32LdrClassData));
  w32class->lpfnWndProc = lpfnWndProc;
  w32class->hInstance = hInstance;
  w32class->widechar = widechar;
  w32class->classname = w32ldr_atom_get_name (atom);
  w32class->menuname = GPOINTER_TO_UINT (menuname) > G_MAXUINT16
                                    ? g_strdup (menuname) : (gchar *)menuname;
  w32class->atom = atom;
  g_hash_table_insert (ghash_class, GUINT_TO_POINTER ((guint)atom), w32class);
  return atom;
}


/******************************************************************************
*                                                                             *
* ja:ウインドウ関数群                                                         *
*                                                                             *
******************************************************************************/
static gboolean
w32wnd_button_press_event (GtkWidget      *widget,
                           GdkEventButton *event,
                           gpointer        user_data)
{
  UINT uMsg = WM_NULL;

  switch ((gint)event->type)
    {
      case GDK_BUTTON_PRESS:
        switch (event->button)
          {
            case 1: uMsg = WM_LBUTTONDOWN; break;   /* ja:左クリック */
            case 2: uMsg = WM_MBUTTONDOWN; break;   /* ja:中クリック */
            case 3: uMsg = WM_RBUTTONDOWN;          /* ja:右クリック */
          }
        break;
      case GDK_2BUTTON_PRESS:
        switch (event->button)
          {
            case 1: uMsg = WM_LBUTTONDBLCLK; break; /* ja:左ダブルクリック */
            case 2: uMsg = WM_MBUTTONDBLCLK; break; /* ja:中ダブルクリック */
            case 3: uMsg = WM_RBUTTONDBLCLK;        /* ja:右ダブルクリック */
          }
        break;
    }
  if (uMsg != WM_NULL)
    SendMessage (widget, uMsg, (event->state & GDK_SHIFT_MASK ? MK_SHIFT : 0)
                        | (event->state & GDK_CONTROL_MASK ? MK_CONTROL : 0)
                        | (event->state & GDK_BUTTON1_MASK ? MK_LBUTTON : 0)
                        | (event->state & GDK_BUTTON2_MASK ? MK_MBUTTON : 0)
                        | (event->state & GDK_BUTTON3_MASK ? MK_RBUTTON : 0),
                                MAKELPARAM ((WORD)event->x, (WORD)event->y));
  return FALSE;
}


static gboolean
w32wnd_button_release_event (GtkWidget      *widget,
                             GdkEventButton *event,
                             gpointer        user_data)
{
  UINT uMsg = WM_NULL;

  switch ((gint)event->type)
    {
      case GDK_BUTTON_RELEASE:
        switch (event->button)
          {
            case 1: uMsg = WM_LBUTTONUP; break; /* ja:左リリース */
            case 2: uMsg = WM_MBUTTONUP; break; /* ja:中リリース */
            case 3: uMsg = WM_RBUTTONUP;        /* ja:右リリース */
          }
    }
  if (uMsg != WM_NULL)
    SendMessage (widget, uMsg, (event->state & GDK_SHIFT_MASK ? MK_SHIFT : 0)
                        | (event->state & GDK_CONTROL_MASK ? MK_CONTROL : 0)
                        | (event->state & GDK_BUTTON1_MASK ? MK_LBUTTON : 0)
                        | (event->state & GDK_BUTTON2_MASK ? MK_MBUTTON : 0)
                        | (event->state & GDK_BUTTON3_MASK ? MK_RBUTTON : 0),
                                MAKELPARAM ((WORD)event->x, (WORD)event->y));
  return FALSE;
}


static gboolean
w32wnd_configure_event (GtkWidget         *widget,
                        GdkEventConfigure *event,
                        gpointer           user_data)
{
  RECT rc;

  if (GTK_IS_WINDOW (widget))
    {
      gint x, y, width, height;

      gtk_window_get_position (GTK_WINDOW (widget), &x, &y);
      gtk_window_get_size (GTK_WINDOW (widget), &width, &height);
      rc.left = x;
      rc.top = y;
      rc.right = x + width;
      rc.bottom = y + height;
    }
  else
    {
      rc.left = event->x;
      rc.top = event->y;
      rc.right = event->x + event->width;
      rc.bottom = event->y + event->height;
    }
  SendMessage (widget, WM_MOVING, WMSZ_BOTTOMRIGHT, GPOINTER_TO_INT (&rc));
  SendMessage (widget, WM_MOVE,
                            0, MAKELPARAM ((WORD)event->x, (WORD)event->y));
  SendMessage (widget, WM_SIZING, WMSZ_BOTTOMRIGHT, GPOINTER_TO_INT (&rc));
  SendMessage (widget, WM_SIZE, SIZE_RESTORED,
                        MAKELPARAM ((WORD)event->width, (WORD)event->height));
  return FALSE;
}


static gboolean
w32wnd_delete_event (GtkWidget *widget,
                     GdkEvent  *event,
                     gpointer   user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (!wd->lpDialogFunc || !wd->lpDialogFunc (widget, WM_COMMAND,
                    MAKEWPARAM (IDCANCEL, BN_CLICKED), GPOINTER_TO_INT (NULL))){
    SendMessage (widget, WM_CLOSE, 0, 0);}
  return TRUE;
}


static gboolean
w32wnd_expose_event (GtkWidget      *widget,
                     GdkEventExpose *event,
                     gpointer        user_data)
{
  SendMessage (widget, WM_PAINT, 0, 0);
  return FALSE;
}


static gboolean
w32wnd_focus_in_event (GtkWidget     *widget,
                       GdkEventFocus *event,
                       gpointer       user_data)
{
  SendMessage (widget, WM_SETFOCUS, 0, 0);
  return FALSE;
}


static gboolean
w32wnd_focus_out_event (GtkWidget     *widget,
                        GdkEventFocus *event,
                        gpointer       user_data)
{
  SendMessage (widget, WM_KILLFOCUS, 0, 0);
  return FALSE;
}


static gboolean
w32wnd_key_press_event (GtkWidget   *widget,
                        GdkEventKey *event,
                        gpointer     user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (event->keyval == GDK_Escape
                && wd->lpDialogFunc && wd->lpDialogFunc (widget, WM_COMMAND,
                    MAKEWPARAM (IDCANCEL, BN_CLICKED), GPOINTER_TO_INT (NULL)))
    return TRUE;
  return FALSE;
}


static gboolean
w32wnd_key_release_event (GtkWidget   *widget,
                          GdkEventKey *event,
                          gpointer     user_data)
{
  return FALSE;
}


static gboolean
w32wnd_motion_notify_event (GtkWidget      *widget,
                            GdkEventMotion *event,
                            gpointer        user_data)
{
  SendMessage (widget, WM_MOUSEMOVE,
                        (event->state & GDK_SHIFT_MASK ? MK_SHIFT : 0)
                        | (event->state & GDK_CONTROL_MASK ? MK_CONTROL : 0)
                        | (event->state & GDK_BUTTON1_MASK ? MK_LBUTTON : 0)
                        | (event->state & GDK_BUTTON2_MASK ? MK_MBUTTON : 0)
                        | (event->state & GDK_BUTTON3_MASK ? MK_RBUTTON : 0),
                                MAKELPARAM ((WORD)event->x, (WORD)event->y));
  return FALSE;
}


static gboolean
w32wnd_scroll_event (GtkWidget      *widget,
                     GdkEventScroll *event,
                     gpointer        user_data)
{
  SendMessage (widget, WM_MOUSEWHEEL,
                    MAKEWPARAM ((event->state & GDK_SHIFT_MASK ? MK_SHIFT : 0)
                        | (event->state & GDK_CONTROL_MASK ? MK_CONTROL : 0)
                        | (event->state & GDK_BUTTON1_MASK ? MK_LBUTTON : 0)
                        | (event->state & GDK_BUTTON2_MASK ? MK_MBUTTON : 0)
                        | (event->state & GDK_BUTTON3_MASK ? MK_RBUTTON : 0),
                        (event->direction == GDK_SCROLL_UP
                                        || event->direction == GDK_SCROLL_LEFT
                                                    ? 1 : -1) * WHEEL_DELTA),
                                MAKELPARAM ((WORD)event->x, (WORD)event->y));
  return FALSE;
}


static void
w32wnd_realize (GtkWidget *widget,
                gpointer   user_data)
{
  W32LdrWindowData *wd;

  g_signal_handlers_disconnect_by_func (G_OBJECT (widget),
                                                    w32wnd_realize, user_data);
  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->csa)
    {
      SendMessageA (widget, WM_CREATE, 0, GPOINTER_TO_INT (wd->csa));
      if (GPOINTER_TO_UINT (wd->csa->lpszName) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszName);
      if (GPOINTER_TO_UINT (wd->csa->lpszClass) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszClass);
      g_free (wd->csa);
      wd->csa = NULL;
    }
  if (wd->csw)
    {
      SendMessageW (widget, WM_CREATE, 0, GPOINTER_TO_INT (wd->csw));
      if (GPOINTER_TO_UINT (wd->csw->lpszName) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszName);
      if (GPOINTER_TO_UINT (wd->csw->lpszClass) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszClass);
      g_free (wd->csw);
      wd->csw = NULL;
    }
}


static void
w32wnd_destroy (GtkWidget *widget,
                gpointer   user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    SendMessageW (widget, WM_DESTROY, 0, 0);
  else
    SendMessageA (widget, WM_DESTROY, 0, 0);
  if (wd->csa)
    {
      if (GPOINTER_TO_UINT (wd->csa->lpszName) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszName);
      if (GPOINTER_TO_UINT (wd->csa->lpszClass) > G_MAXUINT16)
        g_free ((gchar *)wd->csa->lpszClass);
      g_free (wd->csa);
    }
  if (wd->csw)
    {
      if (GPOINTER_TO_UINT (wd->csw->lpszName) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszName);
      if (GPOINTER_TO_UINT (wd->csw->lpszClass) > G_MAXUINT16)
        g_free ((gunichar2 *)wd->csw->lpszClass);
      g_free (wd->csw);
    }
  g_free (wd->classname);
  g_free (wd->child);
  g_free (wd);
  g_object_set_data (G_OBJECT (widget), "user_data", NULL);
}


HWND
w32ldr_CreateWindowEx (DWORD           dwExStyle,
                       const gchar    *classname,
                       const gchar    *windowname,
                       DWORD           dwStyle,
                       const gint      x,
                       const gint      y,
                       const gint      width,
                       const gint      height,
                       HWND            hWndParent,
                       HMENU           hMenu,
                       HINSTANCE       hInstance,
                       LPVOID          lpvParam,
                       const gboolean  widechar)
{
  guint16 atom;
  gint i;
  GtkWidget *widget = NULL;
  W32LdrWindowData *wd;
  struct _control
  {
    gchar *name;
    GtkWidget * (*func) (const gchar    *windowname,
                         const guint32   style,
                         const guint32   exstyle,
                         const gint      width,
                         const gint      height,
                         const guint16   id,
                         HWND            hWndParent,
                         HINSTANCE       hInstance,
                         const gboolean  widechar);
  } control[] = {{"BUTTON",             w32ldr_control_create_button},
                 {"EDIT",               w32ldr_control_create_edit},
                 {"STATIC",             w32ldr_control_create_static},
                 {"LISTBOX",            w32ldr_control_create_list},
                 {"SCROLLBAR",          w32ldr_control_create_scroll},
                 {"COMBOBOX",           w32ldr_control_create_combo},
                 {"ToolbarWindow32",    w32ldr_control_create_tool},
                 {"msctls_statusbar32", w32ldr_control_create_stat},
                 {"SysPropertySheet",   w32ldr_control_create_prsht},
                 {"SysListView32",      w32ldr_control_create_lview},
                 {"SysTreeView32",      w32ldr_control_create_tview},
                 {"msctls_progress32",  w32ldr_control_create_prgrs},
                 {"msctls_updown32",    w32ldr_control_create_updwn},
                 {NULL,                 NULL}};

  atom = w32ldr_atom_find_name (classname);
  for (i = 0; control[i].func; i++)
    if (atom > 0 && w32ldr_atom_find_name (control[i].name) == atom)
      {
        widget = control[i].func (windowname,
                                  dwStyle,
                                  dwExStyle,
                                  width,
                                  height,
                                  GPOINTER_TO_UINT (hMenu),
                                  hWndParent,
                                  hInstance,
                                  widechar);
        break;
      }
  if (!control[i].func)
    {
      W32LdrClassData *w32class;
      W32LdrOverlapData *od;

      /* ja:登録クラスを検索 */
      w32class = ghash_class ? g_hash_table_lookup (ghash_class,
                                        GUINT_TO_POINTER ((guint)atom)) : NULL;
      od = g_malloc (sizeof (W32LdrOverlapData));
      wd = &od->wd;
      wd->dwStyle = dwStyle;
      wd->dwExStyle = dwExStyle;
      wd->hInstance = hInstance;
      wd->lpfnWndProc = w32class ? w32class->lpfnWndProc
                                 : widechar ? DefWindowProcW : DefWindowProcA;
      wd->hWndParent = hWndParent;
      wd->wID = GPOINTER_TO_UINT (hMenu);
      wd->dwUserData = 0;
      wd->lpDialogFunc = NULL;
      wd->lResult = -1;
      wd->dwUser = 0;
      wd->csa = NULL;
      wd->csw = NULL;
      wd->dwInitParam = 0;
      wd->resizable = TRUE;
      wd->widechar = widechar;
      wd->classname = w32class ? g_strdup (w32class->classname) : NULL;
      wd->focus = NULL;
      wd->child = NULL;
      od->table = NULL;
      od->fixed = NULL;
      od->menu = NULL;
      if (GPOINTER_TO_UINT (hMenu) > G_MAXUINT16)
        {
          if (GTK_IS_MENU_BAR (hMenu))
            od->menu = hMenu;
          else
            DestroyMenu (hMenu);
        }
      if (!od->menu && w32class && w32class->menuname)
        {
          LPWSTR lpMenuName;

          lpMenuName = GPOINTER_TO_UINT (w32class->menuname) > G_MAXUINT16
                ? g_utf8_to_utf16 (w32class->menuname, -1, NULL, NULL, NULL)
                : (LPWSTR)w32class->menuname;
          od->menu = LoadMenuW (w32class->hInstance, lpMenuName);
          if (GPOINTER_TO_UINT (lpMenuName) > G_MAXUINT16)
            g_free (lpMenuName);
        }
      od->hscroll = NULL;
      od->vscroll = NULL;
      if (dwStyle & WS_CHILD)
        {
          widget = dwStyle & WS_DLGFRAME ? gtk_fixed_new ()
                                         : gtk_drawing_area_new ();
          if (width != CW_USEDEFAULT && height != CW_USEDEFAULT)
            gtk_widget_set_size_request (widget, width, height);
        }
      else
        {
          gchar *title;

          /* ja:フィクスト */
          od->fixed = gtk_fixed_new ();
          gtk_widget_show (od->fixed);
          /* ja:スクロールバー */
          if (dwStyle & WS_HSCROLL)
            {
              od->hscroll = gtk_hscrollbar_new
                    (GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 0, 1, 1, 1)));
              gtk_widget_show (od->hscroll);
            }
          if (dwStyle & WS_VSCROLL)
            {
              od->vscroll = gtk_vscrollbar_new
                    (GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 0, 1, 1, 1)));
              gtk_widget_show (od->vscroll);
            }
          /* ja:テーブル
                最上段からメニュー,クライアント,スクロールバー,ステータス */
          od->table = gtk_table_new (2, 4, FALSE);
          if (od->menu)
            gtk_table_attach (GTK_TABLE (od->table), od->menu, 0, 2, 0, 1,
                                            GTK_EXPAND | GTK_FILL, 0, 0, 0);
          gtk_table_attach_defaults (GTK_TABLE (od->table), od->fixed,
                    od->hscroll && (dwExStyle & WS_EX_LEFTSCROLLBAR) ? 1 : 0,
                    od->hscroll && !(dwExStyle & WS_EX_LEFTSCROLLBAR) ? 1 : 2,
                    1, 2);
          if (od->vscroll)
            gtk_table_attach (GTK_TABLE (od->table), od->vscroll,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 0 : 1,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 1 : 2,
                                    1, 2, 0, GTK_EXPAND | GTK_FILL, 0, 0);
          if (od->hscroll)
            gtk_table_attach (GTK_TABLE (od->table), od->hscroll,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 1 : 0,
                                    dwExStyle & WS_EX_LEFTSCROLLBAR ? 2 : 1,
                                    2, 3, GTK_EXPAND | GTK_FILL, 0, 0, 0);
          gtk_widget_show (od->table);
          if (dwStyle & DS_MODALFRAME)
            {
              widget = gtk_dialog_new ();
              gtk_dialog_set_has_separator (GTK_DIALOG (widget), FALSE);
              gtk_widget_hide (GTK_DIALOG (widget)->action_area);
              gtk_container_add (GTK_CONTAINER (GTK_DIALOG (widget)->vbox),
                                                                    od->table);
            }
          else
            {
              widget = gtk_window_new (dwStyle & WS_POPUP ? GTK_WINDOW_POPUP
                                                        : GTK_WINDOW_TOPLEVEL);
              gtk_container_add (GTK_CONTAINER (widget), od->table);
            }
          gtk_window_set_destroy_with_parent (GTK_WINDOW (widget), TRUE);
          gtk_window_set_position (GTK_WINDOW (widget),
                                                GTK_WIN_POS_CENTER_ON_PARENT);
          gtk_window_set_resizable (GTK_WINDOW (widget),
                                                    dwStyle & WS_THICKFRAME);
          title = w32ldr_atom_get_string (windowname);
          if (title)
            {
              gtk_window_set_title (GTK_WINDOW (widget), title);
              g_free (title);
            }
          if (hWndParent)
            gtk_window_set_transient_for (GTK_WINDOW (widget),
                                                    GTK_WINDOW (hWndParent));
          g_signal_connect (G_OBJECT (widget), "button-press-event",
                            G_CALLBACK (w32wnd_button_press_event), NULL);
          g_signal_connect (G_OBJECT (widget), "button-release-event",
                            G_CALLBACK (w32wnd_button_release_event), NULL);
          g_signal_connect (G_OBJECT (widget), "configure-event",
                            G_CALLBACK (w32wnd_configure_event), NULL);
          g_signal_connect (G_OBJECT (widget), "delete-event",
                            G_CALLBACK (w32wnd_delete_event), NULL);
          g_signal_connect_after (G_OBJECT (widget), "expose-event",
                            G_CALLBACK (w32wnd_expose_event), NULL);
          g_signal_connect (G_OBJECT (widget), "focus-in-event",
                            G_CALLBACK (w32wnd_focus_in_event), NULL);
          g_signal_connect (G_OBJECT (widget), "focus-out-event",
                            G_CALLBACK (w32wnd_focus_out_event), NULL);
          g_signal_connect (G_OBJECT (widget), "key-press-event",
                            G_CALLBACK (w32wnd_key_press_event), NULL);
          g_signal_connect (G_OBJECT (widget), "key-release-event",
                            G_CALLBACK (w32wnd_key_release_event), NULL);
          g_signal_connect (G_OBJECT (widget), "motion-notify-event",
                            G_CALLBACK (w32wnd_motion_notify_event), NULL);
          g_signal_connect (G_OBJECT (widget), "scroll-event",
                            G_CALLBACK (w32wnd_scroll_event), NULL);
          gtk_widget_add_events (widget, GDK_POINTER_MOTION_MASK
                            | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
                            | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
                            | GDK_FOCUS_CHANGE_MASK | GDK_SCROLL_MASK);
        }
      g_object_set_data (G_OBJECT (widget), "user_data", wd);
    }
  if (!widget)
    return NULL;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    {
      wd->csw = g_malloc (sizeof (CREATESTRUCTW));
      wd->csw->lpCreateParams = lpvParam;
      wd->csw->hInstance = hInstance;
      wd->csw->hMenu = hMenu;
      wd->csw->hwndParent = hWndParent;
      wd->csw->cy = height;
      wd->csw->cx = width;
      wd->csw->y = y;
      wd->csw->x = x;
      wd->csw->style = dwStyle;
      wd->csw->lpszName = GPOINTER_TO_UINT (windowname) > G_MAXUINT16
                        ? g_utf8_to_utf16 (windowname, -1, NULL, NULL, NULL)
                        : (gunichar2 *)windowname;
      wd->csw->lpszClass = GPOINTER_TO_UINT (classname) > G_MAXUINT16
                        ? g_utf8_to_utf16 (classname, -1, NULL, NULL, NULL)
                        : (gunichar2 *)classname;
      wd->csw->dwExStyle = dwExStyle;
    }
  else
    {
      wd->csa = g_malloc (sizeof (CREATESTRUCTA));
      wd->csa->lpCreateParams = lpvParam;
      wd->csa->hInstance = hInstance;
      wd->csa->hMenu = hMenu;
      wd->csa->hwndParent = hWndParent;
      wd->csa->cy = height;
      wd->csa->cx = width;
      wd->csa->y = y;
      wd->csa->x = x;
      wd->csa->style = dwStyle;
      wd->csa->lpszName = GPOINTER_TO_UINT (windowname) > G_MAXUINT16
                                ? w32ldr_utf8_to_mb (windowname) : windowname;
      wd->csa->lpszClass = GPOINTER_TO_UINT (classname) > G_MAXUINT16
                                ? w32ldr_utf8_to_mb (classname) : classname;
      wd->csa->dwExStyle = dwExStyle;
    }
  if (GTK_IS_WINDOW (widget))
    {
      if (x != CW_USEDEFAULT && y != CW_USEDEFAULT)
        gtk_window_move (GTK_WINDOW (widget), x, y);
      if (width != CW_USEDEFAULT && height != CW_USEDEFAULT)
        gtk_window_set_default_size (GTK_WINDOW (widget), width, height);
    }
  else if (hWndParent && !gtk_widget_get_parent (widget))
    {
      GtkWidget *fixed;
      W32LdrOverlapData *od;

      if (wd->resizable && width != CW_USEDEFAULT && height != CW_USEDEFAULT)
        gtk_widget_set_size_request (widget, width, height);
      od = g_object_get_data (G_OBJECT (hWndParent), "user_data");
      i = 0;
      if (od->wd.child)
        while (od->wd.child[i])
          i++;
      od->wd.child = g_realloc (od->wd.child, sizeof (GtkWidget *) * (i + 2));
      od->wd.child[i] = widget;
      od->wd.child[i + 1] = NULL;
      /* ja:直近のラベルにアクセラレータを設定する */
      if (wd->focus && !GTK_IS_BUTTON (widget) && !GTK_IS_LABEL (widget))
        while (--i >= 0)
          if (GTK_IS_LABEL (od->wd.child[i])
            && !gtk_label_get_mnemonic_widget (GTK_LABEL (od->wd.child[i])))
            {
              gtk_label_set_mnemonic_widget (GTK_LABEL (od->wd.child[i]),
                                                                    wd->focus);
              break;
            }
      /* ja:フィクストに配置する */
      fixed = GTK_IS_FIXED (hWndParent) ? hWndParent : od->fixed;
      if (fixed)
        gtk_fixed_put (GTK_FIXED (fixed), widget, x, y);
    }
  g_signal_connect (G_OBJECT (widget), "realize",
                                            G_CALLBACK (w32wnd_realize), NULL);
  g_signal_connect_after (G_OBJECT (widget), "destroy",
                                            G_CALLBACK (w32wnd_destroy), NULL);
  if (dwStyle & WS_DISABLED)
    gtk_widget_set_sensitive (widget, FALSE);
  if (dwStyle & WS_VISIBLE)
    gtk_widget_show (widget);
  if (GTK_IS_WINDOW (widget))
    gtk_widget_realize (widget);
  return widget;
}
