/*!
******************************************************************************

	@file	main.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "cpu.h"
#include "task.h"
#include "mboot.h"
#include "syscall.h"
#include "pic.h"
#include "lapic.h"
#include "dmac.h"
#include "pci.h"
#include "disp.h"
#include "acpi.h"
#include "timer.h"
#include "keyboard.h"
#include "uart.h"
#include "usb.h"
#include "disk.h"
#include "fs.h"
#include "pfemu.h"
#include "shell.h"

const char * vsun86_ver_string = "Vsun86 version 0.01 ("__DATE__" "__TIME__")";

#ifndef _VSUN86_PCSIM
EXTERN_C u8 vmm_int_stack[VMM_INT_STACK_SIZE];
EXTERN_C u8 vmm_stack[VMM_STACK_SIZE];
#else	//_VSUN86_PCSIM
u8 vmm_int_stack[VMM_INT_STACK_SIZE];
u8 vmm_stack[VMM_STACK_SIZE];
#endif	//_VSUN86_PCSIM

EXTERN_C int vmm_main( void *mboot_info )
{
#ifndef _VSUN86_PCSIM
	// Initialize UART (for Debug)
	const u8 uart_flags = UART_DATA_8BITS | UART_STOP_1BIT | UART_PARITY_NONE;
	if ( !uart_init( 115200, uart_flags ) )
		return -1;
#endif	//!_VSUN86_PCSIM

	// Initialize CPU
	if ( !cpu_init() )
		return -1;

#ifndef _VSUN86_PCSIM
	// Check Multiboot Information
	if ( !mboot_init( mboot_info ) )
		return -1;
#else	//_VSUN86_PCSIM
	(void)mboot_info;
#endif	//_VSUN86_PCSIM

	// Initialize Task Manager
	if ( !task_init() )
		return -1;

	// Initialize PIC
	if ( !pic_init() )
		return -1;

#ifndef _VSUN86_PCSIM
	// Initialize Local APIC
	if ( !lapic_init() )
		return -1;

	// Initialize ACPI
	if ( !acpi_init() )
		return -1;
#endif	//!_VSUN86_PCSIM

	// Initialize Legacy Devices
	if ( !timer_init() )
		return -1;
#ifndef _VSUN86_PCSIM
	if ( !dmac_init() )
		return -1;
#endif	//!_VSUN86_PCSIM

	// Initialize FS Manager
	if ( !fs_init() )
		return -1;

	// Initialize Disk Manager
	if ( !disk_init() )
		return -1;

#ifndef _VSUN86_PCSIM
	// Initialize USB Device Manager
	if ( !usb_init() )
		return -1;

	// Initialize PCI Devices
	if ( !pci_init() )
		return -1;
#endif	//!_VSUN86_PCSIM

	// Initialize Display Manager
	if ( !disp_init() )
		return -1;

	// Initialize Shell
	if ( !shell_init() )
		return -1;

	vmm_printf( VMM_INFO, "\n%s\n\n", vsun86_ver_string );

#ifndef _VSUN86_PCSIM
	// Initialize Platform Emulator
	if ( !pfemu_init() )
		vmm_printf( VMM_WARNING, "Warning: Platform Emulator initialization failed.\n" );
#endif	//!_VSUN86_PCSIM

	// Start Task Manager
	if ( !task_start() )
		return -1;

	return 0;
}

#ifndef _VSUN86_PCSIM
void abort( void )
{
	vmm_printf( VMM_ERROR, "ABORT !!!\n" );
	cpu_disable_interrupt();
	while ( 1 )
		halt();		// 永久ループ
}
#endif	//!_VSUN86_PCSIM

#ifndef _VSUN86_PCSIM
u32 virt_to_phys( void *p )
{
	const u32 va = (u32)p;
	const u32 pte = x86_get_pte( p );

	if ( !(pte & PTE_PRESENT) )
		return 0;

	return (pte & 0xFFFFF000) | (va & 0x00000FFF);
}
#endif	//_VSUN86_PCSIM

#ifdef	_VSUN86_PCSIM

#include <stdio.h>

static bool terminate = false;

EXTERN_C int main( int argc, char *argv[] )
{
	(void)argc;
	(void)argv;

	if ( 0 != SDL_Init( SDL_INIT_VIDEO | SDL_INIT_TIMER ) ) {
		fprintf( stderr, "SDL_Init() failed.\n" );
		return -1;
	}
	if ( 0 != SDL_EnableKeyRepeat( SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL ) ) {
		fprintf( stderr, "SDL_EnableKeyRepeat() failed.\n" );
		goto failed;
	}

	if ( 0 != vmm_main( NULL ) ) {
		fprintf( stderr, "vmm_main() failed.\n" );
		goto failed;
	}

	while ( !terminate ) {
		SDL_Event ev;
		if ( !SDL_WaitEvent( &ev ) ) {
			fprintf( stderr, "SDL_WaitEvent() failed.\n" );
			goto failed;
		}
		switch ( ev.type )
		{
		case SDL_QUIT:
			terminate = true;
			break;

		case SDL_KEYDOWN:
			if ( cpu_get_interrupt_flag() ) {
				cpu_disable_interrupt();
				if ( !task_send_event( TASK_ID_KEYBOARD, MSG_VKEY_DOWN, ev.key.keysym.sym, ev.key.keysym.mod ) ) {
					fprintf( stderr, "task_send_event( TASK_ID_KEYBOARD, MSG_VKEY_DOWN ) failed.\n" );
					goto failed;
				}
				cpu_enable_interrupt();
			}
			break;

		case SDL_KEYUP:
			if ( cpu_get_interrupt_flag() ) {
				cpu_disable_interrupt();
				if ( !task_send_event( TASK_ID_KEYBOARD, MSG_VKEY_UP, ev.key.keysym.sym, ev.key.keysym.mod ) ) {
					fprintf( stderr, "task_send_event( TASK_ID_KEYBOARD, MSG_VKEY_UP ) failed.\n" );
					goto failed;
				}
				cpu_enable_interrupt();
			}
			break;

		case SDL_VSUN86_VIRQ:
			if ( cpu_get_interrupt_flag() ) {
				cpu_disable_interrupt();
				if ( ev.user.code < 8 ) {
					if ( !(pic_get_imr( PIC_MASTER ) & (1<<ev.user.code )) )
						irq_handler( (u8)(IRQ_START_VECTOR + ev.user.code) );
				}
				else {
					if ( !(pic_get_imr( PIC_SLAVE ) & (1<<(ev.user.code-8))) )
						irq_handler( (u8)(IRQ_START_VECTOR + ev.user.code) );
				}
				cpu_enable_interrupt();
			}
			break;
		}
	}

	SDL_Quit();
	return 0;

failed:
	SDL_Quit();
	return -1;
}

#endif	//_VSUN86_PCSIM
