/*!
******************************************************************************

	@file	pci.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "pci.h"
#include "irq.h"
#include "io.h"
#include "usb/host/uhci.h"
#include "usb/host/ohci.h"
#include "usb/host/ehci.h"
#include "printf.h"

static void pci_probe( PCI_DEVICE * );
static bool pci_get_device_name( PCI_DEVICE *, const char **, const char ** );

bool pci_init( void )
{
	PCI_DEVICE dev;
	u32 data;
	u32 num = 0;

	vmm_printf( VMM_DEBUG, "\nInitializing PCI Devices ...\n\n" );

	for ( u16 bus_id=0; bus_id<PCI_BUS_MAX; bus_id++ ) {
		for ( u8 dev_id=0; dev_id<PCI_DEV_MAX; dev_id++ ) {
			for ( u8 func_id=0; func_id<PCI_FUNC_MAX; func_id++ ) {
				dev.dev_addr = (bus_id  << PCI_BUS_SHIFT )
							 | (dev_id  << PCI_DEV_SHIFT )
							 | (func_id << PCI_FUNC_SHIFT);

				// 0x00-0x01 : Vendor ID
				// 0x02-0x03 : Device ID
				data = pci_read_config( &dev, 0x00 );
				if ( data == 0xFFFFFFFF )
					continue;
				dev.vendor_id = (u16)(data      );
				dev.device_id = (u16)(data >> 16);
				num++;

				// 0x08      : Revision
				// 0x09-0x0B : Class Code
				data = pci_read_config( &dev, 0x08 );
				dev.revision_id = (u8)data;
				dev.class_code  = data >> 8;

				// 0x0C : Cache Line Size
				// 0x0D : Latency Timer
				// 0x0E : Header Type
				// 0x0F : BIST
				data = pci_read_config( &dev, 0x0C );
				dev.header_type = (u8)(data >> 16);
				if ( dev.header_type & 0x80 )
				{	// マルチファンクションデバイス
					dev.is_multi_func = true;
					dev.header_type &= 0x7F;
				}
				else
				{	// シングルファンクションデバイス
					dev.is_multi_func = false;
				}

				// 0x10-0x27 : Base Address Registers
				for ( int i=0; i<6; i++ ) {
					data = pci_read_config( &dev, 0x10 + (i<<2) );
					dev.base_addr[i] = data;
				}

				// 0x3C : Interrupt Line
				// 0x3D : Interrupt Pin
				// 0x3E : MIN_GNT
				// 0x3F : MAX_LAT
				data = pci_read_config( &dev, 0x3C );
				dev.interrupt_line = (u8)(data      );
				dev.interrupt_pin  = (u8)(data >>  8);
				dev.min_gnt		   = (u8)(data >> 16);
				dev.max_lat		   = (u8)(data >> 24);

				pci_probe( &dev );

				if ( (func_id == 0) && !dev.is_multi_func )
				{	// マルチファンクションデバイスではないのでここでおしまい
					break;
				}
			}
		}
	}

	vmm_printf( VMM_DEBUG, "finished.\n\n" );

	if ( num == 0 )
		return false;

	return true;
}

static void pci_probe( PCI_DEVICE *dev )
{
	vmm_printf( VMM_DEBUG, "%02x:%02x:%02x %04x:%04x ",
				PCI_BUS_NUMBER( dev ), PCI_DEV_NUMBER( dev ), PCI_FUNC_NUMBER( dev ),
				dev->vendor_id, dev->device_id );

	const char *vendor_name = NULL, *device_name = NULL;

	switch ( PCI_CLASS_TYPE( dev ) )
	{	// デバイスを初期化する
	case PCI_CLASS_TYPE_MASS_STORAGE:
		switch ( dev->class_code & 0xFFFF00 )
		{
		case PCI_CLASS_IDE_CONTROLLER:
			vmm_printf( VMM_DEBUG, "IDE Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		default:
			vmm_printf( VMM_DEBUG, "Unknown Mass Storage Controller (class=%06x)\n", dev->class_code );
			break;
		}
		break;

	case PCI_CLASS_TYPE_NETWORK:
		switch ( dev->class_code )
		{
	   	case PCI_CLASS_ETHERNET_ADAPTER:
			vmm_printf( VMM_DEBUG, "Ethernet Adapter\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		default:
			vmm_printf( VMM_DEBUG, "Unknown Network Controller (class=%06x)\n", dev->class_code );
			break;
		}
		break;

	case PCI_CLASS_TYPE_DISPLAY:
		switch ( dev->class_code )
		{
	   	case PCI_CLASS_VGA_CONTROLLER:
			vmm_printf( VMM_DEBUG, "VGA Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		default:
			vmm_printf( VMM_DEBUG, "Unknown Display Controller (class=%06x)\n", dev->class_code );
			break;
		}
		break;

	case PCI_CLASS_TYPE_MULTIMEDIA:
		switch ( dev->class_code )
		{
		case PCI_CLASS_MM_VIDEO_CONTROLLER:
			vmm_printf( VMM_DEBUG, "Multimedia Video Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_MM_AUDIO_CONTROLLER:
			vmm_printf( VMM_DEBUG, "Multimedia Audio Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_TEL_DEVICE:
			vmm_printf( VMM_DEBUG, "Computer Telephony Device\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_AUDIO_DEVICE:
			vmm_printf( VMM_DEBUG, "Audio Device\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		default:
			vmm_printf( VMM_DEBUG, "Unknown Multimedia Controller (class=%06x)\n", dev->class_code );
			break;
		}
		break;

	case PCI_CLASS_TYPE_BRIDGE:
		switch ( dev->class_code )
		{
	   	case PCI_CLASS_HOST_BRIDGE:
			vmm_printf( VMM_DEBUG, "Host Bridge\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_ISA_BRIDGE:
			vmm_printf( VMM_DEBUG, "ISA Bridge\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_PCI_BRIDGE:
			vmm_printf( VMM_DEBUG, "PCI Bridge\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_PCI_BRIDGE_SUBTRACT:
			vmm_printf( VMM_DEBUG, "PCI Bridge (Subtractive decode)\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_OTHER_BRIDGE:
			vmm_printf( VMM_DEBUG, "Bridge\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		default:
			vmm_printf( VMM_DEBUG, "Unknown Bridge (class=%06x)\n", dev->class_code );
			break;
		}
		break;

	case PCI_CLASS_TYPE_PERIPHERAL:
		switch ( dev->class_code )
		{
		case PCI_CLASS_SYSTEM_PERIPHERAL:
			vmm_printf( VMM_DEBUG, "System Peripheral\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		default:
			vmm_printf( VMM_DEBUG, "Unknown Peripheral (class=%06x)\n", dev->class_code );
			break;
		}
		break;

	case PCI_CLASS_TYPE_SERIAL_BUS:
		switch ( dev->class_code )
		{
		case PCI_CLASS_IEEE1394_CONTROLLER:
			vmm_printf( VMM_DEBUG, "IEEE 1394 Host Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_IEEE1394_OHCI:
			vmm_printf( VMM_DEBUG, "IEEE 1394 Host Controller (OHCI)\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		case PCI_CLASS_USB_UHCI_CONTROLLER:
			vmm_printf( VMM_DEBUG, "USB Universal Host Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			uhci_probe( dev );
			break;

		case PCI_CLASS_USB_OHCI_CONTROLLER:
			vmm_printf( VMM_DEBUG, "USB Open Host Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			ohci_probe( dev );
			break;

		case PCI_CLASS_USB_EHCI_CONTROLLER:
			vmm_printf( VMM_DEBUG, "USB 2.0 Enhanced Host Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			ehci_probe( dev );
			break;

		case PCI_CLASS_SMBUS_CONTROLLER:
			vmm_printf( VMM_DEBUG, "SMBus Controller\n" );
			if ( pci_get_device_name( dev, &vendor_name, &device_name ) )
				vmm_printf( VMM_DEBUG, "%s %s\n", vendor_name, device_name );
			break;

		default:
			vmm_printf( VMM_DEBUG, "Unknown Serial Bus Controller (class=%06x)\n", dev->class_code );
			break;
		}
		break;

	default:
		vmm_printf( VMM_DEBUG, "Unknown Device (class=%06x)\n", dev->class_code );
		break;
	}

	vmm_printf( VMM_DEBUG, "\n" );
}

void pci_write_config( PCI_DEVICE *dev, u8 addr, u32 data )
{
	u32 config_addr = 0x80000000 | dev->dev_addr | addr;
	outd( IO_PCI_CONFIG_ADDR, config_addr );
	outd( IO_PCI_CONFIG_DATA, data );
	config_addr &= 0x7FFFFFFF;
	outd( IO_PCI_CONFIG_ADDR, config_addr );
}

u32 pci_read_config( PCI_DEVICE *dev, u8 addr )
{
	u32 config_addr = 0x80000000 | dev->dev_addr | addr;
	outd( IO_PCI_CONFIG_ADDR, config_addr );
	u32 data = ind( IO_PCI_CONFIG_DATA );
	config_addr &= 0x7FFFFFFF;
	outd( IO_PCI_CONFIG_ADDR, config_addr );

	return data;
}

bool pci_irq_register( PCI_DEVICE *dev, IRQ_HANDLER proc, void *args )
{
	if ( dev->interrupt_pin == 0x00 )
		return false;	// デバイスは割り込みを利用しない

	const u8 irq = dev->interrupt_line;
	return irq_register( irq, IRQ_TRIGGER_LEVEL, proc, args );
}

static bool pci_get_device_name( PCI_DEVICE *dev, const char **vendor_name, const char **device_name )
{
	if ( (dev == NULL) || (vendor_name == NULL) || (device_name == NULL) )
		return false;

	*vendor_name = NULL;
	*device_name = NULL;

	switch ( dev->vendor_id )
	{
	case PCI_VENDOR_ATI:
		*vendor_name = "ATI Technologies Inc";
		switch ( dev->device_id )
		{
		case 0x4380:	*device_name = "SB600 Non-Raid-5 SATA";										break;
		case 0x4381:	*device_name = "SB600 Raid-5 SATA";											break;
		case 0x4382:	*device_name = "SB600 AC97 Audio";											break;
		case 0x4383:	*device_name = "SBx00 Azalia (Intel HDA)";									break;
		case 0x4384:	*device_name = "SBx00 PCI to PCI Bridge";									break;
		case 0x4385:	*device_name = "SBx00 SMBus Controller";									break;
		case 0x4386:	*device_name = "SB600 USB Controller (EHCI)";								break;
		case 0x4387:	*device_name = "SB600 USB (OHCI0)";											break;
		case 0x4388:	*device_name = "SB600 USB (OHCI1)";											break;
		case 0x4389:	*device_name = "SB600 USB (OHCI2)";											break;
		case 0x438A:	*device_name = "SB600 USB (OHCI3)";											break;
		case 0x438B:	*device_name = "SB600 USB (OHCI4)";											break;
		case 0x438C:	*device_name = "SB600 IDE";													break;
		case 0x438D:	*device_name = "SB600 PCI to LPC Bridge";									break;
		case 0x438E:	*device_name = "SB600 AC97 Modem";											break;
		case 0x7910:	*device_name = "RS690 Host Bridge";											break;
		case 0x7911:	*device_name = "RS690 Host Bridge";											break;
		case 0x7912:	*device_name = "RS690 PCI to PCI Bridge (Internal gfx)";					break;
		case 0x7913:	*device_name = "RS690 PCI to PCI Bridge (PCI Express Graphics Port 0)";		break;
		case 0x7915:	*device_name = "RS690 PCI to PCI Bridge (PCI Express Port 1)";				break;
		case 0x7916:	*device_name = "RS690 PCI to PCI Bridge (PCI Express Port 2)";				break;
		case 0x7917:	*device_name = "RS690 PCI to PCI Bridge (PCI Express Port 3)";				break;
		case 0x7919:	*device_name = "Radeon X1200 Series Audio Controller";						break;
		case 0x791E:	*device_name = "RS690 [Radeon X1200 Series]";								break;
		case 0x791F:	*device_name = "RS690M [Radeon X1200 Series]";								break;
		}
		break;

	case PCI_VENDOR_AMD:
		*vendor_name = "Advanced Micro Devices [AMD]";
		switch( dev->device_id )
		{
		case 0x1100:	*device_name = "K8 [Athlon64/Opteron] HyperTransport Technology Configuration";		break;
		case 0x1101:	*device_name = "K8 [Athlon64/Opteron] Address Map";									break;
		case 0x1102:	*device_name = "K8 [Athlon64/Opteron] DRAM Controller";								break;
		case 0x1103:	*device_name = "K8 [Athlon64/Opteron] Miscellaneous Control";						break;
		case 0x2000:	*device_name = "[PCnet32 LANCE]";													break;
		}
		break;

	case PCI_VENDOR_APPLE:
		*vendor_name = "Apple Computer Inc.";
		switch ( dev->device_id )
		{
		case 0x003F:	*device_name = "KeyLargo/Intrepid USB";		break;
		}
		break;

	case PCI_VENDOR_REALTEK:
		*vendor_name = "Realtek Semiconductor Co., Ltd.";
		switch ( dev->device_id )
		{
		case 0x8167:	*device_name = "RTL-8110SC/8169SC Gigabit Ethernet";						break;
		case 0x8168:	*device_name = "RTL8111/8168B PCI Express Gigabit Ethernet controller";		break;
		case 0x8169:	*device_name = "RTL-8169 Gigabit Ethernet";									break;
		}
		break;

	case PCI_VENDOR_VIA:
		*vendor_name = "VIA Technologies, Inc.";
		switch ( dev->device_id )
		{
		case 0x3044:	*device_name = "VT6306 Fire II IEEE 1394 OHCI Link Layer Controller";	break;
		}
		break;

	case PCI_VENDOR_VIA_2:
		*vendor_name = "VIA Technologies Inc.";
		switch ( dev->device_id )
		{
		case 0x1724:	*device_name = "VT1720/24 [Envy24PT/HT] PCI Multi-Channel Audio Controller";	break;
		}
		break;

	case PCI_VENDOR_VMWARE:
		*vendor_name = "VMware Inc";
		switch ( dev->device_id )
		{
		case 0x0405:	*device_name = "Abstract SVGA II Adapter";			break;
		case 0x0770:	*device_name = "Abstract USB2 EHCI Controller";		break;
		}
		break;

	case PCI_VENDOR_INTEL:
		*vendor_name = "Intel Corporation";
		switch ( dev->device_id )
		{
		case 0x1237:	*device_name = "440FX - 82441FX PMC [Natoma]";								break;
		case 0x265C:	*device_name = "82801FB/FBM/FR/FW/FRW (ICH6 Family) USB2 EHCI Controller";	break;
		case 0x7000:	*device_name = "430TX - 82439TX MTXC";										break;
		case 0x7110:	*device_name = "82371AB/EB/MB PIIX4 ISA";									break;
		case 0x7111:	*device_name = "82371AB/EB/MB PIIX4 IDE";									break;
		case 0x7112:	*device_name = "82371AB/EB/MB PIIX4 USB";									break;
		case 0x7113:	*device_name = "82371AB/EB/MB PIIX4 ACPI";									break;
		case 0x7190:	*device_name = "440BX/ZX/DX - 82443BX/ZX/DX Host bridge";					break;
		case 0x7191:	*device_name = "440BX/ZX/DX - 82443BX/ZX/DX AGP bridge";					break;
		}
		break;

	case PCI_VENDOR_INNOTEK:
		*vendor_name = "InnoTek Systemberatung GmbH";
		switch ( dev->device_id )
		{
		case 0xBEEF:	*device_name = "VirtualBox Graphics Adapter";	break;
		case 0xCAFE:	*device_name = "VirtualBox Guest Service";		break;
		}
		break;
	}

	if ( (*vendor_name == NULL) || (*device_name == NULL) )
		return false;

	return true;
}
