/*!
******************************************************************************

	@file	task.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "task.h"
#include "atomic.h"
#include "timer.h"
#include "keyboard.h"
#include "uart.h"
#include "disp.h"
#include "shell.h"

#include <time.h>
#ifdef	_VSUN86_PCSIM
#include <stdlib.h>
#endif	//_VSUN86_PCSIM

#define TASK_EVENT_MAX	32

static TASK task_info[TASK_ID_MAX];
static bool task_switch_flag;
static bool task_lock_flag;
static TASK_EVENT task_events[TASK_ID_MAX][TASK_EVENT_MAX];
static TASK task_null[TASK_STATE_MAX];
static TASK *task_queue_head[TASK_STATE_MAX];
static TASK *task_queue_tail[TASK_STATE_MAX];
static TASK *task_running;

static volatile int task_lock_processing = 0;

static TASK * task_get_cur_task( void );
static void task_init_state( TASK *task );
static void task_set_state( TASK *task, TASK_STATE state );
static void task_entry( void );
static bool task_switch_notifier( void * );
static void sleep_task( void * );

static struct {
	TASK_ID			id;
	const char *	name;
	TASK_PROC		proc;
	void *			args;
	u32				stack_size;
} const task_profile[] =
{
	{	TASK_ID_TIMER,		"TIMER",	timer_task,		NULL,	4096	},
	{	TASK_ID_KEYBOARD,	"KEYBOARD",	keyboard_task,	NULL,	4096	},
#ifndef _VSUN86_PCSIM
	{	TASK_ID_UART,		"UART",		uart_task,		NULL,	4096	},
#endif	//!_VSUN86_PCSIM
	{	TASK_ID_SHELL,		"SHELL",	shell_task,		NULL,	65536	},
	{	TASK_ID_INVALID,	NULL,		NULL,			NULL,	0		}
};

#define TASK_STACK			task_stack
#define TASK_STACK_SIZE		0x00400000		// 4MB
static u8  task_stack[TASK_STACK_SIZE];
static u32 task_stack_remain;

bool task_init( void )
{
	memset( task_info, 0, sizeof(task_info) );
	task_stack_remain = TASK_STACK_SIZE;

	for ( int i=0; i<TASK_ID_MAX; i++ )
		task_info[i].id = TASK_ID_INVALID;

	memset( task_null, 0, sizeof(task_null) );
	for ( int i=0; i<TASK_STATE_MAX; i++ ) {
		if ( i == TASK_STATE_BUSY ) {
			task_queue_head[i] = NULL;
			task_queue_tail[i] = NULL;
			if ( !task_create( TASK_ID_SLEEP, "SLEEP", sleep_task, NULL, 4096 ) )
				return false;
			task_running = &task_info[TASK_ID_SLEEP];
		}
		else {
			TASK *task = &task_null[i];
			task->id   = TASK_ID_INVALID;
			task->next = task;
			task->prev = task;
			task_queue_head[i] = task;
			task_queue_tail[i] = task;
		}
	}

	for ( int i=0; task_profile[i].id != TASK_ID_INVALID; i++ )
	{	// タスクを作る
		if ( !task_create( task_profile[i].id,
						   task_profile[i].name,
						   task_profile[i].proc,
						   task_profile[i].args,
						   task_profile[i].stack_size ) )
			return false;
	}

	task_switch_flag = false;

	return true;
}

bool task_start( void )
{
	// システム時刻を初期化する
	_timezone = 9 * 3600;	// UTC+9 (JST)
	timer_rtc_to_system_time( true );

	// 20msごとにタスク切り替えを通知する
	if ( TIMER_ID_INVALID == timer_add( 20, task_switch_notifier, NULL, true ) ) {
		vmm_printf( VMM_ERROR, "TASK: timer_add() failed.\n" );
		return false;
	}

	return true;
}

bool task_create( TASK_ID id, const char *name, TASK_PROC proc, void *args, u32 stack_size )
{
	const u8 *stack = &TASK_STACK[task_stack_remain - stack_size];

	if ( id >= TASK_ID_MAX )
		return false;	// タスクIDが不正

	if ( stack_size < 4 )
		return false;	// 最低でも4バイトは必要

	if ( task_stack_remain < stack_size )
		return false;	// スタックが確保できない

	TASK *task = &task_info[id];
	if ( (task->id != TASK_ID_INVALID) && (task->state != TASK_STATE_HALT) )
		return false;	// 同じIDのタスクを複数作ろうとした

	task_stack_remain -= stack_size;

	CPU_CONTEXT *context = cpu_alloc_context( (void *)task_entry, stack, stack_size );
	if ( context == NULL )
		return false;

	task->context	= context;
	task->id		= id;
	task->name		= name;
	task->proc		= proc;
	task->args		= args;
	task->events	= task_events[id];
	task->event_rp	= 0;
	task->event_wp	= 0;
	task_init_state( task );

//	set_gdt_desc( SEG_TSS( id ),   (u32)tss,  sizeof(TSS)-1, DESC_TYPE_TSS32  );
//	set_gdt_desc( SEG_FS_GS( id ), (u32)task, sizeof(TSS)-1, DESC_TYPE_DATA32 );

	return true;
}

static TASK * task_get_cur_task( void )
{
	return task_running;
}

static void task_init_state( TASK *task )
{
	const TASK_STATE state = TASK_STATE_BUSY;

	TASK *head = task_queue_head[state];
	TASK *tail = task_queue_tail[state];

	if ( head == NULL )
	{	// キューの先頭にセットする
		task->prev = task;
		task->next = task;
		task_queue_head[state] = task;
		task_queue_tail[state] = task;
	}
	else
	{	// キューに挿入する
		TASK *p = head;
		while ( p != tail )
		{	// 挿入しようとしているタスクより優先度の低いタスクを探す
			if ( task->id < p->id )
				break;	// 優先度の低いタスクが見つかった
			p = p->next;
		}
		TASK *prev = p->prev;
		TASK *next = p;
		prev->next = task;
		next->prev = task;
		task->prev = prev;
		task->next = next;
		if ( p == head )
			task_queue_head[state] = task;
	}

	task->state	= state;
}

static void task_set_state( TASK *task, TASK_STATE state )
{
//	vmm_printf( VMM_DEBUG, "[%-8s]: %d->%d\n", task->name, task->state, state );

	if ( task == NULL )
		abort();
	if ( (task->prev == NULL) || (task->next == NULL) )
		abort();

	do {
		// 現在のステートのキューから外す
		const TASK_STATE old_state = task->state;
		TASK *old_prev = task->prev;
		TASK *old_next = task->next;
		old_prev->next = old_next;
		old_next->prev = old_prev;
		if ( task == task_queue_head[old_state] )
			task_queue_head[old_state] = old_next;
		if ( task == task_queue_tail[old_state] )
			task_queue_tail[old_state] = old_prev;

		if ( (state < 0) || (state >= TASK_STATE_MAX) )
			break;

		// 新しいステートのキューに入れる
		TASK *head = task_queue_head[state];
		TASK *tail = task_queue_tail[state];
		TASK *p = head;
		while ( p != tail )
		{	// 挿入しようとしているタスクより優先度の低いタスクを探す
			if ( task->id < p->id )
				break;	// 優先度の低いタスクが見つかった
			p = p->next;
		}
		TASK *prev = p->prev;
		TASK *next = p;
		prev->next = task;
		next->prev = task;
		task->prev = prev;
		task->next = next;
		if ( p == head )
			task_queue_head[state] = task;

		task->state	= state;
	} while (0);
}

static void task_entry( void )
{
	TASK *task = task_get_cur_task();

	task->start_tick = timer_get_tick_count();
	task->proc( task->args );
#ifndef _VSUN86_PCSIM
	if ( task->id == TASK_ID_VM )
	{	// SHELLタスクにメッセージを送る
		task_send_event( TASK_ID_SHELL, MSG_SHELL_VM_EXIT, 0, 0 );
	}
#endif	//!_VSUN86_PCSIM
	task_switch( true, TASK_STATE_HALT );
}

static bool task_switch_notifier( void *args )
{
	(void)args;

	task_switch_flag = true;
	return false;	// タイマを自動的に再セットする
}

void task_switch( bool force, TASK_STATE cur_task_next_state )
{
	static volatile int processing = 0;

	if ( task_lock_flag && !force )
	{	// タスクロック中にタイマ満了でのタスク切り替えは禁止する
		vmm_printf( VMM_DEBUG, "task_switch(): cannot switch task. (locked)\n" );
		return;
	}

	if ( force )
	{	// 強制的にタスクを切り替える
		if ( !try_lock( &processing ) )
		{	// すでに切り替え中→想定外
			vmm_printf( VMM_DEBUG, "task_switch(): cannot switch task. (already processing)\n" );
			abort();
		}
	}
	else
	{	// タイマ満了によりタスクを切り替える
		if ( !task_switch_flag )
		{	// タスク切り替え不要→何もしない
			return;
		}
		if ( !try_lock( &processing ) )
		{	// すでに切り替え中→何もしない
			vmm_printf( VMM_DEBUG, "task_switch(): cannot switch task. (already processing)\n" );
			return;
		}
	}

	task_switch_flag = false;

	// タスクを切り替える
	TASK *cur_task = task_get_cur_task();
	if ( cur_task == NULL ) {
		vmm_printf( VMM_DEBUG, "task_switch(): task_get_cur_task() failed.\n" );
		abort();
	}
	TASK *new_task = cur_task->next;
	if ( new_task == cur_task ) {
//		vmm_printf( VMM_DEBUG, "task_switch(): \"%-8s\"(%02x)->\"%-8s\"(%02x)\n",
//					cur_task->name, cur_task->id, new_task->name, new_task->id );
		unlock( &processing );
		return;
	}
	if ( new_task->id >= TASK_ID_MAX ) {
		vmm_printf( VMM_ERROR, "task_switch(): cannot select next task.\n" );
		abort();
	}
	new_task->start_tick   = timer_get_tick_count();
	cur_task->elapsed_time += new_task->start_tick - cur_task->start_tick;
	if ( cur_task_next_state != TASK_STATE_BUSY )
		task_set_state( cur_task, cur_task_next_state );

//	vmm_printf( VMM_DEBUG, "task_switch(): \"%-8s\"(%02x)->\"%-8s\"(%02x) >>> %04x:%08x\n",
//				cur_task->name, cur_task->id, new_task->name, new_task->id,
//				SEG_TSS( new_task->id ), new_task->tss->eip );

	task_running = new_task;
	unlock( &processing );
	cpu_switch_context( cur_task->context, new_task->context );
}

#if 0
bool task_switch_fpu_regs( void )
{
#ifndef _VSUN86_PCSIM
	static u16 tr_fpu = SEG_VMM_TSS;
	static TSS *tss_fpu = &task_vmm_tss;

	u16 tr_cur;
	X86_STR( tr_cur );
	if ( tr_cur == SEG_V86_TSS )
		return false;
	__ASM__( "clts" );
	if ( tr_cur == tr_fpu )
		return true;

	TSS *tss_cur;
	if ( tr_cur == SEG_VMM_TSS )
		tss_cur = &task_vmm_tss;
	else
		tss_cur = task_info[(tr_fpu - SEG_TSS(0)) / 0x20].tss;

	FPU_REGS *old_regs = &tss_fpu->fpu;
	FPU_REGS *new_regs = &tss_cur->fpu;
	__ASM__(
		"fnsave		(%0)	\n\t"
		"frstor		(%1)	\n\t"
		:: "r"(old_regs), "r"(new_regs)
	);

	/*
	vmm_printf( VMM_DEBUG, "task_switch_fpu_regs(): tss=%04x/%04x\n", tr_fpu, tr_cur );
	vmm_printf( VMM_DEBUG, "fpu(old): cw=%04x, sw=%04x, tw=%04x\n",
				old_regs->cw, old_regs->sw, old_regs->tw );
	vmm_printf( VMM_DEBUG, "fpu(new): cw=%04x, sw=%04x, tw=%04x\n",
				new_regs->cw, new_regs->sw, new_regs->tw );
	*/

	tr_fpu  = tr_cur;
	tss_fpu = tss_cur;
#endif	//!_VSUN86_PCSIM

	return true;
}
#endif

bool task_wait_event( TASK_EVENT *event )
{
	TASK *task;

	if ( event == NULL )
		return false;

	task_lock();
	{
		// イベントが来るまで他のタスクを実行しながら待機する
		task = task_get_cur_task();
		if ( task->event_cnt == 0 ) {
			task_unlock();
			task_switch( true, TASK_STATE_WAIT );
			task_lock();
		}

		// イベントが来たので実行を再開する
		TASK_EVENT *p = &(task->events[task->event_rp++]);
		if ( task->event_rp >= TASK_EVENT_MAX )
			task->event_rp = 0;
		task->event_cnt--;
		event->from	= p->from;
		event->msg	= p->msg;
		event->arg1	= p->arg1;
		event->arg2	= p->arg2;
	}
	task_unlock();

	return true;
}

bool task_send_event( TASK_ID id, u32 msg, u32 arg1, u32 arg2 )
{
	TASK *from, *to;

	if ( id >= TASK_ID_MAX )
		return false;

	to = &task_info[id];
	if ( to->events == NULL )
		return false;

	task_lock();
	{	// イベントをキューに入れる
		if ( to->event_cnt >= TASK_EVENT_MAX ) {
			task_unlock();
			return false;
		}
		from = task_get_cur_task();
		TASK_EVENT *event = &(to->events[to->event_wp++]);
		if ( to->event_wp >= TASK_EVENT_MAX )
			to->event_wp = 0;
		event->from	= from->id;
		event->msg	= msg;
		event->arg1	= arg1;
		event->arg2	= arg2;
		to->event_cnt++;
		task_set_state( to, TASK_STATE_BUSY );
	}
	task_unlock();

	return true;
}

static void sleep_task( void *args )
{
	(void)args;

#ifndef _VSUN86_PCSIM
	while (1)
		halt();
#endif	//!_VSUN86_PCSIM
}

void task_lock( void )
{
	lock( &task_lock_processing );

	if ( task_lock_flag ) {
		vmm_printf( VMM_ERROR, "task_lock(): already locked.\n" );
		abort();
	}
	task_lock_flag = true;

	unlock( &task_lock_processing );
}

void task_unlock( void )
{
	lock( &task_lock_processing );

	if ( !task_lock_flag ) {
		vmm_printf( VMM_ERROR, "task_unlock(): already unlocked.\n" );
		abort();
	}
	task_lock_flag = false;

	unlock( &task_lock_processing );
}

void task_dump( void )
{
	for ( int i=0; i<TASK_ID_MAX; i++ ) {
		if ( task_info[i].id != TASK_ID_INVALID ) {
			vmm_printf( VMM_INFO, "[%-8s] state: %d, elapsed_time: %lld sec\n",
						task_info[i].name, task_info[i].state, task_info[i].elapsed_time / 1000LL );
		}
	}
}

static bool task_wakeup_notifier( void *args )
{
	TASK *task = (TASK *)args;

	// タスクをBUSY状態に戻す
	task_set_state( task, TASK_STATE_BUSY );

	return true;
}

void task_sleep( u32 msec )
{
	TASK *task = task_get_cur_task();
	if ( TIMER_ID_INVALID == timer_add( msec, task_wakeup_notifier, task, true ) ) {
		vmm_printf( VMM_ERROR, "task_sleep() failed.\n" );
		return;
	}

	cpu_disable_interrupt();
	{
		// タスクをIDLE状態にしてタスクを切り替える
		task_switch( true, TASK_STATE_IDLE );
	}
	cpu_enable_interrupt();
}
