/*!
******************************************************************************

	@file	hub.cpp

	Copyright (C) 2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "usb.h"
#include "hub.h"
#include "timer.h"
#include "printf.h"

#define USB_HUB_CMD_TIMEOUT		5000	// [ms]

static bool usb_hub_get_descriptor( USB_DEVICE *dev, USB_HUB_DESC *desc, size_t buf_len );
//static bool usb_hub_get_status( USB_DEVICE *dev, u32 *status );
static bool usb_hub_get_port_status( USB_DEVICE *dev, u8 port, u32 *status );
//static bool usb_hub_set_feature( USB_DEVICE *dev, u16 feature );
static bool usb_hub_set_port_feature( USB_DEVICE *dev, u8 port, u16 feature );
//static bool usb_hub_clear_feature( USB_DEVICE *dev, u16 feature );
//static bool usb_hub_clear_port_feature( USB_DEVICE *dev, u8 port, u16 feature );

bool usb_hub_init( USB_DEVICE *dev, USB_HCD *child_hcd )
{
	u8 hub_desc_buf[256];
	USB_HUB_DESC *hub_desc = (USB_HUB_DESC *)hub_desc_buf;
	if ( !usb_hub_get_descriptor( dev, hub_desc, sizeof(hub_desc_buf) ) ) {
		vmm_printf( VMM_DEBUG, "usb_hub(%d): GetHubDescriptor failed.\n", dev->func_addr );
		return false;
	}

	if ( hub_desc->bNbrPorts == 0 ) {
		vmm_printf( VMM_DEBUG, "usb_hub(%d): No Ports.\n" );
		return false;
	}

	u32 port_status;
	for ( u8 i=0; i<hub_desc->bNbrPorts; i++ )
	{
		if ( !usb_hub_get_port_status( dev, i, &port_status ) ) {
			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] GetPortStatus failed.\n", dev->func_addr, i );
			return false;
		}
//		vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] GetPortStatus OK. (status=%08x)\n",
//					dev->func_addr, i, port_status );

		// ポートに電源が供給されているか確認する
		if ( !(port_status & USB_HUB_PORTST_POWER) )
		{	// 電源が入ってないので電源を入れる
//			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] PortPower=0\n", dev->func_addr, i );
			if ( !usb_hub_set_port_feature( dev, i, USB_HUB_PORT_POWER ) ) {
				vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] SetPortFeature( PORT_POWER ) failed.\n",
							dev->func_addr, i );
				return false;
			}
//			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] SetPortFeature( PORT_POWER ) OK.\n", dev->func_addr, i );
			timer_usleep( hub_desc->bPwrOn2PwrGood * 2000 );
			if ( !usb_hub_get_port_status( dev, i, &port_status ) ) {
				vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] GetPortStatus failed.\n", dev->func_addr, i );
				return false;
			}
//			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] GetPortStatus OK. (status=%08x)\n",
//						dev->func_addr, i, port_status );
		}

		// ポートにデバイスがつながっているか確認する
		if ( !(port_status & USB_HUB_PORTST_CONNECTION) ) {
			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] device not found. (status=%08x)\n",
						dev->func_addr, i, port_status );
			continue;
		}
		vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] device found. (status=%08x)\n",
					dev->func_addr, i, port_status );

		// ポートをリセットする
		if ( !usb_hub_set_port_feature( dev, i, USB_HUB_PORT_RESET ) ) {
			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] SetPortFeature( PORT_RESET ) failed.\n",
						dev->func_addr, i );
			return false;
		}
		vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] SetPortFeature( PORT_RESET ) OK.\n",
					dev->func_addr, i );

		int timeout = 1000;
		while ( timeout > 0 )
		{	// リセットが完了するまで待つ
			if ( !usb_hub_get_port_status( dev, i, &port_status ) ) {
				vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] GetPortStatus failed.\n", dev->func_addr, i );
				return false;
			}
			if ( !(port_status & USB_HUB_PORTST_RESET) )
				break;
			timer_usleep( 1000 );
			timeout--;
		}
		if ( timeout <= 0 )
		{	// 一定時間内にリセットが完了しなかった
			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] reset failed.\n", dev->func_addr, i );
			return false;
		}
		vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] reset OK.\n", dev->func_addr, i );

		// デバイスを初期化する
		const bool low_speed = (port_status & USB_HUB_PORTST_LOW_SPEED)? true:false;
		USB_DEVICE *usb_dev = usb_alloc_device( child_hcd, dev->hcd_priv, low_speed );
		if ( usb_dev == NULL )
			continue;
		if ( !usb_register( usb_dev, child_hcd ) ) {
			vmm_printf( VMM_DEBUG, "usb_hub(%d): [port %d] usb_register() failed.\n", dev->func_addr, i );
			continue;
		}
	}

	return true;
}

static bool usb_hub_get_descriptor( USB_DEVICE *dev, USB_HUB_DESC *desc, size_t buf_len )
{
	return usb_ctrl_msg( dev, USB_REQ_DIR_IN | USB_REQ_TYPE_CLASS | USB_REQ_DEVICE,
						 USB_HUB_GET_DESCRIPTOR, USB_DT_HUB, 0, desc, buf_len, USB_HUB_CMD_TIMEOUT );
}

/*
static bool usb_hub_get_status( USB_DEVICE *dev, u32 *status )
{
	return usb_ctrl_msg( dev, USB_REQ_DIR_IN | USB_REQ_TYPE_CLASS | USB_REQ_DEVICE,
						 USB_HUB_GET_STATUS, 0, 0, status, 4, USB_HUB_CMD_TIMEOUT );
}
*/

static bool usb_hub_get_port_status( USB_DEVICE *dev, u8 port, u32 *status )
{
	return usb_ctrl_msg( dev, USB_REQ_DIR_IN | USB_REQ_TYPE_CLASS | USB_REQ_OTHER,
						 USB_HUB_GET_STATUS, 0, port + 1, status, 4, USB_HUB_CMD_TIMEOUT );
}

/*
static bool usb_hub_set_feature( USB_DEVICE *dev, u16 feature )
{
	return usb_ctrl_msg( dev, USB_REQ_DIR_OUT | USB_REQ_TYPE_CLASS | USB_REQ_DEVICE,
						 USB_HUB_SET_FEATURE, feature, 0, NULL, 0, USB_HUB_CMD_TIMEOUT );
}
*/

static bool usb_hub_set_port_feature( USB_DEVICE *dev, u8 port, u16 feature )
{
	return usb_ctrl_msg( dev, USB_REQ_DIR_OUT | USB_REQ_TYPE_CLASS | USB_REQ_OTHER,
						 USB_HUB_SET_FEATURE, feature, port + 1, NULL, 0, USB_HUB_CMD_TIMEOUT );
}

/*
static bool usb_hub_clear_feature( USB_DEVICE *dev, u16 feature )
{
	return usb_ctrl_msg( dev, USB_REQ_DIR_OUT | USB_REQ_TYPE_CLASS | USB_REQ_DEVICE,
						 USB_HUB_CLEAR_FEATURE, feature, 0, NULL, 0, USB_HUB_CMD_TIMEOUT );
}

static bool usb_hub_clear_port_feature( USB_DEVICE *dev, u8 port, u16 feature )
{
	return usb_ctrl_msg( dev, USB_REQ_DIR_OUT | USB_REQ_TYPE_CLASS | USB_REQ_OTHER,
						 USB_HUB_CLEAR_FEATURE, feature, port + 1, NULL, 0, USB_HUB_CMD_TIMEOUT );
}
*/
