package jp.sf.wagashi.anmitsu;

import java.util.HashMap;

import jp.sf.wagashi.IEditorListener;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IPaintPositionManager;
import org.eclipse.jface.text.IPainter;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.ITextViewerExtension5;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.StyledTextContent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.ui.internal.editors.text.EditorsPlugin;
import org.eclipse.ui.internal.texteditor.PropertyEventDispatcher;
import org.eclipse.ui.texteditor.AbstractDecoratedTextEditorPreferenceConstants;
import org.eclipse.ui.texteditor.ITextEditor;


/**
 * A painter for drawing visible characters for (invisible) whitespace 
 * characters.
 * 
 * @since 3.4
 */
public class LinePainter implements IPainter, PaintListener, DisposeListener {
	
	/** Indicates whether this painter is active. */
	private boolean fIsActive= false;
	/** The source viewer this painter is attached to. */
	private ITextViewer fTextViewer;
	/** The viewer's widget. */
	private StyledText fTextWidget;
	/** Tells whether the advanced graphics sub system is available. */
	private boolean fIsAdvancedGraphicsPresent;

	private boolean fShowHighlightCurrentLine;
	
	private boolean fEnableLinePainter;
	
	private boolean fShowTab;
	private boolean fShowHalfSpace;
	private boolean fShowFullSpace;
	private boolean fShowLineFeed;
	private boolean fShowEOF;
	private boolean fIsCutSelectionText;
	
	private Color fTabColor;
	private Color fHalfSpaceColor;
	private Color fFullSpaceColor;
	private Color fLineFeedColor;
	private Color fEOFColor;
	private Color fHighlightCurrentColor;

	private PropertyEventDispatcher fEditorDispatcher = null;
	private PropertyEventDispatcher fLinePainterDispatcher = null;
	
	private HashMap<String,Color> fColorMap = new HashMap<String,Color>();
	
    final IPreferenceStore fStore = EditorToolsPlugin.getDefault().getPreferenceStore();
    final IPreferenceStore fEditorStore = EditorsPlugin.getDefault().getPreferenceStore();

	private IPropertyChangeListener fLinePainterPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fEnableLinePainter = (Boolean)event.getNewValue();
    		fTextWidget.redraw();
    	}
    };
	
	private IPropertyChangeListener fTabPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fShowTab = (Boolean)event.getNewValue();
    		fTextWidget.redraw();
    	}
    };
	
	private IPropertyChangeListener fHalfSpacePropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fShowHalfSpace = (Boolean)event.getNewValue();
    		fTextWidget.redraw();
    	}
    };

	private IPropertyChangeListener fFullSpacePropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fShowFullSpace = (Boolean)event.getNewValue();
    		fTextWidget.redraw();
    	}
    };
    
	private IPropertyChangeListener fLineFeedPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fShowLineFeed = (Boolean)event.getNewValue();
    		fTextWidget.redraw();
    	}
    };
    
	private IPropertyChangeListener fEOFPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fShowEOF = (Boolean)event.getNewValue();
    		fTextWidget.redraw();
    	}
    };
    
	private IPropertyChangeListener fCutSelectionTextPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fIsCutSelectionText = (Boolean)event.getNewValue();
    		fTextWidget.redraw();
    	}
    };

	private IPropertyChangeListener fTabColorPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fTabColor = new Color(fTextWidget.getDisplay(), PreferenceConverter.getColor(fStore, EditorToolsConstants.TAB_COLOR));
    		fColorMap.put(EditorToolsConstants.TAB_COLOR, fTabColor);
    		fTextWidget.redraw();
    	}
    };
    

	private IPropertyChangeListener fHalfSpaceColorPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fHalfSpaceColor = new Color(fTextWidget.getDisplay(), PreferenceConverter.getColor(fStore, EditorToolsConstants.HALF_SPACE_COLOR));
    		fTextWidget.redraw();
    	}
    };
    
	private IPropertyChangeListener fFullSpaceColorPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fFullSpaceColor = new Color(fTextWidget.getDisplay(), PreferenceConverter.getColor(fStore, EditorToolsConstants.FULL_SPACE_COLOR));
    		fTextWidget.redraw();
    	}
    };

	private IPropertyChangeListener fLineFeedColorPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fLineFeedColor = new Color(fTextWidget.getDisplay(), PreferenceConverter.getColor(fStore, EditorToolsConstants.LINE_FEED_COLOR));
    		fTextWidget.redraw();
    	}
    };
    
	private IPropertyChangeListener fEOFColorPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		System.out.println("fEOFColor");
    		fEOFColor = new Color(fTextWidget.getDisplay(), PreferenceConverter.getColor(fStore, EditorToolsConstants.EOF_COLOR));
    		fTextWidget.redraw();
    	}
    };
        

    private IPropertyChangeListener fHighlightCurrentLinePropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
    		fShowHighlightCurrentLine = (Boolean)event.getNewValue();
    	}
    };

    private IPropertyChangeListener fHighlightCurrentColorPropertyChangeListener = new IPropertyChangeListener(){
    	public void propertyChange(PropertyChangeEvent event) {
            fHighlightCurrentColor = new Color(fTextWidget.getDisplay(), PreferenceConverter.getColor(fEditorStore, AbstractDecoratedTextEditorPreferenceConstants.EDITOR_CURRENT_LINE_COLOR));
    	}
    };
    
    public LinePainter(ITextViewer textViewer) {
		super();
		fTextViewer = textViewer;
		fTextWidget = textViewer.getTextWidget();
		
		propertyChange();
        addPropertyEventDispatcher();
		
	}
	
	private void addPropertyEventDispatcher(){
        fLinePainterDispatcher = new PropertyEventDispatcher(fStore);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.ENABLE_LINE_PAINTER, fLinePainterPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.TAB, fTabPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.HALF_SPACE, fHalfSpacePropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.FULL_SPACE, fFullSpacePropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.LINE_FEED, fLineFeedPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.LINE_FEED, fLineFeedPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.EOF, fEOFPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.CUT_SELECTION_TEXT, fCutSelectionTextPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.TAB_COLOR, fTabColorPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.HALF_SPACE_COLOR, fHalfSpaceColorPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.FULL_SPACE_COLOR, fFullSpaceColorPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.LINE_FEED_COLOR, fLineFeedColorPropertyChangeListener);
        fLinePainterDispatcher.addPropertyChangeListener(EditorToolsConstants.EOF_COLOR, fEOFColorPropertyChangeListener);
	
        fEditorDispatcher = new PropertyEventDispatcher(fEditorStore);
        fEditorDispatcher.addPropertyChangeListener(AbstractDecoratedTextEditorPreferenceConstants.EDITOR_CURRENT_LINE, fHighlightCurrentLinePropertyChangeListener);
        fEditorDispatcher.addPropertyChangeListener(AbstractDecoratedTextEditorPreferenceConstants.EDITOR_CURRENT_LINE_COLOR, fHighlightCurrentColorPropertyChangeListener);
	}
	/*
	 * @see org.eclipse.jface.text.IPainter#dispose()
	 */
	public void dispose() {
		if(fEditorDispatcher != null){
			fEditorDispatcher.dispose();
			fEditorDispatcher = null;
		}
		if(fLinePainterDispatcher != null){
			fLinePainterDispatcher.dispose();
			fLinePainterDispatcher = null;
		}
		fTabColor = null;
		fHalfSpaceColor = null;
		fFullSpaceColor = null;
		fLineFeedColor = null;
		fEOFColor = null;
		fHighlightCurrentColor = null;
		fTextViewer= null;
		fTextWidget= null;
		
	}

	/*
	 * @see org.eclipse.jface.text.IPainter#paint(int)
	 */
	public void paint(int reason) {
		if(!fEnableLinePainter){
			return;
		}
		try{
			IDocument document= fTextViewer.getDocument();
			if (document == null) {
				deactivate(false);
				return;
			}
			if (!fIsActive) {
				fIsActive= true;
				fTextWidget.addPaintListener(this);
				fTextWidget.redraw();
			} else if (reason == CONFIGURATION || reason == INTERNAL) {
				fTextWidget.redraw();
			} else if (reason == TEXT_CHANGE) {
				// redraw current line only
				try {
					IRegion lineRegion =
						document.getLineInformationOfOffset(getDocumentOffset(fTextWidget.getCaretOffset()));
					int widgetOffset= getWidgetOffset(lineRegion.getOffset());
					int charCount= fTextWidget.getCharCount();
					int redrawLength= Math.min(lineRegion.getLength(), charCount - widgetOffset);
					if (widgetOffset >= 0 && redrawLength > 0) {
						fTextWidget.redrawRange(widgetOffset, redrawLength, true);
					}
				} catch (BadLocationException e) {
					// ignore
				}
			}
		}catch(Exception e){System.out.println(e);}
	}

	/*
	 * @see org.eclipse.jface.text.IPainter#deactivate(boolean)
	 */
	public void deactivate(boolean redraw) {
		if (fIsActive) {
			fIsActive= false;
			fTextWidget.removePaintListener(this);
			if (redraw) {
				fTextWidget.redraw();
			}
		}
	}

	/*
	 * @see org.eclipse.jface.text.IPainter#setPositionManager(org.eclipse.jface.text.IPaintPositionManager)
	 */
	public void setPositionManager(IPaintPositionManager manager) {
		// no need for a position manager
	}

	/*
	 * @see org.eclipse.swt.events.PaintListener#paintControl(org.eclipse.swt.events.PaintEvent)
	 */
	public void paintControl(PaintEvent event) {
		if (fTextWidget != null && fEnableLinePainter) {
			handleDrawRequest(event.gc, event.x, event.y, event.width, event.height);
		}
	}

	/**
	 * Draw characters in view range.
	 * 
	 * @param gc
	 * @param x
	 * @param y
	 * @param w
	 * @param h
	 */
	private void handleDrawRequest(GC gc, int x, int y, int w, int h) {
		int lineCount= fTextWidget.getLineCount();
		int startLine= (y + fTextWidget.getTopPixel()) / fTextWidget.getLineHeight();
		int endLine= (y + h - 1 + fTextWidget.getTopPixel()) / fTextWidget.getLineHeight();
		if (startLine <= endLine && startLine < lineCount) {
			int startOffset= fTextWidget.getOffsetAtLine(startLine);
			int endOffset =
				endLine < lineCount - 1 ? fTextWidget.getOffsetAtLine(endLine + 1) : fTextWidget.getCharCount();
			if (fIsAdvancedGraphicsPresent) {
				int alpha= gc.getAlpha();
				gc.setAlpha(255);
				handleDrawRequest(gc, startOffset, endOffset);
				gc.setAlpha(alpha);
			} else
				handleDrawRequest(gc, startOffset, endOffset);
		}
	}

	/**
	 * Draw characters of content range.
	 * 
	 * @param gc the GC
	 * @param startOffset inclusive start index
	 * @param endOffset exclusive end index
	 */
	private void handleDrawRequest(GC gc, int startOffset, int endOffset) {
		StyledTextContent content= fTextWidget.getContent();
		int length= endOffset - startOffset;
		String text= content.getTextRange(startOffset, length);
		for (int textOffset= 0; textOffset <= length; ++textOffset) {
			if (textOffset < length) {
				char c= text.charAt(textOffset);
				switch (c) {
					case ' ' :
						drawHalfSpace(gc, startOffset + textOffset);
						break;
					case '\u3000' :
						drawFullSpace(gc, startOffset + textOffset);
						break;
					case '\t' :
						drawTab(gc, startOffset + textOffset);
						break;
					case '\n' :
						drawLineFeed(gc, startOffset + textOffset);
						break;
					default :
						break;
				}
			}
		}
		if(content.getLineAtOffset(endOffset) == (content.getLineCount() - 1)){
			drawEOF(gc, endOffset);
		}
	}

	/**
	 * Check if the given widget line is a folded line.
	 * 
	 * @param widgetLine  the widget line number
	 * @return <code>true</code> if the line is folded
	 */
	private boolean isFoldedLine(int widgetLine) {
		if (fTextViewer instanceof ITextViewerExtension5) {
			ITextViewerExtension5 extension= (ITextViewerExtension5)fTextViewer;
			int modelLine= extension.widgetLine2ModelLine(widgetLine);
			int widgetLine2= extension.modelLine2WidgetLine(modelLine + 1);
			return widgetLine2 == -1;
		}
		return false;
	}

	private void drawHalfSpace(GC gc, int offset){
		StyledTextContent content = fTextWidget.getContent();

		if (!isFoldedLine(content.getLineAtOffset(offset))) {
			Color[] color = getColors(offset, EditorToolsConstants.HALF_SPACE_COLOR);
			Point pos = fTextWidget.getLocationAtOffset(offset);
			Point pos2 = fTextWidget.getLocationAtOffset(offset+1);

			gc.setForeground(color[0]);
			gc.setBackground(color[1]);
			
			int charWidth = pos2.x - pos.x;
			int count = charWidth >= 7 ? 3 : (charWidth - 1) >> 1; 
			int startx = charWidth >= 7 ? (charWidth - 7) >> 1 : 0;
			int height = fTextWidget.getLineHeight() - 1;
			
			gc.fillRectangle(pos.x, pos.y, pos2.x - pos.x, fTextWidget.getLineHeight());

			if(fShowHalfSpace){
				gc.drawPoint(pos.x + startx, pos.y + height - 2);
				gc.drawPoint(pos.x + startx, pos.y + height - 4);
				gc.drawPoint((count << 1) + pos.x + startx, pos.y + height - 2);
				gc.drawPoint((count << 1) + pos.x + startx, pos.y + height - 4);
				for(int i = 0; i < count; i++){
					gc.drawPoint(pos.x + startx + 1 + (i << 1), pos.y + height - 1);
				}
			}
		}
	}
	
	private void drawFullSpace(GC gc, int offset){
		StyledTextContent content = fTextWidget.getContent();

		if (!isFoldedLine(content.getLineAtOffset(offset))) {
			Color[] color = getColors(offset, EditorToolsConstants.FULL_SPACE_COLOR);
			Point pos = fTextWidget.getLocationAtOffset(offset);
			Point pos2 = fTextWidget.getLocationAtOffset(offset+1);

			gc.setForeground(color[0]);
			gc.setBackground(color[1]);

			int charWidth = pos2.x - pos.x;
			int startx = (charWidth >> 1) - 4;
			int starty = (fTextWidget.getLineHeight() >> 1) - 4;
			
			gc.fillRectangle(pos.x, pos.y, pos2.x - pos.x, fTextWidget.getLineHeight());

			if(fShowFullSpace){
				for(int i = 0; i < 4; i++){
					gc.drawPoint(pos.x + startx, 2 + pos.y + starty + (i << 1));
					gc.drawPoint(pos.x + startx + 8, 2 + pos.y + starty + (i << 1));
				}
				for(int i = 0; i < 4; i++){
					gc.drawPoint(1 + pos.x + startx + (i << 1), 1 + pos.y + starty);
					gc.drawPoint(1 + pos.x + startx + (i << 1), 9 + pos.y + starty);
				}
			}
		}
	}
	
	private void drawTab(GC gc, int offset){
		StyledTextContent content = fTextWidget.getContent();

		if (!isFoldedLine(content.getLineAtOffset(offset))) {
			int height = fTextWidget.getLineHeight() >> 1;
			Color[] color = getColors(offset, EditorToolsConstants.TAB_COLOR);
			Point pos = fTextWidget.getLocationAtOffset(offset);
			Point pos2 = fTextWidget.getLocationAtOffset(offset+1);

			gc.setForeground(color[0]);
			gc.setBackground(color[1]);
	
			gc.fillRectangle(pos.x, pos.y, pos2.x - pos.x, fTextWidget.getLineHeight());
			
			if(fShowTab){
				gc.drawPoint(pos.x + 1, pos.y + height - 1);
				gc.drawPoint(pos.x + 1, pos.y + height + 1);
				gc.drawPoint(pos.x + 2, pos.y + height);
				gc.drawPoint(pos.x + 3, pos.y + height);
			}
		}
	}
	
	/**
	 * Draw line feed
	 * 
	 * @param gc
	 * @param offset the widget offset
	 */
	private void drawLineFeed(GC gc, int offset){
		FontMetrics fontMetrics= gc.getFontMetrics();
		
		int width = fTextWidget.getBounds().width;
		int lineFeedWidth = 9;
		Point pos = fTextWidget.getLocationAtOffset(offset);
		StyledTextContent content = fTextWidget.getContent();
		if (!isFoldedLine(content.getLineAtOffset(offset))) {
			int fontBaseline = fTextWidget.getLineHeight() - fontMetrics.getDescent() - 1;
			Color[] color = getLineFeedColors(offset);
			gc.setForeground(color[0]);
			gc.setBackground(color[1]);

			if(fIsCutSelectionText){
				gc.fillRectangle(pos.x + lineFeedWidth, pos.y, width - (pos.x + lineFeedWidth), fTextWidget.getLineHeight());
				if(color[2] != null){
					gc.setBackground(color[2]);
				}
				gc.fillRectangle(pos.x, pos.y, lineFeedWidth, fTextWidget.getLineHeight());
			}
			
			if(fShowLineFeed){
				gc.drawPoint(pos.x + 1, pos.y + fontBaseline);
				gc.drawLine(pos.x + 2, pos.y + fontBaseline - 1, pos.x + 2, pos.y + fontBaseline + 1);
				gc.drawLine(pos.x + 3, pos.y + fontBaseline - 2, pos.x + 3, pos.y + fontBaseline + 2);
				gc.drawLine(pos.x + 4, pos.y + fontBaseline, pos.x + 5, pos.y + fontBaseline);
				gc.drawLine(pos.x + 6, pos.y + fontBaseline - 1, pos.x + 6, pos.y + fontBaseline - 6);
			}
		}
	}

	private void drawEOF(GC gc, int offset){
		int width = fTextWidget.getBounds().width;
		Point pos = fTextWidget.getLocationAtOffset(offset);
		Color bg;
		if(fTextWidget.getLineAtOffset(offset) == fTextWidget.getLineAtOffset(fTextWidget.getCaretOffset()) &&
		   fShowHighlightCurrentLine){
			bg = fHighlightCurrentColor;
		}else{
			bg = fTextWidget.getBackground();
		}
		gc.setForeground(fEOFColor);
		gc.setBackground(bg);

		gc.fillRectangle(pos.x, pos.y, width - pos.x, fTextWidget.getLineHeight());
		if(fShowEOF){
			gc.drawString("[EOF]", pos.x, pos.y);
		}
	}
	
	/**
	 * Convert a document offset to the corresponding widget offset.
	 * 
	 * @param documentOffset
	 * @return widget offset
	 */
	private int getWidgetOffset(int documentOffset) {
		if (fTextViewer instanceof ITextViewerExtension5) {
			ITextViewerExtension5 extension= (ITextViewerExtension5)fTextViewer;
			return extension.modelOffset2WidgetOffset(documentOffset);
		}
		IRegion visible= fTextViewer.getVisibleRegion();
		int widgetOffset= documentOffset - visible.getOffset();
		if (widgetOffset > visible.getLength()) {
			return -1;
		}
		return widgetOffset;
	}

	private Color[] getColors(int offset, String key){
		Color[] colors = new Color[2];
		Point selection = fTextWidget.getSelection();
		
		colors[0] = fColorMap.get(key);
		if (offset >= selection.x && offset < selection.y) {
			colors[1] = fTextWidget.getSelectionBackground();
		}else{
			if(fTextWidget.getLineAtOffset(offset) == fTextWidget.getLineAtOffset(fTextWidget.getCaretOffset()) &&
			   fTextWidget.getLineAtOffset(selection.x) == fTextWidget.getLineAtOffset(selection.y) &&
			   fShowHighlightCurrentLine){
				colors[1] = fHighlightCurrentColor;
			}else{
				colors[1] = fTextWidget.getBackground();	
			}
		}
		return colors;
	}

	private Color[] getLineFeedColors(int offset){
		Color[] colors = new Color[3];
		Point selection = fTextWidget.getSelection();
		if (offset >= selection.x && offset < selection.y) {
			if(fIsCutSelectionText){
				colors[0] = fTextWidget.getBackground();
				colors[1] = fTextWidget.getBackground();
				colors[2] = fLineFeedColor;
			} else{
				colors[0] = fLineFeedColor;
				colors[1] = fLineFeedColor;
				colors[2] = fTextWidget.getBackground();
			}
		}else{
			colors[0] = fLineFeedColor;
			if(fTextWidget.getLineAtOffset(offset) == fTextWidget.getLineAtOffset(fTextWidget.getCaretOffset()) &&
			   fTextWidget.getLineAtOffset(selection.x) == fTextWidget.getLineAtOffset(selection.y) &&
			   fShowHighlightCurrentLine){
				colors[1] = fHighlightCurrentColor;
			}else{
				colors[1] = fTextWidget.getBackground();	
			}
		}
		return colors;
	}
	
	/**
	 * Convert a widget offset to the corresponding document offset.
	 * 
	 * @param widgetOffset
	 * @return document offset
	 */
	private int getDocumentOffset(int widgetOffset) {
		if (fTextViewer instanceof ITextViewerExtension5) {
			ITextViewerExtension5 extension= (ITextViewerExtension5)fTextViewer;
			return extension.widgetOffset2ModelOffset(widgetOffset);
		}
		IRegion visible= fTextViewer.getVisibleRegion();
		if (widgetOffset > visible.getLength()) {
			return -1;
		}
		return widgetOffset + visible.getOffset();
	}

	public void propertyChange(){
		Device device = null;
		if(fTextWidget != null){
			device = fTextWidget.getDisplay();
		}
		
        IPreferenceStore store = EditorToolsPlugin.getDefault().getPreferenceStore();
		fEnableLinePainter = store.getBoolean(EditorToolsConstants.ENABLE_LINE_PAINTER);
		fShowTab = store.getBoolean(EditorToolsConstants.TAB);
		fShowHalfSpace = store.getBoolean(EditorToolsConstants.HALF_SPACE);
		fShowFullSpace = store.getBoolean(EditorToolsConstants.FULL_SPACE);
		fShowLineFeed = store.getBoolean(EditorToolsConstants.LINE_FEED);
		fShowEOF = store.getBoolean(EditorToolsConstants.EOF);
		fIsCutSelectionText = store.getBoolean(EditorToolsConstants.CUT_SELECTION_TEXT);
		
		fTabColor = new Color(device, PreferenceConverter.getColor(store, EditorToolsConstants.TAB_COLOR));
		fHalfSpaceColor = new Color(device, PreferenceConverter.getColor(store, EditorToolsConstants.HALF_SPACE_COLOR));
		fFullSpaceColor = new Color(device, PreferenceConverter.getColor(store, EditorToolsConstants.FULL_SPACE_COLOR));
		fLineFeedColor = new Color(device, PreferenceConverter.getColor(store, EditorToolsConstants.LINE_FEED_COLOR));
		fEOFColor = new Color(device, PreferenceConverter.getColor(store, EditorToolsConstants.EOF_COLOR));
		
		fColorMap.clear();
		fColorMap.put(EditorToolsConstants.TAB_COLOR, fTabColor);
		fColorMap.put(EditorToolsConstants.HALF_SPACE_COLOR, fHalfSpaceColor);
		fColorMap.put(EditorToolsConstants.FULL_SPACE_COLOR, fFullSpaceColor);
		fColorMap.put(EditorToolsConstants.LINE_FEED_COLOR, fLineFeedColor);
		fColorMap.put(EditorToolsConstants.EOF_COLOR, fEOFColor);
		
		store = EditorsPlugin.getDefault().getPreferenceStore();
		fShowHighlightCurrentLine = store.getBoolean(AbstractDecoratedTextEditorPreferenceConstants.EDITOR_CURRENT_LINE);
		fHighlightCurrentColor = new Color(device, PreferenceConverter.getColor(store, AbstractDecoratedTextEditorPreferenceConstants.EDITOR_CURRENT_LINE_COLOR));

	}

	public void widgetDisposed(DisposeEvent event) {
        fLinePainterDispatcher.removePropertyChangeListener(fLinePainterPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fTabPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fHalfSpacePropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fFullSpacePropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fLineFeedPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fLineFeedPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fEOFPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fCutSelectionTextPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fTabColorPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fHalfSpaceColorPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fFullSpaceColorPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fLineFeedColorPropertyChangeListener);
        fLinePainterDispatcher.removePropertyChangeListener(fEOFColorPropertyChangeListener);
        fEditorDispatcher.removePropertyChangeListener(fHighlightCurrentLinePropertyChangeListener);
        fEditorDispatcher.removePropertyChangeListener(fHighlightCurrentColorPropertyChangeListener);
	}

}
