/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Flags.Flag;

import com.wankuma.commons.lang.SystemPropertiesUtils;
import com.wankuma.mail.AttachmentFile;
import com.wankuma.mail.Header;
import com.wankuma.mail.Importance;
import com.wankuma.mail.Mail;
import com.wankuma.mail.MailAddress;
import com.wankuma.mail.MailBody;
import com.wankuma.mail.MailException;
import com.wankuma.mail.MailReceiverProtocol;
import com.wankuma.mail.MessageIdGenerator;
import com.wankuma.mail.ReceivedHeader;
import com.wankuma.mail.ReceivedMail;
import com.wankuma.mail.helper.SimpleHeader;
import com.wankuma.mail.helper.SimpleMail;
import com.wankuma.mail.helper.SimpleMailAddress;

/**
 * 日本語処理に対応した{@link com.wankuma.mail.MailReceiver}の実装です。<br>
 * 内部では<a href="http://java.sun.com/products/javamail/">Java Mail</a>を利用しています。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
class JapaneseMailReceiver extends AbstractJavaMailMailReceiver {

	static {
		String mapping = "x-windows-iso2022jp/ISO-2022-JP";
		String key = "sun.nio.cs.map";

		String property = SystemPropertiesUtils.getString(key);
		if (property == null) {
			property = mapping;
		} else {
			if (property.indexOf(mapping) == -1) {
				property = property + "," + mapping;
			}
		}

		SystemPropertiesUtils.setProperty(key, mapping);
	}

	private static class ReceivedHeaderImpl extends SimpleHeader implements
			ReceivedHeader {

		private static Pattern receivedHeaderPattern = Pattern.compile(
				"^from\\s+(\\S+?)\\s+.*by\\s+(\\S+?)\\s+.*$", Pattern.DOTALL);

		private String by;

		private String from;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param name
		 *            名前
		 * @param value
		 *            値
		 */
		ReceivedHeaderImpl(String name, String value) {
			super(name, value);

			Matcher matcher = ReceivedHeaderImpl.receivedHeaderPattern
					.matcher(value);
			if (matcher.matches()) {
				this.from = matcher.group(1);
				this.by = matcher.group(2);
			}
		}

		/**
		 * @see com.wankuma.mail.ReceivedHeader#getBy()
		 */
		@Override
		public String getBy() {
			return this.by;
		}

		/**
		 * @see com.wankuma.mail.ReceivedHeader#getFrom()
		 */
		@Override
		public String getFrom() {
			return this.from;
		}
	}

	private static class ReceivedMailImpl implements ReceivedMail {

		private Mail mail;

		private Message message;

		private Date receivedDate;

		private ReceivedHeader[] receivedHeaders;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param mail
		 *            メール
		 * @param receivedHeaders
		 *            Receivedヘッダの配列
		 * @param message
		 *            メッセージ
		 * @param receivedDate
		 *            受信日時
		 */
		ReceivedMailImpl(Mail mail, ReceivedHeader[] receivedHeaders,
				Message message, Date receivedDate) {
			this.mail = mail;
			this.receivedHeaders = receivedHeaders;
			this.message = message;
			this.receivedDate = receivedDate;
		}

		/**
		 * @see com.wankuma.mail.ReceivedMail#delete()
		 */
		@Override
		public void delete() throws MailException {
			try {
				this.message.setFlag(Flag.DELETED, true);
			} catch (MessagingException e) {
				throw new MailException(e);
			}
		}

		/**
		 * @see com.wankuma.mail.Mail#getAttachmentFiles()
		 */
		public AttachmentFile[] getAttachmentFiles() {
			return this.mail.getAttachmentFiles();
		}

		/**
		 * @see com.wankuma.mail.Mail#getBcc()
		 */
		public MailAddress[] getBcc() {
			return this.mail.getBcc();
		}

		/**
		 * @see com.wankuma.mail.Mail#getBody()
		 */
		public MailBody getBody() {
			return this.mail.getBody();
		}

		/**
		 * @see com.wankuma.mail.Mail#getCc()
		 */
		public MailAddress[] getCc() {
			return this.mail.getCc();
		}

		/**
		 * @see com.wankuma.mail.Mail#getEnvelopeTo()
		 */
		public String[] getEnvelopeTo() {
			return this.mail.getEnvelopeTo();
		}

		/**
		 * @see com.wankuma.mail.Mail#getFrom()
		 */
		public MailAddress getFrom() {
			return this.mail.getFrom();
		}

		/**
		 * @see com.wankuma.mail.Mail#getHeaders()
		 */
		public Header[] getHeaders() {
			return this.mail.getHeaders();
		}

		/**
		 * @see com.wankuma.mail.Mail#getImportance()
		 */
		public Importance getImportance() {
			return this.mail.getImportance();
		}

		/**
		 * @see com.wankuma.mail.Mail#getMessageIdGenerator()
		 */
		public MessageIdGenerator getMessageIdGenerator() {
			return this.mail.getMessageIdGenerator();
		}

		/**
		 * @see com.wankuma.mail.ReceivedMail#getReceivedDate()
		 */
		@Override
		public Date getReceivedDate() {
			return (Date) this.receivedDate.clone();
		}

		/**
		 * @see com.wankuma.mail.ReceivedMail#getRecievedHeaders()
		 */
		@Override
		public ReceivedHeader[] getRecievedHeaders() {
			return this.receivedHeaders.clone();
		}

		/**
		 * @see com.wankuma.mail.Mail#getReplayTo()
		 */
		public MailAddress[] getReplayTo() {
			return this.mail.getReplayTo();
		}

		/**
		 * @see com.wankuma.mail.Mail#getReturnPath()
		 */
		public String getReturnPath() {
			return this.mail.getReturnPath();
		}

		/**
		 * @see com.wankuma.mail.Mail#getSentDate()
		 */
		public Date getSentDate() {
			return this.mail.getSentDate();
		}

		/**
		 * @see com.wankuma.mail.Mail#getSubject()
		 */
		public String getSubject() {
			return this.mail.getSubject();
		}

		/**
		 * @see com.wankuma.mail.Mail#getTo()
		 */
		public MailAddress[] getTo() {
			return this.mail.getTo();
		}
	}

	/**
	 * インスタンスを構築します。
	 * 
	 * @param protocol
	 * @param properties
	 */
	JapaneseMailReceiver(MailReceiverProtocol protocol, Properties properties) {
		super(protocol, properties);
	}

	/**
	 * @see com.wankuma.mail.javamail.AbstractJavaMailMailReceiver#convertMessage(javax.mail.Message)
	 */
	@Override
	protected ReceivedMail convertMessage(Message message) throws IOException,
			MessagingException {
		ReceivedHeader[] receivedHeaders = this.getReceivedHeaders(message);
		Mail mail = this.toMail(message);
		Date receivedDate = this.getReceivedDate(message);

		return new ReceivedMailImpl(mail, receivedHeaders, message,
				receivedDate);
	}

	private Date getReceivedDate(Message message) throws MessagingException,
			IOException {
		Date date = message.getReceivedDate();
		if (date == null) {
			date = new Date();
		}

		return date;
	}

	private ReceivedHeader[] getReceivedHeaders(Message message)
			throws MessagingException {
		String[] values = null;
		try {
			values = message.getHeader("Received");
		} catch (MessagingException e) {
			return new ReceivedHeader[0];
		}

		List<ReceivedHeader> receivedHeaders = new ArrayList<ReceivedHeader>();
		if (values != null) {
			for (String value : values) {
				receivedHeaders.add(new ReceivedHeaderImpl("Received", value));
			}
		}

		return receivedHeaders.toArray(new ReceivedHeader[receivedHeaders
				.size()]);
	}

	private MailAddress[] parseMailAddress(String addresses) throws IOException {
		List<MailAddress> result = new ArrayList<MailAddress>();

		String[] tokens = new String[2];
		OUTER: for (String address : addresses.split(",")) {
			int idx = 0;
			for (String a : address.split("\\n|\\r|\\t| ")) {
				if (idx == 2) {
					continue OUTER;
				}

				a = a.trim();
				if (a.length() == 0) {
					continue;
				}
				int index = a.indexOf(":");
				if (index > -1) {
					a = a.substring(index + 1);
				}
				index = a.indexOf(";");
				if (index > -1) {
					a = a.substring(0, index);
				}
				if (a.length() == 0) {
					continue;
				}
				tokens[idx++] = a;
			}

			String addr;
			String name;
			if (idx == 1) {
				name = null;
				addr = tokens[0];
			} else if (idx == 2) {
				name = tokens[0];

				addr = tokens[1];
			} else {
				continue;
			}
			addr = addr.replace("<", "");
			addr = addr.replace(">", "");

			if (name != null) {
				result.add(new SimpleMailAddress(addr, DecodeUtils
						.decodeText(name)));
			} else {
				result.add(new SimpleMailAddress(addr));
			}
		}

		return result.toArray(new MailAddress[result.size()]);
	}

	private Mail toMail(Message message) throws MessagingException, IOException {
		SimpleMail mail = new SimpleMail();

		// date
		mail.setSentDate(message.getSentDate());

		Enumeration<?> e = message.getAllHeaders();
		int n = 0;
		while (e.hasMoreElements()) {
			javax.mail.Header h = (javax.mail.Header) e.nextElement();
			if (h.getName().equals("To")) {
				this.parseMailAddress(h.getValue());
			}
			if (h.getName().equals("From")) {
				this.parseMailAddress(h.getValue());
			}
			if (h.getName().equals("Return-Path")) {
				this.parseMailAddress(h.getValue());
			}
		}

		// TODO
		return new SimpleMail();
	}
}
