/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

import com.wankuma.commons.io.StreamUtils;
import com.wankuma.commons.util.Base64;

/**
 * メールのデコードをサポートするユーティリティです。<br>
 * <dl>
 * <dt>謝辞</dt>
 * <dd> このクラスは<a href="http://www.sk-jp.com/java/library/index.html">Shin
 * Kinoshita's Home</a>にて提供されている、 <a
 * href="http://www.sk-jp.com/java/library/skutility/index.html">Utilities</a>及び、<a
 * href="http://www.sk-jp.com/java/library/skmail/index.html">Message Access
 * Utilities</a>から、<br>
 * 一部のコードを利用させていただいております。 Shin Kinoshita氏には深く感謝いたします。</dd>
 * </dl>
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */

final class DecodeUtils {

	private static class QDecoderStream extends FilterInputStream {

		/**
		 * インスタンスを構築します。
		 * 
		 * @param in
		 *            入力ストリーム
		 */
		QDecoderStream(InputStream in) {
			super(new PushbackInputStream(in, 2));
		}

		/**
		 * @see java.io.FilterInputStream#read()
		 */
		@Override
		public int read() throws IOException {
			int c = this.in.read();

			if (c == 95) {
				return 32;
			}

			if (c == 61) {
				return DecodeUtils.parseInt((byte) this.in.read(),
						(byte) this.in.read());
			}

			return c;
		}

		/**
		 * @see java.io.FilterInputStream#read(byte[], int, int)
		 */
		@Override
		public int read(byte[] buf, int off, int len) throws IOException {
			int i = 0;

			while (true) {
				if (i >= len) {
					break;
				}

				int c = this.read();
				if (c == -1) {
					if (i == 0) {
						i = -1;
					}
					break;
				}

				buf[off + i] = (byte) c;
				i++;
			}

			return i;
		}
	}

	private static final class RFC2047Decoder {

		/**
		 * デコードを行います。
		 * 
		 * @param source
		 *            ソース
		 * @return デコード結果
		 * 
		 * @throws IOException
		 *             デコードに失敗した場合
		 */
		static String decode(String source) throws IOException {
			return new RFC2047Decoder(source).get();
		}

		private StringBuilder buf;

		private int endIndex;

		private byte[] pooledBytes;

		private String pooledCES;

		private int pos = 0;

		private String source;

		private int startIndex;

		private RFC2047Decoder(String source) throws IOException {
			this.source = source;
			this.buf = new StringBuilder(source.length());
			this.parse();
		}

		private String get() {
			return this.buf.toString();
		}

		private boolean hasEncodedWord() {
			this.startIndex = this.source.indexOf("=?", this.pos);
			if (this.startIndex == -1) {
				return false;
			}
			this.endIndex = this.source.indexOf("?=", this.startIndex + 2);
			if (this.endIndex == -1) {
				return false;
			}

			this.endIndex += 2;
			return true;
		}

		private void parse() throws IOException {
			while (this.hasEncodedWord()) {
				String work = this.source.substring(this.pos, this.startIndex);
				if (DecodeUtils.indexOfNonLWSP(work, 0, false) > -1) {
					this.sweepPooledBytes();
					this.buf.append(work);
				}
				this.parseWord();
			}
			this.sweepPooledBytes();
			this.buf.append(this.source.substring(this.pos));
		}

		private void parseWord() {
			try {
				int start = this.startIndex + 2;
				int end = this.source.indexOf('?', start);
				if (end == this.endIndex - 2) {
					throw new RuntimeException();
				}

				String ces = this.source.substring(start, end);
				if (!Charset.isSupported(ces)) {
					ces = "JISAutoDetect";
				}
				start = end + 1;
				end = this.source.indexOf('?', start);
				if (end == this.endIndex - 2) {
					throw new RuntimeException();
				}

				String tes = this.source.substring(start, end);
				byte[] bytes = DecodeUtils.decodeByTES(this.source.substring(
						end + 1, this.endIndex - 2), tes);

				if (ces.equals(this.pooledCES)) {
					byte[] word = new byte[this.pooledBytes.length
							+ bytes.length];
					System.arraycopy(this.pooledBytes, 0, word, 0,
							this.pooledBytes.length);
					System.arraycopy(bytes, 0, word, this.pooledBytes.length,
							bytes.length);
					this.pooledBytes = word;
				} else {
					this.sweepPooledBytes();
					this.pooledCES = ces;
					this.pooledBytes = bytes;
				}
			} catch (Exception e) {
				this.buf.append(this.source.substring(this.startIndex,
						this.endIndex));
			}

			this.pos = this.endIndex;
		}

		private void sweepPooledBytes() throws IOException {
			if (this.pooledBytes == null) {
				return;
			}
			this.buf.append(new String(this.pooledBytes, this.pooledCES));
			this.pooledCES = null;
			this.pooledBytes = null;
		}
	}

	private static byte[] decodeByTES(String str, String tes)
			throws IOException {
		StringBuilder builder = new StringBuilder(str);
		int n;
		while ((n = DecodeUtils.indexOfLWSP(builder, 0, false, (char) 0)) > -1) {
			builder.deleteCharAt(n);
		}

		int len = builder.length();
		if (tes.equalsIgnoreCase("B") && len % 4 != 0) {
			switch (4 - len % 4) {
			case 1:
				builder.append('=');
				break;
			case 2:
				builder.append("==");
				break;
			case 3:
				if (builder.charAt(len - 1) != '=') {
					builder.append("===");
				} else {
					builder.deleteCharAt(len - 1);
				}
				break;
			default:
				break;
			}
		}

		if (tes.equalsIgnoreCase("B")) {
			return Base64.decode(builder.toString().toCharArray());
		} else if (tes.equalsIgnoreCase("Q")) {
			byte[] bytes = DecodeUtils.getBytes(builder.toString());
			ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			StreamUtils.transfer(new QDecoderStream(bis), baos);

			return baos.toByteArray();
		} else {
			throw new UnsupportedEncodingException(tes);
		}
	}

	/**
	 * JAVADOC
	 * 
	 * @param source
	 * @return 1
	 * @throws IOException
	 */
	static String decodeText(String source) throws IOException {
		if (source == null) {
			return null;
		}

		if (source.indexOf('\u001b') >= 0) {
			return new String(source.getBytes("ISO-8859-1"), "ISO-2022-JP");
		}

		String decodedText = RFC2047Decoder.decode(source);
		if (decodedText.indexOf('\u001b') >= 0) {
			return new String(decodedText.getBytes("ISO-8859-1"), "ISO-2022-JP");
		}

		return decodedText;
	}

	private static byte[] getBytes(String str) {
		char[] chars = str.toCharArray();
		byte[] bytes = new byte[chars.length];

		for (int i = 0; i < chars.length; i++) {
			bytes[i] = (byte) chars[i];
		}

		return bytes;
	}

	private static int indexOfLWSP(CharSequence source, int startIndex,
			boolean decrease, char additionalDelimiter) {
		int inc = decrease ? -1 : 1;
		int len = source.length();

		for (int i = startIndex; i >= 0 && i < len; i += inc) {
			char c = source.charAt(i);
			if (DecodeUtils.isLWSP(c) || c == additionalDelimiter) {
				return i;
			}
		}

		return -1;
	}

	private static int indexOfNonLWSP(CharSequence source, int startIndex,
			boolean decrease) {
		int inc = decrease ? -1 : 1;
		int len = source.length();

		for (int i = startIndex; i >= 0 && i < len; i += inc) {
			char c = source.charAt(i);
			if (!DecodeUtils.isLWSP(c)) {
				return i;
			}
		}

		return -1;
	}

	private static boolean isLWSP(char c) {
		return c == '\r' || c == '\n' || c == ' ' || c == '\t';
	}

	private static int parseInt(byte b1, byte b2) throws NumberFormatException {
		int result = 0;
		int digit = Character.digit((char) b1, 16);
		if (digit < 0) {
			return -1;
		}
		result = -digit;

		digit = Character.digit((char) b2, 16);
		if (digit < 0) {
			return -1;
		}
		result *= 16;
		result -= digit;

		return -result;
	}

	private DecodeUtils() {
	}
}
