/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.util.Properties;

import com.wankuma.commons.lang.Assertion;
import com.wankuma.mail.MailSender;
import com.wankuma.mail.MailSenderFactory;

/**
 * 日本語処理に対応したメール送信が可能な{@link MailSenderFactory}の実装です。<br>
 * このファクトリによって生成されるインスタンスは、<a
 * href="http://java.sun.com/products/javamail/">Java Mail</a>を利用しています。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public class JapaneseMailSenderFactory implements MailSenderFactory {

	/**
	 * 標準のSTMPのポートを示します。
	 */
	public static final int DEFAULT_SMTP_PORT = 25;

	/**
	 * @see com.wankuma.mail.MailSenderFactory#getMailSender(java.util.Properties)
	 */
	@Override
	public MailSender getMailSender(Properties properties) {
		Assertion.notNull("properties", properties);

		return new JapaneseMailSender(properties);
	}

	/**
	 * @see com.wankuma.mail.MailSenderFactory#getMailSender(java.lang.String)
	 */
	@Override
	public MailSender getMailSender(String host) {
		Assertion.notNull("host", host);

		return this.getMailSender(host, -1);
	}

	/**
	 * @see com.wankuma.mail.MailSenderFactory#getMailSender(java.lang.String,
	 *      int)
	 */
	@Override
	public MailSender getMailSender(String host, int port) {
		Assertion.notNull("host", host);

		int port0 = port;
		if (port0 < 0 || port0 > 65535) {
			port0 = JapaneseMailSenderFactory.DEFAULT_SMTP_PORT;
		}

		Properties properties = new Properties();
		properties.setProperty("mail.smtp.host", host);
		properties.setProperty("mail.smtp.port", Integer.toString(port0));

		return this.getMailSender(properties);
	}
}
