/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail;

import java.util.Date;

/**
 * メールを表現します。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public interface Mail {

	/**
	 * 添付ファイルを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。
	 * 
	 * @return 添付ファイルの配列
	 */
	AttachmentFile[] getAttachmentFiles();

	/**
	 * BCCで送信するメールアドレスを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。<br>
	 * {@link #getEnvelopeTo()}で取得可能なメールアドレスが存在する場合、実際の送信先としては利用されません。
	 * 
	 * @return メールアドレスの配列
	 */
	MailAddress[] getBcc();

	/**
	 * メール本文を返します。
	 * 
	 * @return メール本文
	 */
	MailBody getBody();

	/**
	 * CCで送信するメールアドレスを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。<br>
	 * {@link #getEnvelopeTo()}で取得可能なメールアドレスが存在する場合、実際の送信先としては利用されません。
	 * 
	 * @return メールアドレスの配列
	 */
	MailAddress[] getCc();

	/**
	 * 実際の送信先のメールアドレスを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。<br>
	 * このメソッドが空ではない配列を返す場合、メールの配信先として使われ、<br>
	 * {@link #getBcc()}、{@link #getCc()}、{@link #getTo()}にて取得可能なメールアドレス宛には送信されません。
	 * 
	 * @return 実際の送信先のメールアドレスの配列
	 */
	String[] getEnvelopeTo();

	/**
	 * 送信元のメールアドレスを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。
	 * 
	 * @return 送信元のメールアドレスの配列
	 */
	MailAddress[] getFrom();

	/**
	 * メールヘッダを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。
	 * 
	 * @return メールヘッダの配列
	 */
	Header[] getHeaders();

	/**
	 * 重要度を返します。<br>
	 * 指定されていない場合は{@code null}を返します。
	 * 
	 * @return 重要度
	 */
	Importance getImportance();

	/**
	 * メッセージIDのジェネレータを返します。<br>
	 * 存在しない場合は{@code null}を返します。
	 * 
	 * @return メッセージIDのジェネレータ
	 */
	MessageIdGenerator getMessageIdGenerator();

	/**
	 * 返信先のメールアドレスを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。
	 * 
	 * @return メールアドレスの配列
	 */
	MailAddress[] getReplayTo();

	/**
	 * 配信エラー時等に利用されるメールの返却先のメールアドレスを返します。
	 * 
	 * @return メールアドレス
	 */
	String getReturnPath();

	/**
	 * 送信日時を返します。
	 * 
	 * @return 送信日時
	 */
	Date getSentDate();

	/**
	 * 件名を返します。<br>
	 * 存在しない場合は{@code null}を返します。
	 * 
	 * @return 件名
	 */
	String getSubject();

	/**
	 * TOで送信するメールアドレスを配列で返します。<br>
	 * 存在しない場合は空の配列を返します。<br>
	 * {@link #getEnvelopeTo()}で取得可能なメールアドレスが存在する場合、実際の送信先としては利用されません。
	 * 
	 * @return メールアドレスの配列
	 */
	MailAddress[] getTo();
}
