/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PushbackInputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.Message;
import javax.mail.MessageAware;
import javax.mail.MessageContext;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.ContentType;
import javax.mail.internet.HeaderTokenizer;
import javax.mail.internet.ParseException;

import com.wankuma.commons.io.StreamUtils;
import com.wankuma.commons.util.Base64;
import com.wankuma.mail.MailException;

/**
 * メールのデコードをサポートするユーティリティです。<br>
 * <dl>
 * <dt>謝辞</dt>
 * <dd> このクラスは<a href="http://www.sk-jp.com/java/library/index.html">Shin
 * Kinoshita's Home</a>にて提供されている、 <a
 * href="http://www.sk-jp.com/java/library/skutility/index.html">Utilities</a>及び、<a
 * href="http://www.sk-jp.com/java/library/skmail/index.html">Message Access
 * Utilities</a>から、<br>
 * 一部のコードを利用させていただいております。 Shin Kinoshita氏には深く感謝いたします。</dd>
 * </dl>
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */

public final class DecodeUtils {

	/**
	 * 添付ファイルのハンドラです。
	 * 
	 * @author Katsunori Koyanagi
	 * @version 1.0
	 */
	static interface AttachmentHandler {

		/**
		 * 添付ファイル示すパートをハンドルします。
		 * 
		 * @param fileName
		 *            ファイル名
		 * @param part
		 *            添付ファイルを示すパート
		 * @throws IOException
		 *             入出力処理に失敗した場合
		 * @throws MessagingException
		 *             メッセージの処理に失敗した場合
		 */
		void handleAttachment(String fileName, Part part) throws IOException,
				MessagingException;
	}

	private static class AttachmentsExtractor implements PartHandler {

		private AttachmentHandler handler;

		private int index;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param handler
		 *            ハンドラ
		 */
		AttachmentsExtractor(AttachmentHandler handler) {
			this.handler = handler;
		}

		private String getFileName(Part part) throws MessagingException,
				IOException {
			String name = DecodeUtils.getFileName(part);
			if (name == null) {
				if (part.isMimeType("message/*")) {
					name = "message" + this.index + ".eml";
				} else {
					name = "file" + this.index + ".tmp";
				}
			}

			return name;
		}

		/**
		 * @see com.wankuma.mail.javamail.DecodeUtils.PartHandler#processPart(javax.mail.Part,
		 *      javax.mail.internet.ContentType)
		 */
		public boolean processPart(Part part, ContentType context)
				throws MessagingException, IOException {
			if (part.getContentType().indexOf("application/applefile") != -1) {
				return true;
			}

			if (part.isMimeType("message/*")) {
				return true;
			}
			if (DecodeUtils.getFileName(part) == null) {
				return true;
			}

			this.index++;
			String fileName = this.getFileName(part);
			this.handler.handleAttachment(fileName, part);
			return true;
		}
	}

	private static class CorrectedContentTypeDataSource implements DataSource,
			MessageAware {

		private String defaultCharset;

		private String forceCharset;

		private DataSource source;

		/**
		 * インスタンスを構築します。
		 */
		CorrectedContentTypeDataSource() {
		}

		/**
		 * @see javax.activation.DataSource#getContentType()
		 */
		public String getContentType() {
			ContentType contentType = null;
			try {
				contentType = new ContentType(this.source.getContentType());
			} catch (ParseException e) {
				return "text/plain; charset=" + this.defaultCharset;
			}
			String specifiedCharset = contentType.getParameter("charset");
			if (specifiedCharset == null) {
				contentType.setParameter("charset", this.defaultCharset);
			} else if (this.forceCharset != null) {
				contentType.setParameter("charset", this.forceCharset);
			}

			String charset = contentType.getParameter("charset");
			if (DecodeUtils.canChangeMapping(charset)) {
				contentType.setParameter("charset", DecodeUtils.MAP_ENCODING);
			}

			return contentType.toString();
		}

		/**
		 * @see javax.activation.DataSource#getInputStream()
		 */
		public InputStream getInputStream() throws IOException {
			return this.source.getInputStream();
		}

		/**
		 * @see javax.mail.MessageAware#getMessageContext()
		 */
		public synchronized MessageContext getMessageContext() {
			if (this.source instanceof MessageAware) {
				return ((MessageAware) this.source).getMessageContext();
			}

			throw new MailException(this.source + " isn't MessageAware.");
		}

		/**
		 * @see javax.activation.DataSource#getName()
		 */
		public String getName() {
			return this.source.getName();
		}

		/**
		 * @see javax.activation.DataSource#getOutputStream()
		 */
		public OutputStream getOutputStream() throws IOException {
			return this.source.getOutputStream();
		}

		private void setDataSource(DataSource newSource) {
			this.source = newSource;
		}

		/**
		 * 標準の文字セットを設定します。
		 * 
		 * @param defaultCharset
		 *            文字セット
		 */
		void setDefaultCharset(String defaultCharset) {
			this.defaultCharset = defaultCharset;
		}

		/**
		 * 指定された文字コードで既存の文字コードを上書きします。
		 * 
		 * @param forceCharset
		 *            強制的に適用する文字コード
		 */
		void setForceCharset(String forceCharset) {
			this.forceCharset = forceCharset;
		}

		/**
		 * パートを設定します。
		 * 
		 * @param part
		 *            パート
		 * @throws MessagingException
		 *             メッセージの処理に失敗した場合
		 */
		void setPart(Part part) throws MessagingException {
			this.setDataSource(part.getDataHandler().getDataSource());
		}
	}

	private static class Encoding {

		private String encoding = "us-ascii";

		private String lang = "";

		/**
		 * インスタンスを構築します。
		 */
		Encoding() {
		}

		/**
		 * エンコーディングを返します。
		 * 
		 * @return エンコーディング
		 */
		String getEncoding() {
			return this.encoding;
		}

		/**
		 * 言語を返します。
		 * 
		 * @return 言語
		 */
		String getLang() {
			return this.lang;
		}

		/**
		 * エンコーディングを設定します。
		 * 
		 * @param encoding
		 *            エンコーディング
		 */
		void setEncoding(String encoding) {
			this.encoding = encoding;
		}

		/**
		 * 言語を設定します。
		 * 
		 * @param lang
		 *            言語
		 */
		void setLang(String lang) {
			this.lang = lang;
		}
	}

	private static class HtmlPartExtractor implements PartHandler {

		private StringBuilder html = null;

		/**
		 * HTMLを返します。
		 * 
		 * @return HTML
		 */
		String getHtml() {
			return this.html == null ? null : this.html.toString();
		}

		/**
		 * @see com.wankuma.mail.javamail.DecodeUtils.PartHandler#processPart(javax.mail.Part,
		 *      javax.mail.internet.ContentType)
		 */
		public boolean processPart(Part part, ContentType context)
				throws MessagingException, IOException {
			if (!part.isMimeType("text/html")) {
				return true;
			}
			if (this.html == null) {
				this.html = new StringBuilder((String) DecodeUtils
						.getContent(part));
			} else {
				String disposition = part.getDisposition();
				if (disposition == null
						|| disposition.equalsIgnoreCase(Part.INLINE)) {
					this.html.append("\r\n\r\n-- inline --\r\n\r\n");
					this.html.append(DecodeUtils.getContent(part));
				}
			}
			return true;
		}

	}

	private static interface PartHandler {

		/**
		 * パートの処理結果をハンドルして処理を行います。
		 * 
		 * @param part
		 *            パート
		 * @param context
		 *            コンテキスト
		 * @return 処理結果
		 * 
		 * @throws MessagingException
		 *             メッセージの処理に失敗した場合
		 * @throws IOException
		 *             入出力処理に失敗した場合
		 */
		boolean processPart(Part part, ContentType context)
				throws MessagingException, IOException;
	}

	private static class PlainPartExtractor implements PartHandler {

		private StringBuilder text = null;

		/**
		 * テキストを返します。
		 * 
		 * @return テキスト
		 */
		String getText() {
			return this.text == null ? null : this.text.toString();
		}

		/**
		 * @see com.wankuma.mail.javamail.DecodeUtils.PartHandler#processPart(javax.mail.Part,
		 *      javax.mail.internet.ContentType)
		 */
		public boolean processPart(Part part, ContentType context)
				throws MessagingException, IOException {
			if (!part.isMimeType("text/plain")) {
				return true;
			}
			if (this.text == null) {
				this.text = new StringBuilder((String) DecodeUtils
						.getContent(part));
			} else {
				String disposition = part.getDisposition();
				if (disposition == null
						|| disposition.equalsIgnoreCase(Part.INLINE)) {
					this.text.append("\r\n\r\n-- inline --\r\n\r\n");
					this.text.append(DecodeUtils.getContent(part));
				}
			}
			return true;
		}
	}

	private static class QDecoderStream extends FilterInputStream {

		/**
		 * インスタンスを構築します。
		 * 
		 * @param in
		 *            入力ストリーム
		 */
		QDecoderStream(InputStream in) {
			super(new PushbackInputStream(in, 2));
		}

		/**
		 * @see java.io.FilterInputStream#read()
		 */
		@Override
		public int read() throws IOException {
			int c = this.in.read();

			if (c == 95) {
				return 32;
			}

			if (c == 61) {
				return DecodeUtils.parseInt((byte) this.in.read(),
						(byte) this.in.read());
			}

			return c;
		}

		/**
		 * @see java.io.FilterInputStream#read(byte[], int, int)
		 */
		@Override
		public int read(byte[] buf, int off, int len) throws IOException {
			int i = 0;

			while (true) {
				if (i >= len) {
					break;
				}

				int c = this.read();
				if (c == -1) {
					if (i == 0) {
						i = -1;
					}
					break;
				}

				buf[off + i] = (byte) c;
				i++;
			}

			return i;
		}
	}

	private static final class RFC2047Decoder {

		/**
		 * デコードを行います。
		 * 
		 * @param source
		 *            ソース
		 * @return デコード結果
		 * 
		 * @throws IOException
		 *             デコードに失敗した場合
		 */
		static String decode(String source) throws IOException {
			return new RFC2047Decoder(source).get();
		}

		private StringBuilder buf;

		private int endIndex;

		private byte[] pooledBytes;

		private String pooledCES;

		private int pos = 0;

		private String source;

		private int startIndex;

		private RFC2047Decoder(String source) throws IOException {
			this.source = source;
			this.buf = new StringBuilder(source.length());
			this.parse();
		}

		private String get() {
			return this.buf.toString();
		}

		private boolean hasEncodedWord() {
			this.startIndex = this.source.indexOf("=?", this.pos);
			if (this.startIndex == -1) {
				return false;
			}
			this.endIndex = this.source.indexOf("?=", this.startIndex + 2);
			if (this.endIndex == -1) {
				return false;
			}

			this.endIndex = this.endIndex + 2;
			return true;
		}

		private void parse() throws IOException {
			while (this.hasEncodedWord()) {
				String work = this.source.substring(this.pos, this.startIndex);
				if (DecodeUtils.indexOfNonLWSP(work, 0, false) > -1) {
					this.sweepPooledBytes();
					this.buf.append(work);
				}
				this.parseWord();
			}
			this.sweepPooledBytes();
			this.buf.append(this.source.substring(this.pos));
		}

		private void parseWord() {
			boolean thrown = false;

			try {
				int start = this.startIndex + 2;
				int end = this.source.indexOf('?', start);
				if (end == this.endIndex - 2) {
					thrown = true;
				} else {
					String ces = this.source.substring(start, end);
					if (!Charset.isSupported(ces)) {
						ces = "JISAutoDetect";
					}
					start = end + 1;
					end = this.source.indexOf('?', start);
					if (end == this.endIndex - 2) {
						thrown = true;
					} else {
						String tes = this.source.substring(start, end);
						byte[] bytes = DecodeUtils.decodeByTES(this.source
								.substring(end + 1, this.endIndex - 2), tes);

						if (ces.equals(this.pooledCES)) {
							byte[] word = new byte[this.pooledBytes.length
									+ bytes.length];
							System.arraycopy(this.pooledBytes, 0, word, 0,
									this.pooledBytes.length);
							System.arraycopy(bytes, 0, word,
									this.pooledBytes.length, bytes.length);
							this.pooledBytes = word;
						} else {
							this.sweepPooledBytes();
							this.pooledCES = ces;
							this.pooledBytes = bytes;
						}
					}
				}
			} catch (Exception e) {
				thrown = true;
			}

			if (thrown) {
				this.buf.append(this.source.substring(this.startIndex,
						this.endIndex));
			}
			this.pos = this.endIndex;
		}

		private void sweepPooledBytes() throws IOException {
			if (this.pooledBytes == null) {
				return;
			}

			String value = null;
			if (DecodeUtils.canChangeMapping(this.pooledCES)) {
				value = new String(this.pooledBytes, DecodeUtils.MAP_ENCODING);
			}
			if (value == null) {
				value = new String(this.pooledBytes, this.pooledCES);
			}

			this.buf.append(value);
			this.pooledCES = null;
			this.pooledBytes = null;
		}
	}

	private static DataHandler correctedDataHandler;

	private static CorrectedContentTypeDataSource correctedDataSource = new CorrectedContentTypeDataSource();

	private static final String JIS = "ISO-2022-JP";

	private static final String LATIN1 = "ISO-8859-1";

	private static final String MAP_ENCODING = "x-windows-iso2022jp";

	private static boolean supportWindowsIso2022JP;

	static {
		DecodeUtils.correctedDataHandler = new DataHandler(
				DecodeUtils.correctedDataSource);
		DecodeUtils.supportWindowsIso2022JP = Charset
				.isSupported(DecodeUtils.MAP_ENCODING);
	}

	private static boolean canChangeMapping(String charset) {
		if (charset == null) {
			return false;
		}
		if (!DecodeUtils.supportWindowsIso2022JP) {
			return false;
		}

		if (!Charset.isSupported(charset)) {
			return false;
		}

		Charset c = Charset.forName(DecodeUtils.JIS);
		if (c.name().equalsIgnoreCase(charset)) {
			return true;
		}
		for (String alias : c.aliases()) {
			if (alias.equalsIgnoreCase(charset)) {
				return true;
			}
		}

		return false;
	}

	private static void checkType(HeaderTokenizer.Token token)
			throws ParseException {
		int t = token.getType();
		if (t != HeaderTokenizer.Token.ATOM
				&& t != HeaderTokenizer.Token.QUOTEDSTRING) {
			throw new ParseException("Illegal token : " + token.getValue());
		}
	}

	private static byte[] decodeByTES(String str, String tes)
			throws IOException {
		StringBuilder builder = new StringBuilder(str);
		int n;
		while ((n = DecodeUtils.indexOfLWSP(builder, 0, false, (char) 0)) > -1) {
			builder.deleteCharAt(n);
		}

		int len = builder.length();
		if (tes.equalsIgnoreCase("B") && len % 4 != 0) {
			switch (4 - len % 4) {
			case 1:
				builder.append('=');
				break;
			case 2:
				builder.append("==");
				break;
			case 3:
				if (builder.charAt(len - 1) != '=') {
					builder.append("===");
				} else {
					builder.deleteCharAt(len - 1);
				}
				break;
			default:
				break;
			}
		}

		if (tes.equalsIgnoreCase("B")) {
			return Base64.decode(builder.toString().toCharArray());
		} else if (tes.equalsIgnoreCase("Q")) {
			byte[] bytes = DecodeUtils.getBytes(builder.toString());
			ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			StreamUtils.transfer(new QDecoderStream(bis), baos);

			return baos.toByteArray();
		} else {
			throw new UnsupportedEncodingException(tes);
		}
	}

	private static String decodeParameterSpciallyJapanese(String s)
			throws IOException {
		return DecodeUtils.decodeText(new String(
				s.getBytes(DecodeUtils.LATIN1), "JISAutoDetect"));
	}

	private static String decodeRFC2231(String s, Encoding encoding,
			boolean isInitialSection) throws ParseException,
			UnsupportedEncodingException {
		StringBuilder builder = new StringBuilder();
		int i = 0;
		if (isInitialSection) {
			int work = s.indexOf('\'');
			if (work > 0) {
				encoding.setEncoding(s.substring(0, work));
				work++;
				i = s.indexOf('\'', work);
				if (i < 0) {
					throw new ParseException("lang tag area was missing.");
				}
				encoding.setLang(s.substring(work, i));
				i++;
			}
		}
		try {
			for (; i < s.length(); i++) {
				if (s.charAt(i) == '%') {
					builder.append((char) Integer.parseInt(s.substring(i + 1,
							i + 3), 16));
					i = i + 2;
					continue;
				}
				builder.append(s.charAt(i));
			}
			return new String(new String(builder).getBytes(DecodeUtils.LATIN1),
					encoding.getEncoding());
		} catch (IndexOutOfBoundsException e) {
			throw new ParseException(s + " :: this string were not decoded.");
		}
	}

	/**
	 * 指定のテキストのデコードを行います。
	 * 
	 * @param text
	 *            テキスト
	 * @return デコード結果
	 * 
	 * @throws IOException
	 *             デコード処理に失敗した場合
	 */
	public static String decodeText(String text) throws IOException {
		if (text == null) {
			return null;
		}

		if (text.indexOf('\u001b') >= 0) {
			try {
				return new String(text.getBytes(DecodeUtils.LATIN1),
						DecodeUtils.MAP_ENCODING);
			} catch (UnsupportedEncodingException e) {
				return new String(text.getBytes(DecodeUtils.LATIN1),
						DecodeUtils.JIS);
			}
		}

		String decodedText = RFC2047Decoder.decode(text);
		if (decodedText.indexOf('\u001b') >= 0) {
			try {
				return new String(decodedText.getBytes(DecodeUtils.LATIN1),
						DecodeUtils.MAP_ENCODING);
			} catch (UnsupportedEncodingException e) {
				return new String(decodedText.getBytes(DecodeUtils.LATIN1),
						DecodeUtils.JIS);
			}
		}

		return decodedText;
	}

	/**
	 * 指定のメッセージを解析して添付ファイルを指定のハンドラへ渡します。
	 * 
	 * @param message
	 *            メッセージ
	 * @param handler
	 *            ハンドラ
	 * @throws IOException
	 *             入出力処理に失敗した場合
	 * @throws MessagingException
	 *             メッセージの処理に失敗した場合
	 */
	static void getAttachmentFiles(Message message, AttachmentHandler handler)
			throws IOException, MessagingException {
		AttachmentsExtractor extractor = new AttachmentsExtractor(handler);
		DecodeUtils.process(message, extractor);
	}

	private static byte[] getBytes(String str) {
		char[] chars = str.toCharArray();
		byte[] bytes = new byte[chars.length];

		for (int i = 0; i < chars.length; i++) {
			bytes[i] = (byte) chars[i];
		}

		return bytes;
	}

	private static Object getContent(Part part) throws MessagingException,
			IOException {
		synchronized (DecodeUtils.correctedDataSource) {
			DecodeUtils.correctedDataSource.setPart(part);
			try {
				DecodeUtils.correctedDataSource
						.setDefaultCharset(DecodeUtils.JIS);
				return DecodeUtils.correctedDataHandler.getContent();
			} catch (UnsupportedEncodingException e) {
				DecodeUtils.correctedDataSource
						.setForceCharset(DecodeUtils.JIS);
				return DecodeUtils.correctedDataHandler.getContent();
			}
		}
	}

	private static String getFileName(Part part) throws MessagingException,
			IOException {
		String[] disposition = part.getHeader("Content-Disposition");
		String filename = null;

		if (disposition != null && disposition.length >= 0) {
			filename = DecodeUtils.getParameter(disposition[0], "filename");
		}

		if (disposition == null || disposition.length < 1 || filename == null) {
			filename = part.getFileName();
			if (filename != null) {
				return DecodeUtils.decodeParameterSpciallyJapanese(filename);
			}

			return null;
		}

		return filename;
	}

	/**
	 * 指定のメッセージからHTMLを抽出して返します。<br>
	 * 存在しない場合は{@code null}を返します。
	 * 
	 * @param message
	 *            メッセージ
	 * @return HTML
	 * 
	 * @throws MessagingException
	 *             メッセージの処理に失敗した場合
	 * @throws IOException
	 *             入出力処理に失敗した場合
	 */
	static String getHtmlContent(Message message) throws MessagingException,
			IOException {
		HtmlPartExtractor htmlPartExtractor = new HtmlPartExtractor();
		DecodeUtils.process(message, htmlPartExtractor);

		return htmlPartExtractor.getHtml();
	}

	private static String getParameter(String header, String name)
			throws ParseException, IOException {
		if (header == null) {
			return null;
		}

		String header1 = DecodeUtils.decodeParameterSpciallyJapanese(header);
		HeaderTokenizer tokenizer = new HeaderTokenizer(header1, ";=\t ", true);
		HeaderTokenizer.Token token;
		StringBuilder builder = new StringBuilder();
		Encoding encoding = new Encoding();
		String n;
		String v;

		while (true) {
			token = tokenizer.next();
			if (token.getType() == HeaderTokenizer.Token.EOF) {
				break;
			}
			if (token.getType() != ';') {
				continue;
			}
			token = tokenizer.next();
			DecodeUtils.checkType(token);
			n = token.getValue();
			token = tokenizer.next();
			if (token.getType() != '=') {
				throw new ParseException("Illegal token : " + token.getValue());
			}
			token = tokenizer.next();
			DecodeUtils.checkType(token);
			v = token.getValue();
			if (n.equalsIgnoreCase(name)) {
				return v;
			}
			int index = name.length();
			if (!n.startsWith(name) || n.charAt(index) != '*') {
				continue;
			}
			int lastIndex = n.length() - 1;
			if (n.charAt(lastIndex) == '*') {
				if (index == lastIndex || n.charAt(index + 1) == '0') {
					builder
							.append(DecodeUtils
									.decodeRFC2231(v, encoding, true));
				} else {
					builder.append(DecodeUtils
							.decodeRFC2231(v, encoding, false));
				}
			} else {
				builder.append(v);
			}
			if (index == lastIndex) {
				break;
			}
		}

		if (builder.length() == 0) {
			return null;
		}
		return builder.toString();
	}

	/**
	 * 指定のメッセージからプレーンテキストを抽出して返します。<br>
	 * 存在しない場合は{@code null}を返します。
	 * 
	 * @param message
	 *            メッセージ
	 * @return プレーンテキスト
	 * 
	 * @throws MessagingException
	 *             メッセージの処理に失敗した場合
	 * @throws IOException
	 *             入出力処理に失敗した場合
	 */

	static String getTextContent(Message message) throws MessagingException,
			IOException {
		PlainPartExtractor plainPartExtractor = new PlainPartExtractor();
		DecodeUtils.process(message, plainPartExtractor);

		return plainPartExtractor.getText();
	}

	private static int indexOfLWSP(CharSequence source, int startIndex,
			boolean decrease, char additionalDelimiter) {
		int inc = decrease ? -1 : 1;
		int len = source.length();

		for (int i = startIndex; i >= 0 && i < len; i = i + inc) {
			char c = source.charAt(i);
			if (DecodeUtils.isLWSP(c) || c == additionalDelimiter) {
				return i;
			}
		}

		return -1;
	}

	private static int indexOfNonLWSP(CharSequence source, int startIndex,
			boolean decrease) {
		int inc = decrease ? -1 : 1;
		int len = source.length();

		for (int i = startIndex; i >= 0 && i < len; i = i + inc) {
			char c = source.charAt(i);
			if (!DecodeUtils.isLWSP(c)) {
				return i;
			}
		}

		return -1;
	}

	private static boolean isLWSP(char c) {
		return c == '\r' || c == '\n' || c == ' ' || c == '\t';
	}

	private static int parseInt(byte b1, byte b2) throws NumberFormatException {
		int result = 0;
		int digit = Character.digit((char) b1, 16);
		if (digit < 0) {
			return -1;
		}
		result = -digit;

		digit = Character.digit((char) b2, 16);
		if (digit < 0) {
			return -1;
		}
		result = result * 16;
		result = result - digit;

		return -result;
	}

	private static void process(Part part, PartHandler handler)
			throws MessagingException, IOException {
		DecodeUtils.process(part, handler, null);
	}

	private static boolean process(Part part, PartHandler handler,
			ContentType context) throws MessagingException, IOException {
		if (part.isMimeType("multipart/*")) {
			Multipart mp = (Multipart) part.getContent();
			ContentType cType = new ContentType(part.getContentType());

			for (int i = 0; i < mp.getCount(); i++) {
				if (!DecodeUtils.process(mp.getBodyPart(i), handler, cType)) {
					return false;
				}
			}
			return true;
		}
		return handler.processPart(part, context);
	}

	private DecodeUtils() {
	}
}
