/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.Flags.Flag;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import com.wankuma.commons.lang.NumberUtils;
import com.wankuma.mail.AttachmentFile;
import com.wankuma.mail.Header;
import com.wankuma.mail.Importance;
import com.wankuma.mail.Mail;
import com.wankuma.mail.MailAddress;
import com.wankuma.mail.MailBody;
import com.wankuma.mail.MailException;
import com.wankuma.mail.MailReceiverProtocol;
import com.wankuma.mail.MessageIdGenerator;
import com.wankuma.mail.ReceivedHeader;
import com.wankuma.mail.ReceivedMail;
import com.wankuma.mail.ReceivedMailInfo;
import com.wankuma.mail.helper.SimpleHeader;
import com.wankuma.mail.helper.SimpleMail;
import com.wankuma.mail.helper.SimpleMailAddress;

/**
 * 日本語処理に対応した{@link com.wankuma.mail.MailReceiver}の実装です。<br>
 * 内部では<a href="http://java.sun.com/products/javamail/">Java Mail</a>を利用しています。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
class JapaneseMailReceiver extends AbstractJavaMailMailReceiver {

	private static class EmptyMailBody implements MailBody {

		/**
		 * インスタンスを構築します。
		 */
		EmptyMailBody() {
		}

		/**
		 * @see com.wankuma.mail.MailBody#getContentType()
		 */
		@Override
		public String getContentType() {
			return "application/octet-stream";
		}

		/**
		 * @see com.wankuma.mail.MailBody#getHeaders()
		 */
		@Override
		public Header[] getHeaders() {
			return new Header[0];
		}

		/**
		 * @see com.wankuma.mail.MailBody#getInputStream()
		 */
		@Override
		public InputStream getInputStream() throws IOException {
			return new ByteArrayInputStream(new byte[0]);
		}

		/**
		 * @see com.wankuma.mail.MailBody#getMessage()
		 */
		@Override
		public String getMessage() {
			return "";
		}
	}

	private static class InternetAddress2 extends InternetAddress {

		private static final long serialVersionUID = -5530942441368910856L;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param address
		 *            アドレス
		 * @throws AddressException
		 *             アドレスの解釈に失敗した場合
		 */
		InternetAddress2(InternetAddress address) throws AddressException {
			super(address.toString());
		}

		/**
		 * @see javax.mail.internet.InternetAddress#getPersonal()
		 */
		@Override
		public String getPersonal() {
			try {
				return DecodeUtils.decodeText(this.encodedPersonal);
			} catch (IOException e) {
				return null;
			}
		}
	}

	private static class PartAttachmentFile implements AttachmentFile {

		private String fileName;

		private Part part;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param fileName
		 *            ファイル名
		 * @param part
		 *            パート
		 */
		PartAttachmentFile(String fileName, Part part) {
			this.part = part;
			this.fileName = fileName;
		}

		/**
		 * @see com.wankuma.mail.AttachmentFile#getFileName()
		 */
		@Override
		public String getFileName() {
			return this.fileName;
		}

		/**
		 * @see com.wankuma.mail.AttachmentFile#getInputStream()
		 */
		@Override
		public InputStream getInputStream() throws IOException {
			try {
				return this.part.getInputStream();
			} catch (MessagingException e) {
				IOException e2 = new IOException();
				e2.initCause(e);
				throw e2;
			}
		}
	}

	private static class ReceivedHeaderImpl extends SimpleHeader implements
			ReceivedHeader {

		private static Pattern receivedHeaderPattern = Pattern.compile(
				"^from\\s+(\\S+?)\\s+.*by\\s+(\\S+?)\\s+.*$", Pattern.DOTALL);

		private String by;

		private String from;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param name
		 *            名前
		 * @param value
		 *            値
		 */
		ReceivedHeaderImpl(String name, String value) {
			super(name, value);

			Matcher matcher = ReceivedHeaderImpl.receivedHeaderPattern
					.matcher(value);
			if (matcher.matches()) {
				this.from = matcher.group(1);
				this.by = matcher.group(2);
			}
		}

		/**
		 * @see com.wankuma.mail.ReceivedHeader#getBy()
		 */
		@Override
		public String getBy() {
			return this.by;
		}

		/**
		 * @see com.wankuma.mail.ReceivedHeader#getFrom()
		 */
		@Override
		public String getFrom() {
			return this.from;
		}
	}

	private static class ReceivedMailImpl implements ReceivedMail {

		private Mail mail;

		private Date receivedDate;

		private ReceivedHeader[] receivedHeaders;

		private long size;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param mail
		 *            メール
		 * @param receivedHeaders
		 *            Receivedヘッダの配列
		 * @param receivedDate
		 *            受信日時
		 * @param size
		 *            サイズ
		 */
		ReceivedMailImpl(Mail mail, ReceivedHeader[] receivedHeaders,
				Date receivedDate, long size) {
			this.mail = mail;
			this.receivedHeaders = receivedHeaders;
			this.receivedDate = receivedDate;
			this.size = size;
		}

		/**
		 * @see com.wankuma.mail.Mail#getAttachmentFiles()
		 */
		public AttachmentFile[] getAttachmentFiles() {
			return this.mail.getAttachmentFiles();
		}

		/**
		 * @see com.wankuma.mail.Mail#getBcc()
		 */
		public MailAddress[] getBcc() {
			return this.mail.getBcc();
		}

		/**
		 * @see com.wankuma.mail.Mail#getBody()
		 */
		public MailBody getBody() {
			return this.mail.getBody();
		}

		/**
		 * @see com.wankuma.mail.Mail#getCc()
		 */
		public MailAddress[] getCc() {
			return this.mail.getCc();
		}

		/**
		 * @see com.wankuma.mail.Mail#getEnvelopeTo()
		 */
		public String[] getEnvelopeTo() {
			return this.mail.getEnvelopeTo();
		}

		/**
		 * @see com.wankuma.mail.Mail#getFrom()
		 */
		public MailAddress[] getFrom() {
			return this.mail.getFrom();
		}

		/**
		 * @see com.wankuma.mail.Mail#getHeaders()
		 */
		public Header[] getHeaders() {
			return this.mail.getHeaders();
		}

		/**
		 * @see com.wankuma.mail.Mail#getImportance()
		 */
		public Importance getImportance() {
			return this.mail.getImportance();
		}

		/**
		 * @see com.wankuma.mail.Mail#getMessageIdGenerator()
		 */
		public MessageIdGenerator getMessageIdGenerator() {
			return this.mail.getMessageIdGenerator();
		}

		/**
		 * @see com.wankuma.mail.ReceivedMail#getReceivedDate()
		 */
		@Override
		public Date getReceivedDate() {
			return (Date) this.receivedDate.clone();
		}

		/**
		 * @see com.wankuma.mail.ReceivedMail#getRecievedHeaders()
		 */
		@Override
		public ReceivedHeader[] getRecievedHeaders() {
			return this.receivedHeaders.clone();
		}

		/**
		 * @see com.wankuma.mail.Mail#getReplayTo()
		 */
		public MailAddress[] getReplayTo() {
			return this.mail.getReplayTo();
		}

		/**
		 * @see com.wankuma.mail.Mail#getReturnPath()
		 */
		public String getReturnPath() {
			return this.mail.getReturnPath();
		}

		/**
		 * @see com.wankuma.mail.Mail#getSentDate()
		 */
		public Date getSentDate() {
			return this.mail.getSentDate();
		}

		/**
		 * @see com.wankuma.mail.ReceivedMail#getSize()
		 */
		@Override
		public long getSize() {
			return this.size;
		}

		/**
		 * @see com.wankuma.mail.Mail#getSubject()
		 */
		public String getSubject() {
			return this.mail.getSubject();
		}

		/**
		 * @see com.wankuma.mail.Mail#getTo()
		 */
		public MailAddress[] getTo() {
			return this.mail.getTo();
		}
	}

	/**
	 * インスタンスを構築します。
	 * 
	 * @param protocol
	 * @param properties
	 */
	JapaneseMailReceiver(MailReceiverProtocol protocol, Properties properties) {
		super(protocol, properties);
	}

	/**
	 * @see com.wankuma.mail.javamail.AbstractJavaMailMailReceiver#convertMessage(javax.mail.Message)
	 */
	@Override
	protected ReceivedMailInfo convertMessage(final Message message)
			throws IOException, MessagingException {

		/**
		 * メール情報の遅延処理用スタブクラス
		 * 
		 * @author Katsunori Koyanagi
		 * @version 1.0
		 */
		class Info {

			private MailAddress from;

			private Date sentDate;

			private String subject;

			MailAddress getFrom() throws MessagingException, IOException {
				if (this.from == null) {
					String[] froms = message.getHeader("From");
					if (froms != null && froms.length > 0) {
						MailAddress[] addresses = JapaneseMailReceiver.this
								.parseMailAddress(froms[0]);
						if (addresses.length > 0) {
							this.from = addresses[0];
						}
					}
				}

				return this.from;
			}

			Date getSentDate() throws MessagingException {
				if (this.sentDate == null) {
					this.sentDate = message.getSentDate();
				}

				return this.sentDate;
			}

			String getSubject() throws IOException, MessagingException {
				if (this.subject == null) {
					String[] subjectHeader = message.getHeader("Subject");
					if (subjectHeader != null && subjectHeader.length > 0) {
						this.subject = DecodeUtils.decodeText(subjectHeader[0]);
					}
				}
				return this.subject;
			}
		}

		final int index = message.getMessageNumber();
		final Info info = new Info();

		return new ReceivedMailInfo() {

			private ReceivedMail mail;

			@Override
			public void delete() throws MailException {
				try {
					message.setFlag(Flag.DELETED, true);
				} catch (MessagingException e) {
					throw new MailException(e);
				}
			}

			@Override
			public MailAddress getFrom() {
				try {
					return info.getFrom();
				} catch (MessagingException e) {
					return null;
				} catch (IOException e) {
					return null;
				}
			}

			@Override
			public int getIndex() {
				return index;
			}

			@Override
			public ReceivedMail getMail() throws MailException {
				if (this.mail == null) {
					try {
						this.mail = JapaneseMailReceiver.this
								.convertMessage0(message);
					} catch (IOException e) {
						throw new MailException(e);
					} catch (MessagingException e) {
						throw new MailException(e);
					}
				}

				return this.mail;
			}

			@Override
			public Date getSentDate() {
				try {
					return info.getSentDate();
				} catch (MessagingException e) {
					return null;
				}
			}

			@Override
			public String getSubject() {
				try {
					return info.getSubject();
				} catch (IOException e) {
					return null;
				} catch (MessagingException e) {
					return null;
				}
			}
		};
	}

	private ReceivedMail convertMessage0(final Message message)
			throws IOException, MessagingException {
		ReceivedHeader[] receivedHeaders = this.getReceivedHeaders(message);
		Mail mail = this.toMail(message);
		Date receivedDate = this.getReceivedDate(message);
		long size = message.getSize();

		message.setFlag(Flag.SEEN, true);

		return new ReceivedMailImpl(mail, receivedHeaders, receivedDate, size);
	}

	private Date getReceivedDate(Message message) throws MessagingException,
			IOException {
		Date date = message.getReceivedDate();
		if (date == null) {
			date = new Date();
		}

		return date;
	}

	private ReceivedHeader[] getReceivedHeaders(Message message)
			throws MessagingException {
		String[] values = null;
		try {
			values = message.getHeader("Received");
		} catch (MessagingException e) {
			return new ReceivedHeader[0];
		}

		List<ReceivedHeader> receivedHeaders = new ArrayList<ReceivedHeader>();
		if (values != null) {
			for (String value : values) {
				receivedHeaders.add(new ReceivedHeaderImpl("Received", value));
			}
		}

		return receivedHeaders.toArray(new ReceivedHeader[receivedHeaders
				.size()]);
	}

	private MailAddress[] parseMailAddress(String addresses)
			throws IOException, MessagingException {
		List<MailAddress> result = new ArrayList<MailAddress>();
		for (InternetAddress address : InternetAddress.parse(addresses, true)) {
			String personal = address.getPersonal();
			String addr = address.getAddress();

			MailAddress mailAddress;
			if (personal == null) {
				mailAddress = new SimpleMailAddress(addr);
			} else {
				personal = new InternetAddress2(address).getPersonal();
				if (personal == null) {
					mailAddress = new SimpleMailAddress(addr);
				} else {
					mailAddress = new SimpleMailAddress(addr, DecodeUtils
							.decodeText(personal));
				}
			}
			result.add(mailAddress);
		}

		return result.toArray(new MailAddress[result.size()]);
	}

	private Mail toMail(Message message) throws MessagingException, IOException {
		final SimpleMail mail = new SimpleMail();

		// date
		mail.setSentDate(message.getSentDate());

		// addresses & headers & message-id & subject & importance
		Enumeration<?> headers = message.getAllHeaders();
		while (headers.hasMoreElements()) {
			javax.mail.Header header = (javax.mail.Header) headers
					.nextElement();

			String name = header.getName();
			String value = header.getValue();

			if (name.equals("To")) {
				for (MailAddress address : this.parseMailAddress(value)) {
					mail.addTo(address);
				}
			}
			if (name.equals("Bcc")) {
				for (MailAddress address : this.parseMailAddress(value)) {
					mail.addBcc(address);
				}
			}
			if (name.equals("Cc")) {
				for (MailAddress address : this.parseMailAddress(value)) {
					mail.addCc(address);
				}
			}
			if (name.equals("Reply-To")) {
				for (MailAddress address : this.parseMailAddress(value)) {
					mail.addReplyTo(address);
				}
			}
			if (name.equals("From")) {
				for (MailAddress address : this.parseMailAddress(value)) {
					mail.addFrom(address);
				}
			}
			if (name.equals("Return-Path")) {
				for (MailAddress address : this.parseMailAddress(value)) {
					mail.setReturnPath(address.getAddress());
				}
			}
			if (name.equals("Message-Id")) {
				mail.setMessageId(value);
			}
			if (name.equals("Subject")) {
				mail.setSubject(DecodeUtils.decodeText(value));
			}
			if (name.equals("X-Priotiry")) {
				switch (NumberUtils.toInt(value)) {
				case 1:
					mail.setImportance(Importance.HIGH);
					break;
				case 2:
					mail.setImportance(Importance.HIGH);
					break;
				case 3:
					mail.setImportance(Importance.NORMAL);
					break;
				case 4:
					mail.setImportance(Importance.LOW);
					break;
				case 5:
					mail.setImportance(Importance.LOW);
					break;
				default:
					break;
				}
			}
			if (name.equals("Priotiry")) {
				for (Importance importance : Importance.values()) {
					if (importance.getPriority().equalsIgnoreCase(value)) {
						mail.setImportance(importance);
						break;
					}
				}
			}
			if (name.equals("Importance")) {
				for (Importance importance : Importance.values()) {
					if (importance.getImportance().equalsIgnoreCase(value)) {
						mail.setImportance(importance);
						break;
					}
				}
			}
			mail
					.addHeader(new SimpleHeader(header.getName(), header
							.getValue()));
		}

		// attachments
		DecodeUtils.getAttachmentFiles(message,
				new DecodeUtils.AttachmentHandler() {
					@Override
					public void handleAttachment(String fileName, Part part)
							throws IOException, MessagingException {
						mail.addAttachmentFile(new PartAttachmentFile(fileName,
								part));
					}
				});

		// body
		String html = DecodeUtils.getHtmlContent(message);
		if (html != null) {
			MailBody body = new JapaneseMailBodyFactory()
					.createHTMLMailBody(html);
			mail.setBody(body);
		}

		if (mail.getBody() == null) {
			String text = DecodeUtils.getTextContent(message);
			if (text != null) {
				MailBody body = new JapaneseMailBodyFactory()
						.createPlainTextMailBody(text);
				mail.setBody(body);
			}
		}

		if (mail.getBody() == null) {
			mail.setBody(new EmptyMailBody());
		}

		return mail;
	}
}
