unit Logs;

interface

uses
  Contnrs, Controls, Classes, SysUtils, Dialogs, Forms, HttpThread, BottleDef,
  DateUtils, SsParser, XDOM_2_3_J3, Graphics, SppList;

type

  TLogType = (ltBottle, ltSystemLog);
  TLogState = (lsUnopened, lsPlaying, lsOpened);

  THasURL = (huUndefined, huYes, huNo);

  TLogItem = class(TPersistent)
  private
    FScript: String;
    FChannel: String;
    FMID: String;
    FLogTime: TDateTime;
    FLogType: TLogType;
    FGhost: String;
    FVotes: Integer;
    FAgreements: Integer;
    FState: TLogState;
    FHasURL: THasURL;
    procedure SetChannel(const Value: String);
    procedure SetLogType(const Value: TLogType);
    procedure SetMID(const Value: String);
    procedure SetScript(const Value: String);
    procedure SetLogTime(const Value: TDateTime);
    procedure SetGhost(const Value: String);
    procedure SetVotes(const Value: Integer);
    procedure SetAgreements(const Value: Integer);
    procedure SetState(const Value: TLogState);
    procedure SetHasURL(const Value: THasURL);
  public
    constructor Create(LogType: TLogType; const MID, Channel, Script,
      Ghost: String; LogTime: TDateTime); overload;
    constructor Create(Source: TLogItem); overload;
    property LogType: TLogType read FLogType write SetLogType;
    property MID: String read FMID write SetMID;
    property Channel: String read FChannel write SetChannel;
    property LogTime: TDateTime read FLogTime write SetLogTime;
    property Script: String read FScript write SetScript;
    property Ghost: String read FGhost write SetGhost;
    property Votes: Integer read FVotes write SetVotes;
    property Agrees: Integer read FAgreements write SetAgreements;
    property State: TLogState read FState write SetState;
    property HasURL: THasURL read FHasURL write SetHasURL;
    procedure Assign(Source: TPersistent); override;
  end;

  TBottleLogSortType = (stLogTime, stChannel, stGhost, stScript, stVote, stAgree);
  TBottleLogLoadFailureEvent = procedure(Sender: TObject; const Message: String) of object;
  TBottleLogHTMLOutputWork = procedure(Sender: TObject; const Count: integer;
    var Canceled: boolean) of object;

  EXMLFileOpenException = class(Exception);

  TBottleLogDownloadCondition = packed record
    IsRange: boolean;
    RecentCount: integer;
    DateLo: TDateTime;
    DateHi: TDateTime;
    MinVote: integer;
    MinAgree: integer;
    Channel: string;
  end;

  TLogXMLThread = class(TThread)
  private
    FList: TObjectList;
    FFileName: String;
    FOnLoadFailure: TBottleLogLoadFailureEvent;
    FOnLoaded: TNotifyEvent;
    FLoadFailureMessage: String;
    procedure SetOnLoaded(const Value: TNotifyEvent);
    procedure SetOnLoadFailure(const Value: TBottleLogLoadFailureEvent);
  protected
    procedure Execute; override;
    procedure DoLoaded;
    procedure DoLoadFailure;
  public
    property OnLoaded: TNotifyEvent read FOnLoaded write SetOnLoaded;
    property OnLoadFailure: TBottleLogLoadFailureEvent
      read FOnLoadFailure write SetOnLoadFailure;
    property List: TObjectList read FList;
    constructor Create(FileName: String);
    destructor Destroy; override;
  end;

  THTMLOutputUseColor = (ucUse, ucNoUse);
  THTMLOutputImageType = (itNone, itBMP, itPNG, itJPG);
  THTMLOutputRange = (orAll, orUpward, orSelected);

  THTMLOutputOptions = record
    ImageDir: String;
    UseColor: THTMLOutputUseColor;
    ImageType: THTMLOutputImageType;
  end;

  TBottleLogList = class(TObjectList)
  private
    FOnLoaded: TNotifyEvent;
    FHttpThread: THTTPDownloadThread;
    FXMLThread: TLogXMLThread;
    FLoadBytes: integer;
    FTitle: String;
    FOnLoadFailure: TBottleLogLoadFailureEvent;
    FOldSortColumn: TBottleLogSortType; // O
    FOldSortDesc: boolean;   // O
    FSelectedIndex: integer;
    FLastDownloadCondition: TBottleLogDownloadCondition;
    FOnLoadWork: TNotifyEvent; // O
    FImagePath: String; // HTMLo͎IMG^ÕpX(΂܂͐΃pX)
    FImageFiles: TStringList;
    FOnHTMLOutputWork: TBottleLogHTMLOutputWork; // d摜o͂Ȃ߂̃`FbJ

    function GetBottles(Index: integer): TLogItem;
    procedure SetOnLoaded(const Value: TNotifyEvent);
    procedure HttpSuccess(Sender: TObject);
    procedure HttpFailure(Sender: TObject);
    procedure HttpWork(Sender: TObject; LoadBytes: integer);
    procedure XMLLoaded(Sener: TObject);
    procedure XMLLoadFailure(Sender: TObject; const Message: String);
    procedure SetTitle(const Value: String);
    procedure SetOnLoadFailure(const Value: TBottleLogLoadFailureEvent);
    procedure SetSelectedIndex(const Value: integer);
    procedure SetOnLoadWork(const Value: TNotifyEvent);
    procedure DoLoadFailure(const ErrorMessage: String);
    function ColorToHex(const Col: TColor): String;
    procedure PrintHTMLBottle(Strs: TStringList; Bottle: TLogItem;
      SsParser: TSsParser; const Options: THTMLOutputOptions);
    function PrintSurfaceImage(Strs: TStringList; const Ghost: String; ID: integer;
      const Options: THTMLOutputOptions): boolean;
    procedure SetOnHTMLOutputWork(const Value: TBottleLogHTMLOutputWork);
  public
    constructor Create(const Title: String);
    destructor Destroy; override;
    procedure SortBottles(LogSortType: TBottleLogSortType);
    function Bottle(MID: String): TLogItem;
    property Bottles[Index: integer]: TLogItem read GetBottles;
    procedure LoadFromWeb(const Cond: TBottleLogDownloadCondition);
    property Title: String read FTitle write SetTitle;
    procedure LoadFromStream(Stream: TStream);
    property OnLoaded: TNotifyEvent read FOnLoaded write SetOnLoaded;
    property OnLoadWork: TNotifyEvent read FOnLoadWork write SetOnLoadWork;
    property OnLoadFailure: TBottleLogLoadFailureEvent read FOnLoadFailure write SetOnLoadFailure;
    property SelectedIndex: integer read FSelectedIndex write SetSelectedIndex;
    procedure AddScriptLog(const Script, Channel, MID, Ghost: String);
    procedure AddSystemLog(const MessageString: String);
    procedure SaveToText(const FileName: String);
    procedure SaveToSstpLog(const FileName: String;
      const WithChannel: boolean = false);
    procedure SaveToXMLFile(const FileName: String);
    procedure LoadFromXMLFile(const FileName: String);
    procedure SaveToHTML(const FileName: String;
      const Options: THTMLOutputOptions; SsParser: TSsParser);
    procedure ExtractUniqueChannels(Target: TStrings);
    procedure ExtractUniqueGhosts(Target: TStrings);
    property OnHTMLOutputWork: TBottleLogHTMLOutputWork read FOnHTMLOutputWork write SetOnHTMLOutputWork;
  end;

var
  ASortType: TBottleLogSortType; //LogCompare֐猩悤Ɉꎞޔp
  ASortDesc: boolean;

function LogCompare(Item1, Item2: Pointer): integer;


////////////////////////////////////////////////////////////////////////////////
implementation

uses StrUtils;

function LogCompare(Item1, Item2: Pointer): Integer;
var
  Log1, Log2: TLogItem;
begin
  Log1 := TLogItem(Item1);
  Log2 := TLogItem(Item2);
  Result := 0;
  case ASortType of
    stLogTime: begin
      Result := AnsiCompareStr(Log2.MID, Log1.MID);
    end;
    stChannel: begin
      Result := AnsiCompareStr(Log1.Channel, Log2.Channel);
    end;
    stGhost: begin
      Result := AnsiCompareStr(Log1.Ghost, Log2.Ghost);
    end;
    stScript: begin
      Result := AnsiCompareStr(Log1.Script, Log2.Script);
    end;
    stVote: begin
      Result := Log2.Votes - Log1.Votes;
    end;
    stAgree: begin
      Result := Log2.Agrees - Log1.Agrees;
    end;
  end;
  if ASortDesc then Result := -Result; //
end;

{ TLogItem }

constructor TLogItem.Create(LogType: TLogType; const MID, Channel, Script,
  Ghost: String; LogTime: TDateTime);
begin
  Self.LogType := LogType;
  Self.MID := MID;
  Self.Script := Script;
  Self.Channel := Channel;
  Self.Ghost := Ghost;
  Self.LogTime := LogTime;
  Self.State := lsUnopened;
end;

procedure TLogItem.SetChannel(const Value: String);
begin
  FChannel := Value;
end;

procedure TLogItem.SetLogType(const Value: TLogType);
begin
  FLogType := Value;
end;

procedure TLogItem.SetMID(const Value: String);
begin
  FMID := Value;
end;

procedure TLogItem.SetScript(const Value: String);
begin
  FScript := Value;
end;

procedure TLogItem.SetLogTime(const Value: TDateTime);
begin
  FLogTime := Value;
end;

procedure TLogItem.SetAgreements(const Value: Integer);
begin
  FAgreements := Value;
end;

procedure TLogItem.SetVotes(const Value: Integer);
begin
  FVotes := Value;
end;

procedure TLogItem.SetGhost(const Value: String);
begin
  FGhost := Value;
end;


procedure TLogItem.SetState(const Value: TLogState);
begin
  FState := Value;
end;

procedure TLogItem.Assign(Source: TPersistent);
var Src: TLogItem;
begin
  if not (Source is TLogItem) then
    inherited
  else begin
    Src := Source as TLogItem;
    self.FScript := Src.FScript;
    self.FChannel := Src.FChannel;
    self.FMID := Src.FMID;
    self.FLogTime := Src.FLogTime;
    self.FLogType := Src.FLogType;
    self.FGhost := Src.FGhost;
    self.FVotes := Src.FVotes;
    self.FAgreements := Src.FAgreements;
    self.FState := Src.FState;
    self.FHasURL := Src.FHasURL;
  end;
end;

constructor TLogItem.Create(Source: TLogItem);
begin
  self.Assign(Source);
end;

procedure TLogItem.SetHasURL(const Value: THasURL);
begin
  FHasURL := Value;
end;

{ TBottleLogList }

procedure TBottleLogList.AddScriptLog(const Script, Channel, MID,
  Ghost: String);
var LogItem: TLogItem;
begin
  LogItem := TLogItem.Create(ltBottle, MID, Channel, Script, Ghost, Now());
  try
    Insert(0, LogItem);
  except
    LogItem.Free;
    raise;
  end;
end;

procedure TBottleLogList.AddSystemLog(const MessageString: String);
var LogItem: TLogItem;
begin
  LogItem := TLogItem.Create(ltSystemLog, '', '', MessageString, '', Now());
  try
    Insert(0, LogItem);
  except
    LogItem.Free;
    raise;
  end;
end;

function TBottleLogList.Bottle(MID: String): TLogItem;
var i: integer;
begin
  Result := nil;
  for i := 0 to Count-1 do
    if (Items[i] as TLogItem).MID = MID then begin
      Result := Items[i] as TLogItem;
      exit;
    end;
end;

function TBottleLogList.ColorToHex(const Col: TColor): String;
var
  RGB: integer;
  R, G, B: byte;
begin
  RGB := ColorToRGB(Col);
  R := RGB and $FF;
  G := (RGB and $FF00) shr 8;
  B := (RGB and $FF0000) shr 16;
  RGB := (R shl 16) or (G shl 8) or B;
  Result := '#' + IntToHex(RGB, 6);
end;

constructor TBottleLogList.Create(const Title: String);
begin
  inherited Create;
  FTitle := Title;
  OwnsObjects := true;
  FSelectedIndex := -1; // IĂȂ
end;

destructor TBottleLogList.Destroy;
begin
  inherited;
  // FHttpThread͎FreeuOvSynchronizeŎQƂ폜̂ŁA
  // FHttpThread <> nil Ȃ݂̂邱Ƃ͊młB
  if FHttpThread <> nil then begin
    FHttpThread.OnSuccess := nil;
    FHttpThread.OnConnectionFailed := nil;
  end;
end;

procedure TBottleLogList.DoLoadFailure(const ErrorMessage: String);
begin
  self.Clear;
  self.AddSystemLog(ErrorMessage);
  if Assigned(FOnLoadFailure) then FOnLoadFailure(self, ErrorMessage);
end;

procedure TBottleLogList.ExtractUniqueChannels(Target: TStrings);
var i: integer;
begin
  // ̃Õ`lo
  // TStrings̎THashedStringList
  for i := 0 to Count-1 do
    if Target.IndexOf(Bottles[i].Channel) < 0 then
      if Bottles[i].Channel <> '' then
        Target.Add(Bottles[i].Channel);
end;

procedure TBottleLogList.ExtractUniqueGhosts(Target: TStrings);
var i: integer;
begin
  // ̃ÕS[Xgo
  // TStrings̎THashedStringList
  for i := 0 to Count-1 do
    if Target.IndexOf(Bottles[i].Ghost) < 0 then
      if Bottles[i].Ghost <> '' then
        Target.Add(Bottles[i].Ghost);
end;

function TBottleLogList.GetBottles(Index: integer): TLogItem;
begin
  Result := Items[Index] as TLogItem;
end;

procedure TBottleLogList.HttpFailure(Sender: TObject);
begin
  if Assigned(FOnLoadFailure) then
    DoLoadFailure('T[o̐ڑɎs܂');
end;

procedure TBottleLogList.HttpSuccess(Sender: TObject);
var Stream: TStringStream;
    StrList: TStringList;
begin
  Stream := TStringStream.Create(FHttpThread.RecvString);
  try
    StrList := TStringList.Create;
    try
      StrList.Text := Stream.DataString;
      if StrList[0] <> 'Result: OK' then begin
        if Assigned(FOnLoadFailure) then
          DoLoadFailure('T[oG[Ԃ܂'#13#10 + Stream.DataString);
      end else begin
        while StrList.Count > 0 do
          if StrList[0] <> '' then StrList.Delete(0) else Break; //wb_[폜
        if StrList.Count > 0 then StrList.Delete(0); //wb_[̋s폜
        Stream.Seek(0, soFromBeginning);
        Stream.Size := Length(StrList.Text);
        Stream.WriteString(StrList.Text);
        Stream.Seek(0, soFromBeginning);
        LoadFromStream(Stream);
      end;
    finally
      StrList.Free;
    end;
  finally
    Stream.Free;
    // nilłȂꍇɂ͎݂̂邱Ƃmɂ
    //  nil ƂĎ݂̂ȂƂ͌Ȃ(FreeOnTerminatê)
    FHttpThread := nil;
  end;
end;

procedure TBottleLogList.HttpWork(Sender: TObject; LoadBytes: integer);
begin
  self.Clear;
  AddSystemLog(Format('_E[hł - %4d KB', [LoadBytes div 1024]));
  FLoadBytes := LoadBytes;
  if Assigned(FOnLoadWork) then FOnLoadWork(self);
end;

procedure TBottleLogList.LoadFromStream(Stream: TStream);
var i: integer;
    agree: integer;
    vote: integer;
    SourceStream: TStringStream;
    Source: TStringList;
    LogItem: TLogItem;
  function S2D (const S: String): TDateTime;
  begin
    Result := EncodeDateTime(
      StrToInt(Copy(S, 1, 4)), StrToInt(Copy(S, 5, 2)), StrToInt(Copy(S, 7, 2)),
      StrToInt(Copy(S, 9, 2)), StrToInt(Copy(S, 11, 2)), StrToInt(Copy(S, 13, 2)), 0);
  end;
begin
  Source := nil;
  SourceStream := nil;
  try
    try
      SourceStream := TStringStream.Create('');
      SourceStream.Seek(0, soFromBeginning);
      SourceStream.CopyFrom(Stream, Stream.Size);
      Source := TStringList.Create;
      Source.Text := SourceStream.DataString;
      i := 0;
      if Source.Count = 0 then begin
        DoLoadFailure('w̃O݂͑܂');
        Exit;
      end;
      if Source[0] <> 'OK' then begin
      end;
      Self.Clear;
      while (i + 7) < Source.Count do begin
        vote := StrToInt(Source[i+5]);
        agree := StrToInt(Source[i+6]);

        //[Eӏw͂Ŏ
        //FIXME FIXME - ͋Z܂AAA
        if vote >= FLastDownloadCondition.MinVote then begin
          if agree >= FLastDownloadCondition.MinAgree then begin
            LogItem := TLogItem.Create(ltBottle, Source[i+1], Source[i+2],
              Source[i+7], Source[i+3], S2D(Source[i]));

            LogItem.Votes  := vote;
            LogItem.Agrees := agree;
            LogItem.State  := lsOpened;
            Self.Add(LogItem);
          end;
        end;

        i := i + 8;
      end;
    finally
      SourceStream.Free;
      Source.Free;
    end;
  except
    On EConvertError do begin
      DoLoadFailure('T[ǒ`sł');
      Self.Clear;
    end;
  end;
  FOldSortColumn := stLogTime;
  FOldSortDesc := false;
  if Assigned(FOnLoaded) then FOnLoaded(Self);
end;

procedure TBottleLogList.LoadFromWeb(const Cond: TBottleLogDownloadCondition);
var Param: String;
begin
  Self.Clear;
  AddSystemLog('_E[hł - T[o牞҂');

  FLastDownloadCondition := Cond;
  with Cond do begin
    if Cond.IsRange then begin
      Param := Format('year_lo=%d&month_lo=%d&day_lo=%d&year_hi=%d&month_hi=%d&day_hi=%d&',
        [YearOf(DateLo), MonthOf(DateLo), DayOf(DateLo),
         YearOf(DateHi), MonthOf(DateHi), DayOf(DateHi)]);
    end else begin
      Param := Format('recent=%d&', [RecentCount]);
    end;
    Param := Param + Format('channel=%s&sort=timedesc&delimiter=%%0D%%0A',
      [ParamsEncode(Channel)]);
  end;
  FHttpThread := THTTPDownloadThread.Create(Pref.BottleServer, Pref.CgiFetchLog, Param);
  if Pref.UseHttpProxy then begin
    FHttpThread.ProxyServer := Pref.ProxyAddress;
    FHttpThread.ProxyPort   := Pref.ProxyPort;
    if Pref.ProxyNeedAuthentication then begin
      FHttpThread.ProxyUser := Pref.ProxyUser;
      FHttpThread.ProxyPass := Pref.ProxyPass;
    end;
  end;
  FHttpThread.FreeOnTerminate := true;
  FHttpThread.OnSuccess := HttpSuccess;
  FHttpThread.OnConnectionFailed := HttpFailure;
  FHttpThread.OnHttpWork := HttpWork;

  FLoadBytes := 0;
  FHttpThread.Resume;
end;

procedure TBottleLogList.LoadFromXMLFile(const FileName: String);
begin
  Self.Clear;
  AddSystemLog('[h...');
  FXMLThread := TLogXMLThread.Create(FileName);
  FXMLThread.FreeOnTerminate := true; // ɏĂ炤
  with FXMLThread do
  begin
    OnLoaded := XMLLoaded;
    OnLoadFailure := XMLLoadFailure;
    Resume;
  end;
end;

procedure TBottleLogList.PrintHTMLBottle(Strs: TStringList;
  Bottle: TLogItem; SsParser: TSsParser; const Options: THTMLOutputOptions);
var
  i: integer;
  InScope1, InSync: boolean;
  Talk, Tag: String;
  sur0, sur1, sur: integer;
  procedure TalkEnd;
  var Cl: String;
  begin
    if Talk = '' then
      Exit;
    if InSync then Cl := 'synchronized'
    else if InScope1 then Cl := 'scope1'
    else Cl := 'scope0';
    if Options.ImageType = itNone then
    begin
      with Strs do
      begin
        Add('  <tr class="' + Cl + '">');
        Add('    <td class="talk">' + XMLEntity(Talk) + '</td>');
        Add('  </tr>');
      end;
    end else
    begin
      with Strs do
      begin
        Add('  <tr class="' + Cl + '">');
        Add('    <td class="surface">');
        if InSync then
        begin
          PrintSurfaceImage(Strs, Bottle.Ghost, Sur0, Options);
          PrintSurfaceImage(Strs, Bottle.Ghost, Sur1, Options);
        end else if InScope1 then
        begin
          PrintSurfaceImage(Strs, Bottle.Ghost, Sur1, Options);
        end else
        begin
          PrintSurfaceImage(Strs, Bottle.Ghost, Sur0, Options);
        end;
        Add('    </td>');
        Add('    <td class="talk">' + Talk + '</td>');
        Add('  </tr>');
      end;
    end;
    Talk := '';
  end; // of sub-procedure TalkEnd
begin
  SsParser.EscapeInvalidMeta := false;
  SsParser.LeaveEscape := false;
  SsParser.InputString := Bottle.Script;
  InScope1 := false;
  InSync := false;
  sur0 := 0;
  sur1 := 10;
  for i := 0 to SsParser.Count-1 do
    case SsParser.MarkUpType[i] of
      mtStr, mtMeta:
        Talk := Talk + XMLEntity(SsParser.Str[i]);
      mtTag:
        begin
          Tag := SsParser.Str[i];
          if (Tag = '\h') then
          begin
            if InScope1 and not InSync then
              TalkEnd;
            InScope1 := false;
          end else if (Tag = '\u') then
          begin
            if not InScope1 and not InSync then
              TalkEnd;
            InScope1 := true;
          end else if Tag = '\_s' then
          begin
            TalkEnd;
            InSync := not InSync;
          end else if SsParser.Match(Tag, '\s%d') = 3 then
          begin
            TalkEnd;
            sur := Ord(Tag[3]) - Ord('0');
            if InSync then
            begin
              sur0 := sur;
              sur1 := sur;
            end else if InScope1 then
              sur1 := sur
            else
              sur0 := sur;
          end else if (SsParser.Match(Tag, '\s[%D]') > 0) or (Tag = '\s[-1]') then
          begin
            TalkEnd;
            sur := StrToInt(SsParser.GetParam(Tag, 1));
            if InSync then
            begin
              sur0 := sur;
              sur1 := sur;
            end else if InScope1 then
              sur1 := sur
            else
              sur0 := sur;
          end else if SsParser.Match(Tag, '\n') >= 2 then
          begin
            if Talk <> '' then
              Talk := Talk + '<br>';
          end else if Tag = '\c' then
          begin
            TalkEnd;
          end;
        end;
    end;
  TalkEnd;
end;

function TBottleLogList.PrintSurfaceImage(Strs: TStringList; const Ghost: String;
  ID: integer; const Options: THTMLOutputOptions): boolean;
var
  Bmp: TBitmap;
  Path, Ext, Name: String;
begin
  Result := false;
  if (ID < 0) or (Ghost = '') then
    Exit;
  Ext := 'bmp';
  Name := Format('%s,%d.%s', [SafeFileName(Ghost), ID, Ext]);
  if FImageFiles.IndexOf(Name) >= 0 then
  begin
    Strs.Add(Format('      <img src="%s\%s" alt="%s,%d">',
      [FImagePath, XMLEntity(Name), XMLEntity(Ghost), ID]));
    Result := true;
  end else
  begin
    Bmp := TBitmap.Create;
    try
      if Spps.TryGetImage(Ghost, ID, Bmp) then
      begin
        Path := Options.ImageDir + PathDelim + Name;
        Bmp.SaveToFile(Path);
        Strs.Add(Format('      <img src="%s\%s" alt="%s,%d">',
          [FImagePath, XMLEntity(Name), XMLEntity(Ghost), ID]));
        Result := true;
        FImageFiles.Add(Name); // 񂩂ĂяoȂĂ悤ɂ
      end else
        Strs.Add(Format('      [%d]', [ID]));
    finally
      Bmp.Free;
    end;
  end;
end;

procedure TBottleLogList.SaveToHTML(const FileName: String;
  const Options: THTMLOutputOptions; SsParser: TSsParser);
var
  i: integer;
  Strs: TStringList;
  ChannelAndGhost: String;
  Cancel: boolean;
begin
  if ExtractFilePath(FileName) = ExtractFilePath(Options.ImageDir) then
    FImagePath := ExtractFileName(Options.ImageDir)
  else
    FImagePath := Options.ImageDir;

  Screen.Cursor := crHourGlass;
  FImageFiles := TStringList.Create;
  try
    Strs := TStringList.Create;
    try
      with Strs do
      begin
        Add('<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN">');
        Add('<html>');
        Add('<style type="text/css"><!--');
        Add('table.bottle td{font-family: monospace}');
        if Options.UseColor = ucUse then
        begin
          Add(Format('p.bottleattr {color: %s}', [ColorToHex(Pref.TalkColorH)]));
          Add(Format('body {background-color: %s}', [ColorToHex(Pref.BgColor)]));
          Add(Format('table.bottle tr.scope0 td.talk{color: %s}', [ColorToHex(Pref.TalkColorH)]));
          Add(Format('table.bottle tr.scope1 td.talk{color: %s}', [ColorToHex(Pref.TalkColorU)]));
          Add(Format('table.bottle tr.synchronized td.talk{color: %s}', [ColorToHex(Pref.TalkColorS)]));
          Add('table.bottle td.surface {text-align: center}');
        end;
        Add('--></style>');
        for i := 0 to Self.Count-1 do
        begin
          if Assigned(FOnHTMLOutputWork) then
          begin
            Cancel := false;
            FOnHTMLOutputWork(Self, i, Cancel);
            if Cancel then
              Exit;
          end;
          if Bottles[i].Ghost <> '' then
            ChannelAndGhost := XMLEntity(Bottles[i].Channel) + '/' + XMLEntity(Bottles[i].Ghost)
          else
            ChannelAndGhost := XMLEntity(Bottles[i].Channel);
          Add(Format('<p class="bottleattr">%s %s [%d %d</p>', [
            FormatDateTime('yyyy/mm/dd hh:nn:ss', Bottles[i].LogTime),
            ChannelAndGhost,
            Bottles[i].Votes,
            Bottles[i].Agrees
          ]));
          Add('<table class="bottle">');
          PrintHTMLBottle(Strs, Bottles[i], SsParser, Options);
          Add('</table>');
          Add('<hr>');
        end;
        Add('</html>');
        SaveToFile(FileName);
      end;
    finally
      Strs.Free;
    end;
  finally
    FImageFiles.Free;
    Screen.Cursor := crArrow;
  end;
end;

procedure TBottleLogList.SaveToSstpLog(const FileName: String;
  const WithChannel: boolean = false);
var i: integer;
    Str: TStringList;
    Item: TLogItem;
    Date: String;
const
  DayStr: array[1..7] of String = ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat');
begin
  Str := nil;
  try
    Str := TStringList.Create;
    for i := 0 to Self.Count - 1 do begin
      Item := Self.Items[i] as TLogItem;
      if Item.LogType = ltBottle then begin
        Date := FormatDateTime('yyyy/mm/dd hh:nn:ss ', Item.LogTime);
        Date := Date + '(' + DayStr[DayOfWeek(Item.LogTime)] + ')';
        if WithChannel then
          Date := Date + ',' + Item.Channel +',SEND,' + Item.Script
        else
          Date := Date + ',0.0.0.0,SEND,' + Item.Script;
        Str.Add(Date);
      end;
    end;
    Str.SaveToFile(FileName);
  finally
    Str.Free;
  end;
end;

procedure TBottleLogList.SaveToText(const FileName: String);
var i: integer;
    Str: TStringList;
begin
  Str := nil;
  try
    Str := TStringList.Create;
    for i := 0 to Self.Count - 1 do
      if (Self.Items[i] as TLogItem).LogType = ltBottle then
        Str.Add((Self.Items[i] as TLogItem).Script);
    Str.SaveToFile(FileName);
  finally
    Str.Free;
  end;
end;

procedure TBottleLogList.SaveToXMLFile(const FileName: String);
var i: integer;
    MessageNode, Child: TdomElement;
    Item: TLogItem;
    DOM: TdomDocument;
    Parser: TXmlToDomParser;
    Impl: TDomImplementation;
    FS: TFileStream;
begin
  Impl := TDomImplementation.create(nil);
  try
    Parser := TXmlToDomParser.create(nil);
    Parser.DOMImpl := Impl;
    try
      try
        DOM := Parser.fileToDom(ExtractFilePath(Application.ExeName)+'xbtl.dat');
        // ͖IFreeȂĂ悢
        with DOM do
        begin
          documentElement.setAttribute('saved',
            FormatDateTime('yy/mm/dd hh:nn:ss', Now));
          documentElement.setAttribute('generator', VersionString);
          documentElement.setAttribute('version', '1.0');
          for i := 0 to Self.Count-1 do begin
            Item := Self.GetBottles(i);
            MessageNode := createElement('message');
            MessageNode.setAttribute('mid', Item.MID);
            documentElement.appendChild(MessageNode);

            // t
            Child := createElement('date');
            Child.appendChild(createTextNode(FormatDateTime('yy/mm/dd hh:nn:ss', Item.LogTime)));
            MessageNode.appendChild(Child);
            // `l
            Child := createElement('channel');
            Child.appendChild(createTextNode(Item.Channel));
            MessageNode.appendChild(Child);
            // XNvg
            Child := createElement('script');
            Child.appendChild(createTextNode(Item.Script));
            MessageNode.appendChild(Child);
            // [
            Child := createElement('votes');
            Child.appendChild(createTextNode(IntToStr(Item.Votes)));
            MessageNode.appendChild(Child);
            // 
            Child := createElement('agrees');
            Child.appendChild(createTextNode(IntToStr(Item.Agrees)));
            MessageNode.appendChild(Child);
            // S[Xg
            Child := createElement('ghost');
            Child.appendChild(createTextNode(Item.Ghost));
            MessageNode.appendChild(Child);

          end;
        end;
        FS := TFileStream.Create(FileName, fmCreate or fmShareExclusive);
        try
          DOM.writeCodeAsShiftJIS(FS);
        finally
          FS.Free;
        end;
      except
        ShowMessage('xbtl.datăCXg[ĂB');
      end;
    finally
      Parser.DOMImpl.freeDocument(DOM);
      Parser.Free;
    end;
  finally
    Impl.Free;
  end;
end;

procedure TBottleLogList.SetOnHTMLOutputWork(
  const Value: TBottleLogHTMLOutputWork);
begin
  FOnHTMLOutputWork := Value;
end;

procedure TBottleLogList.SetOnLoaded(const Value: TNotifyEvent);
begin
  FOnLoaded := Value;
end;

procedure TBottleLogList.SetOnLoadFailure(
  const Value: TBottleLogLoadFailureEvent);
begin
  FOnLoadFailure := Value;
end;

procedure TBottleLogList.SetOnLoadWork(const Value: TNotifyEvent);
begin
  FOnLoadWork := Value;
end;

procedure TBottleLogList.SetSelectedIndex(const Value: integer);
begin
  FSelectedIndex := Value;
end;

procedure TBottleLogList.SetTitle(const Value: String);
begin
  FTitle := Value;
end;

procedure TBottleLogList.SortBottles(LogSortType: TBottleLogSortType);
begin
  if FOldSortColumn = LogSortType then
    ASortDesc := not FOldSortDesc
  else begin
    ASortDesc := false;
  end;
  ASortType := LogSortType;
  Self.Sort(LogCompare);
  FOldSortColumn := ASortType;
  FOldSortDesc := ASortDesc;
end;

procedure TBottleLogList.XMLLoaded(Sener: TObject);
begin
  Self.Assign(FXMLThread.List);
  if Assigned(FOnLoaded) then FOnLoaded(Self);
  FXMLThread := nil;
end;

procedure TBottleLogList.XMLLoadFailure(Sender: TObject;
  const Message: String);
begin
  if Assigned(FOnLoadFailure) then
    DoLoadFailure(Message);
end;

{ TLogXMLThread }

constructor TLogXMLThread.Create(FileName: String);
begin
  inherited Create(true);
  FList := TObjectList.Create(false); // OwnsObject = false (!!)
  FFileName := FileName;
end;

destructor TLogXMLThread.Destroy;
begin
  FList.Free;
  inherited;
end;

procedure TLogXMLThread.DoLoaded;
begin
  if Assigned(FOnLoaded) then
    FOnLoaded(self);
end;

procedure TLogXMLThread.DoLoadFailure;
begin
  if Assigned(FOnLoadFailure) then
    FOnLoadFailure(self, FLoadFailureMessage);
end;

procedure TLogXMLThread.Execute;
var i, j, votes, agrees: integer;
    Time: TDateTime;
    ANode, Child: TdomElement;
    Item: TLogItem;
    DOM: TdomDocument;
    Parser: TXmlToDomParser;
    Impl: TDomImplementation;
    Str, mid, channel, script, ghost: String;
begin
  FList.Clear;
  Impl := TDomImplementation.create(nil);
  try
    Parser := TXmlToDomParser.create(nil);
    Parser.DOMImpl := Impl;
    try
      try
        DOM := Parser.fileToDom(FFileName); //͖IFreeȂĂ悢
        DOM.normalize;
        if not DOM.validate(nil, erReplace) then
          raise EXMLFileOpenException.Create('Lȃ{gO`ł͂܂B');
        with DOM do
        begin
          if DocumentElement = nil then
          begin
            FLoadFailureMessage := 'LȌ`ł͂܂B' +
             '[g^O܂';
            Synchronize(DoLoadFailure);
            Exit;
          end;
          if DocumentElement.nodeName <> 'bottlelog' then
          begin
            FLoadFailureMessage := 'LȌ`ł͂܂B' +
              'bottlelog܂';
            Synchronize(DoLoadFailure);
            Exit;
          end;
          Str :=  DocumentElement.getAttribute('version');
          if Str <> '1.0' then
          begin
            FLoadFailureMessage := Format('LȌ`ł͂܂B' +
              '̃Ot@C̃o[W(%s)͓ǂݍ߂܂', [Str]);
            Synchronize(DoLoadFailure);
            Exit;
          end;
          for i := 0 to DocumentElement.childNodes.length-1 do
          begin
            if documentElement.childNodes.item(i).nodeType <> ntElement_Node then
              Continue;
            ANode := documentElement.childNodes.item(i) as TdomElement;
            if ANode.nodeName <> 'message' then
              Continue;
            mid := ANode.getAttribute('mid');
            channel := '';
            script := '';
            ghost := '';
            votes := 0;
            agrees := 0;
            Time := Now;
            for j := 0 to ANode.childNodes.length-1 do
            begin
              if ANode.childNodes.item(j).nodeType <> ntElement_Node then
                Continue;
              Child := ANode.childNodes.item(j) as TdomElement;
              if Child.nodeName = 'channel' then
                channel := Trim(Child.textContent)
              else if Child.nodeName = 'script' then
                script := Trim(Child.textContent)
              else if Child.nodeName = 'ghost' then
                ghost := Trim(Child.textContent)
              else if Child.nodeName = 'votes' then
                votes := StrToIntDef(Child.textContent, 0)
              else if Child.nodeName = 'agrees' then
                agrees := StrToIntDef(Child.textContent, 0)
              else if Child.nodeName = 'date' then
                TryStrToDateTime(Trim(Child.textContent), Time);
            end;
            Item := TLogItem.Create(ltBottle, mid, channel, script, ghost, time);
            Item.Votes  := votes;
            Item.Agrees := agrees;
            Item.State := lsOpened;
            try
              FList.Add(Item);
            except
              Item.Free;
            end;
          end;
          Synchronize(DoLoaded);
        end;
      except
        on E: EParserException do
        begin
          FLoadFailureMessage := 'XML̓G[B' + E.Message;
          Synchronize(DoLoadFailure);
          Exit;
        end;
        on E: Exception do
        begin
          FLoadFailureMessage := 'XMLI[vɃG[܂B' +
            E.Message;
          Synchronize(DoLoadFailure);
          Exit;
        end;
      end;
    finally
      Parser.DOMImpl.freeDocument(DOM);
      Parser.Free;
    end;
  finally
    Impl.Free;
  end;
end;

procedure TLogXMLThread.SetOnLoaded(const Value: TNotifyEvent);
begin
  FOnLoaded := Value;
end;

procedure TLogXMLThread.SetOnLoadFailure(const Value: TBottleLogLoadFailureEvent);
begin
  FOnLoadFailure := Value;
end;

end.
